/***************************************************************************
    file	         : kb_eventlog.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>

#include	<qwidget.h>
#include	<qdatetime.h>


#include	"kb_classes.h"
#include	"kb_location.h"
#include	"kb_gui.h"
#include	"kb_node.h"
#include	"kb_notifier.h"
#include	"kb_viewer.h"
#include	"kb_options.h"
#include	"kbase.h"



#ifndef 	_WIN32
#include	"kb_eventlog.moc"
#else
#include	"kb_eventlog.h"
#endif



static	QString	timeNow ()
{
	QTime	now	= QTime::currentTime() ;
	return	QString().sprintf("%02d:%02d:%02d", now.hour(), now.minute(), now.second()) ;
}


static	GUIElement	eventLogGUI[] =
{
{	GTAction,	KB::GRNone,	true,	"Clear log",		"editclear",	ACCEL(NoAccel),	SLOT(clearLog ()),"KB_clear",	0,	"Clear log"		},
{	GTNone,		KB::GRNone,	false,	0,			0,		ACCEL(NoAccel),	0,		0,		0,	0			}
}	;


/*  ------------------------------------------------------------------  */

/*  KBEventLogEntry							*/
/*  KBEventLogEntry							*/
/*		: Constructor for event log entry			*/
/*  parent	: QListView *	  : Parent list view			*/
/*  after	: QListViewItem * : Insert after			*/
/*  count	: uint		  : Event count				*/
/*  type	: const QString & : Type (event/slot)			*/
/*  objName	: const QString & : Object Name				*/
/*  objType	: const QString & : Object type				*/
/*  eventName	: const QString & : Event name				*/
/*  argc	: uint		  : Number of arguments			*/
/*  argv	: const KBValue * : Vector of arguments			*/
/*  (returns)	: KBEventLogEntry :					*/

KBEventLogEntry::KBEventLogEntry
	(	QListView	*parent,
		QListViewItem	*after,
		uint		count,
		const QString	&type,
		const QString	&objName,
		const QString	&objType,
		const QString	&eventName,
		uint		argc,
		const KBValue	*argv
	)
	:
	QListViewItem
	(	parent,
		after,
		QString("%1").arg(count),
		timeNow(),
		type,
		objName,
		objType,
		eventName
	),
	m_extra	(false)
{
	if ((argc > 0) && (argv != 0))
	{
		/* Limit the number of arguments saved to the limit	*/
		/* but set the extra flag to note when there were more.	*/
		if (argc > KBOptions::getLogMaxArgs())
		{	argc	= KBOptions::getLogMaxArgs() ;
			m_extra	= true	  ;
		}

		/* For each argument, store the raw text (truncated if	*/
		/* they are too long) and the associated type.		*/
		for (uint idx = 0 ; idx < argc ; idx += 1)
		{
			QString	s  = argv[idx].getRawText() ;
			if (s.length() > KBOptions::getLogMaxArgLen())
			{	s.truncate(KBOptions::getLogMaxArgLen()) ;
				s.append  (" ...."  ) ;
			}
			m_args .append (s) ;
			m_types.append (argv[idx].getType()->getDescrip()) ;
		}
	}
}

/*  KBEventLogEntry							*/
/*  showArgs	: Show arguments					*/
/*  lv		: QListView *	: List view into which to display	*/
/*  (returns)	: void		:					*/

void	KBEventLogEntry::showArgs
	(	QListView	*lv
	)
{
	lv->clear	()   ;
	lv->setSorting	(-1) ;

	QListViewItem *last  = 0 ;

	for (uint arg = 0 ; arg < m_args.count() ; arg += 1)
		last = new QListViewItem
				(	lv,
					last,
					QString("%1").arg(arg),
					m_types[arg],
					m_args [arg]
				)	;

	if (m_extra)
		last = new QListViewItem (lv, last, "", "....", "....") ;
}



/*  ------------------------------------------------------------------  */

/*  KBEventLog								*/
/*  KBEventLog	: Constructor for text logging windows			*/
/*  parent	: QWidget *	  : Parent widget			*/
/*  caption	: const QString & : Caption				*/
/*  action	: KToggleAction*  : Show/hide action for this window	*/
/*  (returns)	: KBEventLog	  :					*/

KBEventLog::KBEventLog
	(	QWidget		*parent,
		const QString	&caption,
		TKToggleAction	*action
	)
	:
	KBasePart (0, parent, WDestructiveClose|WStyle_NormalBorder),
	m_split	  (new QSplitter (Qt::Vertical, m_partWidget)),
	m_args	  (new QListView (m_split)),
	m_log	  (new QListView (m_split)),
	m_action  (action)
{
	m_topWidget = m_split ;

	m_args ->addColumn	("",	      60) ;
	m_args ->addColumn      ("Type",      60) ;
	m_args ->addColumn	("Argument", 400) ; 
	m_args ->setSorting	(-1) ;

	m_log  ->addColumn      ("Count",     60) ;
	m_log  ->addColumn      ("Time",      60) ;
	m_log  ->addColumn      ("Type",      60) ;
	m_log  ->addColumn      ("ObjType",  100) ;
	m_log  ->addColumn      ("ObjName",  100) ;
	m_log  ->addColumn      ("Event",    100) ;
	m_log  ->addColumn      ("OK",        40) ;
	m_log  ->addColumn      ("Result",   250) ;
	
	m_log  ->setSorting	(-1) ;

	m_split->show		() ;

	TKConfig *config = TKConfig::getConfig() ;
	config->setGroup ("Event Log") ;

	QSize		size	= config->readSizeEntry   ("geometry" ) ;
	QValueList<int> depths	= config->readIntListEntry("depths"   ) ;
	QValueList<int>	aw	= config->readIntListEntry("argWidths") ;
	QValueList<int>	lw	= config->readIntListEntry("logWidths") ;

	if (size.isEmpty()) size = QSize(500, 500) ;
	while (depths.count() < 2) depths.append(0) ;
	if (depths[0] == 0) depths[0] = 200 ;
	if (depths[1] == 0) depths[1] = 200 ;

	setLocalGUISpec (&eventLogGUI[0])   ;
	setGUI (m_gui = new KBaseGUI (this, this, "rekallui.text.log")) ;

#if	__KB_EMBEDDED
	m_partWidget->showMaximized () ;
#else
	m_partWidget->resize	(size.width(), size.height(), true, false) ;
#endif
	m_partWidget->show   	()	   ;
	m_partWidget->setCaption(caption ) ;

	if (aw.count() >= 3)
	{	m_args->setColumnWidth (0, aw[0]) ;
		m_args->setColumnWidth (1, aw[1]) ;
		m_args->setColumnWidth (2, aw[2]) ;
	}
	if (lw.count() >= 6)
	{	m_log ->setColumnWidth (0, lw[0]) ;
		m_log ->setColumnWidth (1, lw[1]) ;
		m_log ->setColumnWidth (2, lw[2]) ;
		m_log ->setColumnWidth (3, lw[3]) ;
		m_log ->setColumnWidth (4, lw[4]) ;
		m_log ->setColumnWidth (5, lw[5]) ;
	}

	extern	KBaseApp *kbaseApp	 ;
	m_split	  ->setSizes	(depths) ;
	kbaseApp  ->addViewer	(this  ) ;

	connect
	(	m_log,
		SIGNAL(doubleClicked(QListViewItem *)),
		SLOT  (clickEvent   (QListViewItem *))
	)	;

	m_count	= 0  ;
	m_last	= 0  ;
}

/*  KBEventLog								*/
/*  ~KBEventLog	: Destructor for text logging window			*/
/*  (returns)	:		:					*/

KBEventLog::~KBEventLog ()
{
}

/*  KBEventLog								*/
/*  logEvent	: Add query to log					*/
/*  type	: const QString & : Type (event/slot)			*/
/*  objName	: const QString & : Object Name				*/
/*  objType	: const QString & : Object type				*/
/*  eventName	: const QString & : Event name				*/
/*  argc	: uint		  : Number of arguments			*/
/*  argv	: const KBValue * : Vector of arguments			*/
/*  (returns)	: void *	  : Pointer at entry			*/

void	*KBEventLog::logEvent
	(	const QString	&type,
		const QString	&objName,
		const QString	&objType,
		const QString	&eventName,
		uint		argc,
		const KBValue	*argv
	)
{
	while (m_log->childCount() >= (int)KBOptions::getLogMaxEvents())
		delete m_log->firstChild() ;

	m_count += 1 ;
	m_last	 = new KBEventLogEntry
		       (	m_log,
				m_last,
		       		m_count,
		       		type,
		       		objName,
		       		objType,
		       		eventName,
		       		argc, argv
		       ) ;

	/* Return a pointer at the entry. This will come back as the	*/
	/* third argument to "logEventResult".				*/
	return	m_last	;
}

/*  KBEventLog								*/
/*  logEventResult							*/
/*		: Add result to event log				*/
/*  resval	: const KBValue & : Result value (or error message)	*/
/*  ok		: bool		  : Event completed successfully	*/
/*  logid	: void *	  : Log entry				*/
/*  (returns)	: void		  :					*/

void	KBEventLog::logEventResult
	(	const KBValue	&resval,
		bool		ok,
		void		*logid
	)
{
	/* Look for the log entry. Optimise this by checking if the	*/
	/* entry is the last one; this will be the case except when one	*/
	/* event triggers another.					*/
	KBEventLogEntry	*entry	= 0 ;

	if (logid != m_last)
	{
		for (QListViewItem *item  = m_log->firstChild () ;
				    item != 0 ;
				    item  = item ->nextSibling())
			if (item == logid)
			{
				entry	= (KBEventLogEntry *)item ;
				break	;
			}
	}
	else	entry	= (KBEventLogEntry *)m_last ;

	if (entry != 0)
	{
		QString	res = resval.getRawText() ;
		if (res.length() > 64) res = res.left(64) + " ..." ;
		entry->setText (6, ok ? "OK" : "Fail" ) ;
		entry->setText (7, res) ;
	}
}


/*  KBEventLog								*/
/*  queryClose	: Close handler						*/
/*  (returns)	: void			:				*/

bool	KBEventLog::queryClose ()
{
	TKConfig	*config = TKConfig::getConfig () ;
	QValueList<int>	aw	;
	QValueList<int>	lw	;

	for (uint ao = 0 ; ao < 3 ; ao += 1)
		aw.append (m_args->columnWidth(ao)) ;

	for (uint lo = 0 ; lo < 8 ; lo += 1)
		lw.append (m_log ->columnWidth(lo)) ;

	config->setGroup   ("Event Log") ;
	config->writeEntry ("geometry",  m_partWidget->size ()) ;
	config->writeEntry ("depths",    m_split     ->sizes()) ;
	config->writeEntry ("argWidths", aw) ;
	config->writeEntry ("logWidths", lw) ;
	config->sync	   () ;


	m_action->setChecked (false) ;
	return	true	;
}

/*  KBEventLog								*/
/*  clickEvent	: User clicks event					*/
/*  item	: QListViewItem *	: Item				*/
/*  (returns)	: void			:				*/

void	KBEventLog::clickEvent
	(	QListViewItem	*item
	)
{
	if (item != 0)
		((KBEventLogEntry *)item)->showArgs (m_args) ;
}

void	KBEventLog::clearLog ()
{
	while (m_log ->childCount() > 0) delete m_log ->firstChild () ;
	while (m_args->childCount() > 0) delete m_args->firstChild () ;
	m_last	= 0 ;
}

