/***************************************************************************
    file	         : kb_copyxml.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<errno.h>
#include	<qapplication.h>
#include	<qxml.h>
#include	<qdom.h>


#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_location.h"
#include	"kb_databuffer.h"

#include	"kb_param.h"

#include	"kb_copyxml.h"
#include	"kb_copyexec.h"



/*  Base64 Encoder/Decoder						*/
/*  ----------------------						*/
/*									*/
/*  This code is taken from LibTomCrypt:				*/
/*									*/
/*		http://libtomcrypt.org/index.html			*/
/*									*/

static const char *codes = 
	"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/" ;

static const uchar base64Map[256] =
{
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
	255, 255, 255, 255, 255, 255, 255,  62, 255, 255, 255,  63,
	 52,  53,  54,  55,  56,  57,  58,  59,  60,  61, 255, 255,
	255, 254, 255, 255, 255,   0,   1,   2,   3,   4,   5,   6,
	  7,   8,   9,  10,  11,  12,  13,  14,  15,  16,  17,  18,
	 19,  20,  21,  22,  23,  24,  25, 255, 255, 255, 255, 255,
	255,  26,  27,  28,  29,  30,  31,  32,  33,  34,  35,  36,
	 37,  38,  39,  40,  41,  42,  43,  44,  45,  46,  47,  48,
	 49,  50,  51, 255, 255, 255, 255, 255, 255, 255, 255, 255,
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
	255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
	255, 255, 255, 255
}	;

bool	b64Needed
	(	cchar		*data,
		uint		dlen
	)
{
	while (dlen > 0)
	{	if ((*data < 0x20) || (*data >= 0x7f))
			return	true ;
		dlen	-= 1 ;
		data	+= 1 ;
	}

	return	false	;
}


void	b64Encode
	(	uchar		*data,
		uint		dlen,
		KBDataBuffer	&dest
	)
{
	uint	i	;
	uint	leven	= 3 * (dlen / 3) ;

	for (i = 0; i < leven; i += 3)
	{
		dest.append (codes[data[0] >> 2]) ;
		dest.append (codes[((data[0] & 0x3) << 4) + (data[1] >> 4)]) ;
		dest.append (codes[((data[1] & 0xf) << 2) + (data[2] >> 6)]) ;
		dest.append (codes[data[2] & 0x3f]) ;
		data += 3;
	}

	if (i < dlen)
	{
		uint	a = data[0] ;
		uint	b = (i + 1 < dlen) ? data[1] : 0 ;
		uint	c = 0 ;

		dest.append (codes[a >> 2]) ;
		dest.append (codes[((a & 3) << 4) + (b >> 4)]) ;
		dest.append ((i + 1 < dlen) ? codes[((b & 0xf) << 2) + (c >> 6)] : '=') ;
		dest.append ('=') ;
	}
}

void	b64Decode
	(	uchar		*data, 
		uint		dlen,
		KBDataBuffer	&dest
	)
{
	unsigned long t, x, y	;
	int 	      g = 3	;

	for (x = y = t = 0; x < dlen; x += 1, data += 1)
	{
		uchar c = base64Map[*data] ;

		if (c == 255)
			continue ;

		if (c == 254)
		{
			c  = 0 ;
			g -= 1 ;
		}

		t = (t<<6) | c ;

		if ((y += 1) == 4)
		{
			dest.append ((uchar)((t>>16)&255)) ;
			if (g > 1)
				dest.append ((uchar)((t>>8)&255)) ;
			if (g > 2)
				dest.append ((uchar)(t&255)) ;

			y = 0 ;
			t = 0 ;
		}
	}
}


/*  ------------------------------------------------------------------	*/

/*  KBCopyXMLSAX							*/
/*  ------------							*/
/*  SAX parser class for reading XML source files into a copier.	*/

class	LIBKBASE_API	KBCopyXMLSAX : public QXmlDefaultHandler
{
protected :

	enum	ParseState
	{
		Initial,
		Data,
		Record,
		Value
	}	;

	KBError		m_lError	;	/* Error message	*/
	bool		m_bGotErr	;	/* Error occurred flag	*/
	ParseState	m_state		;	/* Parser state		*/

	QString		m_baseTag	;
	QString		m_rowTag	;
	QStringList	&m_names	;

	KBCopyBase	*m_dest		;
	KBValue		*m_values	;
	int		m_nCols		;
	bool		m_base64	;

	int		m_nRows		;

	KBCopyExec	*m_exec		;
	KBDataBuffer	m_data		;

	void	setErrMessage	(const QXmlParseException       &) ;
	void	setErrMessage	(const QString &, const QString &) ;
	void	setErrMessage	(const QString &, ParseState	 ) ;

public	:

	KBCopyXMLSAX
	(	const QString	&,
		const QString	&,
		QStringList	&,
		KBCopyBase 	*,
		KBValue		*,
		int		,
		KBCopyExec	*
	)	;

	virtual	bool	characters	(const QString &) ;
	virtual	bool	startElement	(const QString &, const QString &, const QString &, const QXmlAttributes &) ;
	virtual	bool	endElement	(const QString &, const QString &, const QString &) ;
	virtual	bool	error		(const QXmlParseException &) ;
	virtual	bool	warning		(const QXmlParseException &) ;
	virtual	bool	fatalError	(const QXmlParseException &) ;

	bool		parse		(QTextStream &) ;

	inline	const KBError	&lastError () { return m_lError  ; }
	inline	int		numRows	   () { return m_nRows	 ; }
}	;


/*  KBCopyXMLSAX							*/
/*  KBCopyXMLSAX: Constructor for parser class				*/
/*  (returns)	: KBCopyXMLSAX	 :					*/

KBCopyXMLSAX::KBCopyXMLSAX
	(	const QString	&baseTag,
		const QString	&rowTag,
		QStringList	&names,
		KBCopyBase	*dest,
		KBValue		*values,
		int		nCols,
		KBCopyExec	*exec
	)
	:
	m_baseTag	(baseTag),
	m_rowTag	(rowTag),
	m_names		(names),
	m_dest		(dest),
	m_values	(values),
	m_nCols		(nCols),
	m_exec	(exec)
{
	m_state		= Initial	;
	m_bGotErr	= false		;
	m_nRows		= 0		;
	m_base64	= false		;

//	fprintf
//	(	stderr,
//		"KBCopyXMLSAX::KBCopyXMLSAX: base=[%s] row=[%s]\n",
//		(cchar *)m_baseTag,
//		(cchar *)m_rowTag
//	)	;
}

/*  KBCopyXMLSAX							*/
/*  parseFile	: Parse document					*/
/*  stream	: QTextStream &	: Input stream				*/
/*  (returns)	: bool		: Success				*/

bool	KBCopyXMLSAX::parse
	(	QTextStream	&stream
	)
{
	QXmlInputSource	 xmlSource (stream)	;
	QXmlSimpleReader xmlReader		;

	xmlReader.setContentHandler(this     )	;
	xmlReader.parse		   (xmlSource)	;

	/* See if an error was logged. If so then delete any parse tree	*/
	/* that was forthcoming, then return failure.			*/
	return	! m_bGotErr ;
}

/*  KBCopyXMLSAX							*/
/*  setErrMessage: Set an error message					*/
/*  e		 : const QXmlParseException &				*/ 
/*				: SAX parser exception information	*/
/*  (returns)	 : void		:					*/

void	KBCopyXMLSAX::setErrMessage
	(	const QXmlParseException &e
	)
{
	m_lError = KBError
		   (	KBError::Error,
			QString  (TR("parsing error at line %2, column %3"))
					.arg(e.lineNumber  ())
					.arg(e.columnNumber()),
			e.message(),
			__ERRLOCN
		   )	  ;
	m_bGotErr  = true ;
}

/*  KBCopyXMLSAX							*/
/*  setErrMessage: Set an error message					*/
/*  msg		 : const QString&: Error message			*/
/*  arg		 : const QString&: Error message argument		*/
/*  (returns)	 : void		 :					*/

void	KBCopyXMLSAX::setErrMessage
	(	const QString	&msg,
		const QString	&arg
	)
{
	m_lError  = KBError
		  (	KBError::Error,
			msg,
			arg,
			__ERRLOCN
		  )	;
	m_bGotErr = true;
}

/*  KBCopyXMLSAX							*/
/*  setErrMessage: Set an error message					*/
/*  msg		 : const QString&: Error message			*/
/*  state	 : ParseState	 : Error state				*/
/*  (returns)	 : void		 :					*/

void	KBCopyXMLSAX::setErrMessage
	(	const QString	&msg,
		ParseState	state
	)
{
	QString	arg	;

	switch (state)
	{
		case Initial	 : arg = "Initial"	; break ;
		case Data	 : arg = "Data"		; break ;
		case Record	 : arg = "Record"	; break ;
		case Value	 : arg = "Value"	; break ;
		default		 :
			arg = QString("Unknown (%1)").arg(state) ;
			break	 ;
	}

	setErrMessage (msg, QString(": state %1").arg(arg)) ;
	m_bGotErr = true;
}

/*  KBCopyXMLSAX							*/
/*  startElement: Handle start of element				*/
/*  URI		: const QString &	: Namespace URI			*/
/*  localName	: const QString &	:				*/
/*  qName	: const QString &	: Element name			*/
/*  attribs	: const QXmlAttributes &: Attribute list		*/
/*  (returns)	: bool			:				*/

bool	KBCopyXMLSAX::startElement
	(	const	QString		&,
		const	QString		&,
		const	QString		&qName,
		const	QXmlAttributes	&attribs
	)
{
//	fprintf
//	(	stderr,
//		"KBCopyXMLSAX::startElement: state %d: [%s]\n",
//		m_state,
//		(cchar *)qName
//	)	;

	switch (m_state)
	{
		case Initial :
			if (qName != m_baseTag)
			{	setErrMessage ("XML error", "expected base tag") ;
				return	false ;
			}
			m_state	= Data	;
			break	;

		case Data   :
			if (qName != m_rowTag)
			{	setErrMessage ("XML error", "expected row tag") ;
				return	false ;
			}
			m_state	= Record;

			{
			for (int idx = 0 ; idx < m_nCols ; idx += 1)
				m_values[idx] = KBValue() ;
			}

			{
			for (int idx = 0 ; idx < attribs.length() ; idx += 1)
			{
				int	slot = m_names.findIndex (attribs.qName(idx)) ;
				if (slot >= 0)
					m_values[slot] = attribs.value(idx) ;
			}
			}
			break	;

		case Record :
			if (m_exec != 0)
				if (m_exec->showProgress (m_nRows))
				{
					m_lError = KBError
						   (	KBError::Warning,
						  	TR("User cancelled copy"),
						  	QString::null,
					  		__ERRLOCN
					  	 )	;
					return	 false	;
				}

			m_state = Value	;
			m_data.clear () ;

			/* See if there is a "dt" attribute which has	*/
			/* been set to "base64", in which case we will	*/
			/* treat the following data as base64 encoded.	*/
			{
			m_base64 = false ;
			for (int idx = 0 ; idx < attribs.length() ; idx += 1)
				if (attribs.qName(idx) == "dt")
					if (attribs.value(idx) == "base64")
						m_base64 = true ;
			}
			break	;

		case Value  :
			setErrMessage
			(	"XML error",
				QString("unexpected tag '%1' in data value").arg(qName)
			)	;
			return	false ;

		default	:
			setErrMessage ("XML error", m_state) ;
			return	false	;
	}

	return	true	;
}

/*  KBCopyXMLSAX							*/
/*  endElement	: Handle end of element					*/
/*  URI		: const QString &	: Namespace URI			*/
/*  localName	: const QString &	:				*/
/*  qName	: const QString &	: Element name			*/
/*  (returns)	: bool			: Success			*/

bool	KBCopyXMLSAX::endElement
	(	const	QString		&,
		const	QString		&,
		const	QString		&qName
	)
{
//	fprintf
//	(	stderr,
//		"KBCopyXMLSAX::endElement: state %d: [%s]\n",
//		m_state,
//		(cchar *)qName
//	)	;

	switch (m_state)
	{
		case Value	 :
			{
			/* If the name exists then store the data as a	*/
			/* new value in the values list. If the base64	*/
			/* flag has been set then we need to decode it	*/
			/* first.					*/
			int	slot = m_names.findIndex (qName) ;
			if (slot >= 0)
			{
				if (m_base64)
				{
					KBDataBuffer	buff ;
					b64Decode ((uchar *)m_data.data(), m_data.length(), buff) ;
					m_values[slot] = KBValue(buff  .data(), buff  .length()) ;
				}
				else	m_values[slot] = KBValue(m_data.data(), m_data.length()) ;
			}
			}

			m_state	  	 = Record ;
			m_base64	 = false  ;
			m_data.clear ()	 ;
			return	true 	 ;

		case Record	:
			if (!m_dest->putRow (m_values, m_nCols))
			{
				m_lError = m_dest->lastError() ;
				return	 false	;
			}
			m_nRows	 += 1	 ;
			m_state	  = Data ;
			return	  true	 ;

		default	:
			break	;
	}

	return	true ;
}

/*  KBCopyXMLSAX							*/
/*  characters	: Handle a block of characters				*/
/*  code	: const QString & : Characters				*/
/*  (returns)	: bool		  : Success				*/

bool	KBCopyXMLSAX::characters
	(	const QString		&code
	)
{
	m_data.append (code) ;
	return	true	;
}

bool	KBCopyXMLSAX::error
	(	const QXmlParseException &e
	)
{
	setErrMessage (e) ;
	return	true	;
}

bool	KBCopyXMLSAX::warning
	(	const QXmlParseException &e
	)
{
	setErrMessage (e) ;
	return	true	;
}

bool	KBCopyXMLSAX::fatalError
	(	const QXmlParseException &e
	)
{
	setErrMessage (e) ;
	return	false	 ;
}

/*  ------------------------------------------------------------------  */

/*  IOError	: Convert QT IO error status to message			*/
/*  status	: int		: Error status				*/
/*  (returns)	: QString	: Error message				*/

static	QString	IOError
	(	int	status
	)
{
	cchar	*qterr	;

	switch (status)
	{
		case IO_Ok		: qterr	= TR("No error")	; break ;
		case IO_ReadError	: qterr	= TR("Read error")	; break ;
		case IO_WriteError	: qterr	= TR("Write error")	; break ;
		case IO_FatalError	: qterr	= TR("Fatal error")	; break ;
		case IO_OpenError	: qterr	= TR("Open error")	; break ;
		case IO_AbortError	: qterr	= TR("Abort")		; break ;
		case IO_TimeOutError	: qterr	= TR("Time-out error")	; break ;
		default			: qterr	= TR("Unknown error")	; break ;
	}

	return	QString("%1: %2").arg(qterr).arg(strerror(errno)) ;
}

static	void	xmlEscape
	(	KBValue		&value,
		QTextStream	&stream
	)
{
	cchar	*dataPtr = value.dataPtr   () ;
	uint	dataLen	 = value.dataLength() ;
	int	ch	 ;

	for (uint idx = 0 ; idx < dataLen ; idx += 1)
	{
		ch = dataPtr[idx] ;

		if (ch == '<')
		{	stream << "&lt;"   ;
			continue  ;
		}
		if (ch == '>')
		{	stream << "&gt;"   ;
			continue  ;
		}
		if (ch == '&')
		{	stream << "&amp;"  ;
			continue  ;
		}
		if (ch == '"')
		{	stream << "&quot;" ;
			continue  ;
		}

		stream.writeRawBytes (&dataPtr[idx], 1) ;
	}
}

/*  ------------------------------------------------------------------  */

/*  KBCopyXML								*/
/*  KBCopyXML	: Constructor for table copier				*/
/*  srce	: bool		: Source, else destination		*/
/*  location	: KBLocation &	: Database location			*/
/*  (returns)	: KBCopyXML	:					*/

KBCopyXML::KBCopyXML
	(	bool		srce,
		KBLocation	&location
	)
	:
	m_srce		(srce),
	m_location	(location)
{
}

/*  KBCopyXML								*/
/*  ~KBCopyXML: Destructor for table copier				*/
/*  (returns)	:		:					*/

KBCopyXML::~KBCopyXML ()
{
}

/*  KBCopyXML								*/
/*  tag		: Get tag for XML output				*/
/*  (returns)	: cchar *	: Tag					*/

cchar	*KBCopyXML::tag ()
{
	return	"xml"	;
}

/*  KBCopyXML								*/
/*  setErrOpt	: Set line error option					*/
/*  erropt	: uint		: Option				*/
/*  (returns)	: void		:					*/

void	KBCopyXML::setErrOpt
	(	uint		erropt
	)
{
	m_erropt = erropt ;
}

/*  KBCopyXML								*/
/*  setFile	: Set source/destination file name			*/
/*  file	: const QString & : File name				*/
/*  (returns)	: void		  :					*/

void	KBCopyXML::setFile
	(	const QString	&file
	)
{
	m_file	= file  ;
}

/*  KBCopyXML								*/
/*  setMainTag	: Set main node tag					*/
/*  mainTag	: const QString & : Tag					*/
/*  (returns)	: void		  :					*/

void	KBCopyXML::setMainTag
	(	const QString	&mainTag
	)
{
	m_mainTag = mainTag ;
}

/*  KBCopyXML								*/
/*  setRowTag	: Set row node tag					*/
/*  rowTag	: const QString & : Tag					*/
/*  (returns)	: void		  :					*/

void	KBCopyXML::setRowTag
	(	const QString	&rowTag
	)
{
	m_rowTag = rowTag  ;
}

/*  KBCopyXML								*/
/*  addField	: Add field definition					*/
/*  name	: const QString & : Tag or attribute name		*/
/*  asattr	: bool		  : Save as attribute			*/
/*  (returns)	: void		  :					*/

void	KBCopyXML::addField
	(	const QString	&name,
		bool		asattr
	)
{
	m_names  .append (name  ) ;
	m_asattr .append (asattr) ;
}

/*  KBCopyXML								*/
/*  getField	: Get specified field					*/
/*  idx		: uint		: Field index				*/
/*  name	: QString &	: Return notional name			*/
/*  asattr	: bool &	: Return as attribute flag		*/
/*  (returns)	: bool		: Indexed field exists			*/

bool	KBCopyXML::getField
	(	uint		idx,
		QString		&name,
		bool		&asattr
	)
{
	if (idx < m_names.count())
	{
		name	= *m_names.at(idx) ;
		asattr	= m_asattr   [idx] ;
		return	true	;
	}

	return	false	;
}

/*  KBCopyXML								*/
/*  set		: Set copier from definition				*/
/*  copy	: QDomElement & : Definition parent			*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBCopyXML::set
	(	QDomElement	&copy,
		KBError		&
	)
{
	QDomElement	element	= copy.namedItem(tag()).toElement() ;

	if (element.isNull ()) return true ;

	reset	()	;

	setErrOpt  (element.attribute ("erropt").toUInt()) ;
	setMainTag (element.attribute ("maintag")) ;
	setRowTag  (element.attribute ("rowtag" )) ;
	setFile    (element.attribute ("file"   )) ;

	QDomNodeList	dFields	= element.elementsByTagName ("field") ;
	for (uint idx = 0 ; idx < dFields.length() ; idx += 1)
	{	QDomElement e = dFields.item(idx).toElement() ;
		m_names  .append (e.attribute("name"  )) ;
		m_asattr .append (e.attribute("asattr") == "Yes") ;
	}

	return	true	;
}

/*  KBCopyXML								*/
/*  valid	: Check validity					*/
/*  error	: KBError &	: Error return				*/
/*  (returns)	: bool		: Valid					*/

bool	KBCopyXML::valid
	(	KBError		&error
	)
{
	if (m_file.isEmpty())
	{
		error	= KBError
			  (	KBError::Error,
				TR("No source or destination file specified"),
				QString::null,
				__ERRLOCN
			  )	;
		return	false	;
	}

	return	true	;
}

/*  KBCopyXML								*/
/*  def		: Get table copy definition				*/
/*  copy	: QDomElement &	: Element to which to attach		*/
/*  (returns)	: void		:					*/

void	KBCopyXML::def
	(	QDomElement	&copy
	)
{
	QDomElement	element	;

	copy   .appendChild  (element = copy.ownerDocument().createElement (tag())) ;

	element.setAttribute ("erropt",  m_erropt ) ;
	element.setAttribute ("maintag", m_mainTag) ;
	element.setAttribute ("rowtag",  m_rowTag ) ;
	element.setAttribute ("file",    m_file   ) ;

	for (uint idx = 0 ; idx < m_names.count() ; idx += 1)
	{
		QDomElement	dField	;
		element.appendChild (dField = element.ownerDocument().createElement ("field")) ;
		dField.setAttribute ("name",   *m_names.at(idx)) ;
		dField.setAttribute ("asattr", m_asattr   [idx] ? "Yes" : "No") ;
	}
}

/*  KBCopyXML								*/
/*  getColumnNames							*/
/*		: Get column names					*/
/*  names	: QStringList &	: List of column names			*/
/*  (returns)	: void		:					*/

void	KBCopyXML::getColumnNames
	(	QStringList	&names
	)
{
	names	= m_names	;
}

/*  KBCopyXML								*/
/*  prepare	: Prepare for execution					*/
/*  paramDict	: const QDict<QString> &				*/
/*				: Paramater dictionary			*/
/*  other	: KBCopyBase *	: Other half of copier			*/
/*  (returns)	: bool		: Success				*/

bool	KBCopyXML::prepare
	(	const QDict<QString>	&paramDict,
		KBCopyBase		*other
	)
{
	m_io.close   ()	  ;
	m_io.setName (m_useFile = paramSub (m_file, paramDict)) ;

	if (!m_io.open (m_srce ? IO_ReadOnly : IO_WriteOnly|IO_Truncate))
	{
		m_lError = KBError
			   (	KBError::Error,
				QString (TR("Unable to open \"%1\"")).arg(m_useFile),
				IOError (m_io.status()),
				__ERRLOCN
			   )	;
		return	 false	;
	}

	m_stream.setDevice (&m_io) ;

	m_useMainTag	= paramSub (m_mainTag, paramDict) ;
	m_useRowTag	= paramSub (m_rowTag,  paramDict) ;

	if (!m_srce)
	{
		QStringList srceNames ;
		other->getColumnNames (srceNames) ;
		m_destNames.clear ()   ;

		for (uint idx = 0 ; idx < m_names.count() ; idx += 1)
			if (m_names[idx] == "<Auto>")
				m_destNames.append (srceNames[idx]) ;
			else	m_destNames.append (m_names  [idx]) ;
	}

	m_nRows	= 0  ;
	return	true ;
}

/*  KBCopyXML								*/
/*  getNumCols	: Get number of columns in copy				*/
/*  (returns)	: int		: Number of columns			*/

int	KBCopyXML::getNumCols ()
{
	return	m_names.count() ;
}

/*  KBCopyXML								*/
/*  getRow	: Get next row						*/
/*  values	: KBValue *	: Value vector				*/
/*  nCols	: uint		: Number of columns available		*/
/*  ok		: bool &	: Execute OK				*/
/*  (returns)	: int		: Actual number of columns		*/

int	KBCopyXML::getRow
	(	KBValue		*,
		uint		,
		bool		&ok
	)
{
	/* Check that this is a source copier, if not then there is an	*/
	/* implementation error.					*/
	if (!m_srce)
	{
		m_lError = KBError
			   (	KBError::Fault,
				TR("Attempt to fetch row from destination copier"),
				QString::null,
				__ERRLOCN
			   )	;
		ok	= false	;
		return	-1	;
	}

	ok	= true	;
	return	-1	;
}

/*  KBCopyXML								*/
/*  putRow	: Put a row of data					*/
/*  values	: KBValue *	: Value vector				*/
/*  aCols	: uint		: Actual number of columns		*/
/*  (returns)	: bool		: Success				*/

bool	KBCopyXML::putRow
	(	KBValue		*values,
		uint		aCols
	)
{
	/* Check that this is a destination copier. An error here is an	*/
	/* implementation problem.					*/
	if (m_srce)
	{
		m_lError = KBError
			   (	KBError::Fault,
				TR("Attempt to insert row into source copier"),
				QString::null,
				__ERRLOCN
			   )	;
		return	false	;
	}

	/* If there are no values this is the initial call before any	*/
	/* real data arrives. Use this to write out the header.		*/
	if (values == 0)
	{
		extern	QString	kbXMLEncoding () ;

		m_stream << QString
			  (	"<?xml version=\"1.0\" encoding=\"%1\"?>\n"
				"<!DOCTYPE %1>\n"
				"<%1>\n"
			  )
			  .arg	(kbXMLEncoding())
			  .arg	(m_useMainTag)
			  .arg	(m_useMainTag) ;

		if (m_io.status() != IO_Ok)
		{
			m_lError = KBError
				   (	KBError::Error,
					QString (TR("Error writing to \"%1\"")).arg(m_file),
					IOError (m_io.status()),
					__ERRLOCN
		  		   )	;
			return	false	;
		}

		return	true	;
	}

	/* Check that we have the right number of columns and take	*/
	/* action accordingly if not. Note the for delimited output,	*/
	/* any number of columns is OK.					*/
	if ((getNumCols() != 0) && (getNumCols() != (int)aCols))
		switch (m_erropt)
		{
			case ErrSkip	:
				/* Just skip this record.		*/
				return	true	;

			case ErrAbort	:
				/* Abort copy with a suitable error.	*/
				m_lError = KBError
					   (	KBError::Error,
						TR("Insufficient output columns"),
						QString	(TR("Expected %1, got %2"))
							.arg(getNumCols())
							.arg(aCols),
						__ERRLOCN
					   )	;
				return	false	;

			default	:
				/* Otherwise pad with nulls. The copier	*/
				/* object will have done this for us.	*/
				break	;
		}



	m_stream << "  <"       ;
	m_stream << m_useRowTag ;

	for (uint idx1 = 0 ; idx1 < m_destNames.count() ; idx1 += 1)
		if (m_asattr[idx1])
		{	m_stream << " "   ;
			m_stream << m_destNames[idx1] ;
			m_stream << "=\"" ;
			xmlEscape(values[idx1], m_stream) ;
			m_stream << "\""  ;
		}

	m_stream << ">\n" ;

	for (uint idx2 = 0 ; idx2 < m_destNames.count() ; idx2 += 1)
		if (!m_asattr[idx2])
		{
			KBValue	&v	= values[idx2]	 ;
			cchar	*data	= v.dataPtr   () ;
			uint	dlen	= v.dataLength() ;
			bool	b64	= b64Needed (data, dlen) ;

			m_stream << "    <" ;
			m_stream << m_destNames[idx2] ;

			if (b64)
				m_stream << " dt=\"base64\"" ;

			m_stream << ">"     ;


			if (b64)
			{
				KBDataBuffer b ;
				b64Encode ((uchar *)data, dlen, b) ;
				m_stream.writeRawBytes (b.data(), b.length()) ;
			}
			else	xmlEscape (values[idx2], m_stream) ;

			m_stream << "</"    ;
			m_stream << m_destNames[idx2] ;
			m_stream << ">\n"   ;
		}

	m_stream << "  </"      ;
	m_stream << m_useRowTag ;
	m_stream << ">\n"       ;

	if (m_io.status() != IO_Ok)
	{
		m_lError = KBError
			   (	KBError::Error,
				QString (TR("Error writing to \"%1\"")).arg(m_file),
				IOError (m_io.status()),
				__ERRLOCN
		  	   )	;
		return	 false	;
	}

	m_nRows	+= 1	;
	return	true	;
}

/*  KBCopyXML								*/
/*  finish	: Finish execution					*/
/*  report	: QString &	: Completion report			*/
/*  (returns)	: bool		: Success				*/

bool	KBCopyXML::finish
	(	QString		&report
	)
{
	if (!m_srce)
		m_stream << QString ("</%1>\n").arg(m_mainTag) ;

	if (m_io.status() != IO_Ok)
	{
		m_lError = KBError
			   (	KBError::Error,
				QString (TR("Error closing \"%1\"")).arg(m_file),
				IOError (m_io.status()),
				__ERRLOCN
		  	   )	;
		return	 false	;
	}

	m_io.close () ;

	if (m_io.status() != IO_Ok)
	{
		m_lError = KBError
			   (	KBError::Error,
				QString (TR("Error closing \"%1\"")).arg(m_file),
				IOError (m_io.status()),
				__ERRLOCN
		  	   )	;
		return	 false	;
	}

	report	= QString(TR("Copied %1 rows")).arg(m_nRows) ;
	return	true	;
}

/*  KBCopyXML								*/
/*  (returns)	: void		:					*/

void	KBCopyXML::reset ()
{
	m_mainTag	= QString::null ;
	m_rowTag	= QString::null ;
	m_file		= QString::null ;

	m_names  .clear () ;
	m_asattr .clear () ;
}


int	KBCopyXML::execute
	(	KBCopyBase	*dest,
		KBValue		*values,
		int		nCols,
		KBCopyExec 	*exec
	)
{
	KBCopyXMLSAX sax
		(	m_useMainTag,
			m_useRowTag,
			m_names,
			dest,
			values,
			nCols,
			exec
		)	;

	if (!sax.parse(m_stream))
	{
		m_lError = sax.lastError() ;
		return	 -1 ;
	}

	return	sax.numRows() ;
}

