/***************************************************************************
    file	         : kb_loader.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<errno.h>

#include	<qstring.h>
#include	<qfile.h>
#include	<qtextstream.h>
#include	<qlayout.h>
#include	<qtimer.h>

#include	"tk_messagebox.h"

#include	"kb_dbinfo.h"
#include	"kb_dblink.h"

#include	"kb_dialog.h"

#include	"kb_copybase.h"
#include	"kb_copytable.h"
#include	"kb_copyxml.h"
#include	"kb_copyexec.h"
#include	"kb_paramsetdlg.h"

#ifdef		_WIN32
#include	"kb_loader.h"
#else
#include	"kb_loader.moc"
#endif



/*  KBLoaderItem							*/
/*  KBLoaderItem: Constructor for loader list item			*/
/*  parent	: QListView *	  : Parent list view			*/
/*  name	: const QString & : Object name				*/
/*  flags	: uint		  : Definition/data flags		*/
/*  exists	: bool		  : True if object exists		*/
/*  (returns)	: KBLoaderItem	  :					*/

KBLoaderItem::KBLoaderItem
	(	QListView	*parent,
		const QString	&name,
		uint		flags,
		bool		exists
	)
	:
	QCheckListItem	(parent, name, QCheckListItem::CheckBox),
	m_name		(name),
	m_flags		(flags),
	m_exists	(exists)
{
	setText
	(	1,
		(flags & FL_TABLEDEF ) != 0 ? TR("Table"   ) :
		(flags & FL_VIEWDEF  ) != 0 ? TR("View"	   ) :
		(flags & FL_SEQDEF   ) != 0 ? TR("Sequence") :
		(flags & FL_TABLEDATA) != 0 ? TR("Table"   ) :
					      TR("Unknown" )
	)	;

	setText (2, exists 		      ? TR("Yes") : TR("No")) ;		
	setText (3, (flags & FL_ANYDEF ) != 0 ? TR("Yes") : TR("No")) ;
	setText (4, (flags & FL_ANYDATA) != 0 ? TR("Yes") : TR("No")) ;

	setText
	(	5,
		((flags & FL_SEQDEF   ) != 0 ? "aa_" :
		 (flags & FL_TABLEDEF ) != 0 ? "bb_" :
		 (flags & FL_TABLEDATA) != 0 ? "bb_" : "cc_") + name
	)	;
}


/*  ------------------------------------------------------------------  */

static	void	getFileList
	(	QDict<uint>	&dict,
		const QString	&dir,
		cchar		*extn,
		uint		flag
	)
{
	QDir		scan	;

	scan.setPath	   (dir )	 ;
	scan.setNameFilter (extn)	 ;
	scan.setFilter	   (QDir::Files) ;
	scan.setSorting	   (QDir::Name ) ;

	QStringList sl = scan.entryList () ;
	for (uint idx = 0 ; idx < sl.count() ; idx += 1)
	{
		const QString &file = sl[idx] ;
		QString	      name  = file.left(file.length() - strlen(extn) + 1) ;
		uint	      *val  = dict.find(name) ;

		if (val == 0)
		{	val	= new uint ;
			*val	= 0 ;
			dict.insert (name, val) ;
		}

		*val	|= flag	;
	}
}

/*  ------------------------------------------------------------------  */


class	LIBKBASE_API	KBCopyExecLoader : public KBCopyExec
{
	KBLoader	*m_loader	;

public	:

	KBCopyExecLoader
	(	KBCopyBase	*,
		KBCopyBase	*,
		KBLoader	*
	)	;

	virtual	bool	showProgress	(int)	;
}	;

KBCopyExecLoader::KBCopyExecLoader
	(	KBCopyBase	*srce,
		KBCopyBase	*dest,
		KBLoader	*loader
	)
	:
	KBCopyExec	(srce, dest),
	m_loader	(loader)
{
}

bool	KBCopyExecLoader::showProgress
	(	int		nRows
	)
{
	return	m_loader->showProgress (nRows) ;
}

/*  ------------------------------------------------------------------  */

/*  KBLoader								*/
/*  KBLoader	: Constructor for database loader			*/
/*  dbInfo	: KBDBInfo *	  : Database information		*/
/*  server	: const QString & : Server name				*/
/*  dir		: const QString & : Output directory			*/
/*  (returns)	: KBLoader	  :					*/

KBLoader::KBLoader
	(	KBDBInfo	*dbInfo,
		const QString	&server,
		const QString	&dir
	)
	:
	_KBDialog	(TR("Load database"), true),
	m_loadAll	(this),
	m_loadDefn	(this),
	m_loadData	(this),
	m_dropObjects	(this),
	m_tableList	(this),
	m_loading	(this),
	m_record	(this),
	m_of		(this),
	m_bOK		(this, "ok"    ),
	m_bCancel	(this, "cancel"),
	m_dbInfo	(dbInfo),
	m_server	(server),
	m_dir		(dir)
{
	QVBoxLayout	*layMain = new QVBoxLayout (this) ;

	layMain->addWidget (&m_loadAll    ) ;
	layMain->addWidget (&m_loadDefn   ) ;
	layMain->addWidget (&m_loadData   ) ;
	layMain->addWidget (&m_dropObjects) ;
	layMain->addWidget (&m_tableList  ) ;

	QHBoxLayout	*layProg = new QHBoxLayout (layMain) ;
	QLabel		*l1	 = new QLabel	   (this) ;
	QLabel		*l2	 = new QLabel	   (this) ;
	QLabel		*l3	 = new QLabel	   (this) ;

	layProg->addWidget (l1) ;
	layProg->addWidget (&m_loading) ;
	layProg->addWidget (l2) ;
	layProg->addWidget (&m_record ) ;
	layProg->addWidget (l3) ;
	layProg->addWidget (&m_of     ) ;

	QHBoxLayout	*layButt = new QHBoxLayout (layMain) ;

	layButt->addStretch()		   ;
	layButt->addWidget (&m_bOK    	 ) ;
	layButt->addWidget (&m_bCancel	 ) ;

#if	! __KB_EMBEDDED
	m_loading.setMinimumWidth (250) ;
#endif
	m_loadAll    .setText (TR("Load all objects"	    )) ;
	m_loadDefn   .setText (TR("Load object definitions" )) ;
	m_loadData   .setText (TR("Load object data"	    )) ;
	m_dropObjects.setText (TR("Replace existing objects")) ;

	m_cancelled	= false	;
	m_finished	= false ;
	m_currItem	= 0	;
	m_index		= 0	;

	m_tableList.addColumn(TR("Name"	     )) ;
	m_tableList.addColumn(TR("Type"	     )) ;
	m_tableList.addColumn(TR("Exists"    )) ;
	m_tableList.addColumn(TR("Definition")) ;
	m_tableList.addColumn(TR("Data"	     )) ;

	m_loading.setFrameStyle (QFrame::Box|QFrame::Plain) ;
	m_loading.setLineWidth  (1) ;
	m_record .setFrameStyle (QFrame::Box|QFrame::Plain) ;
	m_record .setLineWidth  (1) ;
	m_of	 .setFrameStyle (QFrame::Box|QFrame::Plain) ;
	m_of	 .setLineWidth  (1) ;

	l1->setText 	(TR("Object"  )) ;
	l1->setAlignment(Qt::AlignRight) ;

	l2->setText	(TR("Record"  )) ;
	l2->setAlignment(Qt::AlignRight) ;

	l3->setText 	(TR("Object"  )) ;
	l3->setAlignment(Qt::AlignRight) ;
}

/*  KBLoader								*/
/*  accept	: User clicks OK button					*/
/*  (returns)	: void		:					*/

void	KBLoader::accept ()
{
	if (!m_finished)
	{
		if (!m_loadDefn.isChecked() && !m_loadData.isChecked())
		{
			TKMessageBox::sorry
			(	0,
				TR("Please select definition and/or data loading"),
				TR("Load Database")
			)	;
			return	;
		}

		if (!m_loadAll.isChecked())
		{
			bool	any = false ;
			QCheckListItem *i = (QCheckListItem *)m_tableList.firstChild() ;
			while (i != 0)
				if (i->isOn())
				{	any	= true ;
					break	;
				}
				else	i = (QCheckListItem *)i->nextSibling() ;

			if (!any)
			{
				TKMessageBox::sorry
				(	0,
					TR("No tables selected for loading"),
					TR("Load Database")
				)	;
				return	;
			}
		}

		m_loadAll    .setEnabled (false) ;
		m_loadDefn   .setEnabled (false) ;
		m_loadData   .setEnabled (false) ;
		m_dropObjects.setEnabled (false) ;
		m_tableList  .setEnabled (false) ;
		m_bOK	     .setEnabled (false) ;

		slotTimer () ;
		return	;
	}

	done	(1) ;
}

/*  KBLoader								*/
/*  clickCancel	: User clicks cancel button				*/
/*  (returns)	: void		:					*/

void	KBLoader::reject ()
{
	if (!m_cancelled)
	{
		m_cancelled = true ;
		done	(0) ;
	}
}

/*  KBLoader								*/
/*  exec	: Execute dumper					*/
/*  (returns)	: int		: Exit code				*/

int	KBLoader::exec	()
{
	if (!m_dbLink.connect (m_dbInfo, m_server))
	{	m_dbLink.lastError().DISPLAY() ;
		return 0 ;
	}

	QDict<uint>	names	   ;
	names.setAutoDelete (true) ;

	getFileList	(names, m_dir, "*.tabledef",  FL_TABLEDEF ) ;
	getFileList	(names, m_dir, "*.viewdef",   FL_VIEWDEF  ) ;
	getFileList	(names, m_dir, "*.seqdef",    FL_SEQDEF	  ) ;
	getFileList	(names, m_dir, "*.tabledata", FL_TABLEDATA) ;

	m_tableList.setSorting (5) ;

	QDictIterator<uint> iter (names) ;
	uint		    *val	;

	while ((val = iter.current()) != 0)
	{
		QString		name	= iter.currentKey() ;
		bool		exists	= false ;

		if	((*val & (FL_TABLEDEF|FL_TABLEDATA)) != 0)
		{
			if (!m_dbLink.tableExists    (name, exists))
			{	m_dbLink.lastError().DISPLAY() ;
				return 0 ;
			}
		}
		else if	((*val & FL_VIEWDEF) != 0)
		{
			if (!m_dbLink.viewExists     (name, exists))
			{	m_dbLink.lastError().DISPLAY() ;
				return 0 ;
			}
		}
		else if	((*val & FL_SEQDEF ) != 0)
		{
			if (!m_dbLink.sequenceExists (name, exists))
			{	m_dbLink.lastError().DISPLAY() ;
				return 0 ;
			}
		}


		new KBLoaderItem
		(	&m_tableList,
			name,
			*val,
			exists
		)	;

		iter += 1 ;
	}

	m_currItem = (KBLoaderItem *)m_tableList.firstChild() ;
	m_index	   = 0 ;

	return	_KBDialog::exec () ;
}

bool	KBLoader::loadXMLSpec
	(	const QString	&name,
		cchar		*extn,
		QDomDocument	&doc,
		KBError		&pError
	)
{
	QFile	file	(m_dir + "/" + name + extn) ;

	if (!file.open (IO_ReadOnly))
	{
		pError	= KBError
			  (	KBError::Error,
				QString	(TR("Cannot open \"%1\"")).arg(file.name()),
				strerror(errno),
				__ERRLOCN
			  )	;
		return	false	;
	}

	if (!doc.setContent (&file))
	{
		pError	= KBError
			  (	KBError::Error,
				QString	(TR("Cannot parse \"%1\"")).arg(file.name()),
				strerror(errno),
				__ERRLOCN
			  )	;
		return	false	;
	}

	return	true	;
}

/*  KBLoader								*/
/*  loadTableDef: Load table definition					*/
/*  name	: const QString & : Table/definition file name		*/
/*  exists	: bool		  : Table already exists		*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBLoader::loadTableDef
	(	const QString	&name,
		bool		exists,
		KBError		&pError
	)
{
	QDomDocument	doc	;
	if (!loadXMLSpec (name, ".tabledef", doc, pError))
		return	false	;

	KBTableSpec	tabSpec  (doc.documentElement().firstChild().toElement()) ;

	if (m_dropObjects.isChecked() && exists)
		if (!m_dbLink.dropTable (name, false))
		{	pError	= m_dbLink.lastError() ;
			return	false	;
		}

	if (!m_dbLink.createTable (tabSpec, false))
	{	pError = m_dbLink.lastError() ;
		return	 false	;
	}

	return	true	;
}

/*  KBLoader								*/
/*  loadViewDef	: Load view definition					*/
/*  name	: const QString & : View/definition file name		*/
/*  exists	: bool		  : View already exists			*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBLoader::loadViewDef
	(	const QString	&name,
		bool		exists,
		KBError		&pError
	)
{
	QDomDocument	doc	;
	if (!loadXMLSpec (name, ".viewdef", doc, pError))
		return	false	;

	KBTableSpec	viewSpec  (doc.documentElement().firstChild().toElement()) ;

	if (m_dropObjects.isChecked() && exists)
		if (!m_dbLink.dropView (name))
		{	pError	= m_dbLink.lastError() ;
			return	false	;
		}


	if (!m_dbLink.createView (viewSpec))
	{	pError = m_dbLink.lastError() ;
		return	 false	;
	}

	return	true	;
}

/*  KBLoader								*/
/*  loadSequenceDef							*/
/*		: Load sequence definition				*/
/*  name	: const QString & : Sequence/definition file name	*/
/*  exists	: bool		  : Sequence already exists		*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBLoader::loadSequenceDef
	(	const QString	&name,
		bool		exists,
		KBError		&pError
	)
{
	QDomDocument	doc	;
	if (!loadXMLSpec (name, ".seqdef", doc, pError))
		return	false	;

	KBSequenceSpec	seqSpec  (doc.documentElement().firstChild().toElement()) ;

	if (m_dropObjects.isChecked() && exists)
		if (!m_dbLink.dropSequence (name))
		{	pError	= m_dbLink.lastError() ;
			return	false	;
		}

	if (!m_dbLink.createSequence (seqSpec))
	{	pError = m_dbLink.lastError() ;
		return	 false	;
	}

	return	true	;
}

/*  KBLoader								*/
/*  loadTableData: Load table data					*/
/*  name	 : const QString & : Table/data file name		*/
/*  exists	 : bool		   : Table already exists		*/
/*  pError	 : KBError &	   : Error return			*/
/*  (returns)	 : bool		   : Success				*/

bool	KBLoader::loadTableData
	(	const QString	&name,
		bool		,
		KBError		&pError
	)
{
	KBTableSpec	tabSpec	(name) ;
	if (!m_dbLink.listFields(tabSpec))
	{	pError	= m_dbLink.lastError() ;
		return	false	;
	}

	/* The work is done using a copier. Create a dummy location and	*/
	/* thence an XML source and an table destination. These can be	*/
	/* appropriately initialised with the table structure.		*/
	KBLocation	location   (m_dbInfo, "copier", m_server, "unnamed") ;
	KBCopyXML	*copyXML   = new KBCopyXML   (true,  location) ;
	KBCopyTable	*copyTable = new KBCopyTable (false, location) ;

	copyXML	 ->setMainTag (name) ;
	copyXML	 ->setRowTag  ("row") ;
	copyXML	 ->setErrOpt  (KBCopyXML::ErrPad) ;
	copyXML	 ->setFile    (m_dir + "/" + name + ".tabledata") ;

	copyTable->setServer  (m_server) ;
	copyTable->setTable   (name) ;
	copyTable->setOption  (KBCopyTable::OptReplace, "") ;


	for (uint idx = 0 ; idx < tabSpec.m_fldList.count() ; idx += 1)
	{
		KBFieldSpec *fSpec = tabSpec.m_fldList.at(idx) ;

		copyXML  ->addField (fSpec->m_name, false) ;
		copyTable->addField (fSpec->m_name) ;
	}

	/* Now create the actual copier and execute it. Most of the	*/
	/* arguments are dummy in this situation.			*/
	KBCopyExecLoader  copy   (copyXML, copyTable, this) ;
	QString	   	  report ;
	int	   	  nRows  ;
	QDict<QString>	  d1	 ;
	QDict<KBParamSet> d2	 ;
	if (!copy.execute
		(	report,
			pError,
			nRows,
			d1, d2,
			false
		))
	{
		return	false	;
	}

	return	true	;
}


bool	KBLoader::loadDetails
	(	KBLoaderItem	*item
	)
{
	const QString	&name	= item->name  () ;
	uint		flags	= item->flags () ;
	bool		exists	= item->exists() ;
	KBError		error	;

	m_loading.setText (name) ;
	m_record .setText (""  ) ;

	m_of.setText
	(	QString(TR("%1 of %2"))
			.arg(m_index + 1)
			.arg(m_tableList.childCount())
	)	;

	m_tableList.ensureItemVisible (m_currItem) ;
	m_tableList.setCurrentItem    (m_currItem) ;

	qApp->processEvents() ;


	if ((flags & (FL_TABLEDEF|FL_TABLEDATA)) != 0)
	{
		/* In the event of an error, display the error and	*/
		/* simulate the user clicking the cancel button, to	*/
		/* terminate the dialog.				*/
		if (m_loadDefn.isChecked() && ((flags & FL_TABLEDEF ) != 0))
			if (!loadTableDef  (name, exists, error))
			{	error.DISPLAY() ;
				return	false	;
			}

		if (m_loadData.isChecked() && ((flags & FL_TABLEDATA) != 0))
			if (!loadTableData (name, exists, error))
			{	error.DISPLAY() ;
				return	false	;
			}

		return	true	;
	}

	if ((flags & FL_VIEWDEF) != 0)
	{
		if (m_loadDefn.isChecked())
			if (!loadViewDef  (name, exists, error))
			{	error.DISPLAY() ;
				return	false	;
			}

		return	true	;
	}

	if ((flags & FL_SEQDEF) != 0)
	{
		if (m_loadDefn.isChecked())
			if (!loadSequenceDef  (name, exists, error))
			{	error.DISPLAY() ;
				return	false	;
			}

		return	true	;
	}

	return	true	;
}

/*  KBLoader								*/
/*  slotTimer	: Start dump of next object				*/
/*  (returns)	: void		:					*/

void	KBLoader::slotTimer ()
{
	/* Look for the next object to dump. The loop will exit if we	*/
	/* run out of objects before finding a candidate.		*/
	while (m_currItem != 0)
	{
		bool	pause	= false	;

		if (m_loadAll.isChecked() || m_currItem->isOn())
		{
			if (!loadDetails (m_currItem))
			{
				reject	() ;
				return	;
			}

			pause	= true	;
		}

		m_currItem = (KBLoaderItem *)m_currItem->nextSibling() ;
		m_index   += 1 ;

		if (pause)
		{
			QTimer::singleShot (200, this, SLOT(slotTimer())) ;
			qApp->processEvents() ;
			return	;
		}
	}

	m_bOK	 .setEnabled (true ) ;
	m_bCancel.setEnabled (false) ;
	m_finished = true ;
}

bool	KBLoader::showProgress
	(	int		nRows
	)
{
	if (nRows % 10 == 0)
	{
		m_record.setText (QString::number(nRows)) ;
		qApp->processEvents() ;
	}

	return	m_cancelled ;
}

/*  loadDatabase: Load database from a set of files			*/
/*  dbInfo	: KBDBInfo *	  : Database information		*/
/*  server	: const QString & : Server name				*/
/*  dir		: const QString & : Destination directory		*/
/*  (returns)	: void		  :					*/

LIBKBASE_API	void	loadDatabase
	(	KBDBInfo	*dbInfo,
		const QString	&server,
		const QString	&dir
	)
{
	KBLoader (dbInfo, server, dir).exec() ;
}
