/* imv.c - Main routines for imv.
 *
 * Copyright (C) 2001  Oskar Liljeblad
 *
 * This file is part of the file renaming utilities (renameutils).
 *
 * This software is copyrighted work licensed under the terms of the
 * GNU General Public License. Please consult the file `COPYING' for
 * details.
 */

#include <config.h>
#include <unistd.h>
#include <signal.h>
#include <stdio.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <errno.h>
#include "common/memory.h"
#include "common/error.h"
#include "common/io-utils.h"
#include "common/string-utils.h"
#include "common/common.h"

#define PROGRAM "imv"

static char *sourcefile = NULL;
static char *first_text = NULL;

static int
insert_first_text(void)
{
	return rl_insert_text(first_text);
}

static void
int_signal_handler(int signal)
{
	puts("");
	warn("no changes made");
	exit(0);
}

static void
display_imv_help(void)
{
	printf("Usage: %s [OPTION]... FILE\n\
Rename a file by editing the destination name using GNU readline.\n\
All options except the options listed below are passed to mv(1).\n\
\n\
      --help                 display this help and exit\n\
      --version              output version information and exit\n\
\n\
Report bugs to <%s>.\n", program_invocation_name, BUG_EMAIL);
}

static void
display_imv_version(void)
{
	printf("%s (%s) %s\n\
Written by %s.\n\
\n\
Copyright (C) %s.\n\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n",
		PROGRAM, PACKAGE, VERSION, AUTHORS, COPYRIGHT);
}

int
main(int argc, char **argv)
{
	struct sigaction action;
	char *newname;
	bool gotend = FALSE;
	int c;
	int len;

	if (argc == 2 && strcmp(argv[1], "--help") == 0) {
		display_imv_help();
		exit(0);
	}
	if (argc == 2 && strcmp(argv[1], "--version") == 0) {
		display_imv_version();
		exit(0);
	}

	memclear(&action, sizeof(sigaction));
	action.sa_handler = int_signal_handler;
	action.sa_flags = SA_RESTART;
	if (sigaction(SIGINT, &action, NULL) < 0)
		die_errno(NULL);
	action.sa_handler = SIG_IGN;
	if (sigaction(SIGQUIT, &action, NULL) < 0)
		die_errno(NULL);

	for (c = 1; c < argc; c++) {
		if (!gotend && strcmp(argv[c], "--") == 0) {
			gotend = TRUE;
		} else if (gotend || argv[c][0] != '-') {
			if (sourcefile != NULL)
				die("too many arguments");
			sourcefile = argv[c];
		}
	}

	if (sourcefile == NULL)
		die("missing file argument");

	len = strlen(sourcefile);
	if (sourcefile[len-1] == '/')
		sourcefile[len-1] = '\0';
	first_text = xstrdup(sourcefile);

	do {
		add_history(sourcefile);
		rl_startup_hook = insert_first_text;
		newname = readline("> ");

		if (newname != NULL
				&& strcmp(newname, "") != 0
				&& strcmp(newname, sourcefile) != 0) {
			char **args;

			if (file_exists(newname)) {
				char *line;
				printf("%s: overwrite `%s'? ", program_invocation_name, newname);
				fflush(stdout);
				line = read_line(stdin);
				if (line == NULL) {
					free(newname);
					newname = NULL;
					break;
				}
				chomp(line);
				if (strcmp(line, "y") != 0) {
					free(line);
					free(first_text);
					first_text = newname;
					continue;
				}
			}

			args = xmalloc(argc + (gotend ? 2 : 3));
			memcpy(args, argv, argc * sizeof(char *));
			if (gotend) {
				args[argc] = newname;
				args[argc+1] = NULL;
			} else {
				args[argc] = "--";
				args[argc+1] = newname;
				args[argc+2] = NULL;
			}
			args[0] = "mv";
			execvp("mv", args);
			die("cannot execute mv");
		}
	} while (FALSE);

	if (newname == NULL)
		puts("");

	warn("no changes made");
	exit(0);
}
