/* qmv.c - Main routines for qmv.
 *
 * Copyright (C) 2001  Oskar Liljeblad
 *
 * This file is part of the file renaming utilities (renameutils).
 *
 * This software is copyrighted work licensed under the terms of the
 * GNU General Public License. Please consult the file `COPYING' for
 * details.
 */

#include <config.h>
#include <unistd.h>
#include <locale.h>
#include <getopt.h>
#include <stdlib.h>
#include <errno.h>
#include "common/io-utils.h"
#include "common/memory.h"
#include "common/string-utils.h"
#include "common/error.h"
#include "common/llist.h"
#include "qmv.h"

bool dummy = FALSE;
bool show_control_chars = FALSE;
bool verbose = FALSE;
LList *renames;
char *renames_directory;
char *editor_program = NULL;
char *edit_filename = NULL;
EditFormat *format = &dual_column_format;
RenamePlan *plan = NULL;

#define SHOW_CTRL_OPT	1003
#define DUMMY_OPT		1004
#define VERSION_OPT		1005
#define HELP_OPT		1006

static struct option qmv_option_table[] = {
	{ "format",				required_argument,	NULL,	'f',			},
	{ "options",			required_argument,	NULL,	'o'				},
	{ "show-control-chars",	no_argument,		NULL,	SHOW_CTRL_OPT	},
	{ "editor",				required_argument,	NULL,	'e'				},
	{ "verbose",			no_argument,		NULL,	'v'				},
	{ "dummy",				no_argument,		NULL,	DUMMY_OPT		},
	{ "interactive",		no_argument,		NULL,	'i'				},
	{ "version",			no_argument,		NULL,	VERSION_OPT		},
	{ "help",				no_argument,		NULL,	HELP_OPT		},
	{ 0,					0,					0,		0				},
};

void
display_qmv_version(void)
{
	printf("%s (%s) %s\n\
Written by %s.\n\
\n\
Copyright (C) %s.\n\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n",
		PROGRAM, PACKAGE, VERSION, AUTHORS, COPYRIGHT);
}

void
display_qmv_help(FILE *out)
{
	fprintf(out, "Usage: %s [OPTION]... [FILE]...\n\
Rename files quickly, editing the file names in a text editor.\n\
\n", program_invocation_name);
	display_ls_help(out);
	fprintf(out, "\n\
Format-related options:\n\
  -f, --format=FORMAT        change edit format of text file\n\
  -o, --options=OPTIONS      pass options to the edit format\n\
      --show-control-chars   show non graphic characters as-is\n\
\n\
Other options:\n\
  -i, --interactive          start in command mode\n\
  -e, --editor=EDITOR        program to edit text file with\n\
  -v, --verbose              be more verbose\n\
      --dummy                don't rename (\"dummy\" mode)\n\
\n\
General options:\n\
      --help                 display this help and exit\n\
      --version              output version information and exit\n\
\n\
Possible values for --format are: `single-column', `dual-column',\n\
and `destination-only'. For a list of available options for each\n\
format, use --options=help.\n\
\n\
Report bugs to <%s>.\n", BUG_EMAIL);
}

bool
set_show_control_chars(bool value)
{
	show_control_chars = value;
	if (!value) {
		if (setlocale (LC_ALL, "") == NULL) {
			warn_errno ("cannot set default locale");
			return FALSE;
		}
	} else {
		if (setlocale (LC_ALL, "C") == NULL) {
			warn_errno ("cannot set C locale");
			return FALSE;
		}
	}
	return TRUE;
}

int
main(int argc, char **argv)
{
	struct option *all_options;
	char *format_options = NULL;
	bool interactive = FALSE;
	int c;

	renames_directory = xstrdup(".");
	renames = llist_new();

	/* Initialize option table, and parse options */
	all_options = append_ls_options(qmv_option_table);
	while (TRUE) {
		c = getopt_long (argc, argv, "aABcdrRStuUXvf:o:e:i",
				qmv_option_table, NULL);
		if (c == -1)
			break;

		switch (c) {
		case '?':
			exit(1);
		case 'i': /* --interactive */
			interactive = TRUE;
			break;
		case 'v': /* --verbose */
			verbose = TRUE;
			break;
		case 'f': /* --format */
			format = find_edit_format_by_name(optarg);
			if (format == NULL)
				die("no such edit format `%s'", optarg); /* OK */
			break;
		case 'o': /* options */
			format_options = optarg;
			break;
		case 'e': /* --editor */
			editor_program = optarg;
			break;
		case SHOW_CTRL_OPT: /* --show-control-chars */
			show_control_chars = TRUE;
			break;
		case DUMMY_OPT:
			dummy = TRUE;
			break;
		case VERSION_OPT:
			display_qmv_version();
			exit(0);
			break;
		case HELP_OPT:
			display_qmv_help(stdout);
			exit(0);
			break;
		default:
			process_ls_option(c);
			break;
		}
	}

	/* Create a temporary file for editing */
	edit_filename = create_temporary_file(PROGRAM);
	if (edit_filename == NULL)
		die_errno("cannot create temporary file"); /* OK */

	/* Find a suitable editor program */
	if (editor_program == NULL
			&& (editor_program = getenv("VISUAL")) == NULL
			&& (editor_program = getenv("EDITOR")) == NULL)
		editor_program = "editor";
	editor_program = xstrdup(editor_program);

	/* Initialize locale if necessary */
	if (!set_show_control_chars(show_control_chars))
		exit(1);

	/* Parse format options */
	if (format_options != NULL) {
		if (!format->parse_options(format_options))
			exit(0);
	}
	
	if (interactive) {
		display_commandmode_header();
		if (argv[optind] != NULL) {
			/*list_command(argv+optind-1);*/
			list_files(argv+optind);
		}
		commandmode_mainloop();
	} else {
		if (list_files(argv+optind)) {
			if (llist_is_empty(renames)) {
				warn("no files to rename");
			} else if (!edit_renames(FALSE)
					|| !llist_is_empty(plan->error)
					|| (!llist_is_empty(plan->ok) && !apply_plan(plan))) {
				printf("Entering interactive mode.\n");
				commandmode_mainloop();
			}
		}
	}

	if (plan != NULL)
		free_plan(plan);

	free_renames(renames);
	free(all_options);
	free(editor_program);
	unlink(edit_filename);	/* Ignore errors */
	free(edit_filename);
	free(renames_directory);

	return 0;
}
