#! /bin/sh
# A script to clean or rebuild files created by autoconf, automake,
# gettext and gnulib. Run autogen.sh --help for usage.

set -e

GNULIBTOOL=/usr/share/gnulib/gnulib-tool
GNULIBPATH=/usr/share/gnulib
GETTEXTPATH=/usr/share/gettext

PACKAGE=renameutils

function contains () {
  local find="$1"
  shift
  for str in "$@"; do
    if [ "$str" = "$find" ]; then
      return 0
    fi
  done
  return 1
}

function expand_module_dependencies () {
  local knownlist="$@"
  local checklist="$@"

  while [ -n "$checklist" ]; do
    newlist=`$GNULIBTOOL --extract-dependencies $checklist`
    checklist=""
    for module in $newlist; do
      if ! contains $module $knownlist; then
         knownlist="$knownlist $module"
         checklist="$checklist $module"
      fi
    done
  done

  echo $knownlist
}

if contains --help "$@"; then
  echo "Usage: $0 --clean|--generate|--help"
  exit
fi

modules=`cat modules.gnulib | egrep -v '^[[:space:]]*#'`
modules=`expand_module_dependencies $modules`
filelist=`$GNULIBTOOL --extract-filelist $modules | sort | uniq`

if contains --clean "$@"; then
  cleanup=1
  if contains --generate "$@"; then
    regenerate=1
  else
    regenerate=0
  fi
else
  cleanup=0
  regenerate=1
fi

if [ "$cleanup" = "1" ]; then
  if [ -f Makefile ]; then
    make maintainer-clean || :
  fi
  echo Removing generated files
  # Generated by autoheader
  rm -f config.h.in config.h.in~
  # Generated by aclocal
  rm -f aclocal.m4
  # Added by automake -a
  rm -f COPYING INSTALL compile config.guess config.sub depcomp install-sh missing
  # Generated by automake
  rm -f Makefile.in
  rm -f lib/Makefile.in
  rm -f m4/Makefile.in
  rm -f src/Makefile.in src/editformats/Makefile.in src/common/Makefile.in
  # Generated by autoconf
  rm -f configure
  # Added by gettextize
  rm -f ABOUT-NLS config.rpath mkinstalldirs
  rm -f po/Makefile.in.in po/Rules-quot po/boldquot.sed
  rm -f po/en@boldquot.header po/en@quot.header
  rm -f po/insert-header.sin po/quot.sed po/remove-potcdate.sin
  # Generated by make at build time
  rm -f po/$PACKAGE.pot
  # Generated manually
  rm -f m4/Makefile.am
  rm -f lib/Makefile.am
  # Added by gnulib
  echo Removing gnulib files
  for file in $filelist; do
    if [ "${file:0:3}" = "m4/" -o "${file:0:4}" = "lib/" ]; then
      rm -f "$file"
    fi
  done
fi

if [ "$regenerate" = "1" ]; then
  echo "Adding symbolic links to gnulib files"
  for file in $filelist; do
    if [ "${file:0:3}" = "m4/" -o "${file:0:4}" = "lib/" ]; then
      rm -f $file
      ln -s $GNULIBPATH/$file $file
    fi
  done

  echo "Creating m4/Makefile.am"
  (cat m4/Makefile.am.head &&
   for file in $filelist; do
     if [ "${file:0:3}" = "m4/" ]; then
       echo "EXTRA_DIST += ${file:3}"
     fi
   done) > m4/Makefile.am

  echo "Creating lib/Makefile.am"
  (cat lib/Makefile.am.head &&
   $GNULIBTOOL --extract-automake-snippet --lib libgnu $modules |
   sed -e "s,lib_SOURCES,libgnu_a_SOURCES,g" -e "s,lib_OBJECTS,libgnu_a_OBJECTS,g") > lib/Makefile.am

  #echo "Running gettextize"
  #gettextize --no-changelog
  echo "Installing gettext files"
  for file in mkinstalldirs config.rpath ABOUT-NLS; do
    rm -f $file
    ln -s $GETTEXTPATH/$file $file
  done
  for file in $GETTEXTPATH/po/*; do
    base=`basename "$file"`
    rm -f po/$base
    ln -s $file po/$base
  done

  echo "Updating gnulib commands in configure.ac"
  (state=0
   cat configure.ac | while IFS="" read line; do
     if [ $state = 0 ]; then
       echo "$line"
       if [ "$line" = "# Gnulib" ]; then
         echo gl_USE_SYSTEM_EXTENSIONS
         $GNULIBTOOL --extract-autoconf-snippet --lib libgnu $modules | grep -v '^[ \t]*$'
         state=1
       fi
     elif [ $state = 1 ]; then
       if [ "$line" = "" ]; then
         echo
         state=0
       fi
     fi
   done) > configure.ac.tmp
  mv configure.ac.tmp configure.ac

  #echo "Running autoreconf"
  #autoreconf

  echo "Running autoheader"
  autoheader

  echo "Running aclocal"
  aclocal -I m4

  echo "Running automake"
  automake -a

  echo "Running autoconf"
  autoconf
fi
