/* commandmode.c - Interpreter and completion for the interactive mode.
 *
 * Copyright (C) 2001  Oskar Liljeblad
 *
 * This file is part of the file renaming utilities (renameutils).
 *
 * This software is copyrighted work licensed under the terms of the
 * GNU General Public License. Please consult the file `COPYING' for
 * details.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif
/* As recommended by autoconf for AC_HEADER_SYS_WAIT */
#include <sys/types.h>
#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
# define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif
/* POSIX */
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_WORDEXP_H
#include <wordexp.h>
#endif
/* C89 */
#include <signal.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
/* POSIX/gnulib */
#include <stdbool.h>
/* GNU Readline */
#include <readline/readline.h>
#include <readline/history.h>
/* gettext */
#include <gettext.h> /* will include <libintl.h> if ENABLE_NLS */
#define _(String) gettext(String)
/* Common */
#include "common/common.h"
#include "common/memory.h"
#include "common/string-utils.h"
#include "common/error.h"
/* qmv */
#include "qmv.h"

struct Command {
	char *word;
	void (*handler)(char **);
};

static int command_compare(const struct Command *c1, const struct Command *c2);
static char *qmv_command_generator(const char *text, int state);
static char **qmv_completion(const char *text, int start, int end);
static void help_command(char **args);
static void quit_command(char **args);
static void version_command(char **args);

/* This array must be sorted by the `word' field */
static struct Command commands[] = {
	{ "apply",      apply_command },
	{ "ed",         edit_command },
	{ "edit",       edit_command },
	{ "exit",       quit_command },
	{ "help",       help_command },
	{ "list",       list_command },
	{ "ls", 	    list_command },
	{ "plan",		plan_command },
	{ "quit",       quit_command },
	{ "retry",		retry_command },
	{ "set",        set_command },
	{ "show",       show_command },
	{ "version",	version_command },
};
static bool running = true;
static bool exit_warned = false;

static int
command_compare(const struct Command *c1, const struct Command *c2)
{
	return strcmp(c1->word, c2->word);
}

static char *
qmv_command_generator(const char *text, int state)
{
	static int c;

	if (state == 0)
		c = 0;

	while (c < sizeof(commands)/sizeof(*commands)) {
		char *name = commands[c].word;
		c++;
		if (starts_with(name, text))
			return xstrdup(name);
	}

	return NULL;
}

static char **
qmv_completion(const char *text, int start, int end)
{
	int idx = word_get_index(rl_line_buffer, start);

	if (idx == 0) {
		return rl_completion_matches(text, qmv_command_generator);
	} else if (idx == 1) {
		char *command = word_get(rl_line_buffer, 0);
		if (strcmp(command, "set") == 0 || strcmp(command, "show") == 0) {
			free(command);
			return rl_completion_matches(text, qmv_variable_generator);
		}
		free(command);
	} else if (idx == 2) {
		char *command = word_get(rl_line_buffer, 0);
		if (strcmp(command, "set") == 0) {
			char *variable = word_get(rl_line_buffer, 1);
			if (strcmp(variable, "format") == 0) {
				free(command);
				free(variable);
				return rl_completion_matches(text, edit_format_generator);
			}
			else if (strcmp(variable, "options") == 0) {
				free(command);
				free(variable);
				return rl_completion_matches(text, format->option_generator);
			}
			free(variable);
		}
		free(command);
	}

	return NULL;
}

void
display_commandmode_header(void)
{
	printf(_("%s (%s) %s\n\
Copyright (C) 2001-2004 Oskar Liljeblad.\n\
This program is free software, covered by the GNU General Public License,\n\
and you are welcome to change it and/or distribute copies of it under\n\
certain conditions. There is absolutely no warranty for this program.\n\
See the COPYING file for details.\n"),
		PROGRAM, PACKAGE, VERSION);
}

void
terminate_program(void)
{
	bool valid = false;

	if (plan != NULL && llist_is_empty(plan->error)) {
		Iterator *it;

		valid = true;
		if (!llist_is_empty(plan->ok)) {
			for (it = llist_iterator(plan->ok); iterator_has_next(it); ) {
				FileSpec *spec = iterator_next(it);
				if (spec->renamed != RENAME_COMPLETE) {
					valid = false;
					break;
				}
			}
			iterator_free(it);
		}
	}
	if (valid || plan == NULL || exit_warned) {
		running = false;
		puts("");
		return;
	}
	puts("exit");
	printf(_("There are unsaved changes.\n"));
	exit_warned = true;
}

static void
int_signal_handler(int signal)
{
	puts("");
	rl_line_buffer[0] = '\0';
	rl_point = 0;
	rl_end = 0;
	rl_forced_update_display();
}

void
commandmode_mainloop(void)
{
	struct sigaction action;

	rl_readline_name = program_invocation_name;
	rl_attempted_completion_function = qmv_completion;
	rl_basic_word_break_characters = " \t\n\"\\'`@$><=;|&{(,";

	memclear(&action, sizeof(sigaction));
	action.sa_handler = int_signal_handler;
	action.sa_flags = SA_RESTART;
	if (sigaction(SIGINT, &action, NULL) < 0)
		die_errno(NULL);
	action.sa_handler = SIG_IGN;
	if (sigaction(SIGQUIT, &action, NULL) < 0)
		die_errno(NULL);

	while (running) {
		char *line;
		char **words;
		wordexp_t result;
		int rc;

		line = readline(PROGRAM "> ");
		if (line == NULL) {
			terminate_program();
			continue;
		}
		add_history(line);
		rc = wordexp(line, &result, WRDE_NOCMD|WRDE_UNDEF);
		if (rc != 0) {
			free(line);
			switch (rc) {
			case WRDE_BADCHAR:
				warn(_("input contains unquoted invalid character"));
				break;
			case WRDE_BADVAL:
				warn(_("variable reference using dollar sign ($) is not allowed"));
				break;
			case WRDE_CMDSUB:
				warn(_("command substitution using backticks (``) is not allowed"));
				break;
			case WRDE_NOSPACE:
				errno = ENOMEM;
				die_errno(NULL);
				break;
			case WRDE_SYNTAX:
				warn(_("syntax error in input"));
				break;
			}
			continue;
		}
		words = result.we_wordv;
		if (words != NULL && *words != NULL) {
			struct Command *result;
			struct Command target;

			target.word = words[0];
			result = bsearch(&target, commands,
					sizeof(commands)/sizeof(*commands),
					sizeof(*commands), (comparison_fn_t) command_compare);
			if (result == NULL)
				warn(_("undefined command `%s'. Try `help'."), words[0]);
			else
				result->handler(words);
		}
		if (words != NULL)
			wordfree(&result);
		free(line);
	}
}

static void
quit_command(char **args)
{
	running = false;
}

static FILE *
launch_pager(pid_t *child_pid)
{
	FILE *pager;
	pid_t pid;
	int pager_fd[2];

	*child_pid = -1;
	if (pipe(pager_fd) != 0) {
		warn_errno(_("cannot start pager"));
		return stdout;
	}

	pid = fork();
	if (pid < 0) {
		warn_errno(_("cannot start pager"));
		return stdout;
	}
	if (pid == 0) {
		if (close(pager_fd[1]) < 0)
			die_errno(_("cannot start pager"));
		if (dup2(pager_fd[0], STDIN_FILENO) < 0)
			die_errno(_("cannot start pager"));
		execlp("pager", "pager", NULL);
		if (errno == ENOENT)
			execlp("more", "more", NULL);
		if (errno == ENOENT)
			execlp("cat", "cat", NULL);
		die_errno(_("cannot start pager"));
	}
	*child_pid = pid;

	if (close(pager_fd[0]) < 0) {
		warn_errno(_("cannot start pager"));
		return stdout;
	}

	pager = fdopen(pager_fd[1], "w");
	if (pager == NULL) {
		warn_errno(_("cannot start pager"));
		return stdout;
	}
	return pager;
}

static bool
close_pager(FILE *pager, pid_t child_pid)
{
	if (pager != stdout && fclose(pager) < 0) {
		warn_errno(_("cannot terminate pager"));
		return false;
	}
	if (child_pid != -1 && waitpid(child_pid, NULL, 0) != child_pid) {
		warn_errno(_("cannot terminate pager"));
		return false;
	}
	return true;
}

static void
help_command(char **args)
{
	FILE *pager;
	pid_t child;

	if (args[1] == NULL) {
		pager = launch_pager(&child);
		fprintf(pager, _("\
ls, list [OPTIONS].. [FILES]..\n\
  Select files to rename. If no files are specified, select all files in\n\
  current directory. Use `help ls' to display a list of allowed options.\n\
ed, edit [all]\n\
  Edit renames in a text editor. If this command has been run before, and\n\
  not `all' is specified, only edit renames with errors in.\n\
plan\n\
  Display the current rename-plan. (This plan is created after `edit'.)\n\
apply\n\
  Apply the current plan, i.e. rename files. Only those files marked as OK\n\
  in the plan will be renamed.\n\
retry\n\
  If some rename failed earlier during `apply', this command will try those\n\
  renames again.\n\
show [VARIABLE]\n\
  Display the value of the specified variable, or all variables if none\n\
  specified.\n\
set VARIABLE VALUE\n\
  Set the value of a variable.\n\
exit, quit\n\
  Exit this program\n\
help [ls|usage]\n\
  If `ls' is specified, display list options. If `usage' is specified,\n\
  display accepted command line options. Otherwise display this help\n\
  message.\n\
version\n\
  Display version information for this program.\n"));
		close_pager(pager, child);
	} else if (strcmp(args[1], "ls") == 0) {
		pager = launch_pager(&child);
		display_ls_help(pager);
		close_pager(pager, child);
	} else if (strcmp(args[1], "usage") == 0) {
		pager = launch_pager(&child);
		display_qmv_help(pager);
		close_pager(pager, child);
	} else {
		printf(_("Usage: help [ls|usage]\n"));
	}
}

static void
version_command(char **args)
{
	display_qmv_version();
	printf(_("\nSend bug reports and questions to <%s>.\n"), PACKAGE_BUGREPORT);
}
