/* destination-only.c - "destination-only" format.
 *
 * Copyright (C) 2001  Oskar Liljeblad
 *
 * This file is part of the file renaming utilities (renameutils).
 *
 * This software is copyrighted work licensed under the terms of the
 * GNU General Public License. Please consult the file `COPYING' for
 * details.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif
/* POSIX/gnulib */
#include <stdbool.h>
/* C89 */
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
/* gettext */
#include <gettext.h> /* will include <libintl.h> if ENABLE_NLS */
#define _(String) gettext(String)
/* common */
#include "common/common.h"
#include "common/io-utils.h"
#include "common/string-utils.h"
#include "common/error.h"
#include "common/llist.h"
#include "common/memory.h"
/* qmv */
#include "qmv.h"

static bool parse_options(char *options);
static char *option_generator(const char *text, int state);
static void output(FILE *out, LList *files);
static bool input(FILE *in, LList *files);

static bool separate = false;
static char *option_array[] = {
	"help",
	"separate",
	NULL
};
enum {
	HELP_OPT,
	SEPARATE_OPT,
};

#define NO_ARGUMENT(name) \
	if (value != NULL) { \
		warn(_("suboption `%s' for --option does not take an argument"), #name); \
		reset_options(); \
		return false; \
	}

EditFormat destination_only_format = {
	"destination-only",
	"do",
	parse_options,
	option_generator,
	output,
	input
};

static void
reset_options()
{
	separate = false;
}

static bool
parse_options(char *options)
{
	reset_options();

	while (*options != '\0') {
		char *value;
		int c = getsubopt(&options, option_array, &value);

		if (c == -1) {
			warn(_("unknown suboption for --option: %s"), value);
			reset_options();
			return false;
		}

		switch (c) {
		case HELP_OPT:
			printf(_("\
Available options for `dual-column' format:\n\
\n\
  separate          put a blank line between all renames\n"));
  			return false;
		case SEPARATE_OPT:
			NO_ARGUMENT(separate);
			separate = true;
			break;
		}
	}

	return true;
}

static char *
option_generator(const char *text, int state)
{
	static int c;
	if (state == 0)
		c = 0;
	while (option_array[c] != NULL) {
		char *name = option_array[c];
		c++;
		if (starts_with(name, text))
			return xstrdup(name);
	}
	return NULL;
}

static void
output(FILE *out, LList *files)
{
	int c = 0;
	Iterator *it;

	for (it = llist_iterator(files); iterator_has_next(it); c++) {
		FileSpec *spec = iterator_next(it);
		char *file;

		file = quote_output_file(spec->new_name);
		fprintf(out, "%s\n", file);
		if (separate && iterator_has_next(it))
			fprintf(out, "\n");
		free(file);
	}
	iterator_free(it);
}

static bool
input(FILE *in, LList *files)
{
	int c = 0;
	Iterator *it;

	for (it = llist_iterator(files); iterator_has_next(it); c++) {
		FileSpec *spec = iterator_next(it);
		char *file_new;
		char *tmp;
		
		file_new = read_line(in);
		if (file_new == NULL) {
			if (ferror(in))
				warn_errno(_("cannot read %s"), edit_filename);
			else
				warn(_("premature end of %s"), edit_filename);
			/* XXX: revert changes */
			iterator_free(it);
			return false;
		}

		if (separate && iterator_has_next(it)) {
			char *empty_line;

			empty_line = read_line(in);
			if (empty_line == NULL) {
				if (ferror(in))
					warn_errno(NULL);
				else
					warn(_("premature end of %s"), edit_filename);
				/* XXX: revert changes */
				iterator_free(it);
				return false;
			}
			chomp(empty_line);
			if (empty_line[0] != '\0') {
				warn(_("expected empty line in file, got `%s'\n"), empty_line);
				/* XXX: revert changes */
				iterator_free(it);
				return false;
			}
			free(empty_line);
		}

		chomp(file_new);

		tmp = dequote_output_file(file_new);
		free(file_new);
		file_new = tmp;

		if (file_new[0] == '\0') {
			warn(_("new filename for %s is empty - not updating name"), spec->old_name);
			/* Don't return */
		} else {
			spec->new_name = file_new;
		}
	}
	iterator_free(it);

	return true;
}
