# Copyright (c) 2002 Red Hat, Inc. All rights reserved.
#
# This software may be freely redistributed under the terms of the
# GNU General Public License.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# TCL code to raise a dialog box for VACUUM & ANALYZE
#
# Written for Red Hat Inc. by Permaine Cheung <pcheung@redhat.com>
#
# Component of: Red Hat Database GUI Administration tool

package require Itcl
package require Itk
package require Iwidgets
package require megaMaintenanceDialog
package provide maintenanceDialogRaiser 1.0

proc maintenanceDialogRaise { args } {

    set errorLevel 0

    # Create the maintenance window

    toplevel .maintenance
    
    # Withdraw the window while we pack it

    wm withdraw .maintenance

    # Disable the main program window

    wm transient .maintenance .
    grab .maintenance

    # Create a frame to hold the maintenanceDialog
    
    frame .maintenance.hostedDialog -relief groove -borderwidth 1

    # Place the maintenanceDialog in the window inside the frame
    
    if {[catch {eval MegaMaintenanceDialog .maintenance.hostedDialog.maintenanceDialog -parentDialog .maintenance $args} errorText]} {
	set errorLevel 1
    }

    # Create and pack OK and Cancel Buttons

    button .maintenance.okButton -text "OK" -underline 0 -command maintenanceDialog_OK
    button .maintenance.cancelButton -text "Cancel" -underline 0 -command maintenanceDialog_Cancel
   
    # Pack the frame
    if { $errorLevel == 0 } {
	pack .maintenance.hostedDialog.maintenanceDialog -fill both -expand y
	pack .maintenance.hostedDialog -side top -fill both -expand y
	pack .maintenance.cancelButton .maintenance.okButton -side right 
	bind .maintenance <Alt-o> {maintenanceDialog_OK; break}
	bind .maintenance <Alt-c> {maintenanceDialog_Cancel; break}
	bind .maintenance.okButton <Return> {maintenanceDialog_OK; break}
	bind .maintenance.cancelButton <Return> {maintenanceDialog_Cancel; break}
	bind .maintenance <Escape> {maintenanceDialog_Cancel; break}
    }

    if { $errorLevel != 0 } {
	# We need to contact the SQL pool to request that it
	# close all database connections so that we can recover from
	# the connection error.
	#
	# Note the SQL pool name is hard coded here.
	SQL_pool closeAllConnections
	# Close the dialog
	maintenanceDialog_Cancel
	# Report the error
	tk_messageBox -icon error -message "An error occurred while raising the dialog box:\n $errorText\nThe operation was aborted." -title "Error"
	return
    } else {
	# Interrogate the maintenanceDialog for the window title
	# and set it
	wm title .maintenance [.maintenance.hostedDialog.maintenanceDialog getTitle]
	wm deiconify .maintenance
    }
}

proc maintenanceDialog_OK {} {
    
    # First, validate the form

    set errorText [.maintenance.hostedDialog.maintenanceDialog validate]

    if {$errorText != "" } {

	# No, so display error message

	tk_messageBox -icon error -title "Error" \
		-message "$errorText"
	
	return
    }

    # Start the busy indicator

    status loadNotify 1

    # Now try and execute the action (apply the form)
    
    if {[catch {set errorText [.maintenance.hostedDialog.maintenanceDialog apply]} errorText2]} {
	tk_messageBox -icon error -title "Error" \
	    -message "An error occurred while processing this directive:\n$errorText2\nThe operation was aborted."
	status loadNotify 0
	return
    }

    # Stop the busy indicator

    status loadNotify 0

    # Check if everything went cleanly

    if {$errorText != "" } {

	# No, so display error message

	tk_messageBox -icon error -title "Error" \
		-message "The backend returned the error:\n $errorText"
    } else {
	
	# Yes, so lower the dialog (just use the cancel procedure)

	maintenanceDialog_Cancel
    }
    
}

proc maintenanceDialog_Cancel {} {
    destroy .maintenance
}

# End of file
