# Copyright (c) 2002 Red Hat, Inc. All rights reserved.
#
# This software may be freely redistributed under the terms of the
# GNU General Public License.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# TCL code to raise the progress dialog while VACUUMing or ANALYZing
#
# Written for Red Hat Inc. by Permaine Cheung <pcheung@redhat.com>
#
# Component of: Red Hat Database GUI Administration tool

package require Itcl
package require Itk
package require Iwidgets
package require progressBarWidget
package provide maintenanceProgress 1.0

proc maintenanceProgress {args} {
global env

    set expectedArgs \
	[list \
	-action\
	-full\
	-freeze\
	-verbose\
	-analyze\
	-dbname\
	-table\
	-columns \
	-user \
	-hostname \
	-password \
	-port \
	]

    foreach option $expectedArgs {
	set options($option) ""
    }

    foreach { option value } $args {
	if {[lsearch -exact $expectedArgs $option] == -1} {
	error "Unexpected option to maintenanceProgress \"$option\" - only expect $expectedArgs"
	}
	set options($option) "$value"
    }

    set optionString "exec psql -c"
    if { $options(-action) == "vacuum" } {
	set optionString "$optionString \"VACUUM "
	if { $options(-full) != "" } {
	    set optionString "$optionString FULL"
	}

	if { $options(-freeze) != "" } {
	    set optionString "$optionString FREEZE"
	}

	if { $options(-analyze) != "" } {
	    set optionString "$optionString ANALYZE"
	}

    }
    if { $options(-action) == "analyze" } {
	set optionString "$optionString \"ANALYZE "
    }

    if { $options(-verbose) != "" } {
	set optionString "$optionString VERBOSE"
    }

    if { $options(-table) != "" } {
	set optionString "$optionString \\\"$options(-table)\\\""
    }

    if { $options(-columns) != "" } {
	set options(-columns) [join [split $options(-columns) "\""] "\\\""]
	set optionString "$optionString ($options(-columns))"
    }

    set optionString "$optionString\""

    if { $options(-hostname) != "" } {
	set optionString "$optionString -h $options(-hostname)"
    }

    if { $options(-port) != "" } {
	set optionString "$optionString -p $options(-port)"
    }

    set optionString "$optionString -d \"$options(-dbname)\""

    if { $options(-user) != "" } {
	set optionString "$optionString -U \"$options(-user)\""
    }

    # puts "$optionString"
    set logfile "$env(HOME)/.rhdb/maintenance.log"
    if { $options(-password) != "" } {
	if {[catch {set procId [eval $optionString > /dev/null 2>$logfile << $options(-password) & ]} result]} {
	    tk_messageBox -icon error -title "Error" -message "$result"
	    return
	}
    } else {
	if {[catch {set procId [eval $optionString > /dev/null 2>$logfile & ]} result]} {
	    tk_messageBox -icon error -title "Error" -message "$result"
	    return
	}
    }

    # if result is not numeric
    if {[string is digit $result] != 1} {
	tk_messageBox -icon error -title "Error" -message "$result"
	return
    }
    set fileID [open $logfile {RDWR CREAT TRUNC}]

    # Create the maintenance window
    toplevel .progressDialog -borderwidth 20
    
    # Withdraw the window while we pack it
    wm withdraw .progressDialog

    # Disable the current window
    wm transient .progressDialog

    # disable the X
    wm protocol .progressDialog WM_DELETE_WINDOW "#"

    grab .progressDialog

    # Progress bar
    ProgressBarWidget .progressDialog.progress -steps 100 -barcolor blue

    if { $options(-action) == "vacuum" } {
	.progressDialog.progress configure -labeltext "Vacuuming ...    "
    } 
    if { $options(-action) == "analyze" } {
	.progressDialog.progress configure -labeltext "Analyzing ...    "
    } 

    if {$options(-verbose) == 1} {
	# Create text box for output if verbose is selected
	iwidgets::scrolledtext .progressDialog.outputArea \
	    -textbackground white -labelpos n -hscrollmode dynamic
	# Create Clear button
	button .progressDialog.clearButton -text "Clear" -underline 4 \
	    -command ".progressDialog.outputArea clear" -state disabled
	# Create Save button
	button .progressDialog.saveButton -text "Save" -underline 0 \
	    -command "saveOutput" -state disabled
	# Create close button
	button .progressDialog.closeButton -text "Close" -underline 0 -command \
	    "destroy .progressDialog
	    catch {destroy .saveOutput}
	    " -state disabled

	if { $options(-action) == "vacuum" } {
	    .progressDialog.outputArea configure -labeltext "Vacuum Output"
	} 
	if { $options(-action) == "analyze" } {
	    .progressDialog.outputArea configure -labeltext "Analyze Output"
	} 
	[.progressDialog.outputArea component horizsb] configure -takefocus 0
	[.progressDialog.outputArea component vertsb] configure -takefocus 0
	pack .progressDialog.progress -side top -expand 1
	pack .progressDialog.outputArea -side top -expand 1 -pady 5
	pack .progressDialog.clearButton .progressDialog.saveButton -side left
	pack .progressDialog.closeButton -side right

    } else {
	# Create close button
	button .progressDialog.closeButton -text "Close" -underline 0 -command \
	    "destroy .progressDialog
	    catch {destroy .saveOutput}
	    " -state disabled

	pack .progressDialog.progress -side top -expand 1
	pack .progressDialog.closeButton -side top
    }

    # set the window title for the progressDialog
    if { $options(-action) == "vacuum" } {
	wm title .progressDialog "VACUUM"
    }
    if { $options(-action) == "analyze" } {
	wm title .progressDialog "ANALYZE"
    }
    wm deiconify .progressDialog

    # start from beginning
    .progressDialog.progress reset
    .progressDialog.progress advance

    if {$options(-verbose) == 1} {
	if {[file size $logfile] == 0} {
	    # wait for 10 seconds and then check if any message comes back
	    after 10000
	    if {[file size $logfile] == 0} {
		tk_messageBox -icon error -title "Error" -message "Connection to server timed out"
		close $fileID
		destroy .progressDialog; catch {destroy .saveOutput}
		return
	    }
	}
	set line [gets $fileID]
	if {[string range $line 0 5] != "NOTICE"} {
	    tk_messageBox -icon error -title "Error" \
		-message "$line"
	    close $fileID
	    destroy .progressDialog
	    return
	}
    } else {
	if {[file size $logfile] != 0} {
	    # some error occurred
	    set line [gets $fileID]
	    if {$line != ""} {
		tk_messageBox -icon error -title "Error" \
		    -message "$line"
		close $fileID
		destroy .progressDialog
		return
	    }
	}
    }

    # start from the begining of file when putting the output into the
    # scrolled text widget.
    set pos 0
    while {1==1} {
	# when the process has finished
	if [catch {exec kill -0 $procId}] {
	    if {$options(-verbose) == 1} {
		set pos [updateOutputArea $fileID $pos]
	    }
	    break
	} else {
	    if {[info exists $options(-columns)]} {
		set interval 100
	    } else {
		if {[info exists $options(-table)]} {
		    set interval 1000
		} else {
		    set interval 10000
		}
	    }

	    if {$options(-verbose) == 1} {
		for {set i 0} { $i < $interval} {incr i} {
			set pos [updateOutputArea $fileID $pos]
		}
	    } else {
		after $interval
	    }
	}
	.progressDialog.progress advance
    }

    # close the file
    close $fileID

    # step to the end
    while {[.progressDialog.progress done] == 0} {
	.progressDialog.progress step
    }
    .progressDialog.closeButton configure -state normal
    if {$options(-verbose) == 1} {
	.progressDialog.saveButton configure -state normal
	.progressDialog.clearButton configure -state normal
	bind .progressDialog <Alt-r> {.progressDialog.outputArea clear; break}
	bind .progressDialog <Alt-s> {saveOutput; break}
	bind .progressDialog.clearButton <Return> {.progressDialog.outputArea clear; break}
	bind .progressDialog.saveButton <Return> {saveOutput; break}
    }

    # enable the X
    wm protocol .progressDialog WM_DELETE_WINDOW {}

    bind .progressDialog.closeButton <Return> {destroy .progressDialog; catch {destroy .saveOutput}; break}
    bind .progressDialog <Alt-c> "destroy .progressDialog; catch \
	{destroy .saveOutput}; break"
    bind .progressDialog <Escape> "destroy .progressDialog; catch \
	{destroy .saveOutput}; break"

#    if {$options(-verbose) == 1} {
#	# create the pop up menu
#	menu .progressDialog.saveMenu -tearoff 0
#	.progressDialog.saveMenu delete 0 end
#	.progressDialog.saveMenu add command -label "Save Output" \
#	    -command "saveOutput"
#	.progressDialog.saveMenu add command -label "Clear" \
#	    -command ".progressDialog.outputArea clear"
#	# bind right click to the menu after 
#	bind [.progressDialog.outputArea component text] <Button-3> \
#	    "tk_popup .progressDialog.saveMenu %X %Y"
#    }
}

proc saveOutput {} {
    # Create the maintenance window
    toplevel .saveOutput
    
    # Withdraw the window while we pack it
    wm withdraw .saveOutput

    # Disable the progress dialog window
    wm transient .saveOutput .progressDialog
    grab .saveOutput

    # Fileselection box to save the output
    iwidgets::fileselectionbox .saveOutput.fileSelection \
	-textbackground white -directory ~/ -childsitepos n

    [[.saveOutput.fileSelection component dirs] component horizsb] \
	configure -takefocus 0
    [[.saveOutput.fileSelection component dirs] component vertsb] \
	configure -takefocus 0

    [[.saveOutput.fileSelection component files] component horizsb] \
	configure -takefocus 0
    [[.saveOutput.fileSelection component files] component vertsb] \
	configure -takefocus 0

    # Create Save/Cancel button
    button .saveOutput.saveButton -text "Save" -underline 0 -command "saveToFile"

    button .saveOutput.cancelButton -text "Cancel" -underline 0 \
	-command "destroy .saveOutput"

    grid configure .saveOutput.fileSelection -row 0 -column 0 -sticky nwe \
	-columnspan 4
    grid configure .saveOutput.saveButton -row 1 -column 2 -sticky nwe
    grid configure .saveOutput.cancelButton -row 1 -column 3 -sticky nwe

    bind .saveOutput <Alt-s> {saveToFile; break}
    bind .saveOutput <Alt-c> {destroy .saveOutput; break}
    bind .saveOutput.saveButton <Return> {saveToFile; break}
    bind .saveOutput.cancelButton <Return> {destroy .saveOutput; break}
    bind .saveOutput <Escape> {destroy .saveOutput; break}

    wm title .saveOutput "Save Output to File"
    wm deiconify .saveOutput
}


proc saveToFile {} {
    set filename [file nativename [.saveOutput.fileSelection get]]
    if {$filename == ""} {
	tk_messageBox -icon error -title "Error" \
	    -message "Please enter a filename" -parent .saveOutput
	return
    } 
    # if the string doesn't start with a /, add the directory in front of it
    if {[string index $filename 0] != "/" && [string index $filename 0] != "~"} {
	set filename "[file dirname \
	[.saveOutput.fileSelection.filter get]]/$filename"
    }

    if {[file exists $filename]} {
	set overwrite [tk_messageBox -icon warning -title "Overwrite" \
	    -message "$filename already exists. Overwrite $filename?" \
	    -type yesnocancel -parent .saveOutput -default no]
	if {$overwrite == "no"} {
	    return
	}
	if {$overwrite == "cancel"} {
	    destroy .saveOutput
	    return
	}
    } 
    catch {.progressDialog.outputArea export $filename} res
    if {$res != ""} {
	tk_messageBox -icon error -title "Error" -message "$res" \
	    -parent .saveOutput
	return
    } 
    destroy .saveOutput
}

proc updateOutputArea { channel pos } {
    seek $channel $pos start
    while {[gets $channel line] >=0 } {
	catch {.progressDialog.outputArea insert end $line\n}
	set pos [tell $channel]
	seek $channel $pos start
    } 
    return $pos
}

# End of file

