///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/config.h"

#ifdef _RHEOLEF_HAVE_MPI

#include "rheolef/space.h"

namespace rheolef {

template <class T>
space_mpi_rep<T>::space_mpi_rep (const geo_basic<T,distributed>& omega, std::string approx)
  : space_rep<T,distributed>::space_rep (omega, approx),
    _idof2par_old_idof(),
    _ext_par_idof2blk_par_iub()
{
    warning_macro ("cstor space_mpi...");
    // P1: old numbering follows old vertex numbering
    size_type ndof = base::get_geo().n_vertex();
    _idof2par_old_idof.resize  (base::_idof2blk_iub.ownership());
    for (size_type idof = 0; idof < ndof; idof++) {
	_idof2par_old_idof [idof] = base::get_geo().iv2par_old_iv (idof);
	trace_macro ("idof2par_old_idof ["<<idof<<"] = "<< _idof2par_old_idof [idof]);
    }
    warning_macro ("cstor space_mpi done");
}
template <class T>
void
space_mpi_rep<T>::freeze_body () const
{
    base::base_freeze_body();
    // -----------------------------------------------------------------------
    // 1) symbolic assembly: 
    // loop on elements & identify some dofs, that are referenced
    // by locally-managed geo_elements, but these dofs are managed
    // by another processor: e.g. dofs associated to vertices on 
    // a partition boundary.
    // -----------------------------------------------------------------------
    // set local numbering with global indexes
    size_type first_par_iu = base::_iu_ownership.first_index();
    size_type first_par_ib = base::_ib_ownership.first_index();
    for (size_type idof = 0, ndof = base::_idof2blk_iub.size(); idof < ndof; idof++) {
        size_type first_par_iub = base::_idof2blk_iub [idof].is_blocked() ? first_par_ib : first_par_iu;
        base::_idof2blk_iub [idof].set_iub (base::_idof2blk_iub[idof].iub() + first_par_iub);
    }
#ifdef TO_CLEAN
    oparstream ops;
    ops.open ("tmp-space-nproc="+itos(comm().size()), "perm");
    base::_idof2blk_iub.put (ops);
    ops.close();
#endif // TO_CLEAN
    std::set<size_type> ext_dof_set;
    size_type i_first = base::ownership().first_index();
    size_type i_last  = base::ownership().last_index();
    for (size_type ie = 0, ne = base::get_geo().size(); ie < ne; ie++) {
        const geo_element& K  = base::get_geo()[ie];
        std::vector<size_type> par_idx;
        set_par_dof (K, par_idx);
        for (size_type iloc = 0, nloc = K.size(); iloc < nloc; iloc++) {
            size_type par_idof = par_idx [iloc];
	    if (par_idof >= i_first && par_idof < i_last) continue;
            ext_dof_set.insert (par_idof);
        }
    }
    // get external dof numbering:
#ifdef TO_CLEAN
    mpi_scatter_map (base::ownership(), base::_idof2blk_iub.begin(), ext_dof_set, _ext_par_idof2blk_par_iub);
#endif // TO_CLEAN
    base::_idof2blk_iub.scatter (ext_dof_set, _ext_par_idof2blk_par_iub);

    // shift localy-managed dofs to local indexes:
    for (size_type idof = 0, ndof = base::_idof2blk_iub.size(); idof < ndof; idof++) {
        size_type first_par_iub = base::_idof2blk_iub [idof].is_blocked() ? first_par_ib : first_par_iu;
        base::_idof2blk_iub [idof].set_iub (base::_idof2blk_iub[idof].iub() - first_par_iub);
    }
}
// ----------------------------------------------------------------------------
// accessors for external dofs
// ----------------------------------------------------------------------------
template <class T>
typename space_mpi_rep<T>::size_type
space_mpi_rep<T>::par_iub (size_type par_idof) const 
{
    base::freeze_guard();
    if (par_idof >= base::ownership().first_index() && par_idof < base::ownership().last_index()) {
        size_type idof = par_idof - base::ownership().first_index();
        size_type idx = base::iub (idof);
        bool      blk = base::is_blocked (idof);
        size_type idx_start = blk ? base::ib_ownership().first_index() : base::iu_ownership().first_index();
        return idx_start + idx;
    }
    // here, par_idof is not managed by current proc: try on external associative table
    typename map_pair_type::const_iterator iter = _ext_par_idof2blk_par_iub.find (par_idof);
    check_macro (iter != _ext_par_idof2blk_par_iub.end(), "unexpected dof="<<par_idof);
    return (*iter).second.iub();
}
template <class T>
bool
space_mpi_rep<T>::par_is_blocked (size_type par_idof) const
{
    base::freeze_guard();
    if (par_idof >= base::ownership().first_index() && par_idof < base::ownership().last_index()) {
        size_type idof = par_idof - base::ownership().first_index();
        return base::is_blocked (idof);
    }
    // here, par_idof is not managed by current proc: try on external associative table
    typename map_pair_type::const_iterator iter = _ext_par_idof2blk_par_iub.find (par_idof);
    check_macro (iter != _ext_par_idof2blk_par_iub.end(), "unexpected par_idof="<<par_idof);
    return (*iter).second.is_blocked();
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class space_mpi_rep<Float>;

} // namespace rheolef
#endif // _RHEOLEF_HAVE_MPI
