///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// Maquette d'une classe geo avec type elements varies (polymorphe)
//  2eme version
//
// MPI: on envoie les elements par groupes de type HPTqtep homogenes
// ce qui fait que ca reste performant du point de vue MPI
//
// limitation:
//  * Les elements ont supposes ici consecutifs.
//  * Pour s'etendre au cas des variantes curvilignes (Hc,...,qc,tc)
//    cela pose un probleme, car les elements touchant la frontiere 
//    ne sont pas consecutifs par numeros.
//    On pourrait renumeroter tous les elements apres l'identification
//    de la frontiere, mais on peut eviter cette renumerotation tardive.
//  * Pour faire sauter l'hypothese des consecutif,
//    on peut aussi envoyer une paire (index,element) puis ranger x(index) = element
//  => c'est l'etape suivante
//
// test: on construit sur le proc=0 et envoie au proc=1 qui depaquette
//
// Author: Pierre.Saramito@imag.fr
//
// Date: 14 dec 2010
//
#include "rheolef/config.h"
#ifndef _RHEOLEF_HAVE_MPI
int main() { return 0; }
#else // _RHEOLEF_HAVE_MPI

#include "rheolef/array.h"
#include <boost/array.hpp>
#include <boost/ptr_container/ptr_container.hpp>
namespace rheolef {
// =====================================================================
class geo_element : boost::noncopyable {
public:
    typedef std::vector<int>::size_type size_type;
    typedef enum {
	p = 0, 
	e = 1, 
	t = 2, 
	q = 3, 
	T = 4, 
	P = 5, 
	H = 6, 
	enum_type_max = 7 
    } enum_type; 

    static char tab_name [enum_type_max];

    virtual ~geo_element () {}
    virtual enum_type type() const = 0;
    virtual char      name() const = 0;
    virtual size_type size() const = 0;
    virtual size_type dimension() const = 0;
    virtual size_type  operator[] (size_type i) const = 0;
    virtual size_type& operator[] (size_type i)       = 0;

    virtual geo_element* do_clone() const = 0;
    geo_element* clone() const { return do_clone(); }
    geo_element* new_clone (const geo_element& K) { return K.clone(); }
};
char geo_element::tab_name [geo_element::enum_type_max] = {'p', 'e', 't', 'q', 'T', 'P', 'H'};
inline
std::ostream&
operator<< (std::ostream& os, const geo_element& K) {
	os << K.name() << "\t";
        for (geo_element::size_type iloc = 0; iloc < K.size(); iloc++) {
	  os << K[iloc];
	  if (iloc < K.size() - 1) os << " ";
	}
	return os;
}
template <class Array>
void
dump (const geo_element& K1, Array& K2) {
  	check_macro (K1.size() == K2.size(), "invalid element sizes: K1.size="<<K1.size()<<" and K2.size="<<K2.size());
        for (geo_element::size_type iloc = 0; iloc < K1.size(); iloc++) {
       	    K2[iloc] = K1[iloc];
	}
} 
class geo_element_t : public geo_element {
    static const enum_type _type =  t;
    static const char      _name = 't';
    static const size_type _size =  3;
    static const size_type _dim  =  2;
public:
    typedef boost::array<size_type,_size> array_type;

    geo_element_t () /* : geo_element() */ {
        for (size_type iloc = 0; iloc < size(); iloc++) {
	    _vertex[iloc] = std::numeric_limits<size_type>::max();
        }
    }
    geo_element_t (size_type a, size_type b, size_type c) : geo_element() {
	_vertex[0] = a;
	_vertex[1] = b;
	_vertex[2] = c;
    } 
    geo_element_t (const array_type& K) {  _vertex = K; }
    geo_element* do_clone() const {
	return new geo_element_t (_vertex[0], _vertex[1], _vertex[2]);
    }
    enum_type type() const      { return _type; };
    char      name() const      { return _name; };
    size_type size() const      { return _size; }
    size_type dimension() const { return _dim; };
    size_type  operator[] (size_type i) const { return _vertex[i]; }
    size_type& operator[] (size_type i)       { return _vertex[i]; }
    template<class Archive>
    void serialize (Archive& ar, const unsigned int version) {
        for (size_type iloc = 0; iloc < size(); iloc++) {
          ar & _vertex[iloc];
	}
    }
// data:
protected:
    array_type _vertex;
};
// =====================================================================
class geo_element_q : public geo_element {
    static const enum_type _type =  q;
    static const char      _name = 'q';
    static const size_type _size =  4;
    static const size_type _dim  =  2;
public:
    typedef boost::array<size_type,_size> array_type;
    geo_element_q () : geo_element() { 
        for (size_type iloc = 0; iloc < size(); iloc++) {
	    _vertex[iloc] = std::numeric_limits<size_type>::max();
        }
    }
    geo_element_q (size_type a, size_type b, size_type c, size_type d) : geo_element() {
	_vertex[0] = a;
	_vertex[1] = b;
	_vertex[2] = c;
	_vertex[3] = d;
    } 
    geo_element_q (const array_type& K) {  _vertex = K; }
    geo_element* do_clone() const {
	return new geo_element_q (_vertex[0], _vertex[1], _vertex[2], _vertex[3]);
    }
    enum_type type() const      { return _type; };
    char      name() const      { return _name; };
    size_type size() const      { return _size; }
    size_type dimension() const { return _dim; };
    size_type  operator[] (size_type i) const { return _vertex[i]; }
    size_type& operator[] (size_type i)       { return _vertex[i]; }
    template<class Archive> 
    void serialize (Archive& ar, const unsigned int version) {
        for (size_type iloc = 0; iloc < size(); iloc++) {
          ar & _vertex[iloc];
	}
    }
// data:
protected:
    array_type _vertex;
};
} // namespace rheolef
// =====================================================================
// Some serializable types, like geo_element, have a fixed amount of data stored at fixed field positions.
// When this is the case, boost::mpi can optimize their serialization and transmission to avoid extraneous 
// copy operations.
// To enable this optimization, we specialize the type trait is_mpi_datatype, e.g.:
namespace boost {
 namespace mpi {
  template <> struct is_mpi_datatype<rheolef::geo_element_t> : mpl::true_ { };
  template <> struct is_mpi_datatype<rheolef::geo_element_q> : mpl::true_ { };
 } // namespace mpi
} // namespace boost

// =====================================================================
// classe polymorphe & MPI
// =====================================================================

using namespace rheolef;
using namespace std;

struct my_polymorphic_geo {
  typedef boost::ptr_vector<geo_element>::size_type size_type;
  typedef boost::ptr_vector<geo_element>::iterator iterator;
  typedef boost::ptr_vector<geo_element>::const_iterator const_iterator;
  my_polymorphic_geo () : _x(), _comm() {
    fill (_count_by_type, _count_by_type+geo_element::enum_type_max, 0);
    if (_comm.rank() == 0) {
      _x.push_back (new geo_element_t (1,2,3));
      _x.push_back (new geo_element_q (11,12,13,14));
      _x.push_back (new geo_element_t (4,5,6));
      _x.push_back (new geo_element_q (15,16,17,18));
      _x.push_back (new geo_element_t (7,8,9));
      _count_by_type [geo_element::t] = 3;
      _count_by_type [geo_element::q] = 2;
    }
  }
  void put () const {
    cout << "geo " << _x.size() << endl;
    for (size_t i = 0; i < _x.size(); i++) {
      const geo_element& K = _x[i];
#ifndef TO_CLEAN
      const std::type_info& rtti = typeid(K);
      cerr << "type " << rtti.name() << ": " << K << endl;
#endif // TO_CLEAN
      cout << K << endl;
    }
  }
  void test_mpi () {
    pair<size_type,size_type> count_tag_pair [size_type(geo_element::enum_type_max)];
    int tag_count = 0;
    if (_comm.rank() == 0) {
      for (size_type i = 0; i < size_type(geo_element::enum_type_max); i++) {
        size_type tag = 100+i;
        count_tag_pair [i] = make_pair(_count_by_type[i], tag);
      }
      _comm.send (1, tag_count, count_tag_pair, size_type(geo_element::enum_type_max));
      // envoi
      vector<pair<size_type,geo_element_t::array_type> > buffer_t (_count_by_type[geo_element::t]);
      vector<pair<size_type,geo_element_q::array_type> > buffer_q (_count_by_type[geo_element::q]);
      vector<size_type> last_by_type (geo_element::enum_type_max, 0);
      for (size_type ie = 0; ie < _x.size(); ie++) {
        const geo_element& K = _x[ie];
      	size_type ibuf = last_by_type[K.type()];
      	last_by_type[K.type()]++;
        switch (K.type()) {
          // -----------------
          // dump triangles:
          // -----------------
          case geo_element::t: {
      	    buffer_t[ibuf].first = ie;
      	    dump (_x[ie], buffer_t[ibuf].second);
	    break;
	  }
          // -----------------
          // dump quadrangles:
          // -----------------
          case geo_element::q: {
      	    buffer_q[ibuf].first = ie;
      	    dump (_x[ie], buffer_q[ibuf].second);
	    break;
	  }
	  default : {
	    error_macro ("unexpected element " << geo_element::tab_name[K.type()]);
	  }
	}
      }
      // -----------------
      // send triangles:
      // -----------------
      if (count_tag_pair[geo_element::t].first != 0) {
        _comm.send (1,
		count_tag_pair[geo_element::t].second,
		buffer_t.begin().operator->(),	
		count_tag_pair[geo_element::t].first);
      }
      // -----------------
      // send quadrangles:
      // -----------------
      if (count_tag_pair[geo_element::q].first != 0) {
        _comm.send (1,
		count_tag_pair[geo_element::q].second,
		buffer_q.begin().operator->(),	
		count_tag_pair[geo_element::q].first);
      }
    } else if (_comm.rank() == 1) {
      _comm.recv (0, tag_count, count_tag_pair, size_type(geo_element::enum_type_max));
      vector<pair<size_type,geo_element_t::array_type> > buffer_t (count_tag_pair[geo_element::t].first);
      vector<pair<size_type,geo_element_q::array_type> > buffer_q (count_tag_pair[geo_element::q].first);
      size_type total_size = 0;
      for (size_type i = 0; i < size_type(geo_element::enum_type_max); i++) {
        _count_by_type[i] = count_tag_pair [i].first;
        total_size += _count_by_type[i];
      }
      _x.reserve (total_size);
      // resize...
#ifdef TODO
      _x.resize (total_size);
#else // TODO
      for (size_type j = 0; j < total_size; j++) {
	_x.push_back (new geo_element_t);
      }
#endif // TODO
      // -----------------
      // recv triangles:
      // -----------------
      if (count_tag_pair[geo_element::t].first != 0) {
        _comm.recv (0,
		count_tag_pair[geo_element::t].second,
		buffer_t.begin().operator->(),	
		count_tag_pair[geo_element::t].first);
        for (size_type ibuf = 0; ibuf < buffer_t.size(); ibuf++) {
          size_type ie = buffer_t[ibuf].first;
      	  //_x.insert (_x.begin()+ie, new geo_element_t (buffer_t[ibuf].second));
	  //warning_macro ("replace["<<ie<<"]="<<buffer_t[ibuf].second);
	  check_macro (!_x.is_null(ie), "is_null "<<ie);
      	  _x.replace (_x.begin()+ie, new geo_element_t (buffer_t[ibuf].second));
        }
      }
      // -----------------
      // recv quadrangles:
      // -----------------
      if (count_tag_pair[geo_element::q].first != 0) {
        _comm.recv (0,
		count_tag_pair[geo_element::q].second,
		buffer_q.begin().operator->(),	
		count_tag_pair[geo_element::q].first);
        for (size_type ibuf = 0; ibuf < buffer_q.size(); ibuf++) {
          size_type ie = buffer_q[ibuf].first;
      	  //_x.insert (_x.begin()+ie, new geo_element_q (buffer_q[ibuf].second));
	  //warning_macro ("replace["<<ie<<"]="<<buffer_q[ibuf].second);
	  check_macro (!_x.is_null(ie), "is_null "<<ie);
      	  _x.replace (_x.begin()+ie, new geo_element_q(buffer_q[ibuf].second));
        }
      }
      put();
    }
  }
// data:
  boost::ptr_vector<geo_element> _x;
  mpi::communicator              _comm;
  size_type _count_by_type [geo_element::enum_type_max];
};
int main(int argc, char**argv) {
  environment parallel(argc, argv);
  mpi::communicator comm;
  if (comm.size() == 1) {
	cerr << "not sequential : use mpirun -np 2 !" << endl;
	return 0;
  }
  my_polymorphic_geo omega;
  omega.test_mpi();
}
#endif // _RHEOLEF_HAVE_MPI
