///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "s_grad_grad.h"
#include "rheolef/ublas_matrix_range.h"

namespace rheolef {
using namespace std;
using namespace ublas;

template<class T, class M>
void 
s_grad_grad<T,M>::operator() (const geo_element& K, ublas::matrix<T>& a) const
{
  check_macro (! base::is_on_band(), "unsupported banded level set extension");

  // 3d: curl(curl(psi)) = - Laplacian(psi) + grad(div(psi))
  //  => not s.d.p matrix but with look at pis with div(psi)=0 : stream function
  if (K.dimension() == 3) {
    size_type n_comp = base::get_first_space().size();
    matrix<T> ai;
    base::build_scalar_grad_grad (K, ai);
    size_type n1 = ai.size1();
    size_type n2 = ai.size2();
    a.resize (n_comp*n1, n_comp*n2);
    a.clear();
    for (size_type i = 0; i < n_comp; i++) {
      mr_set (a, range(i*n1,(i+1)*n1), range(i*n2,(i+1)*n2), ai);
    }
    return;
  }
  // 1d or 2d plane: coincide with grad_grad
  // axisymmetric case: drop the 'r' weight also (Batchelor variant)
  base::set_use_coordinate_system_weight (false);
  base::build_scalar_grad_grad (K, a);
  base::set_use_coordinate_system_weight (true);
}
template<class T, class M>
void
s_grad_grad<T,M>::initialize () const
{
  base::set_n_derivative(2);

  // suppose also that multi-component spaces are homogeneous,
  // i.e. that all components have the same approx
  check_macro (base::get_first_space().valued_tag() == base::get_second_space().valued_tag(),
    "incompatible spaces for the `s_grad_grad' form.");
  if (base::is_weighted()) {
    space_constant::valued_type valued = base::_wh.valued_tag();
    if (valued != space_constant::scalar) {
        error_macro ("invalid weight for the `s_grad_grad' form");
    }
  }
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class s_grad_grad<Float,sequential>;

#ifdef _RHEOLEF_HAVE_MPI
template class s_grad_grad<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
