# ifndef _RHEO_BASIC_POINT_H
# define _RHEO_BASIC_POINT_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
//    point_basic<T> : set of coordinates of type T
//
//    author: Pierre.Saramito@imag.fr
//
#include "rheolef/compiler.h"
#include <sstream>
#ifdef _RHEOLEF_USE_NEW_CODE
#include "rheolef/distributed.h"
#endif // _RHEOLEF_USE_NEW_CODE

namespace rheolef { 

/*Class:point
NAME:  @code{point} - vertex of a mesh
@clindex point
@clindex geo
DESCRIPTION:
  Defines geometrical vertex as an array of coordinates.
  This array is also used as a vector of the three dimensional
  physical space.
End:
*/

//<point:
template <class T>
class point_basic {
    public:

// typedefs:

	typedef size_t size_type;
	typedef T      element_type;
	typedef T      scalar_type;
	typedef T      float_type;

// allocators:

	explicit point_basic () { _x[0] = T();  _x[1] = T();  _x[2] = T(); } 
	
	explicit point_basic (
		const T& x0, 
		const T& x1 = 0, 
		const T& x2 = 0)
			{ _x[0] = x0; _x[1] = x1; _x[2] = x2; }
	
	template <class T1>
	point_basic<T>(const point_basic<T1>& p)
		{ _x[0] = p._x[0]; _x[1] = p._x[1]; _x[2] = p._x[2]; }

	template <class T1>
	point_basic<T>& operator = (const point_basic<T1>& p)
		{ _x[0] = p._x[0]; _x[1] = p._x[1]; _x[2] = p._x[2]; return *this; }

// accessors:

	T& operator[](int i_coord)	        { return _x[i_coord%3]; }
	const T&  operator[](int i_coord) const { return _x[i_coord%3]; }
	T& operator()(int i_coord)	        { return _x[i_coord%3]; }
	const T&  operator()(int i_coord) const { return _x[i_coord%3]; }

	// interface for CGAL library inter-operability:
	const T& x() const { return _x[0]; }
	const T& y() const { return _x[1]; }
	const T& z() const { return _x[2]; }
	T& x(){ return _x[0]; }
	T& y(){ return _x[1]; }
	T& z(){ return _x[2]; }

// inputs/outputs:

	std::istream& get (std::istream& s, int d = 3)
	{
	    switch (d) {
	    case 0 : _x[0] = _x[1] = _x[2] = 0; return s;
	    case 1 : _x[1] = _x[2] = 0; return s >> _x[0];
	    case 2 : _x[2] = 0; return s >> _x[0] >> _x[1];
	    default: return s >> _x[0] >> _x[1] >> _x[2];
	    }
	}
	// output
	std::ostream& put (std::ostream& s, int d = 3) const;

// algebra:

	bool operator== (const point_basic<T>& v) const
		{ return _x[0] == v[0] && _x[1] == v[1] && _x[2] == v[2]; }

	bool operator!= (const point_basic<T>& v) const
		{ return !operator==(v); }

	point_basic<T>& operator+= (const point_basic<T>& v)
		{ _x[0] += v[0]; _x[1] += v[1]; _x[2] += v[2]; return *this; }

	point_basic<T>& operator-= (const point_basic<T>& v)
		{ _x[0] -= v[0]; _x[1] -= v[1]; _x[2] -= v[2]; return *this; }

	point_basic<T>& operator*= (const T& a)
		{ _x[0] *= a; _x[1] *= a; _x[2] *= a; return *this; }

	point_basic<T>& operator/= (const T& a)
		{ _x[0] /= a; _x[1] /= a; _x[2] /= a; return *this; }

	point_basic<T> operator+ (const point_basic<T>& v) const
		{ return point_basic<T> (_x[0]+v[0], _x[1]+v[1], _x[2]+v[2]); }

	point_basic<T> operator- () const
		{ return point_basic<T> (-_x[0], -_x[1], -_x[2]); }

	point_basic<T> operator- (const point_basic<T>& v) const
		{ return point_basic<T> (_x[0]-v[0], _x[1]-v[1], _x[2]-v[2]); }

	point_basic<T> operator* (const T& a) const
		{ return point_basic<T> (a*_x[0], a*_x[1], a*_x[2]); }
	point_basic<T> operator* (int a) const
		{ return operator* (T(a)); }

	point_basic<T> operator/ (const T& a) const
		{ return operator* (T(1)/T(a)); }

	point_basic<T> operator/ (point_basic<T> v) const
		{ return point_basic<T> (_x[0]/v[0], _x[1]/v[1], _x[2]/v[2]); }

// data:
// protected:
	T _x[3];
// internal:
  	static T _my_abs(const T& x) { return (x > T(0)) ? x : -x; }
};
typedef point_basic<Float> point;

// algebra:
template<class T>
inline
point_basic<T>
operator* (int a, const point_basic<T>& u)
{
  return u.operator* (T(a));
}
template<class T>
inline
point_basic<T>
operator* (const T& a, const point_basic<T>& u)
{
  return u.operator* (a);
}
template<class T>
inline
point_basic<T>
vect (const point_basic<T>& v, const point_basic<T>& w)
{
  return point_basic<T> (
	v[1]*w[2]-v[2]*w[1],
	v[2]*w[0]-v[0]*w[2],
	v[0]*w[1]-v[1]*w[0]);
}
// metrics:
template<class T>
inline
T dot (const point_basic<T>& x, const point_basic<T>& y)
{
  return x[0]*y[0]+x[1]*y[1]+x[2]*y[2]; 
}
template<class T>
inline
T norm2 (const point_basic<T>& x)
{
  return dot(x,x);
}
template<class T>
inline
T norm (const point_basic<T>& x)
{
  return sqrt(norm2(x));
}
template<class T>
inline
T dist2 (const point_basic<T>& x,  const point_basic<T>& y)
{
  return norm2(x-y);
}
template<class T>
inline
T dist (const point_basic<T>& x,  const point_basic<T>& y)
{
  return norm(x-y);
}
template<class T>
inline
T dist_infty (const point_basic<T>& x,  const point_basic<T>& y)
{
  return max(point_basic<T>::_my_abs(x[0]-y[0]),
         max(point_basic<T>::_my_abs(x[1]-y[1]),
             point_basic<T>::_my_abs(x[2]-y[2])));
}
template <class T>
T vect2d (const point_basic<T>& v, const point_basic<T>& w);

template <class T>
T mixt (const point_basic<T>& u, const point_basic<T>& v, const point_basic<T>& w);

// robust(exact) floating point predicates: return the sign of the value as (0, > 0, < 0)
// formally: orient2d(a,b,x) = vect2d(a-x,b-x)
template <class T>
int
sign_orient2d (
  const point_basic<T>& a,
  const point_basic<T>& b,
  const point_basic<T>& c);

template <class T>
int
sign_orient3d (
  const point_basic<T>& a,
  const point_basic<T>& b,
  const point_basic<T>& c,
  const point_basic<T>& d);

// compute also the value:
template <class T>
T orient2d(
  const point_basic<T>& a,
  const point_basic<T>& b,
  const point_basic<T>& c);

// formally: orient3d(a,b,c,x) = mixt3d(a-x,b-x,c-x)
template <class T>
T orient3d(
  const point_basic<T>& a,
  const point_basic<T>& b,
  const point_basic<T>& c,
  const point_basic<T>& d);

template <class T>
std::string ptos (const point_basic<T>& x, int d = 3);

// ccomparators: lexicographic order
template<class T, size_t d>
bool
lexicographically_less (const point_basic<T>& a, const point_basic<T>& b)
{
  for (typename point_basic<T>::size_type i = 0; i < d; i++) {
    if (a[i] < b[i]) return true;
    if (a[i] > b[i]) return false;
  }
  return false; // equality
}
//>point:

// -------------------------------------------------------------------------------------
// inline'd
// -------------------------------------------------------------------------------------
/// @brief helper for point_basic<T> & tensor_basic<T>: get basic T type
template<class T> struct scalar_traits                  { typedef T type; };
template<class T> struct scalar_traits<point_basic<T> > { typedef T type; };
template<class T> struct  float_traits<point_basic<T> > { typedef typename float_traits<T>::type type; };

// input/output
template<class T>
inline
std::ostream&
point_basic<T>::put (std::ostream& s, int d) const
{
    switch (d) {
    case 0 : return s;
    case 1 : return s << _x[0];
    case 2 : return s << _x[0] << " " << _x[1];
    default: return s << _x[0] << " " << _x[1] << " " << _x[2];
    }
}
template<class T>
inline
std::istream& 
operator >> (std::istream& s, point_basic<T>& p)
{
    return s >> p[0] >> p[1] >> p[2];
}
template<class T>
inline
std::ostream& 
operator << (std::ostream& s, const point_basic<T>& p)
{
    return s << p[0] << " " << p[1] << " " << p[2];
}
template<class T>
std::string
ptos (const point_basic<T>& x, int d = 3)
{
  std::ostringstream ostrstr;
  x.put (ostrstr, d);
  return ostrstr.str();
}

#ifdef _RHEOLEF_HAVE_DOUBLEDOUBLE
template<>
inline
std::ostream& 
point_basic<doubledouble>::put (std::ostream& s, int d) const
{
    switch (d) {
    case 0 : return s;
    case 1 : return s << double(_x[0]);
    case 2 : return s << double(_x[0]) << " " << double(_x[1]);
    default: return s << double(_x[0]) << " " << double(_x[1]) << " " << double(_x[2]);
    }
}
#endif // _RHEOLEF_HAVE_DOUBLEDOUBLE
// ----------------------------------------------------------
// point extra: inlined
// ----------------------------------------------------------
#define def_point_function2(f,F)	\
template<class T>			\
inline					\
point_basic<T>				\
f (const point_basic<T>& x)		\
{					\
  point_basic<T> y;			\
  for (size_t i = 0; i < 3; i++)	\
    y[i] = F(x[i]);			\
  return y;				\
}

#define def_point_function(f)	def_point_function2(f,f)

def_point_function(sqr)
def_point_function(sqrt)
def_point_function(log)
def_point_function(log10)
def_point_function(exp)
#undef def_point_function2
#undef def_point_function

template<class T1, class T2>
inline
point_basic<T1>
operator/ (const T2& a, const point_basic<T1>& x)
{
  point_basic<T1> y;
  for (size_t i = 0; i < 3; i++)
    if (x[i] != 0) y[i] = a/x[i];
  return y;
}
// vect2d() and mixt() are deduced from:
template <class T>
inline
T
vect2d (const point_basic<T>& v, const point_basic<T>& w)
{
  return orient2d (point_basic<T>(), v, w);
}
template <class T>
inline
T
mixt (const point_basic<T>& u, const point_basic<T>& v, const point_basic<T>& w)
{
  return orient3d (point_basic<T>(), u, v, w);
}

}// namespace rheolef
// -------------------------------------------------------------
// serialization
// -------------------------------------------------------------
#ifdef _RHEOLEF_USE_NEW_CODE
#ifdef _RHEOLEF_HAVE_MPI
namespace boost {
 namespace serialization {
  template<class Archive, class T>
  void serialize (Archive& ar, class rheolef::point_basic<T>& x, const unsigned int version) {
    ar & x[0];
    ar & x[1];
    ar & x[2];
  }
 } // namespace serialization
} // namespace boost

// Some serializable types, like geo_element, have a fixed amount of data stored at fixed field positions.
// When this is the case, boost::mpi can optimize their serialization and transmission to avoid extraneous copy operations.
// To enable this optimization, we specialize the type trait is_mpi_datatype, e.g.:
namespace boost {
 namespace mpi {
  // TODO: when point_basic<T> is not a simple type, such as T=doubledouble or T=gmp, etc
  template <>
  struct is_mpi_datatype<rheolef::point_basic<double> > : mpl::true_ { };
 } // namespace mpi
} // namespace boost
#endif // _RHEOLEF_HAVE_MPI
#endif // _RHEOLEF_USE_NEW_CODE

#endif /* _RHEO_BASIC_POINT_H */
