///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
/*
OVERVIEW:
  1. unary operations
     1.1. unary operators: +a -a
  2. binary operations
     2.1. binary operators: a+b, a-b
     2.2. binary f*a, a/f where f is a constant
     2.3. binary f*a, a/f where f is a field expression
     2.4. binary trial*test operations : u*v, dot(u,v), ddot(u,v)

SUMMARY:
   form_vf_expr: bilinear expressions with respect to a test and a trial function:

        trial u (Xh); test v (Xh);
	form m = integrate (u*v);

   The u & v can be replaced by any field_vf_expr (recursive definition), involving
   linear operators as tr(v), or derivation, as grad(v), div(v), etc.
*/
#define _NO_FIELD_VF_MAIN
#include "field_vf_expr_ops_make.cc" // without main()
// *************************************************************************************
// part 1. unary operations
// *************************************************************************************

// =====================================================================================
// chap 1.1. unary operators: +a, -a
// =====================================================================================
void vf2_unop (const op& o, s details) {
  // op(e)
  cout << "template<class Expr>" << endl;
  vf2_unop_base (o, details,
	"form_vf_expr<Expr>"); 
}
// -----------------------------------------------------------------------------
void vf2_all_unops (const vector<op>& Op, s details) {
  for (size_t i = 0; i < Op.size(); i++) {
    vf2_unop (Op[i], details);
  }
}
// *************************************************************************************
// part 2. binary operations
// *************************************************************************************

// =====================================================================================
// chap 2.1. binary operators: a+b, a-b
// =====================================================================================
void vf2_addop (const op& o, s details) {

    // a+b
    cout << "template<class Expr1, class Expr2>" << endl;
    vf2_binop_base (o, details,
	"form_vf_expr<Expr1>", "",
	"form_vf_expr<Expr2>"); 
}
// -----------------------------------------------------------------------------
void vf2_all_addop (const vector<op>& Op, s details) {
  for (size_t i = 0; i < Op.size(); i++) {
    vf2_addop (Op[i], details);
  }
}
// =====================================================================================
// chap 2.2. binary f*a, a/f where f is a constant
// =====================================================================================
void vf2_multop_by_constant (const op& o, const vector<v::type>& Types, s details) {
  for (size_t i = 1; i <=2; ++i) {
    // v*c, c*v
    // v/c but skip c/v (nonlinear)
    if (o.tag == "divides" && i == 1) continue;

    for (vector<v::type>::const_iterator iter = Types.begin(), last = Types.end(); iter != last; ++iter) {
      v::type t = *iter;
    
      // skip type errors as v/tensor or dot(v,tensor):
      if (!o.have_arg (i, t)) continue;
    
      cout << "template<class Expr>" << endl;
      vf2_binop_by_constant_bind (o, details, i,
	valued(t,"typename Expr::scalar_type"),
	"form_vf_expr<Expr>");
    }
  }
}
// -----------------------------------------------------------------------------
void vf2_all_multops_by_constant (const vector<op>& Op,  const vector<v::type>& Types, s details) {
  for (size_t i = 0; i < Op.size(); i++) {
    vf2_multop_by_constant (Op[i], Types, details);
  }
}
// =====================================================================================
// chap 2.3. binary f*a, a/f where f is a field expresion
// =====================================================================================
void vf2_multop_by_field (const op& o, const vector<v::type>& Types, s details) {
  for (size_t i = 1; i <=2; ++i) {
    // v*f, f*v
    // v/f but skip f/v (nonlinear)
    if (o.tag == "divides" && i == 1) continue;

    // -------------
    // field
    // -------------
    cout << "template<class T, class M, class Expr>" << endl;
    vf2_multop_by_field_bind (o, details, i,
	"field_basic<T,M>",
	"field_expr_terminal_field<T,M>",
	"form_vf_expr<Expr>");

    // -----------------------------
    // field_expr: linear expr
    // -----------------------------
    cout << "template<class FExpr, class Expr>" << endl;
    vf2_multop_by_field_bind (o, details, i,
	"field_expr<FExpr>",
	"field_expr_terminal_field<typename FExpr::scalar_type,typename FExpr::memory_type>",
	"form_vf_expr<Expr>");

    // -----------------------------
    // field_nl_expr: nonlinear expr
    // -----------------------------
    cout << "template<class FExpr, class Expr>" << endl;
    vf2_multop_by_field_bind (o, details, i,
	"field_nonlinear_expr<FExpr>",
	"",
	"form_vf_expr<Expr>");

    for (vector<v::type>::const_iterator iter = Types.begin(), last = Types.end(); iter != last; ++iter) {
      v::type t = *iter;

      if (!o.have_arg (i, t)) continue;
      // ----------------------------------------
      // field_functor, Result=Float,point,tensor
      // ----------------------------------------
      cout << "template<class Fun, class Expr>" << endl;
      vf2_multop_by_field_bind (o, details, i,
	"field_functor<Fun,"+valued(t,"typename Expr::scalar_type")+" >",
	"field_expr_terminal_function<Fun>",
	"form_vf_expr<Expr>");

      // ----------------------------------------
      // class-function, Result=Float
      // ----------------------------------------
      // true functions are not distinguished by the return type => skip ambiguities
      // by fixing Result=Float : point & tensor valued true functions are not supported yet
      if (t != v::s) continue; 
      s r = valued(t,"Float");

      cout << "template<class Expr>" << endl;
      vf2_multop_by_field_bind (o, details, i,
	"function<"+r+"(const point&)>",
	"field_expr_terminal_function<function<"+r+"(const point&)> >",
	"form_vf_expr<Expr>");
    }
  }
}
// -----------------------------------------------------------------------------
void vf2_all_multops_by_field (const vector<op>& Op, const vector<v::type>& Types, s details) {
  for (size_t i = 0; i < Op.size(); i++) {
    vf2_multop_by_field (Op[i], Types, details);
  }
}
// -----------------------------------------------------------------------------
void vf2_all_multops (const vector<op>& Op, const vector<v::type>& Types, s details) {
  vf2_all_multops_by_constant (Op, Types, details);
  vf2_all_multops_by_field    (Op, Types, details);
}
// =====================================================================================
// chap 2.4. binary trial*test operations : u*v, dot(u,v), ddot(u,v)
// =====================================================================================
void vf2_multop_trial_test_base (const op& o, s details, s arg1, s w_arg1, s arg2, s w_arg2="") {
  if (w_arg1 == "") w_arg1 = arg1;
  if (w_arg2 == "") w_arg2 = arg2;
  s x = (arg1 == w_arg1) ? "x" : "arg1_t(x)";
  s y = (arg2 == w_arg2) ? "y" : "arg2_t(y)";
  cout << "inline" << endl
       << "form_vf_expr<" << endl
       << "  form_vf_expr_bf_field<" << endl
       << "      " << details << "::" << o.tag << "" << endl
       << "     ," << w_arg1 << endl
       << "     ," << w_arg2 << endl
       << "    >" << endl
       << "  >" << endl
       << add_operator(o.name) << " (const " << arg1 << "& x, const " << arg2 << "& y)" << endl
       << "{" << endl
       << "  typedef " << details << "::" << o.tag        << " op_t;" << endl
       << "  typedef " << w_arg1                          << " arg1_t;" << endl
       << "  typedef " << w_arg2                          << " arg2_t;" << endl
       << "  typedef form_vf_expr_bf_field<op_t,arg1_t,arg2_t> expr_t;" << endl
       << "  return  form_vf_expr<expr_t>(expr_t(op_t(), "<<x<<", "<<y<<"));" << endl
       << "}" << endl;
}
// -----------------------------------------------------------------------------
void vf2_multop_trial_test (const op& o, s details) {

    // u*v
    cout << "template<class T, class M, class VfTag>" << endl;
    vf2_multop_trial_test_base (o, details,
	"test_basic<T,M,VfTag>", "",
	"test_basic<T,M,typename details::dual_vf_tag<VfTag>::type>"); 

    // expr(u)*v
    cout << "template<class T, class M, class Expr, class VfTag>" << endl;
    vf2_multop_trial_test_base (o, details,
	"field_vf_expr<Expr,VfTag>", "",
	"test_basic<T,M,typename details::dual_vf_tag<VfTag>::type>"); 

    // u*expr(u)
    cout << "template<class T, class M, class Expr, class VfTag>" << endl;
    vf2_multop_trial_test_base (o, details,
	"test_basic<T,M,VfTag>", "",
	"field_vf_expr<Expr,typename details::dual_vf_tag<VfTag>::type>"); 

    // expr(u)*expr(v)
    cout << "template<class Expr1, class Expr2, class VfTag>" << endl;
    vf2_multop_trial_test_base (o, details,
	"field_vf_expr<Expr1,VfTag>", "",
	"field_vf_expr<Expr2,typename details::dual_vf_tag<VfTag>::type>"); 
}
// -----------------------------------------------------------------------------
void vf2_all_multop_trial_test (const vector<op>& Op, s details) {
  for (size_t i = 0; i < Op.size(); i++) {
    vf2_multop_trial_test (Op[i], details);
  }
}
// =====================================================================================
int main() {
// =====================================================================================
  const s details = "details";
  const bool linear = true;
  header("FORM_VF", "form_vf");

  vector<v::type> Types;
  Types.push_back(v::s);
  Types.push_back(v::p);
  Types.push_back(v::t);

  vector<op> Unop;
  Unop.push_back (op("+", "unary_plus", linear));
  Unop.push_back (op("-", "negate",     linear));
  vf2_all_unops (Unop, details);

  vector<op> Addop;
  Addop.push_back (op("+", "plus",  linear));
  Addop.push_back (op("-", "minus", linear));
  vf2_all_addop (Addop, details);

  vector<op> Multop;
  Multop.push_back (op("*",    "multiplies"));
  Multop.push_back (op("dot",  v::p));
  Multop.push_back (op("ddot", v::t));
  vf2_all_multop_trial_test (Multop, details);

  vector<op> Fmultop;
  Fmultop.push_back (op("*",    "multiplies"));
  Fmultop.push_back (op("/",    "divides", v::all, v::s));
  Fmultop.push_back (op("dot",  v::p));
  Fmultop.push_back (op("ddot", v::t));
  vf2_all_multops (Fmultop, Types, details);

  footer("FORM_VF", "form_vf");
}
