///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// geo bamg inputs / outputs
//
// author:
//      Pierre.Saramito@imag.fr
//
// date: 2 may 2001
//

// ============================================================================
//  includes
// ============================================================================

#include "rheolef/georep.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h" // i/o utility
#include "rheolef/field.h"
#include "rheolef/form.h"
#include "rheolef/tiny_element.h"
using namespace std;
namespace rheolef { 


// ============================================================================
// bamg input (2d)
// ============================================================================

istream&
get_nodes_bamg (
	istream& s, 
    	georep::iterator_node iter_p, 
    	georep::iterator_node last_p, 
    	point& xmin, 
	point& xmax,
        vector<domain>& domains,
        vector<size_t>& vertice_domain_bamg_number)
{
  typedef georep::size_type size_type;
  xmin[0] = xmin[1] =  numeric_limits<Float>::max();
  xmax[0] = xmax[1] = -numeric_limits<Float>::max();
  xmin[2] = xmax[2] = 0;
  map<size_type, vector<tiny_element> > domain_map;
  size_t vertice_number = 0;
  tiny_element vertice;
  vertice.set_name('p');
  for (; iter_p != last_p; iter_p++, vertice_number++) {
        georep::nodelist_type::reference p = (*iter_p);
	p.get(s, 2);
	// get vertice domain number and build 0d-domains
        size_t vertice_domain_number; 
        s >> vertice_domain_number; 
        if (vertice_domain_number != 0) {
          // skip zero vertice domain number, that is reserved for unmarked vertices
          vertice[0] = vertice_number;
          domain_map[vertice_domain_number].push_back (vertice);
	}
        xmin[0] = std::min(p[0], xmin[0]);
	xmax[0] = std::max(p[0], xmax[0]);
	xmin[1] = std::min(p[1], xmin[1]);
	xmax[1] = std::max(p[1], xmax[1]);
  }
  // copy domain_map into domains (table)
  vertice_domain_bamg_number.resize (domain_map.size()); 
  domains.resize (domain_map.size());
  vector<domain>::iterator iter_domain = domains.begin();
  size_type i_dom = 0;
  for (map<size_type, vector<tiny_element> >::const_iterator iter_domain_map = domain_map.begin();
      iter_domain_map != domain_map.end(); iter_domain_map++, iter_domain++, i_dom++) {
      vertice_domain_bamg_number[i_dom] = (*iter_domain_map).first;
      const vector<tiny_element>& dom      = (*iter_domain_map).second;
      if (dom.size() == 0) continue;
      (*iter_domain).set(dom.begin(), dom.size(), 0, "unnamed_vertice"+itos(i_dom));
  }
  return s;
}
//
// get boundary edges
//
istream&
get_boundary_edges_bamg (istream& is, vector<domain>& domains,
    vector<size_t>& edge_domain_bamg_number)
{
  typedef georep::size_type size_type;
  size_type n_bdr_edges;
  is >> n_bdr_edges;
  map<size_t, vector<pair<size_type,size_type> > > domain_map;
  for (size_type j = 0; is.good() && j < n_bdr_edges; j++) {

    size_type r, p0, p1;
    is >> p0 >> p1 >> r;
    p0--; p1--; // shift indices start from 1 to 0 
    domain_map[r].push_back (make_pair(p0,p1));
  }
  if (!is.good()) {
      error_macro ("a problem occurs while loading a bamg mesh");
  }
  // copy domain_map into domains (table)
  edge_domain_bamg_number.resize (domain_map.size()); 
  domains.resize (domain_map.size());
  vector<domain>::iterator iter_domain = domains.begin();
  size_t i_dom = 0;
  for (map<size_t, vector<pair<size_type,size_type> > >::const_iterator iter_domain_map = domain_map.begin();
      iter_domain_map != domain_map.end(); iter_domain_map++, iter_domain++, i_dom++) {
      edge_domain_bamg_number[i_dom] = (*iter_domain_map).first;
      const vector<pair<size_type,size_type> >& dom = (*iter_domain_map).second;
      if (dom.size() == 0) continue;
      (*iter_domain).set(dom.begin(), dom.size(), "unnamed"+itos(i_dom));
  }
  return is;
}
istream&
georep::get_bamg(istream& s, bool check_optional_domain_name) 
{
  // bamg has only boundary edges that are numbered
  // and have domain numbers 
  // => this is only a version 1 mesh file format type
  //    not consistent for P2 elements
  _version = 1;

  typedef georep::size_type size_type;
  if (!s) error_macro("bad input stream for geo.");
  bool verbose = iorheo::getverbose(s);
  _dim = 2;
  //
  // get coordinates
  //
  string label;
  size_type n_vert = numeric_limits<size_type>::max();
  size_type n_elt  = numeric_limits<size_type>::max();
  _x.resize(0);
  vector<domain> vertice_domain;
  vector<domain> edge_domain;
  vector<size_t> vertice_domain_bamg_number;
  vector<size_t> edge_domain_bamg_number;
  resize(0);

  while (s.good() && label != "End") {

      s >> label;

      if (label == "Vertices") {

  	 s >> n_vert;
         _x.resize(n_vert);
         get_nodes_bamg (s, begin_node(), end_node(), _xmin, _xmax, vertice_domain, vertice_domain_bamg_number);
         _count_geo [0] = _count_element [0] = n_vert;
         if (!s.good()) return s;

      } else if (label == "Edges") {
         //
         // get boundary edges (domains)
         //
  	 get_boundary_edges_bamg (s, edge_domain, edge_domain_bamg_number);
      } else if (label == "Triangles" || label == "Quadrilaterals") {
  	 //
  	 // get elements
  	 //
  	 s >> n_elt;
  	 size_type old_size = size();
  	 resize(old_size + n_elt); // make physical copies of elements if triangles + quad
	 size_type n_vert_per_elt = (label == "Triangles") ?  3  :  4;

  	 size_type K_idx = 0;
         georep::iterator iter_elt = begin() + old_size;
  	 georep::iterator last_elt = end();
	 for (; s.good() && iter_elt != last_elt; iter_elt++, K_idx++) {

      	    geo_element& K = (*iter_elt);
      	    K.set_variant (n_vert_per_elt, _dim);
      	    K.set_index(K_idx) ;
      	    for (size_type i = 0 ; i < n_vert_per_elt; i++) {
		s >> K[i];
        	K[i]--;
		if (n_vert != numeric_limits<size_type>::max() && K[i] >= n_vert) {
	    	    error_macro("bamg input: element " << K_idx+1 << ": "
		        << i+1 << "-th vertice index "
              	        << K[i] << " out of range 0:" << n_vert-1);
		}
      	    }
            // TODO: get region number and build 2d-domains
            size_type dummy; s >> dummy;
           _count_element [K.variant()]++;
      	   _count_geo     [_dim]++;
	 }
      }
  }
  // -------------------------------------------------------------------
  // reduce vertice domains:
  // in bamg files, boundary vertices have by default the domain
  // number of the 1d edge domain number.
  // we delete it and conserve just vertice domain that have specifi number
  // -------------------------------------------------------------------
  size_t reduced_vertice_domain_size = vertice_domain_bamg_number.size();
  for (size_t i = 0; i < vertice_domain_bamg_number.size(); i++) {
    bool founded_as_edge_domain = false;
    // heavy search, not optimal since index are sorted
    // but domain table size are small in general
    for (size_t j = 0; ! founded_as_edge_domain && j < edge_domain_bamg_number.size(); j++) {
      if (edge_domain_bamg_number[j] == vertice_domain_bamg_number[i]) {
        founded_as_edge_domain = true;
      }
    }
    if (founded_as_edge_domain) {
      vertice_domain_bamg_number[i] = numeric_limits<size_type>::max();
      reduced_vertice_domain_size--;
    }
  }
  vector<domain> reduced_vertice_domain (reduced_vertice_domain_size);
  vector<size_type> reduced_vertice_domain_bamg_number (reduced_vertice_domain_size);
  size_t i_reduced = 0;
  for (size_t i = 0; i < vertice_domain_bamg_number.size(); i++) {
    if (vertice_domain_bamg_number[i] != numeric_limits<size_type>::max()) {
      reduced_vertice_domain [i_reduced] = vertice_domain[i];
      reduced_vertice_domain_bamg_number [i_reduced] = vertice_domain_bamg_number[i];
      i_reduced++;
    }
  }
  // ---------------------------------------------------------------
  // check extension to optional domain names
  // ---------------------------------------------------------------
  if (!check_optional_domain_name) return s;
  char c;
  s >> ws >> c; // skip white and grab a char
  // have "EdgeDomainNames" or "VerticeDomainNames" ?
  // bamg mesh may be followed by field data and such, so be carrefull...
  while (c == 'E' || c == 'V') {
      s.unget(); // put char back
      if (c == 'E') {
        if (!scatch(s,"EdgeDomainNames")) return s;
        // ---------------------------------------------------------------
        // get edge domain names
        // ---------------------------------------------------------------
        size_type n_dom_edge;
        s >> n_dom_edge;
        if (n_dom_edge != edge_domain.size()) {
           string numbers;
           for (size_t j = 0; j < edge_domain_bamg_number.size(); j++) {
              numbers = numbers + itos(edge_domain_bamg_number[j]) + " ";
           }
           warning_macro("geo input: "
	    << n_dom_edge << " domain names founded while "
            << edge_domain.size() << " bamg edge domains numbers are defined: "
            << numbers);
           cerr << endl;
           error_macro("HINT: check domain name file (.dmn)");
        }
        size_t start_edge_domain = _domlist.size();
        _domlist.resize(_domlist.size() + n_dom_edge);
        for (size_type k = 0; k < n_dom_edge; k++) {
          string name;
          s >> name;
          edge_domain[k].set_name(name);
          _domlist.at(start_edge_domain+k) = edge_domain[k];
        }
      } else if (c == 'V') {
        if (!scatch(s,"VerticeDomainNames")) return s;
        // ---------------------------------------------------------------
        // get vertice domain names
        // ---------------------------------------------------------------
        size_type n_dom_vertice;
        s >> n_dom_vertice;
        if (n_dom_vertice != reduced_vertice_domain.size()) {
           string numbers;
           for (size_t j = 0; j < reduced_vertice_domain_bamg_number.size(); j++) {
              numbers = numbers + itos(reduced_vertice_domain_bamg_number[j]) + " ";
           }
           warning_macro("geo input: "
	    << n_dom_vertice << " domain names founded while "
            << reduced_vertice_domain.size() << " bamg vertice domains numbers are defined: "
            << numbers);
           cerr << endl;
           error_macro("HINT: check domain name file (.dmn)");
        }
        size_t start_vertice_domain = _domlist.size();
        _domlist.resize(_domlist.size() + n_dom_vertice);
        for (size_type k = 0; k < n_dom_vertice; k++) {
          string name;
          s >> name;
          reduced_vertice_domain[k].set_name(name);
          _domlist.at(start_vertice_domain+k) = reduced_vertice_domain[k];
        }
      }
      s >> ws >> c; // skip white and grab a char
  }
  // ---------------------------------------------------------------
  // check extension to optional domain names
  // ---------------------------------------------------------------
  return s;
}
// =======================================================================
// bamg output
// =======================================================================

typedef georep::size_type size_type;
typedef map <size_type,  size_type > vertex_bdr_type;
//          (i_vertex, i_bdr_vertex)

static 
void
insert_vertex_bdr (
    vertex_bdr_type& vertex_bdr, 
    vertex_bdr_type& vertex_corner,
    size_type i_vertex, size_type i_domain)
{
    vertex_bdr_type::iterator p = vertex_bdr.find(i_vertex);
    if (p == vertex_bdr.end()) {
	size_type i_bdr_vertex = vertex_bdr.size();
	// the second size_t is now used to remember the domain to which i_vertex
	// belongs, it will then be overwritten to number the boundary vertices.
	vertex_bdr.insert(make_pair(i_vertex,i_domain));
    }
    else
    if ( (*p).second != i_domain )
    // this vertex appears in several different domains: it is a corner point
    vertex_corner.insert(make_pair(i_vertex,numeric_limits<size_type>::max()));
}
ostream& 
georep::put_bamg (ostream& os) const
{
  bool verbose = iorheo::getverbose(os);
  size_t AngleCorner = iorheo::getanglecorner(os);
  if (dimension() != 2) {
      error_macro ("bamg output: " << dimension() << "D geometries not supported.");
  }
  //
  // header
  //
  os << "MeshVersionFormatted\n0\n\n"
     << "Dimension\n2\n\n"
     << "Identifier\n"
     << "\"G=" << name() << ".cad;1, Date: 01/02/28 19:43 29s\"\n\n"
     << "Geometry\n"
     << "\"" << name() << ".cad\"\n\n"
     << "Vertices\n"
     << n_node() << endl;
  //
  // vertices
  //
  for (const_iterator_node iter_p = begin_node(); iter_p != end_node(); iter_p++) {
      const point& p = *iter_p;
      p.put (os, 2);
      os << " 1" << endl ;
  }
  os << endl;
  //
  // boundary edges
  //
  size_type n_bdr_edge = 0;
  domlist_type::const_iterator last = _domlist.end();
  for (domlist_type::const_iterator iter = _domlist.begin(); iter != last; iter++) {
    if ((*iter).dimension() == dimension() - 1) n_bdr_edge += (*iter).size();
  }
  os << "Edges\n"
     << n_bdr_edge << endl;
  size_type i_domain = 0;
  vertex_bdr_type vertex_bdr;
  vertex_bdr_type vertex_corner;
  for (domlist_type::const_iterator iter = _domlist.begin(); iter != last; iter++, i_domain++) {
    const domain& d = *iter;
    if (d.dimension() == dimension() -1) {
      for (domain::const_iterator iter_s = d.begin(); iter_s != d.end(); iter_s++) {
          const geo_element& S = *iter_s;
          os << S[0]+1 << " " << S[1]+1 << " " << i_domain+1 << endl;
          insert_vertex_bdr (vertex_bdr, vertex_corner, S[0], i_domain);
          insert_vertex_bdr (vertex_bdr, vertex_corner, S[1], i_domain);
      }
    }
  }
  // set bdr_vertices numbering
  size_type i_bdr_vertex = 0;
  for (vertex_bdr_type::iterator p = vertex_bdr.begin(); p != vertex_bdr.end(); p++) {
    (*p).second = i_bdr_vertex++;
  }
  os << "\nCrackedEdges\n0\n";
  //
  // elements
  //
  if (n_triangle() != 0) {
     os << "\nTriangles\n" << n_triangle() << endl;
     for (const_iterator iter_k = begin(); iter_k != end(); iter_k++) {
       const geo_element& K = *iter_k;
       if (K.size() != 3) continue;
       os << K[0]+1 << " " << K[1]+1 << " " << K[2]+1 << " 1" << endl;
     }
  }
  if (n_quadrangle() != 0) {
     os << "\nQuadrilaterals\n" << n_quadrangle() << endl;
     for (const_iterator iter_k = begin(); iter_k != end(); iter_k++) {
       const geo_element& K = *iter_k;
       if (K.size() != 4) continue;
       os << K[0]+1 << " " << K[1]+1 << " " << K[2]+1 << " " << K[3]+1 << " 1" << endl;
     }
  }
  os << "\nSubDomainFromMesh" << endl
     << "1" << endl
     << "3 1 1 1" << endl << endl
     ;
  // -----------------------------
  // relation with cad description
  // -----------------------------
  os << "VertexOnGeometricVertex" << endl
     << vertex_bdr.size() << endl;
  i_bdr_vertex = 0;
  for (vertex_bdr_type::iterator p = vertex_bdr.begin();
	p != vertex_bdr.end(); p++) {
    size_type i_vertex = (*p).first;
    os << i_vertex+1 << " " << i_bdr_vertex+1 << endl;
    i_bdr_vertex++;
  }
  os << endl
     << "VertexOnGeometricEdge" << endl
     << "0" << endl
     << endl
     << "EdgeOnGeometricEdge" << endl
     << n_bdr_edge << endl;
  size_type i_edge = 0;
  for (domlist_type::const_iterator iter = _domlist.begin(); iter != last; iter++) {
    const domain& d = *iter;
    for (domain::const_iterator iter_s = d.begin(); iter_s != d.end(); iter_s++) {
      const geo_element& S = *iter_s;
      os << i_edge+1 << " " << i_edge+1 << endl;
      i_edge++;
    }
  }
  os << "\nEnd\n";
  // -----------------------------
  // cad description
  // -----------------------------
  string cad_name = name() + ".cad";
  // NOTE : the file may be RE-CREATED because EdgeOnGeometricEdge specify the
  // curvilinear coordinate of an edge on the geometric boundary : this is not
  // yet supported. Instead, in the new .cad file, each edge is a .geo edge.
  // This is BAD for mesh adaptation on curved boundaries, but there is no better
  // solution yet.
  ofstream cad (cad_name.c_str());
  if (verbose) cerr << "! file \"" << cad_name << "\" created." << endl;
  //
  // cad header
  //
  cad << setprecision(numeric_limits<Float>::digits10) 
      << "MeshVersionFormatted\n0\n\n"
      << "AngleOfCornerBound\n" << AngleCorner << "\n\n"
      << "Dimension\n2\n\n"
      << "Vertices\n"
      << vertex_bdr.size() << endl;
  //
  // cad vertice domain numbers
  //
  i_domain = 0;
  vector<size_t> vertice_domain_bamg_number(n_node(), 1);
  for (domlist_type::const_iterator iter = _domlist.begin(); iter != last; iter++, i_domain++) {
    const domain& d = *iter;
    if (d.dimension() == 0) { 
      for (domain::const_iterator iter_s = d.begin(); iter_s != d.end(); iter_s++) {
        const geo_element& vertex = *iter_s;
        size_type i_vertex = vertex[0];
        vertice_domain_bamg_number[i_vertex] = i_domain + 1;
      }
    }
  }
  //
  // cad vertices
  //
  georep::const_iterator_node x = begin_node();
  for (vertex_bdr_type::iterator p = vertex_bdr.begin();
	p != vertex_bdr.end(); p++) {
    size_type i_vertex = (*p).first;
    cad << x[i_vertex][0] << " " << x[i_vertex][1] << " "
        << vertice_domain_bamg_number[i_vertex] << endl;
  }
#ifdef BUG_BAMG
  //
  //
  // cad corners
  //
  if (vertex_corner.size()>0)
   {
	  cad << endl << "Corners " << vertex_corner.size() << endl;
	  size_t i_corner=0;
	  for (vertex_bdr_type::iterator p = vertex_corner.begin();
		p != vertex_corner.end(); p++, i_corner++) {
	    size_type i_vertex = (*p).first;
	    cad << (*p).first+1 << endl;
	  }
   }
#endif // BUG_BAMG
  //
  // cad edges
  //
  cad << endl
      << "Edges\n"
      << n_bdr_edge << endl;

  i_domain = 0;
  for (domlist_type::const_iterator iter = _domlist.begin(); iter != last; iter++, i_domain++) {
    const domain& d = *iter;
    if (d.dimension() == dimension() -1) {
      for (domain::const_iterator iter_s = d.begin(); iter_s != d.end(); iter_s++) {
        const geo_element& S = *iter_s;
        vertex_bdr_type::iterator p0 = vertex_bdr.find(S[0]);
        vertex_bdr_type::iterator p1 = vertex_bdr.find(S[1]);
        assert_macro (p0 != vertex_bdr.end() && p1 != vertex_bdr.end(),
          "problem with vertices during bamg cad output");
        size_type i_vertex_bdr_0 = (*p0).second;
        size_type i_vertex_bdr_1 = (*p1).second;
        cad << i_vertex_bdr_0+1 << " " << i_vertex_bdr_1+1 << " " << i_domain+1 << endl;
      }
    }
  }
  /* Jocelyn:
   *   NOW FILLING HOLES: Multi-domain meshes are not compatible with that...
   *   TODO: add an option
   *   last: for domains with holes, do not Fill it !
   * Ibrahim & Pierre:
   *   We do not thank you for this dirty hack...
   *   We try now to solve the holes in mesh with the elegant Euler formulae:
   *       N_element + N_vertices - N_edges = 1 - N_holes
   *   and thus the presence of holes is detectable and solved !
   *   This patch has been tested only with one hole...
   */
  size_t n_hole = n_edge() - (n_triangle() + n_quadrangle()) - n_node() + 1; 
  if (n_hole != 0) {
    cad << endl
        << "SubDomain 1" << endl
        << "2 1 -1 0" << endl
        << endl;
  }
  if (n_hole > 1) {
    warning_macro ("domain with " << n_hole << " has not be yet tested with bamg support");
  }
  cad.close();
  return os;
}
}// namespace rheolef
