/*
 * Copyright (C) 2009 Chase Douglas
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.
 */

#include "Connection.h"
#include "InputDevice.h"

InputDevice::InputDevice(Connection *_connection) :
    connection(_connection),
    created(false) {
    connection->setParent(this);

    connect(connection, SIGNAL(readyRead()), SLOT(connectionReadyRead()));
    connect(connection, SIGNAL(disconnected()), SLOT(deleteLater()));
}

// This must be overridden in a subclass
bool InputDevice::create() {
    return FALSE;
}

void InputDevice::connectionReadyRead() {
    while (connection->bytesAvailable() >= (signed)sizeof(rinput_message_t)) {
        rinput_message_t message;
        qint64 len;

        len = connection->read((char *)&message, sizeof(message));
        if (len < 0) {
            qWarning("Warning: Received error while reading a message");
            return;
        }
        if (len < (signed)sizeof(message)) {
            qCritical("Error: Failed to read entire message from client");
            rinput_message_t errorMessage = { RINPUT_ERROR, { RINPUT_INTERNAL_ERROR } };
            hton_rinput(&errorMessage);
            connection->write(QByteArray((const char *)&errorMessage, sizeof(errorMessage)));
            connection->disconnect();
            return;
        }

        qDebug("Read message from client");

        ntoh_rinput(&message);

        handleMessage(message);
    }
}

// This must be overridden in a subclass
void InputDevice::handleMessage(rinput_message_t &message) {}
