//Export.cpp, Copyright (c) 2002-2007 R.Lackner
//export graph files
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
#include "rlplot.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <fcntl.h>				//file open flags
#include <sys/stat.h>			//I/O flags
#ifdef USE_WIN_SECURE
	#include <share.h>			//I/O flags
#endif

#ifdef _WINDOWS
	#include <io.h>					//for read/write
#else
	#define O_BINARY 0x0
	#include <unistd.h>
#endif

extern char TmpTxt[];
extern Default defs;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// export to *.wmf file (windows meta file)
// this code is based on information from the following books
// G. Born, 'Referenzhandbuch Dateiformate', 
//     Addison-Wesley ISBN 3-89319-815-6
// T. Hogan, 'Die PC-referenz fr Programmierer',
//     Microsoft Press: Systema Verlag GmbH ISBN 3-89390-250-3
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
typedef struct {
	unsigned short mtType, mtHeader, mtVersion, mtSize0, mtSize1, mtNoObj;
	unsigned mtMaxRec;
	unsigned short mtnoPar;
}wmf_header;

class ExportWMF:public anyOutput {
public:
	HatchOut *hgo;

	ExportWMF(GraphObj *g, char *FileName, float res, DWORD flags);
	~ExportWMF();
	bool SetLine(LineDEF *lDef);
	bool SetFill(FillDEF *fill);
	bool SetTextSpec(TextDEF *set);
	bool StartPage();
	bool EndPage();
	bool oCircle(int x1, int y1, int x2, int y2, char* nam = 0L);
	bool oPolyline(POINT * pts, int cp, char *nam = 0L);
	bool oRectangle(int x1, int y1, int x2, int y2, char *nam = 0L);
	bool oSolidLine(POINT *p);
	bool oTextOut(int x, int y, char *txt, int cb);
	bool oPolygon(POINT *pts, int cp, char *nam = 0L);

private:
	int oFile;
	unsigned file_size, rec_size;
	wmf_header header;
	unsigned short currGDIobj, maxGDIobj;
	unsigned short hPen, hBrush, hFont;
	GraphObj *go;
	char *name;

	unsigned short wmfCreateSolidBrush(DWORD color);
	unsigned short wmfCreateFontIndirect(short, short, short, short, short, unsigned char,
		unsigned char, unsigned char, unsigned char, unsigned char, unsigned char,
		unsigned char, unsigned char, char *);
	unsigned short wmfCreateSolidPen(DWORD color, unsigned short width);
	void wmfDeleteObject(unsigned short obj);
	void wmfEllipse(unsigned short, unsigned short, unsigned short, unsigned short);
	void wmfPolyline(POINT *pts, unsigned short cp);
	void wmfPolygon(POINT *pts, unsigned short cp);
	void wmfRectangle(unsigned short, unsigned short, unsigned short, unsigned short);
	void wmfSelectObject(unsigned short o);
	void wmfSetBkColor(DWORD col);
	void wmfSetBkMode(unsigned m);
	void wmfSetTextAlign(unsigned a);
	void wmfSetTextColor(DWORD col);
	void wmfTextOut(unsigned short, unsigned short, char *, unsigned short);
};

ExportWMF::ExportWMF(GraphObj *g, char *FileName, float res, DWORD flags)
{
	currGDIobj = maxGDIobj = 0;
	hgo =0L;
	DeskRect.left = DeskRect.top = 0;
	DeskRect.right = DeskRect.bottom = 0x4fffffff;
	dFillCol = 0xffffffffL;
	hPen = 0xffff, hBrush = 0xffff, hFont = 0xffff;
	hres = vres = res;
	go = g;
	if(FileName)name = (char*)memdup(FileName, (int)strlen(FileName)+1, 0);
	else name = 0L;
	oFile = 0;
	rec_size = 28;
	header.mtType = 1;
	header.mtHeader = 9;
	header.mtVersion = 0x300;
	header.mtSize0 = 1000;
	header.mtSize1 = 0;
	header.mtNoObj = 64;
	header.mtMaxRec = 100;
	header.mtnoPar = 0;
}

ExportWMF::~ExportWMF()
{
	if(hgo) delete hgo;
	if(name) free(name);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//overloaded functions from the anyOutput class
bool
ExportWMF::SetLine(LineDEF *lDef)
{
	unsigned short iw;

	if(hPen == 0xffff || lDef->width != LineWidth || lDef->width != LineWidth || 
		lDef->pattern != dPattern || lDef->color != dLineCol) {
		LineWidth = lDef->width;
		iw = (unsigned short)(0.5 + un2fix(lDef->width));
		dPattern = lDef->pattern;
		RLP.finc = 256.0/((float)(un2fix(lDef->patlength*8.0)));
		RLP.fp = 0.0;
		if(iLine == iw && dLineCol == lDef->color && hPen != 0xffff) return true;
		iLine = iw;
		dLineCol = lDef->color;
		if(hPen != 0xffff) wmfDeleteObject(hPen);
		iw = iw > 0 ? iw : 1;
		hPen = wmfCreateSolidPen(dLineCol, iw);
		wmfSelectObject(hPen);
		}
	return true;
}

bool
ExportWMF::SetFill(FillDEF *fill)
{
	if(!fill) return false;
	if((fill->type & 0xff) != FILL_NONE) {
		if(!hgo) hgo = new HatchOut(this);
		if(hgo) hgo->SetFill(fill);
		}
	else {
		if(hgo) delete hgo;
		hgo = 0L;
		}
	if(dFillCol != fill->color) {
		if(hBrush != 0xffff) wmfDeleteObject(hBrush);
		hBrush = wmfCreateSolidBrush(dFillCol = fill->color);
		wmfSelectObject(hBrush);
		}
	dFillCol2 = fill->color2;
	return true;
}

bool
ExportWMF::SetTextSpec(TextDEF *set)
{
	unsigned char lfcs, lfpaf;
	char *face;
	bool IsModified, RetVal;

	switch(set->Font){
	case FONT_HELVETICA:
	default:
		lfcs = 0;
		lfpaf = 2 | 2<<4;
		face = "Arial";
		break;
	case FONT_TIMES:
		lfcs = 0;
		lfpaf = 2 | 1<<4;
		face = "Times New Roman";
		break;
	case FONT_COURIER:
		lfcs = 2;
		lfpaf = 1 | 3<<4;
		face = "Courier New";
		break;
		}
	if(!set->iSize && set->fSize > 0.001f) set->iSize = un2iy(set->fSize);
	if(!set->iSize) return false;
	if(hFont == 0xffff || TxtSet.iSize != set->iSize || TxtSet.Style != set->Style ||
		TxtSet.RotBL != set->RotBL || TxtSet.RotCHAR != set->RotCHAR ||
		TxtSet.Font != set->Font || TxtSet.fSize != set->fSize) IsModified = true;
	else IsModified = false;
	RetVal = anyOutput::SetTextSpec(set);
	if (IsModified && RetVal) {
		hFont = wmfCreateFontIndirect(TxtSet.iSize, 0, 
			iround(TxtSet.RotBL*10), iround(TxtSet.RotBL*10), 
			(TxtSet.Style & TXS_BOLD) ? 700 : 400,
			(TxtSet.Style & TXS_ITALIC) ? 1 : 0,
			(TxtSet.Style & TXS_UNDERLINE) ? 1 : 0,
			0, lfcs, 0, 0, 2, lfpaf, face);
		}
	if(hFont != 0xffff) wmfSelectObject(hFont);
	return true;
}

bool 
ExportWMF::StartPage()
{
	if(!go) return false;
	if(name) {
#ifdef USE_WIN_SECURE
		if(0 !=(_sopen_s(&oFile, name, _O_RDWR | _O_BINARY | _O_CREAT | _O_TRUNC, _SH_DENYNO,
			_S_IWRITE | _S_IREAD))) {
#else
		if(-1 ==(oFile = open(name, O_RDWR | O_BINARY | O_CREAT | O_TRUNC,
			S_IWRITE | S_IREAD))) {
#endif
			ErrorBox("Could not open output file");
			return false;
			}
		}
	else {
		oFile = 2;		//stdout
		}
	VPorg.fy = -un2fiy(go->GetSize(SIZE_GRECT_TOP));
	VPorg.fx = -un2fix(go->GetSize(SIZE_GRECT_LEFT));
	write(oFile, &header, 18);
	return true;
}

bool
ExportWMF::EndPage()
{
	unsigned short end_token[3] = {3, 0, 0};

	write(oFile, &end_token, 6);
	file_size = lseek(oFile, 0L, SEEK_CUR);
	lseek(oFile, 0L, SEEK_SET);
	header.mtSize0 = (file_size>>1)&0xffff;
	header.mtSize1 = file_size>>17;
	header.mtNoObj = maxGDIobj;
	header.mtMaxRec = rec_size;
	write(oFile, &header, 18);
	oFile = close(oFile);
	return true;
}

bool
ExportWMF::oCircle(int x1, int y1, int x2, int y2, char* nam)
{
	wmfEllipse(x1, y1, x2, y2);
	if(hgo) return hgo->oCircle(x1, y1, x2, y2);
	return true;
}

bool
ExportWMF::oPolyline(POINT * pts, int cp, char * nam)
{
	int i;

	if(cp < 1) return false;
	if (dPattern) for (i = 1; i < cp; i++) PatLine(pts[i-1], pts[i]);
	else wmfPolyline(pts, cp);
	return true;
}

bool
ExportWMF::oRectangle(int x1, int y1, int x2, int y2, char *nam)
{
	wmfRectangle(x1, y1, x2, y2);
	if(hgo) return hgo->oRectangle(x1, y1, x2, y2, 0L);
	return true;
}

bool
ExportWMF::oSolidLine(POINT *p)
{
	wmfPolyline(p, 2);
	return true;
}

bool
ExportWMF::oTextOut(int x, int y, char *txt, int cb)
{
	if(!txt || !txt[0]) return false;
	if(hFont != 0xffff) wmfSelectObject(hFont);
	wmfSetTextColor(TxtSet.ColTxt);
	wmfSetBkColor(TxtSet.ColBg);
	wmfSetTextAlign(((TxtSet.Align & TXA_HRIGHT) ? 2 : (TxtSet.Align &
		TXA_HCENTER) ? 6 : 0) | ((TxtSet.Align & TXA_VBOTTOM) ?	8 : 0));
	wmfSetBkMode(TxtSet.Mode ? 1 : 2);
	wmfTextOut(x, (TxtSet.Align & TXA_VCENTER) ? y - TxtSet.iSize/2 : y, txt, 
		(unsigned short)((cb > 0) ? cb : strlen(txt)));
	return true;
}

bool
ExportWMF::oPolygon(POINT *pts, int cp, char *nam)
{
	wmfPolygon(pts, cp);
	if(hgo) return hgo->oPolygon(pts, cp);
	return true;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//templates for wmf file records
typedef struct{
	unsigned Size;
	unsigned short Id;
	unsigned short Value;
}wmfObjShort;

typedef struct{
	unsigned Size;
	unsigned short Id;
}wmfObjCol;

typedef struct{
	unsigned Size;
	unsigned short Id;
	unsigned short Style;
	DWORD Col;
	unsigned short Hatch;
}wmfLogBrush;

typedef struct{
	unsigned Size;
	unsigned short Id;
	short lfh, lfw, lfesc, lfori, lfwei;
	unsigned char lfita, lfund, lfsto, lfcse, lfopre, lfclp, lfqua, lfpaqu;
	char face[32];
}wmfLogFont;

typedef struct{
	unsigned Size;
	unsigned short Id;
	unsigned short Style;
	unsigned Width;
	DWORD col;
}wmfLogPen;

typedef struct{
	unsigned Size;
	unsigned short Id;
	unsigned short x1, y1, x2, y2;
}wmfRect;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//the following routines resemble the corresponding GDI calls
unsigned short
ExportWMF::wmfCreateSolidBrush(DWORD color)
{
	wmfLogBrush lb = {7, 0x2fc, 0, color, 0};

	write(oFile, &lb, 14);
	currGDIobj++;
	maxGDIobj = currGDIobj > maxGDIobj ? currGDIobj : maxGDIobj;
	return currGDIobj-1;
}

unsigned short
ExportWMF::wmfCreateFontIndirect(short lfHeight, short lfWidth, short lfEscapement,
	short lfOrientation, short lfWeight, unsigned char lfItalic, unsigned char lfUnderline,
	unsigned char lfStrikeOut, unsigned char lfCharSet, unsigned char lfOutPrecision, 
	unsigned char lfClipPrecision, unsigned char lfQuality, unsigned char lfPitchAndFamily,
	char *FaceName)
{
	wmfLogFont lf = {28, 0x2fb, lfHeight, lfWidth, lfEscapement, lfOrientation, lfWeight,
		lfItalic, lfUnderline, lfStrikeOut, lfCharSet, lfOutPrecision, lfClipPrecision,
		lfQuality, lfPitchAndFamily, "Arial"};

	if(FaceName && FaceName[0]) rlp_strcpy(lf.face, 32, FaceName);
	write(oFile, &lf, 56);
	currGDIobj++;
	maxGDIobj = currGDIobj > maxGDIobj ? currGDIobj : maxGDIobj;
	return currGDIobj-1;
}

unsigned short
ExportWMF::wmfCreateSolidPen(DWORD color, unsigned short width)
{
	wmfLogPen lp = {8, 0x2fa, 0, width, color};

	write(oFile, &lp, 16);
	currGDIobj++;
	maxGDIobj = currGDIobj > maxGDIobj ? currGDIobj : maxGDIobj;
	return currGDIobj-1;
}

void
ExportWMF::wmfDeleteObject(unsigned short obj)
{
	wmfObjShort wo = {4, 0x1f0, obj};

	if(currGDIobj == obj+1) {
//		write(oFile, &wo, 8);
//		currGDIobj--;
		}
}

void
ExportWMF::wmfEllipse(unsigned short ix1, unsigned short iy1, unsigned short ix2, unsigned short iy2)
{
	wmfRect rc = {7, 0x418, iy2, ix2, iy1, ix1};
	
	write(oFile, &rc, 14);
}

void
ExportWMF::wmfPolyline(POINT *pts, unsigned short cp)
{
	wmfObjShort pl = {4+cp*2, 0x325, cp};
	unsigned short v[2];
	int i;

	write(oFile, &pl, 8);
	for(i = 0; i < cp; i++) {
		v[0] = (unsigned short)pts[i].x;
		v[1] = (unsigned short)pts[i].y;
		write(oFile, &v, 4);
		}
	if(pl.Size > rec_size) rec_size = pl.Size;
}

void
ExportWMF::wmfPolygon(POINT *pts, unsigned short cp)
{
	wmfObjShort pl = {4+cp*2, 0x324, cp};
	unsigned short v[2];
	int i;

	write(oFile, &pl, 8);
	for(i = 0; i < cp; i++) {
		v[0] = (unsigned short)pts[i].x;
		v[1] = (unsigned short)pts[i].y;
		write(oFile, &v, 4);
		}
	if(pl.Size > rec_size) rec_size = pl.Size;
}

void
ExportWMF::wmfRectangle(unsigned short ix1, unsigned short iy1, unsigned short ix2, unsigned short iy2)
{
	wmfRect rc = {7, 0x41B, iy2, ix2, iy1, ix1};
	
	write(oFile, &rc, 14);
}

void
ExportWMF::wmfSelectObject(unsigned short o)
{
	wmfObjShort so  = {4, 0x12D, o};

	write(oFile, &so, 8);
}

void
ExportWMF::wmfSetBkColor(DWORD col)
{
	wmfObjCol co = {5, 0x201};

	write(oFile, &co, 6);
	write(oFile, &col, 4);
}

void
ExportWMF::wmfSetBkMode(unsigned m)
{
	wmfObjShort mo = {5, 0x102, m & 0xffff};
	unsigned short p;

	write(oFile, &mo, 8);
	p = m>>16;
	write(oFile, &p, 2);
}

//cmSetMapMode()
//cmSetPolyFillMode()

void
ExportWMF::wmfSetTextAlign(unsigned a)
{
	wmfObjShort ao = {5, 0x12E, a & 0xffff};
	unsigned short p;

	write(oFile, &ao, 8);
	p = a>>16;
	write(oFile, &p, 2);
}

void
ExportWMF::wmfSetTextColor(DWORD col)
{
	wmfObjCol tc = {5, 0x209};

	write(oFile, &tc, 6);
	write(oFile, &col, 4);
}

//cmSetWindowExt()
//cmSetWindowOrg()

void
ExportWMF::wmfTextOut(unsigned short ix1, unsigned short iy1, char *txt, unsigned short cb)
{
	wmfObjShort to  = {6, 0x521, cb};
	unsigned short le, v[2] = {iy1, ix1};

	le = cb &1 ? cb+1 : cb;
	to.Size += le>>1;
	write(oFile, &to, 8);
	write(oFile, txt, le);
	write(oFile, &v, 4);	
	if(to.Size > rec_size) rec_size = to.Size;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Entry point to export graph to Windows Meta File
void DoExportWmf(GraphObj *g, char *FileName, float res, DWORD flags)
{
	ExportWMF *ex;
	

	InfoBox("The export of Windos metafile (*.wmf) is deprecated.\n\nThis feature will be removed in future\nversions of RLPlot!\n\n");
	ex = new ExportWMF(g, FileName, res, flags);
	if(ex->StartPage()) {
		g->DoPlot(ex);
		ex->EndPage();
		}
	delete(ex);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// export to *.svg file (scalable vector graphic)
// this code is based on information from the following books
// H. Spona, 'Das Einsteigerseminar SVG-Webgrafiken mit XML',
//     vmi, ISBN 3-8266-7181-3
// M. Salathe, 'SVG Scalabe Vector Graphics ...fr professionelle Einsteiger',
//     M&T, ISBN 3-8272-6188-0
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
class ExportSVG:public anyOutput {
public:
	HatchOut *hgo;

	ExportSVG(GraphObj *g, char *FileName, DWORD flags);
	~ExportSVG();
	bool SetLine(LineDEF *lDef);
	bool SetFill(FillDEF *fill);
	bool SetTextSpec(TextDEF *set);
	bool StartPage();
	bool EndPage();
	bool oCircle(int x1, int y1, int x2, int y2, char* nam = 0L);
	bool oPolyline(POINT * pts, int cp, char *nam = 0L);
	bool oRectangle(int x1, int y1, int x2, int y2, char *nam = 0L);
	bool oSolidLine(POINT *p);
	bool oTextOut(int x, int y, char *txt, int cb);
	bool oTextOutW(int x, int y, w_char *txt, int cb);
	bool oPolygon(POINT *pts, int cp, char * nam = 0L);

private:
	int iLineWidth, cb_out;
	bool bUseGroupLine, bOutputPending;
	GraphObj *go;
	char *name, indent[80], output[120], tHatchStyle[80];
	FILE *oFile;
	DWORD flags;

	bool com_TextOut(int x, int y, char *txt, int cb);
	void Indent(bool ind);
	void AddToOutput(char *txt, int len);
	char *ColName(DWORD col);
};

ExportSVG::ExportSVG(GraphObj *g, char *FileName, DWORD flg)
{
	hgo =0L;
	DeskRect.left = DeskRect.top = 0;
	DeskRect.right = DeskRect.bottom = 0x4fffffff;
	dFillCol = 0xffffffffL;
	hres = vres = 1000.0f;
	go = g;
	if(FileName)name = (char*)memdup(FileName, (int)strlen(FileName)+1, 0);
	else name = 0L;
	oFile = 0L;
	flags = flg;
	rlp_strcpy(indent, 80, "   ");
	rlp_strcpy(tHatchStyle, 80, "style=\"stroke:black; stroke-width:1\"");
	bUseGroupLine = false;
}

ExportSVG::~ExportSVG()
{
	if(hgo) delete hgo;
	if(name) free(name);
}

bool
ExportSVG::SetLine(LineDEF *lDef)
{
	LineWidth = lDef->width;
	if(1 >(iLineWidth  = iround(un2fix(lDef->width)))) iLineWidth = 1;
	dPattern = lDef->pattern;
	dLineCol = lDef->color;
	RLP.finc = (float)(256.0/un2fix(lDef->patlength*8.0));
	RLP.fp = 0.0;
	return true;
}

bool
ExportSVG::SetFill(FillDEF *fill)
{
	int iL; 

	if(!fill) return false;
	if((fill->type & 0xff) != FILL_NONE) {
		if(!hgo) hgo = new HatchOut(this);
		if(hgo) hgo->SetFill(fill);
		if(fill->hatch) {
			if(1 >(iL  = iround(un2fix(fill->hatch->width)))) iL = 1;
#ifdef USE_WIN_SECURE
			sprintf_s(tHatchStyle, 80, "style=\"fill:none; stroke:%s; stroke-width:%d\"",
				ColName(fill->hatch->color), iL);
#else
			sprintf(tHatchStyle, "style=\"fill:none; stroke:%s; stroke-width:%d\"",
				ColName(fill->hatch->color), iL);
#endif
			}
		}
	else {
		if(hgo) delete hgo;
		hgo = 0L;
		}
	dFillCol = fill->color;
	dFillCol2 = fill->color2;
	return true;
}

bool
ExportSVG::SetTextSpec(TextDEF *set)
{
	if(set->fSize > 0.0) {
		if((set->Style & TXS_SUPER) || (set->Style & TXS_SUB))
			set->iSize = un2iy(set->fSize * 0.71);
		else set->iSize = un2iy(set->fSize);
		}
	if(!set->iSize) return false;
	return anyOutput::SetTextSpec(set);
}

bool 
ExportSVG::StartPage()
{
	int w, h;

	if(!go) return false;
	w = un2ix(go->GetSize(SIZE_GRECT_RIGHT) - go->GetSize(SIZE_GRECT_LEFT))/10;
	h = un2iy(go->GetSize(SIZE_GRECT_BOTTOM) - go->GetSize(SIZE_GRECT_TOP))/10;
	w++; h++;
	if(name) {
#ifdef USE_WIN_SECURE
		fopen_s(&oFile, name, "w");
#else
		oFile = fopen(name, "w");
#endif
		if(!oFile) {
			ErrorBox("Could not open\noutput file!");
			return false;
			}
		}
	else oFile = stdout;
	if(flags & 0x01) fprintf(oFile, "Content-Type: image/svg+xml\n\n");
	VPorg.fy = -un2fiy(go->GetSize(SIZE_GRECT_TOP));
	VPorg.fx = -un2fix(go->GetSize(SIZE_GRECT_LEFT));
	fprintf(oFile, "<?xml version=\"1.0\"?>\n"
		"<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 20001102//EN\"\n"
		"   \"http://www.w3.org/TR/2000/CR-SVG-20001102/DTD/svg-20001102.dtd\">\n");
	fprintf(oFile, "<svg %s width=\"%d\" height=\"%d\" style=\"stroke-linecap:round\">\n", 
		defs.svgAttr ? defs.svgAttr : "", w, h);
	if(defs.svgScript) {
		fprintf(oFile, "<defs>\n<script type=\"text/ecmascript\"><![CDATA[\n");
		fprintf(oFile, "\n%s\n", defs.svgScript);
		fprintf(oFile, "\n]]></script>\n</defs>\n\n");
		}
	fprintf(oFile, "<g transform=\"scale(0.1)\" style=\"font-family:Helvetica\">\n");
	return true;
}

bool
ExportSVG::EndPage()
{
	fprintf(oFile, "</g>\n</svg>\n");
	fclose (oFile);
	return true;
}

bool
ExportSVG::oCircle(int x1, int y1, int x2, int y2, char* nam)
{
	if(x1 > x2) Swap(x1, x2);
	if(y1 > y2) Swap(y1, y2);

	if(hgo){
		fprintf(oFile, "%s<g>  <!-- %s with pattern -->\n", indent, 
			(x2-x1) == (y2-y1) ? "circle" : "ellipse");
		Indent(true);
		}
	fprintf(oFile, "%s<%s%s%s%s cx=\"%d\" cy=\"%d\" ", indent,
		(x2-x1) == (y2-y1) ? "circle" : "ellipse", nam? " name=\"" : "", 
		nam ? nam : "", nam ? "\"" : "", (x1+x2)/2, (y1+y2)/2);
	if((x2-x1) == (y2-y1)) fprintf(oFile, "r=\"%d\"", (x2-x1)/2);
	else fprintf(oFile, "rx=\"%d\" ry=\"%d\"", (x2-x1)/2, (y2-y1)/2);
	fprintf(oFile, " style=\"fill:%s; stroke:%s; stroke-width:%d\"/>\n",
		ColName(dFillCol), ColName(dLineCol), iLineWidth);
	if(hgo) {
		fprintf(oFile, "%s<g %s>  <!-- hatch -->\n", indent, tHatchStyle);
		Indent(true);
		bUseGroupLine = true;
		hgo->oCircle(x1, y1, x2, y2);
		Indent(false);
		bUseGroupLine = false;
		fprintf(oFile, "%s</g>\n", indent);
		Indent(false);
		fprintf(oFile, "%s</g>\n", indent);
		}
	return true;
}

bool
ExportSVG::oPolyline(POINT *pts, int cp, char *nam)
{
	int i, cb;
	char tmptxt[40];

	if(cp < 2) return false;
	if (dPattern){
		fprintf(oFile, "%s<g style=\"stroke:%s; stroke-width:%d; stroke-linecap:round\">"
			"<!-- pattern line -->\n", indent, ColName(dLineCol), iLineWidth);
		Indent(true);
		bUseGroupLine = true;
		for (i = 1; i < cp; i++) PatLine(pts[i-1], pts[i]);
		Indent(false);
		fprintf(oFile, "%s</g>\n", indent);
		bUseGroupLine = false;
		}
	else {
		if(cp == 2) return oSolidLine(pts);
		bOutputPending = false;
		cb_out = sprintf(output, "<polyline points=\""); 
		for(i = 0; i < cp; i++) {
			cb = sprintf(tmptxt, "%d %d ", pts[i].x, pts[i].y);
			AddToOutput(tmptxt, cb);
			}
		if(cb_out) output[cb_out-1] = '"';
		if(!bUseGroupLine) {
			cb = rlp_strcpy(tmptxt, 120, " style = \"fill:none; ");
			AddToOutput(tmptxt, cb);
			cb = rlp_strcpy(tmptxt, 120, "; stroke:");
			cb += rlp_strcpy(tmptxt+cb, 120-cb, ColName(dLineCol));
			cb += rlp_strcpy(tmptxt+cb, 120-cb, "; ");
			AddToOutput(tmptxt, cb);
#ifdef USE_WIN_SECURE
			cb = sprintf_s(tmptxt, 120, "stroke-width:%d\"/>",iLineWidth);
#else
			cb = sprintf(tmptxt, "stroke-width:%d\"/>",iLineWidth);
#endif
			AddToOutput(tmptxt, cb);
			}
		else AddToOutput("/>", 2);
		fprintf(oFile, "%s%s\n", indent, output);
		if(bOutputPending)Indent(false);
		}
	return true;
}

bool
ExportSVG::oRectangle(int x1, int y1, int x2, int y2, char *nam)
{
	if(x1 > x2) Swap(x1, x2);
	if(y1 > y2) Swap(y1, y2);
	if(hgo){
		fprintf(oFile, "%s<g>  <!-- rectangle with pattern -->\n", indent);
		Indent(true);
		}
	fprintf(oFile, "%s<rect%s%s%s x=\"%d\" y=\"%d\" width=\"%d\" height=\"%d\" "
		"style=\"fill:%s; stroke:%s; stroke-width:%d\"/>\n",
		indent, nam? " name=\"" : "", nam ? nam : "", nam ? "\"" : "",
		x1, y1, x2-x1-1, y2-y1-1, ColName(dFillCol), ColName(dLineCol), iLineWidth);
	if(hgo) {
		fprintf(oFile, "%s<g %s>  <!-- hatch -->\n", indent, tHatchStyle);
		Indent(true);
		bUseGroupLine = true;
		hgo->oRectangle(x1, y1, x2, y2, 0L);
		Indent(false);
		bUseGroupLine = false;
		fprintf(oFile, "%s</g>\n", indent);
		Indent(false);
		fprintf(oFile, "%s</g>\n", indent);
		}
	return true;
}

bool
ExportSVG::oSolidLine(POINT *p)
{
	if(bUseGroupLine) fprintf(oFile, "%s<line x1=\"%d\" y1=\"%d\" x2=\"%d\" y2=\"%d\"/>\n",
		indent, p[0].x, p[0].y, p[1].x, p[1].y);
	else fprintf(oFile, "%s<line x1=\"%d\" y1=\"%d\" x2=\"%d\" y2=\"%d\" "
		"style=\"stroke:%s; stroke-width:%d\"/>\n",
		indent, p[0].x, p[0].y, p[1].x, p[1].y, ColName(dLineCol), iLineWidth);
	return true;
}

bool
ExportSVG::com_TextOut(int x, int y, char *txt, int cb)
{
	int c, h, ix, iy, dy;
	char tmptxt[120];

	if(!txt || !txt[0]) return false;
	else h = TxtSet.iSize;
	if(TxtSet.Align & TXA_VCENTER) iy = y + h/3;
	else if(TxtSet.Align & TXA_VBOTTOM) iy = y;
	else iy = y + iround(h * 0.8);
	ix = x;		dy = 0;
	if(TxtSet.Style & TXS_SUB) {
		if((TxtSet.Align & TXA_VCENTER) == TXA_VCENTER) dy = un2iy(TxtSet.fSize*0.4);
		else if((TxtSet.Align & TXA_VBOTTOM) == TXA_VBOTTOM) dy = un2iy(TxtSet.fSize*0.2);
		else if((TxtSet.Align & TXA_VTOP) == TXA_VTOP) dy = un2iy(TxtSet.fSize*.6);
		}
	else if(TxtSet.Style & TXS_SUPER) {
		if((TxtSet.Align & TXA_VCENTER) == TXA_VCENTER) dy = -un2iy(TxtSet.fSize*0.4);
		else if((TxtSet.Align & TXA_VBOTTOM) == TXA_VBOTTOM) dy = -un2iy(TxtSet.fSize*0.6);
		else if((TxtSet.Align & TXA_VTOP) == TXA_VTOP) dy = -un2iy(TxtSet.fSize*0.2);
		}
#ifdef USE_WIN_SECURE
	cb_out = sprintf_s(output, 120, "<text x=\"%d\" y=\"%d\" dy=\"%d\" ", ix, iy, dy);
#else
	cb_out = sprintf(output, "<text x=\"%d\" y=\"%d\" dy=\"%d\" ", ix, iy, dy);
#endif
	if(fabs(TxtSet.RotBL) >.01 || fabs(TxtSet.RotCHAR) >.01) {
#ifdef USE_WIN_SECURE
		cb_out += sprintf_s(output+cb_out, 120-cb_out, "transform=\"rotate(%.0f,%d,%d)\" ", -TxtSet.RotBL, ix, iy);
#else
		cb_out += sprintf(output+cb_out,"transform=\"rotate(%.0f,%d,%d)\" ", -TxtSet.RotBL, ix, iy);
#endif
		}
	c = rlp_strcpy(tmptxt, 140, "style=\"font-family:");
	switch(TxtSet.Font) {
	case FONT_TIMES:	c += rlp_strcpy(tmptxt+c, 120-c, "Times;");		break;
	case FONT_COURIER:	c += rlp_strcpy(tmptxt+c, 120-c, "Courier;");	break;
	default:			c += rlp_strcpy(tmptxt+c, 120-c, "Helvetica;");	break;
		}
	if(TxtSet.Style & TXS_ITALIC) c += rlp_strcpy(tmptxt+c, 120-c, " font-style:italic;");
	if(TxtSet.Style & TXS_BOLD) c += rlp_strcpy(tmptxt+c, 120-c, " font-weight:bold;");
	if(TxtSet.Style & TXS_UNDERLINE) c += rlp_strcpy(tmptxt+c, 120-c, " text-decoration:underline;");
	AddToOutput(tmptxt, c);
#ifdef USE_WIN_SECURE
	c = sprintf_s(tmptxt, 120, " fill:%s; stroke:%s; ", ColName(TxtSet.ColTxt), ColName(TxtSet.ColTxt));
#else
	c = sprintf(tmptxt, " fill:%s; stroke:%s; ", ColName(TxtSet.ColTxt), ColName(TxtSet.ColTxt));
#endif
	AddToOutput(tmptxt, c);
#ifdef USE_WIN_SECURE
	c = sprintf(tmptxt, "font-size:%d; text-anchor:%s \">", h, 
		(TxtSet.Align & TXA_HRIGHT) ? "end" : (TxtSet.Align & TXA_HCENTER) ? "middle":"start");
#else
	c = sprintf(tmptxt, "font-size:%d; text-anchor:%s \">", h, 
		(TxtSet.Align & TXA_HRIGHT) ? "end" : (TxtSet.Align & TXA_HCENTER) ? "middle":"start");
#endif
	AddToOutput(tmptxt, c);
	if((strlen(indent)+strlen(txt)+cb_out) <110) cb_out += rlp_strcpy(output+cb_out, 120-cb_out, txt);
	else {
		fprintf(oFile, "%s%s\n", indent, output);
		cb_out=rlp_strcpy(output, 120, txt);
		}
	if((strlen(indent) + cb_out) <104) 
		fprintf(oFile, "%s%s</text>\n", indent, output);
	else {
		fprintf(oFile, "%s%s\n", indent, output);
		fprintf(oFile, "</text>\n");
		}
	return true;
}

bool
ExportSVG::oTextOut(int x, int y, char *txt, int cb)
{
	char *nt;

	if(!txt || !txt[0]) return false;
	nt = str2xml(txt, TxtSet.Font == FONT_GREEK);
	return com_TextOut(x, y, nt, cb);
}

bool
ExportSVG::oTextOutW(int x, int y, w_char *txt, int cb)
{
	int i, j;
	wchar_t wc;
	char c;

	for(i = j = 0; txt[i]; i++) {
		switch(txt[i]) {
			case '"':
				j += rlp_strcpy(TmpTxt+j, TMP_TXT_SIZE-j, "&quot;");
				break;
			case '&':
				j += rlp_strcpy(TmpTxt+j, TMP_TXT_SIZE-j, "&amp;");
				break;
			case '<':
				j += rlp_strcpy(TmpTxt+j, TMP_TXT_SIZE-j, "&lt;");
				break;
			case '>':
				j += rlp_strcpy(TmpTxt+j, TMP_TXT_SIZE-j, "&gt;");
				break;
			default:
				if(txt[i] > 255) {
#ifdef USE_WIN_SECURE
					j += sprintf_s(TmpTxt+j, TMP_TXT_SIZE-j, "&#%d;", txt[i]);
#else
					j += sprintf(TmpTxt+j, "&#%d;", txt[i]);
#endif
					}
				else if(txt[i] > 127) {
					c = (char)txt[i];
					if(mbtowc(&wc, &c, 1) >0) 
#ifdef USE_WIN_SECURE
						j += sprintf_s(TmpTxt+j, TMP_TXT_SIZE-j, "&#%d;", ((unsigned short)wc));
#else
						j += sprintf(TmpTxt+j, "&#%d;", ((unsigned short)wc));
#endif
					}
				else TmpTxt[j++] = (char)txt[i];
				break;
			}
		}
	TmpTxt[j++] = 0;
	return com_TextOut(x, y, TmpTxt, j-1);
}

bool
ExportSVG::oPolygon(POINT *pts, int cp, char *nam)
{
	int i, cb;
	char tmptxt[40];

	if(cp <3) return false;
	if(hgo){
		fprintf(oFile, "%s<g>  <!-- polygon with pattern -->\n", indent);
		Indent(true);
		}
	bOutputPending = false;
#ifdef USE_WIN_SECURE
	cb_out = sprintf_s(output, 120, "<polygon%s%s%s points=\"",
		nam? " name=\"" : "", nam ? nam : "", nam ? "\"" : ""); 
#else
	cb_out = sprintf(output, "<polygon%s%s%s points=\"",
		nam? " name=\"" : "", nam ? nam : "", nam ? "\"" : ""); 
#endif
	for(i = 0; i < cp; i++) {
#ifdef USE_WIN_SECURE
		cb = sprintf_s(tmptxt, 40, "%d %d ", pts[i].x, pts[i].y);
#else
		cb = sprintf(tmptxt, "%d %d ", pts[i].x, pts[i].y);
#endif
		AddToOutput(tmptxt, cb);
		}
	if(cb_out) output[cb_out-1] = '"';
#ifdef USE_WIN_SECURE
	cb = sprintf(tmptxt, "style=\"fill:%s; ", ColName(dFillCol));
	AddToOutput(tmptxt, cb);
	cb = sprintf(tmptxt, "stroke:%s; ", ColName(dLineCol));
	AddToOutput(tmptxt, cb);
	cb = sprintf(tmptxt, "stroke-width:%d\"/>",iLineWidth);
	AddToOutput(tmptxt, cb);
#else
	cb = sprintf(tmptxt, "style=\"fill:%s; ", ColName(dFillCol));
	AddToOutput(tmptxt, cb);
	cb = sprintf(tmptxt, "stroke:%s; ", ColName(dLineCol));
	AddToOutput(tmptxt, cb);
	cb = sprintf(tmptxt, "stroke-width:%d\"/>",iLineWidth);
	AddToOutput(tmptxt, cb);
#endif
	if(output)fprintf(oFile, "%s%s\n", indent, output);
	if(bOutputPending)Indent(false);
	if(hgo) {
		fprintf(oFile, "%s<g %s>  <!-- hatch -->\n", indent, tHatchStyle);
		Indent(true);
		bUseGroupLine = true;
		hgo->oPolygon(pts, cp);
		Indent(false);
		bUseGroupLine = false;
		fprintf(oFile, "%s</g>\n", indent);
		Indent(false);
		fprintf(oFile, "%s</g>\n", indent);
		}
	return true;
}

void
ExportSVG::Indent(bool ind)
{
	int i = (int)strlen(indent);

	if(i > 20 && ind) return;
	if(ind) strcat(indent, "   ");
	else if(i>2) indent[i-3] = 0;
}

void
ExportSVG::AddToOutput(char *txt, int len)
{
	if(!txt || !txt[0]) return;
	if(!len) len = (int)strlen(txt);
	if((len + cb_out + strlen(indent)) < 110){
		cb_out += rlp_strcpy(output+cb_out, 120, txt);
		}
	else {
		fprintf(oFile, "%s%s\n", indent, output);
		if(!bOutputPending) Indent(true);
		bOutputPending = true;
		cb_out = rlp_strcpy(output, 120, txt);
		}
}

char *
ExportSVG::ColName(DWORD col)
{
	static char txt1[20], txt2[20];
	static int sw;

	switch(col) {
	case 0x00000000:			return "black";
	case 0x000000ff:			return "red";
	case 0x0000ff00:			return "lime";
	case 0x0000ffff:			return "yellow";
	case 0x00ff0000:			return "blue";
	case 0x00ff00ff:			return "magenta";
	case 0x00ffff00:			return "cyan";
	case 0x00ffffff:			return "white";
		}
	sw++;
	if(sw & 0x01) {
		sprintf(txt1, "rgb(%d,%d,%d)", col & 0xff, (col>>8) & 0xff, (col>>16) &0xff);
		return txt1;
		}
	else {
		sprintf(txt2, "rgb(%d,%d,%d)", col & 0xff, (col>>8) & 0xff, (col>>16) &0xff);
		return txt2;
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Entry point to export graph to *.svg
void DoExportSvg(GraphObj *g, char *FileName, DWORD flags)
{
	ExportSVG *ex;
	
	ex = new ExportSVG(g, FileName, flags);
	if(ex->StartPage()) {
		g->DoPlot(ex);
		ex->EndPage();
		}
	HideTextCursor();	
	delete(ex);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// export to *.eps file (encapsulated post script).
// This code is based on information from the following book
// G. Born, 'Referenzhandbuch Dateiformate', 
//     Addison-Wesley ISBN 3-89319-815-6
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
class ExportEPS:public anyOutput {
public:
	HatchOut *hgo;

	ExportEPS(GraphObj *g, char *FileName, DWORD flags);
	~ExportEPS();
	bool SetLine(LineDEF *lDef);
	bool SetFill(FillDEF *fill);
	bool SetTextSpec(TextDEF *set);
	bool StartPage();
	bool EndPage();
	bool oCircle(int x1, int y1, int x2, int y2, char* nam = 0L);
	bool oPolyline(POINT * pts, int cp, char *nam = 0L);
	bool oRectangle(int x1, int y1, int x2, int y2, char *nam = 0L);
	bool oSolidLine(POINT *p);
	bool oTextOut(int x, int y, char *txt, int cb);
	bool oPolygon(POINT *pts, int cp, char *nam = 0L);

private:
	RECT BoundBox;
	fRECT SolLines[2000];
	int nSolLines;
	GraphObj *go;
	char *name, FontName[30];
	FILE *oFile;
	DWORD CurrCol;
	bool bFontChange;

	float ix2eps(int x);
	float iy2eps(int y);
	char *col2eps(DWORD color);
	void FlushSolLines();
	void AddSolLine(float x1, float y1, float x2, float y2);
};

ExportEPS::ExportEPS(GraphObj *g, char *FileName, DWORD flags)
{
	hgo =0L;		nSolLines = 0;			DeskRect.left = DeskRect.top = 0;
	DeskRect.right = DeskRect.bottom = 0x4fffffff;
	dFillCol = 0xffffffffL;					hres = vres = 720.0f;
	go = g;			bFontChange = false;	oFile = 0L;
	if(FileName)name = strdup(FileName);	else name = 0L;
}

ExportEPS::~ExportEPS()
{
	if(hgo) delete hgo;
	if(name) free(name);
}

bool
ExportEPS::SetLine(LineDEF *lDef)
{
	if(LineWidth != lDef->width || dLineCol != lDef->color) {
		FlushSolLines();
		LineWidth = lDef->width;
		CurrCol = dLineCol = lDef->color;
		fprintf(oFile, "\nnewpath %.1f setlinewidth %s ",
			un2fix(LineWidth)/10.0f, col2eps(dLineCol));
		}
	dPattern = lDef->pattern;
	RLP.finc = (float)(256.0/un2fix(lDef->patlength*8.0));
	RLP.fp = 0.0;
	return true;
}

bool
ExportEPS::SetFill(FillDEF *fill)
{
	if(!fill) return false;
	if((fill->type & 0xff) != FILL_NONE) {
		if(!hgo) hgo = new HatchOut(this);		if(hgo) hgo->SetFill(fill);
		}
	else {
		if(hgo) delete hgo;						hgo = 0L;
		}
	dFillCol = fill->color;						dFillCol2 = fill->color2;
	return true;
}

bool
ExportEPS::SetTextSpec(TextDEF *set)
{
	int cb;

	if(set->fSize > 0.0) {
		if((set->Style & TXS_SUPER) || (set->Style & TXS_SUB)) set->iSize = un2iy(set->fSize * 0.71);
		else set->iSize = un2iy(set->fSize);
		}
	if(!set->iSize) return false;
	anyOutput::SetTextSpec(set);
	switch(TxtSet.Font) {
	case FONT_TIMES:	cb = rlp_strcpy(FontName, 30, "(Times");		break;		//Serif
	case FONT_COURIER:	cb = rlp_strcpy(FontName, 30, "(Courier");		break;		//fixed spaced
	default:			cb = rlp_strcpy(FontName, 30, "(Helvetica");	break;		//Sans Serif	
		}
	if(TxtSet.Style & TXS_BOLD) cb += rlp_strcpy(FontName+cb, 30-cb, "-Bold");
	if(TxtSet.Style & TXS_ITALIC) cb += rlp_strcpy(FontName+cb, 30-cb, "-Italic");
	cb += rlp_strcpy(FontName+cb, 30-cb, ")");		bFontChange = true;
	return true;
}

bool 
ExportEPS::StartPage()
{
	time_t ti;
	
	if(!go) return false;
	ti = time(0L);
	BoundBox.top = BoundBox.left = 0;
	BoundBox.right = un2ix(go->GetSize(SIZE_GRECT_RIGHT) - go->GetSize(SIZE_GRECT_LEFT))/10;
	BoundBox.bottom = un2iy(go->GetSize(SIZE_GRECT_BOTTOM) - go->GetSize(SIZE_GRECT_TOP))/10;
	BoundBox.right++;	BoundBox.bottom++;
	if(name) {
		oFile = fopen(name, "w");
		if(!oFile) {
			ErrorBox("Could not open\noutput file!");
			return false;
			}
		}
	else oFile = stdout;
	VPorg.fy = -un2fiy(go->GetSize(SIZE_GRECT_TOP));
	VPorg.fx = -un2fix(go->GetSize(SIZE_GRECT_LEFT));
	fprintf(oFile, "%%!PS-Adobe-1.0 EPSF-3.0\n"
		"%%%%BoundingBox: %d %d %d %d\n", BoundBox.left, BoundBox.top,
		BoundBox.right, BoundBox.bottom);
	fprintf(oFile, "%%%%Title: %s\n", name);
	fprintf(oFile,"%%%%Creator: RLPlot version "SZ_VERSION"\n");
	fprintf(oFile,"%%%%CreationDate: %s", ctime(&ti));
	fprintf(oFile, "%%%%Pages: 1\n%%%%DocumentFonts: (atend)\n");
	fprintf(oFile, "%%%%EndComments\n"
		"%%%%BeginProlog\n"
		"%%%%EndProlog\n"
		"%%%%Page: 1 1");
	return true;
}

bool
ExportEPS::EndPage()
{
	fprintf(oFile, "\nshowpage\n%%%%Trailer\n");
	fprintf(oFile, "%%%%DocumentFonts: Helvetica\n");
	fprintf(oFile, "%%%%EOF\n");
	fclose (oFile);
	return true;
}

bool
ExportEPS::oCircle(int x1, int y1, int x2, int y2, char* nam)
{
	FlushSolLines();
	if((x2 - x1) == (y2 -y1)) {
		fprintf(oFile, "\nnewpath %.1f %.1f %.1f 0 360 arc ", ix2eps((x1+x2)/2), 
			iy2eps((y1+y2)/2), (float)(x2-x1)/20.0f);
		fprintf(oFile, "%s fill", col2eps(CurrCol = dFillCol));
		if(hgo){
			hgo->oCircle(x1, y1, x2, y2);
			FlushSolLines();
			}
		fprintf(oFile, "\nnewpath %.1f %.1f %.1f 0 360 arc ", ix2eps((x1+x2)/2), 
			iy2eps((y1+y2)/2), (float)(x2-x1)/20.0f);
		fprintf(oFile, "%s stroke", col2eps(CurrCol = dLineCol));
		}
	else if(x1 != x2 && y1 != y2){
		fprintf(oFile, "\ngsave %.1f %.1f translate", (ix2eps((x1+x2)>>1)),
			(iy2eps((y1+y2)>>1)));
		if(abs(x2-x1) > abs(y2-y1)) {
			fprintf(oFile, " 1 %lf scale", fabs(((double)(y2-y1))/((double)(x2-x1))));
			fprintf(oFile, " 0 0 %.1lf 0 360 arc ", fabs(((double)(x2-x1))/20.0));
			fprintf(oFile, "%s fill", col2eps(CurrCol = dFillCol));
			fprintf(oFile, "\n 0 0 %.1lf 0 360 arc ", fabs(((double)(x2-x1))/20.0));
			}
		else {
			fprintf(oFile, " %lf 1 scale", fabs(((double)(x2-x1))/((double)(y2-y1))));
			fprintf(oFile, " 0 0 %.1lf 0 360 arc ", fabs(((double)(y2-y1))/20.0));
			fprintf(oFile, "%s fill", col2eps(CurrCol = dFillCol));
			fprintf(oFile, "\n 0 0 %.1lf 0 360 arc ", fabs(((double)(y2-y1))/20.0));
			}
		fprintf(oFile, "%s stroke grestore", col2eps(CurrCol = dLineCol));
		if(hgo){
			hgo->oCircle(x1, y1, x2, y2);
			FlushSolLines();
			}
		}
	return true;
}

bool
ExportEPS::oPolyline(POINT * pts, int cp, char *nam)
{
	int i, j;

	if(cp <1) return false;
	if (dPattern){
		for (i = 1; i < cp; i++) PatLine(pts[i-1], pts[i]);
		return true;
		}
	else if(cp == 2) return oSolidLine(pts);
	FlushSolLines();
	if(CurrCol != dLineCol) {
		fprintf(oFile, "\n%s ", col2eps(CurrCol = dLineCol));
		}
	for(i = cp-1, j = 0; i >= 0; i--, j++) {
		if(!(j & 0x07)) fprintf(oFile, "\n");
		fprintf(oFile, " %.1f %.1f", ix2eps(pts[i].x), iy2eps(pts[i].y));
		}
	fprintf(oFile, " moveto %d {lineto} repeat stroke ", cp-1);
	return true;
}


bool
ExportEPS::oRectangle(int x1, int y1, int x2, int y2, char *nam)
{
	FlushSolLines();
	fprintf(oFile, "\n%.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f moveto 4 {lineto}"
		" repeat %s fill", ix2eps(x1), iy2eps(y1), ix2eps(x1), iy2eps(y2), ix2eps(x2), iy2eps(y2), 
		ix2eps(x2), iy2eps(y1), ix2eps(x1), iy2eps(y1), col2eps(CurrCol = dFillCol));
	if(hgo) hgo->oRectangle(x1, y1, x2, y2, 0L);
	fprintf(oFile, "\n%.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f moveto 4 {lineto}"
		" repeat %s stroke", ix2eps(x1), iy2eps(y1), ix2eps(x1), iy2eps(y2), ix2eps(x2), iy2eps(y2), 
		ix2eps(x2), iy2eps(y1), ix2eps(x1), iy2eps(y1), col2eps(CurrCol = dLineCol));
	return true;
}

bool
ExportEPS::oSolidLine(POINT *p)
{
	if(CurrCol != dLineCol) {
		FlushSolLines();
		fprintf(oFile, "\n%s ", col2eps(CurrCol = dLineCol));
		}
	AddSolLine(ix2eps(p[0].x), iy2eps(p[0].y), ix2eps(p[1].x), iy2eps(p[1].y));
	return true;
}

bool
ExportEPS::oTextOut(int x, int y, char *txt, int cb)
{
	int h, ix, iy, w;
	float fx, fy, lw;

	FlushSolLines();	if(!txt || !txt[0]) return true;
	oGetTextExtent(txt, cb, &w, &h);
	if(bFontChange)	{
		fprintf(oFile, "\n%s findfont %d scalefont setfont ", FontName, TxtSet.iSize/10);
		bFontChange = false;
		}
	if(TxtSet.Align & TXA_VCENTER) iy = y + h/3;
	else if(TxtSet.Align & TXA_VBOTTOM) iy = y;
	else iy = y + iround(h*.8);
	if(TxtSet.Align & TXA_HRIGHT) ix = x-w;
	else if(TxtSet.Align & TXA_HCENTER) ix = x-w/2;
	else ix = x;			lw = (float)(iy-y)/150.0f;
	fprintf(oFile,"\n");
	if(fabs(TxtSet.RotBL) >.01 || fabs(TxtSet.RotCHAR) >.01) {
		if(TxtSet.Style & TXS_SUB) iy += un2iy(TxtSet.fSize*0.6);
		else if(TxtSet.Style & TXS_SUPER) iy -= un2iy(TxtSet.fSize*.2);
		fprintf(oFile, "gsave %.1f %.1f translate %f rotate %.1f %.1f moveto\n",
			ix2eps(x), iy2eps(y), TxtSet.RotBL, (float)(ix-x)/10.0f, (float)(iy-y)/-10.0f);
		fprintf(oFile, "%s ", col2eps(CurrCol = TxtSet.ColTxt));
		fprintf(oFile, "(%s) show ", txt);
		if(TxtSet.Style & TXS_UNDERLINE) {
			fprintf(oFile, "\ncurrentpoint %.1f exch pop moveto", (float)(iy-y)/-10.0f - lw*1.2);
			fprintf(oFile, " 0 %.1f lineto %s %.1f setlinewidth stroke ", (float)(iy-y)/-10.0f -lw*1.2,
				col2eps(TxtSet.ColTxt), lw);
			}
		fprintf(oFile, "grestore\n");
		}
	else {
		if(TxtSet.Style & TXS_SUB) iy += un2iy(TxtSet.fSize*0.6);
		else if(TxtSet.Style & TXS_SUPER) iy -= un2iy(TxtSet.fSize*.2);
		fx = ix2eps(ix);			fy = iy2eps(iy);
		fprintf(oFile, "%s ", col2eps(CurrCol = TxtSet.ColTxt));
		fprintf(oFile,"%.1f %.1f moveto (%s) show ", fx, fy, txt);
		if(TxtSet.Style & TXS_UNDERLINE) {
			fprintf(oFile, "\ncurrentpoint %.1f exch pop moveto", fy - lw*1.2);
			fprintf(oFile, " %.1f %.1f lineto %s %.1f setlinewidth stroke\n", fx, fy - lw*1.2, 
				col2eps(TxtSet.ColTxt), lw);
			}
		}
	return true;
}

bool
ExportEPS::oPolygon(POINT *pts, int cp, char *nam)
{
	int i, j;

	if(cp <1) return false;
	if(cp == 2) return oSolidLine(pts);
	FlushSolLines();
	for(i = cp-1, j = 0; i >= 0; i--, j++) {
		if(!(j & 0x07)) fprintf(oFile, "\n");
		fprintf(oFile, " %.1f %.1f", ix2eps(pts[i].x), iy2eps(pts[i].y));
		}
	fprintf(oFile, " moveto %d {lineto} repeat %s fill ", cp-1, col2eps(CurrCol = dFillCol));
	if(hgo) hgo->oPolygon(pts, cp);
	return oPolyline(pts, cp);
}

float
ExportEPS::ix2eps(int x)
{
	return (float)x/10.0f;
}

float
ExportEPS::iy2eps(int y)
{
	return (float)y/-10.0f + (float)BoundBox.bottom;
}

char *
ExportEPS::col2eps(DWORD color)
{
	static char txt[50];
	float r, g, b;

	r = (float)(color & 0xff)/255.0f;
	g = (float)((color>>8)&0xff)/255.0f;
	b = (float)((color>>16)&0xff)/255.0f;
#ifdef USE_WIN_SECURE
	sprintf_s(txt, 50, "%g %g %g setrgbcolor", r, g, b);
#else
	sprintf(txt, "%g %g %g setrgbcolor", r, g, b);
#endif
	return txt;
}

void
ExportEPS::FlushSolLines()
{
	int i, j;
	
	if(nSolLines <1) {
		nSolLines = 0;
		return;
		}
	if(nSolLines == 1) {
		fprintf(oFile, "\n%.1f %.1f moveto %.1f %.1f lineto stroke ", 
			SolLines[0].Ymin, SolLines[0].Ymax, SolLines[0].Xmin, SolLines[0].Xmax);
		nSolLines = 0;
		return;
		}
	for(i = nSolLines-1, j = 0; i >=0; i--, j++) {
		if(!(j & 0x03)) fprintf(oFile, "\n");
		fprintf(oFile, " %.1f %.1f %.1f %.1f", SolLines[i].Xmin, SolLines[i].Xmax,
			SolLines[i].Ymin, SolLines[i].Ymax);
		}
	if(j > 8 && ((j & 0x3) >=2 || (j & 0x03) == 0)) fprintf(oFile, "\n");
	fprintf(oFile, " %d {moveto lineto} repeat stroke ", nSolLines);
	nSolLines = 0;
}

void
ExportEPS::AddSolLine(float x1, float y1, float x2, float y2)
{
	if(nSolLines >= 2000) FlushSolLines();
	SolLines[nSolLines].Ymin = x1;	SolLines[nSolLines].Ymax = y1;
	SolLines[nSolLines].Xmin = x2;	SolLines[nSolLines].Xmax = y2;
	nSolLines++;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Entry point to export graph to *.eps
void DoExportEps(GraphObj *g, char *FileName, DWORD flags)
{
	ExportEPS *ex;

	ex = new ExportEPS(g, FileName, flags);
	if(ex->StartPage()) {
		g->DoPlot(ex);
		ex->EndPage();
		}
	delete(ex);
}
