/*  readline.c: interacting with the GNU readline library */

/*  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License , or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; see the file COPYING.  If not, write to
    the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

    You may contact the author by:
       e-mail:  hlub@knoware.nl
*/


#include "rlwrap.h"

/* global vars */
int remember_for_completion =  FALSE; /* whether we should put al words from in/output on the list */ 
int use_completion_list = FALSE;      /* use our own completion wordlist  */
struct rl_state  saved_rl_state ={"","",0,0}; /* saved state of readline */
static char return_key;               /* Key pressed to enter line */



char previous_line[BUFFSIZE]="";      /* previous input line */

/* forward declarations */
void line_handler(char * );
void my_add_history(char *);
int my_accept_line(int,int);

static void my_redisplay(void);

void init_readline(char *prompt) {
  rl_add_defun ("rlwrap_accept_line", my_accept_line, -1);
  rl_add_defun ("insert_close", rl_insert_close, -1);
   
  rl_bind_key('\n', my_accept_line); 
  rl_bind_key('\r', my_accept_line);

  rl_bind_key(')', rl_insert_close);
  rl_bind_key('}', rl_insert_close);
  rl_bind_key(']', rl_insert_close);

  rl_initialize(); /* This has to happen *before* we set rl_redisplay_function, otherwise
		      readline will not bother to call tgetent(), will be agnostic about terminal
		      capabilities and hence not be able to honour e.g. a set horizontal-scroll-mode off
		      in .inputrc */
  using_history();
  rl_redisplay_function = my_redisplay;
  rl_already_prompted = 1;  /* we'll always write our own prompt */
  rl_completion_entry_function = (rl_compentry_func_t  *) & my_completion_function;
    
}


void save_rl_state (struct rl_state *state) {
  
  strncpy (state ->text, rl_line_buffer, BUFFSIZE-1);  /* save text*/
  strncpy(state -> prompt, rl_prompt, BUFFSIZE-1);
  state -> point = rl_point;                           /* and point    */
  if (state->  already_saved) 
    return;
  state -> already_saved = 1;
  rl_delete_text(0, rl_end);                           /* clear line  (after prompt) */
  rl_redisplay();                                      /* and redisplay */
  rl_callback_handler_remove ();                       /* restore old term settings */
  rl_deprep_terminal();
  /*  backspace(strlen(state -> prompt));                 go back to start of line   */
}
 
void restore_rl_state (struct rl_state *state) {
  char *newprompt;
  newprompt = state -> prompt; 

  DPRINTF1(DEBUG_READLINE, "restore prompt: '%s'", newprompt);
  rl_expand_prompt (newprompt); /* We have to call this because rl_already_prompted is set */
  
  mirror_slaves_echo_mode(); /* don't show passwords etc */	    
  rl_callback_handler_install (newprompt , &line_handler);
  rl_insert_text(state -> text);
  rl_point = state -> point;
  state -> already_saved = 0;
  cr();
  rl_forced_update_display ();
  rl_prep_terminal(1);
}

void line_handler(char * line) {
  if (line == NULL) {                               /* EOF, forward it  */
    DPRINTF1(DEBUG_READLINE,"EOF detected, writing character %d", term_eof);
    write(master_pty_fd, &term_eof, 1); 
  } else { 
    if (*line && strlen(line) > 1 && redisplay)  {  /* don't save empty lines,
						       single keystrokes or passwords*/
         my_add_history (line);
    }  
    write(master_pty_fd, line, strlen(line));
    DPRINTF2(DEBUG_READLINE, "writing %d bytes %s", strlen(line), mangle_string_for_debug_log(line,40)); 
    write(master_pty_fd, (return_key ? &return_key : "\n"), 1);
    free(line);                                      /* we're done with it  */
    
    return_key = 0;
    within_line_edit = FALSE;
    
    rl_callback_handler_remove ();
    set_echo(FALSE);
    saved_rl_state.text[0]= '\0';
    saved_rl_state.prompt[0] = '\0';
    saved_rl_state.point = 0;
    saved_rl_state.already_saved = TRUE;
    
  }
}

/* this function (drop-in replacement for readline's own accept-line())
   will be bound to RETURN key: */
int my_accept_line(int count, int key) {
  rl_point = 0;                             /* leave cursor on predictable place */
  my_redisplay();
  rl_done = 1;
  return_key = (char) key;  
  return 0;
}




void my_passwordfield_redisplay() {
  char *p;
  int l;
  clear_line();
  write(STDOUT_FILENO, rl_prompt, strlen(rl_prompt)); 
  /* write(STDOUT_FILENO, " ",1); */
  for(p=rl_line_buffer,l = 0; *p; p++, l++) 
    write(STDOUT_FILENO, "*", 1);
  backspace(l - rl_point); 
}

static void my_redisplay() {
  if (redisplay) {
    rl_redisplay();
  } else {
    my_passwordfield_redisplay();
  }
}




void my_add_history (char * line) { /* add line to history list, avoiding duplicates */
  if (strncmp (line, previous_line, BUFFSIZE-1)) {
    strncpy (previous_line, line, BUFFSIZE-1);
    add_history (line);               
  }
}
  















