/*  term.c: terminal handling, cursor movement etc. */

/*  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License , or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; see the file COPYING.  If not, write to
    the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

    You may contact the author by:
       e-mail:  hlub@knoware.nl
*/


#include "rlwrap.h"

/*global vars */
char term_eof;          /* end_of_file char */
char term_stop;         /* stop (suspend) key */
char *term_backspace;   /* backspace control seq  (or 0, if none defined in terminfo) */
int redisplay = 1;

struct termios saved_terminal_settings; /* original terminal settings */
int terminal_settings_saved = FALSE;    /*  saved_terminal_settings is valid */
struct winsize window_size;             /* current window size */

static char *term_cr;          /* carriage return (or 0, if none defined in terminfo) */
static char * term_clear_line; 



void 
init_terminal (void)        /* save term settings and determine term type */ 
{
  char *term_name;
  char *term_buf;
  char *term_string_buf; 
  
  if (! isatty (STDIN_FILENO))
    myerror("stdin is not a tty");
  if (tcgetattr(STDIN_FILENO, &saved_terminal_settings) < 0)
    myerror ("tcgetattr error on stdin");
  else
    terminal_settings_saved = TRUE;
  if (ioctl (STDIN_FILENO, TIOCGWINSZ, &window_size) < 0)
    myerror("Could not get winsize");
  /* unsetenv("TERM");    */ 
  
  /* init some variables: */
  term_name = getenv ("TERM");

  
  if (!term_name)
    term_name = "dumb";
  term_buf = (char*) mymalloc (4096);
  term_string_buf = (char *) mymalloc(2048);

  
  term_backspace = NULL;
  term_cr        = NULL;
  term_clear_line = NULL;

#ifdef HAVE_TERM_H
  if (tgetent (term_buf, term_name) > 0) {
    term_backspace  = tgetstr ("le", &term_string_buf);
    term_cr         = tgetstr ("cr", &term_string_buf);
    term_clear_line = tgetstr ("dl1", &term_string_buf);
  }
#endif  
  
  term_eof = saved_terminal_settings.c_cc[VEOF];
  term_stop = saved_terminal_settings.c_cc[VSTOP];

  DPRINTF2(DEBUG_TERMIO,"term_eof=%d term_stop=%d", term_eof, term_stop);
}


void set_echo(int yes)
{
  
  struct termios pterm = saved_terminal_settings;
  pterm.c_lflag &= ~ICANON;
  pterm.c_lflag &= ~ECHO;
  pterm.c_lflag |= ISIG;
  pterm.c_cc[VMIN] = 1;
  pterm.c_cc[VTIME]=0;
  if (yes)
    pterm.c_lflag |= ECHO;
  else /* no */
    pterm.c_lflag &= ~ECHO;
  if (tcsetattr(STDIN_FILENO, TCSANOW, &pterm) < 0 && errno != ENOTTY)
    ; /* myerror ("cannot prepare terminal (tcsetattr error on stdin)"); */
}


#ifndef HAVE_TERM_H
#define tputs(a,b,c) 
#endif

void 
backspace (int count) 
{
  int i;
  if (term_backspace)
    for (i = 0; i < count; i++)
      tputs (term_backspace, 1, my_putchar);
  else
    for (i = 0; i < count; i++)
      my_putchar ('\b');
}

void 
cr () 
{
  if (term_cr)
    tputs (term_cr, 1, my_putchar);
  else
    my_putchar('\r');
}

void 
clear_line() 
{
  int i;
  int width = window_size.ws_col;
  char *p, *spaces;
  cr();
  if (term_clear_line)
    tputs (term_clear_line, 1, my_putchar);
  else {
    spaces = (char *) mymalloc(width);
    for (i=0, p = spaces; i< width; i++, p++)
      *p = ' ';
    write(STDOUT_FILENO, spaces, width);
    free((void *) spaces);
  }
  cr();
}


int 
my_putchar 
(int c) {
  char ch = c;
  ssize_t nwritten; 
  nwritten = write(STDOUT_FILENO, &ch, 1);
  return (nwritten == -1 ? -1 : c);
}








