//rolestack.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010-2019
 *
 *  This file is part of roard a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "roard.h"

#define STACKSIZE 16

static struct rolestack g_rolestack[STACKSIZE];

void rolestack_init(void) {
 int i;

 for (i = 0; i < STACKSIZE; i++) {
  g_rolestack[i].index  = i;
  g_rolestack[i].role   = ROAR_ROLE_UNKNOWN;
  g_rolestack[i].action = RS_MIX;
 }

 g_rolestack[0].role = ROAR_ROLE_NONE;
}

void print_rolestack(void) {
 int i;

 printf("Index Role             Action\n");
 printf("-----------------------------\n");
 for (i = 0; i < STACKSIZE && g_rolestack[i].role != ROAR_ROLE_UNKNOWN; i++) {
  printf("%-5i %-16s %s\n", g_rolestack[i].index,
          roar_role2str(g_rolestack[i].role), rolestack_action2str(g_rolestack[i].action));
 }
}

const struct rolestack * rolestack_get_role(int role) {
 int i;

 ROAR_DBG("rolestack_get_role(role=%i) = ?", role);

 if ( role == -1 )
  return &(g_rolestack[0]);

 for (i = STACKSIZE - 1; i >= 0; i--)
  if ( g_rolestack[i].role == role )
   return &(g_rolestack[i]);

 return &(g_rolestack[0]);
}

int rolestack_push(const struct rolestack * role) {
 int i;

 for (i = 0; i < STACKSIZE; i++) {
  if ( g_rolestack[i].role == ROAR_ROLE_UNKNOWN ) {
   memcpy(&(g_rolestack[i]), role, sizeof(g_rolestack[i]));
   g_rolestack[i].index = i;
   return 0;
  }
 }

 roar_err_set(ROAR_ERROR_NOMEM);
 return -1;
}

enum rs_action rolestack_str2action(const char * str) {
 if ( str == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if (!strcasecmp(str, "mix")) {
  return RS_MIX;
 } else if (!strcasecmp(str, "kick")) {
  return RS_KICK;
 } else if (!strcasecmp(str, "mute")) {
  return RS_MUTE;
 } else if (!strcasecmp(str, "pause")) {
  return RS_PAUSE;
 }

 return RS_ERROR;
}

const char * rolestack_action2str(enum rs_action action) {
 switch (action) {
  case RS_ERROR: return "(error)"; break;
  case RS_MIX:   return "mix";     break;
  case RS_MUTE:  return "mute";    break;
  case RS_PAUSE: return "pause";   break;
  case RS_KICK:  return "kick";    break;
 }

 return "(unknown)";
}

const struct rolestack * rolestack_parse(const char * str) {
 static struct rolestack ret;
 char * role, * action;

 if ( str == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return NULL;
 }

 role = roar_mm_strdup(str);
 if ( role == NULL )
  return NULL;

 if ( (action = strstr(role, ":")) == NULL ) {
  roar_mm_free(role);
  roar_err_set(ROAR_ERROR_BADMSG);
  return NULL;
 }

 *action = 0;
  action++;

 ret.index  = -1;
 ret.role   = roar_str2role(role);
 ret.action = rolestack_str2action(action);

 roar_mm_free(role);

 if ( ret.role == -1 || ret.action == RS_ERROR ) {
  roar_err_set(ROAR_ERROR_BADMSG);
  return NULL;
 }

 return &ret;
}

//ll
