//service-about.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2013
 *
 *  This file is part of roard a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include <roard/include/roard.h>

static ssize_t __list(int * ids, size_t len) {
 struct roar_client * c;
 size_t i, j;

 if ( g_counters.cur.clients > len ) {
  roar_err_set(ROAR_ERROR_NOSPC);
  return -1;
 }

 for (j = i = 0; i < ROAR_CLIENTS_MAX && j < len; i++)
  if ( clients_get(i, &c) == 0 )
   ids[j++] = i;

 return j;
}

static ssize_t __num(enum roar_service_num what) {
 switch (what) {
  case ROAR_SERVICE_NUM_DEFAULT:
  case ROAR_SERVICE_NUM_CURRENT:
    return g_counters.cur.clients;
   break;
  case ROAR_SERVICE_NUM_BUFFER:
    if ( ((double)g_counters.cur.clients * 1.5) > ROAR_CLIENTS_MAX )
     return ROAR_CLIENTS_MAX;
    return (double)g_counters.cur.clients * 1.25;
   break;
#ifndef DEBUG
  default: break;
#endif
 }
 roar_err_set(ROAR_ERROR_NOTSUP);
 return -1;
}

static int __get(int id, struct roar_client * client) {
 struct roar_client * c;

 if ( client == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( clients_get(id, &c) == -1 )
  return -1;

 memcpy(client, c, sizeof(struct roar_client));
 client->acl = NULL;

 return 0;
}

static int __status(int id) {
 struct roar_dl_librarypara * pluginpara;
 const struct roard_proto_handle * proto;
 struct roar_client_server * cs;
 struct roar_vio_calls vio;
 int ret, err;

 if ( clients_get_server(id, &cs) == -1 )
  return -1;

 if ( (proto = clients_get_protohandle(ROAR_CLIENT(cs)->proto)) == NULL )
  return -1;

 if ( proto->lhandle != NULL )
  roar_dl_context_restore(proto->lhandle);

 pluginpara = roar_dl_getpara(proto->lhandle);

 switch (proto->type) {
  case ROARD_PROTO_TYPE_COMMON:
    if ( proto->impl.common->status == NULL ) {
     ret = -1;
     err = ROAR_ERROR_NOSYS;
    } else {
     roar_vio_open_fh_socket(&vio, clients_get_fh(id));
     ret = proto->impl.common->status(id, &vio, &(cs->outbuf), &(cs->protoinst), proto->para, proto->paralen, pluginpara);
    }
   break;
  default:
    ret = -1;
    err = ROAR_ERROR_NOSYS;
   break;
 }

 if ( pluginpara != NULL )
  roar_dl_para_unref(pluginpara);

 if ( proto->lhandle != NULL )
  roar_dl_context_store(proto->lhandle);

 roar_err_set(err);
 return ret;
}

static int __set_ids(int id, int clear, int pid, int uid, int gid) {
 struct roar_client * c;

 // just to check if the client exist.
 if ( clients_get(id, &c) == -1 )
  return -1;

 // TODO: add proper error handling here.

 if ( pid != -1 || clear )
  clients_set_pid(id, pid);

 if ( uid != -1 || clear )
  clients_set_uid(id, uid);

 if ( gid != -1 || clear )
  clients_set_gid(id, gid);

 return 0;
}

static int __set_proto(int id, int proto) {
 (void)id, (void)proto;

 roar_err_set(ROAR_ERROR_NOSYS);
 return -1;
}


static struct roar_service_client api = {
 .list = __list,
 .num  = __num,
 .get  = __get,
 .kick = (int(*)(int id, int error, const char * msg))clients_delete,
 .status = __status,

 .set_ids = __set_ids,
 .set_name = clients_set_name,
 .set_proto = __set_proto, /* dummy */
 .exec = client_stream_exec
};

ROAR_DL_PLUGIN_REG_SERVICES_GET_API(get_api, api)

static const struct roar_dl_service service[1] = {
 {
  .appname = NULL,
  .appabi = NULL,
  .servicename = ROAR_SERVICE_CLIENT_NAME,
  .serviceabi = ROAR_SERVICE_CLIENT_ABI,
  .description = "Access to client management",
  .flags = ROAR_DL_SERVICE_FLAGS_NONE,
  .userdata = NULL,
  .get_api = get_api
 }
};

ROAR_DL_PLUGIN_REG_SERVICES(service);

ROAR_DL_PLUGIN_START(service_client) {
 ROAR_DL_PLUGIN_META_PRODUCT_NIV("service-client", ROAR_VID_ROARAUDIO, ROAR_VNAME_ROARAUDIO);
 ROAR_DL_PLUGIN_META_VERSION(ROAR_VERSION_STRING);
 ROAR_DL_PLUGIN_META_LICENSE_TAG(GPLv3_0);
 ROAR_DL_PLUGIN_META_CONTACT_FLNE("Philipp", "Schafft", "ph3-der-loewe", "lion@lion.leolix.org");
 ROAR_DL_PLUGIN_META_DESC("This plugin provides an interface to client control features of roard.");
 ROAR_DL_PLUGIN_REG_FNFUNC(ROAR_DL_FN_SERVICE);
} ROAR_DL_PLUGIN_END

//ll
