//roar-config.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2008-2013
 *
 *  This file is part of roarclients a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include <roaraudio.h>

static const struct {
 const char * name;
 const char * cflags;
 const char * libs;
} flags[] = {
 // native/own libs:
 {"roar",      ROAR_CFLAGS, ROAR_LIBS        }, // NOTE: libroar *MUST* be the first entry
 {"roardsp",   ROAR_CFLAGS, ROAR_LIBS_DSP    },
 {"roarmidi",  ROAR_CFLAGS, ROAR_LIBS_MIDI   },
 {"roarlight", ROAR_CFLAGS, ROAR_LIBS_LIGHT  },
 {"roareio",   ROAR_CFLAGS, ROAR_LIBS_EIO    },
 // comp libs:
 {"roaresd",   ROAR_CFLAGS, ROAR_LIBS_C_ESD  },
 {"esd",       ROAR_CFLAGS, ROAR_LIBS_C_ESD  },
 {"roarartsc", ROAR_CFLAGS, ROAR_LIBS_C_ARTSC},
 {"artsc",     ROAR_CFLAGS, ROAR_LIBS_C_ARTSC},
 {"roarpulse", ROAR_CFLAGS, ROAR_LIBS_C_PULSE},
 {"pulse",     ROAR_CFLAGS, ROAR_LIBS_C_PULSE},
 {"roarpulse-simple", ROAR_CFLAGS, ROAR_LIBS_C_PULSE_SIMPLE},
 {"pulse-simple",     ROAR_CFLAGS, ROAR_LIBS_C_PULSE_SIMPLE},
 {"roarsndio", ROAR_CFLAGS, ROAR_LIBS_C_SNDIO},
 {"sndio",     ROAR_CFLAGS, ROAR_LIBS_C_SNDIO},
 {"roaryiff",  ROAR_CFLAGS, ROAR_LIBS_C_YIFF },
 {"Y2",        ROAR_CFLAGS, ROAR_LIBS_C_YIFF },
 {NULL, NULL, NULL}
}, * flags_ptr = NULL;

struct version {
 int major, minor, revision;
};

static struct version parse_version(const char * version) {
 struct version ret = {-1, -1, -1};
 char * next;

 if ( !strcasecmp(version, "current") )
  version = ROAR_VERSION_COMMON;

 ret.major    = strtoll(version, &next, 0);
 ret.minor    = strtoll(next+1, &next, 0);
 ret.revision = strtoll(next+1, &next, 0);

 return ret;
}

static int compare_versions_eq(struct version a, struct version b) {
 return a.major == b.major && a.minor == b.minor && a.revision == b.revision;
}

static int compare_versions_gt(struct version a, struct version b) {
 if ( a.major > b.major )
  return 1;
 if ( a.minor > b.minor )
  return 1;
 if ( a.revision > b.revision )
  return 1;
 return 0;
}

static int compare_versions_ge(struct version a, struct version b) {
 return compare_versions_gt(a, b) || compare_versions_eq(a, b);
}

static int compare_versions_lt(struct version a, struct version b) {
 return compare_versions_gt(b, a);
}

static int compare_versions_le(struct version a, struct version b) {
 return compare_versions_lt(a, b) || compare_versions_eq(a, b);
}

static const struct {
 const char * op;
 int neg;
 int (*func)(struct version a, struct version b);
} compare_versions_funcs[] = {
 {"eq", 0, compare_versions_eq},
 {"ne", 1, compare_versions_eq},
 {"gt", 0, compare_versions_gt},
 {"ge", 0, compare_versions_ge},
 {"lt", 0, compare_versions_lt},
 {"le", 0, compare_versions_le}
};

static int compare_versions(const char * a, const char * op, const char * b) {
 struct version va = parse_version(a);
 struct version vb = parse_version(b);
 size_t i;
 int ret;

 ROAR_DBG("compare_versions(a='%s', op='%s', b='%s'): {%i, %i, %i} <%s> {%i, %i, %i}\n", a, op, b, va.major, va.minor, va.revision, op, vb.major, vb.minor, vb.revision);

 for (i = 0; i < (sizeof(compare_versions_funcs)/sizeof(*compare_versions_funcs)); i++) {
  if ( !strcasecmp(compare_versions_funcs[i].op, op) ) {
   ret = compare_versions_funcs[i].func(va, vb);
   if ( compare_versions_funcs[i].neg )
    ret = ret ? 0 : 1;

   return ret ? 0 : 32;
  }
 }

 ROAR_ERR("compare_versions(*): Operator \"%s\" not found.", op);
 return 1;
}

void print_path(const char * name, int null_as_universal, const char * product, const char * provider) {
 char * path = roar_libroar_get_path(name, null_as_universal, product, provider);
 if ( path == NULL ) {
  fprintf(stderr, "Error: Can not get path: %s: %s\n", name, roar_errorstring);
  return;
 }
 printf("%s\n", path);
 roar_mm_free(path);
}

void list_path(int null_as_universal, const char * product, const char * provider) {
 const char * names[32];
 char * path;
 size_t offset = 0;
 ssize_t ret;
 ssize_t i;
 const char * err;
 size_t errlen;

 while (1) {
  ret = roar_libroar_list_path(names, sizeof(names)/sizeof(*names), offset);
  if ( ret < 1 )
   break;
  offset += (size_t)ret;

  for (i = 0; i < ret; i++) {
   path = roar_libroar_get_path(names[i], null_as_universal, product, provider);
   if ( path == NULL && roar_error == ROAR_ERROR_INVAL )
    path = roar_libroar_get_path(names[i], 0, product, NULL);
   if ( path == NULL && roar_error == ROAR_ERROR_INVAL )
    path = roar_libroar_get_path(names[i], 0, NULL, NULL);
   if ( path == NULL ) {
    err = roar_errorstring;
    errlen = roar_mm_strlen(err) + roar_mm_strlen("<Error: >") + 1;
    path = roar_mm_malloc(errlen);
    if ( path == NULL )
     continue;
    snprintf(path, errlen, "<Error: %s>", err);
   }

   printf("%-24s: %s\n", names[i], path);
   roar_mm_free(path);
  }
 }
}

void print_renderpath(const char * path) {
 char buf[1024];

 if ( roar_env_render_path_r(buf, sizeof(buf), path) != 0 ) {
  fprintf(stderr, "Error: Can not render path: %s: %s\n", path, roar_errorstring);
  return;
 }
 printf("%s\n", buf);
}

void usage (void) {
 printf("Usage: roar-config --version\n"
        "       roar-config --compare-versions VERSIONA OPERATOR VERSIONB\n"
        "       roar-config [{--output-pc|--output-normal}] [--libs] [--cflags] [LIB]\n"
        "       roar-config [--product PRODUCT] [--provider PROVIDER] [--universal] {--path PATH|--list-path}\n"
        "       roar-config --render-path PATH\n");

 printf("\nOptions:\n\n");

 printf("  --help              - Show this help\n"
        "  --version           - Show version of library\n"
        "  --compare-versions A OP B\n"
        "                      - Compare version A against B with operator OP.\n"
        "  --path NAME         - Print path NAME\n"
        "  --list-path         - Print all known paths\n"
        "  --product PRODUCT   - Product string for --path/--list-path\n"
        "  --provider PROVIDER - Provider string for --path/--list-path\n"
        "  --universal         - Use universal path for --path/--list-path\n"
        "  --render-path PATH  - Print a rendered path\n"
        "  --libs              - Show linker flags (-lxxx) needed to link library\n"
        "  --cflags            - Show compiler flags needed to link library\n"
        "  --output-pc         - Output PC format\n"
        "  --output-normal     - Output in \"classical\" format\n"
       );

}

#define _strcat(buf, n) strncat(buf, n, sizeof(buf)-1-strlen(buf))

int main (int argc, char * argv[]) {
 enum { NORMAL, PC } mode = NORMAL;
 int null_as_universal = 0;
 const char * product = NULL;
 const char * provider = NULL;
 int i, h;
 int cflags = 0;
 int libs   = 0;
 char buf[1024] = {0};

 if ( argc == 1 ) {
  usage();
  return 0;
 }

 for (i = 1; i < argc; i++) {
  if ( !strcmp(argv[i], "--version") ) {
   printf("%s\n", ROAR_VERSION_COMMON);
  } else if ( !strcmp(argv[i], "--help") || !strcmp(argv[i], "-h") ) {
   usage();
   return 0;
  } else if ( !strcmp(argv[i], "--compare-versions") ) {
   return compare_versions(argv[i+1], argv[i+2], argv[i+3]);
  } else if ( !strcmp(argv[i], "--product") ) {
   product = argv[++i];
  } else if ( !strcmp(argv[i], "--provider") ) {
   provider = argv[++i];
  } else if ( !strcmp(argv[i], "--universal") ) {
   null_as_universal = 1;
  } else if ( !strcmp(argv[i], "--path") ) {
   print_path(argv[++i], null_as_universal, product, provider);
  } else if ( !strcmp(argv[i], "--list-path") ) {
   list_path(null_as_universal, product, provider);
  } else if ( !strcmp(argv[i], "--render-path") ) {
   print_renderpath(argv[++i]);
  } else if ( !strcmp(argv[i], "--libs") ) {
   libs   = 1;
  } else if ( !strcmp(argv[i], "--cflags") ) {
   cflags = 1;
  } else if ( !strcmp(argv[i], "--output-normal") ) {
   mode = NORMAL;
  } else if ( !strcmp(argv[i], "--output-pc") ) {
   mode = PC;
  } else if ( flags_ptr == NULL ) {
   if ( !strncmp(argv[i], "lib", 3) )
    argv[i] += 3;

   for (h = 0; flags[h].name != NULL; h++) {
    if ( !strcasecmp(argv[i], flags[h].name) )
     flags_ptr = &(flags[h]);
   }

   if ( flags_ptr == NULL ) {
    fprintf(stderr, "Unknown lib: %s\n", argv[i]);
    return 2;
   }
  } else {
   fprintf(stderr, "Unknown option: %s\n", argv[i]);
   usage();
   return 1;
  }
 }

 if ( flags_ptr == NULL )
  flags_ptr = &(flags[0]);

 switch (mode) {
  case NORMAL:
    if ( cflags || libs ) {
     if ( cflags )
      _strcat(buf, flags_ptr->cflags);

     if ( libs )
      _strcat(buf, flags_ptr->libs);

     puts(buf);
    }
   break;
  case PC:
    printf("prefix=");
    print_path("prefix", 0, NULL, NULL);
    printf("exec_prefix=${prefix}\n");
    printf("libdir=");
    print_path("prefix-lib", 0, NULL, NULL);
    printf("includedir=");
    print_path("prefix-inc", 0, NULL, NULL);
    printf("\n");
    printf(
           "Name: lib%s\n"
           "Description: lib%s is part of RoarAudio Sound System\n"
           "Version: %s\n"
//           "Requires: libroar\n"
           "Conflicts:\n"
           "Libs: -L${libdir} %s\n"
           "Cflags: -I${includedir} %s\n",
           flags_ptr->name,
           flags_ptr->name,
           ROAR_VERSION_COMMON,
           flags_ptr->libs,
           flags_ptr->cflags
          );
   break;
 }

 return 0;
}

//ll
