//ff_xspf.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2009-2010
 *
 *  This file is part of RoarAudio PlayList Daemon,
 *  a playlist management daemon for RoarAudio.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "rpld.h"

#ifdef HAVE_LIB_XML2
#include <ctype.h>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xmlreader.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>
#include <libxml/uri.h>

#define META_BASE_URL "http://roaraudio.keep-cool.org/spec/xspf/meta-%s.txt"

static int ff_xspf_write(void * context, const char * buffer, int len) {
 return roar_vio_write(context, (void*)buffer, len);
}

xmlNodePtr new_text_node(xmlNodePtr parent, const char * name, const char * value) {
 xmlNodePtr ret = xmlNewNode(NULL, (const xmlChar *)name);

 if ( ret == NULL )
  return NULL;

 xmlAddChild(ret, xmlNewText((const xmlChar *)value));

 xmlAddChild(parent, ret);

 return ret;
}

xmlNodePtr new_meta_text_node(xmlNodePtr parent, const char * name, const char * value) {
 xmlNodePtr ret = xmlNewNode(NULL, (const xmlChar *)"meta");
 char buf[256];

 if ( ret == NULL )
  return NULL;

 snprintf(buf, sizeof(buf), META_BASE_URL, name);

 xmlSetProp(ret, (xmlChar *)"rel", (const xmlChar *)buf);

 xmlAddChild(ret, xmlNewText((const xmlChar *)value));

 xmlAddChild(parent, ret);

 return ret;
}

int ff_xspf_pl_export(struct fformat_handle * handle, struct roar_vio_calls * vio, struct rpld_playlist  * pl) {
 struct rpld_playlist_entry * cur    = rpld_pl_get_first(pl);
 int                          parent = rpld_pl_get_parent(pl);
 char buf[RPLD_MAX_PLF_LEN+64];
 xmlDocPtr doc;
 xmlNodePtr rootnode, tracklist, track;
 xmlOutputBufferPtr output;
 xmlChar * location;

 (void)handle;

 output = xmlAllocOutputBuffer(NULL);

 if ( output == NULL )
  return -1;

 output->context       = vio;
 output->writecallback = ff_xspf_write;
 output->closecallback = NULL;

 doc = xmlNewDoc((xmlChar *)"1.0");

 if ( doc == NULL )
  return -1;

 doc->charset = XML_CHAR_ENCODING_UTF8;
 doc->encoding = xmlStrdup((xmlChar *)"UTF-8");

 rootnode = xmlNewNode(NULL, (xmlChar *)"playlist");
 xmlSetProp(rootnode, (xmlChar *)"version", (xmlChar *)"1");
 xmlSetProp(rootnode, (xmlChar *)"xmlns", (xmlChar *)"http://xspf.org/ns/0/");

 xmlDocSetRootElement(doc, rootnode);

 if ( rpld_pl_get_name(pl) != NULL )
  new_text_node(rootnode, "title", rpld_pl_get_name(pl));

 if (parent != 0 && parent != -1) {
  snprintf(buf, sizeof(buf), "%d", parent);
  new_meta_text_node(rootnode, "parent-playlist-id", buf);
  new_meta_text_node(rootnode, "parent-playlist-name", rpld_pl_get_name(rpld_pl_get_by_id(parent)));
 }

 tracklist = xmlNewNode(NULL, (xmlChar *)"trackList");
 xmlAddChild(rootnode, tracklist);

 while ( cur != NULL ) {
  track = xmlNewNode(NULL, (xmlChar *)"track");

  if (cur->io.filename[0] != 0) {
   if ( isalpha(cur->io.filename[0]) ) {
    new_text_node(track, "location", cur->io.filename);
   } else {
    location = xmlPathToURI((const xmlChar *)cur->io.filename);
    snprintf(buf, sizeof(buf), "file://%s", location);
    xmlFree(location);
    new_text_node(track, "location", buf);
   }
  }

#ifdef HAVE_LIB_UUID
  strcpy(buf, "tantalos://");
  uuid_unparse(cur->uuid, buf+11);
  new_text_node(track, "location", buf); // as alternative location to the one above, if any.
  new_text_node(track, "identifier", buf);
#endif

  if (cur->meta.album[0] != 0)
   new_text_node(track, "album", cur->meta.album);

  if (cur->meta.title[0] != 0)
   new_text_node(track, "title", cur->meta.title);

  if (cur->meta.artist[0] != 0)
   new_text_node(track, "creator", cur->meta.artist);

  if (cur->meta.performer[0] != 0)
   new_meta_text_node(track, "performer", cur->meta.performer);

  if (cur->meta.version[0] != 0)
   new_meta_text_node(track, "version", cur->meta.version);

  if (cur->meta.tracknum) {
   snprintf(buf, sizeof(buf), "%d", (int)cur->meta.tracknum);
   new_text_node(track, "trackNum", buf);
  }

  if (cur->meta.discid != 0 && cur->meta.discid != -1) {
   snprintf(buf, sizeof(buf), "0x%.8X", (int)cur->meta.discid);
   new_meta_text_node(track, "discid", buf);
  }

  if (cur->meta.genre != -1)
   new_meta_text_node(track, "genre", roar_meta_strgenre(cur->meta.genre));

  if (cur->likeness) {
   snprintf(buf, sizeof(buf), "%f", (double)cur->likeness);
   new_meta_text_node(track, "likeness", buf);
  }

  if (cur->codec != -1) {
   new_meta_text_node(track, "codec", roar_codec2str(cur->codec));
  }

  if (cur->length) {
   snprintf(buf, sizeof(buf), "%d", (int)(cur->length * 1000));
   new_text_node(track, "duration", buf);
  }

  xmlAddChild(tracklist, track);
  cur = cur->list.next;
 }

 xmlSaveFormatFileTo(output, doc, "UTF-8", 1);

 xmlFreeDoc(doc);

 return 0;
}

int ff_xspf_pl_import(struct fformat_handle * handle, struct roar_vio_calls * vio, struct rpld_playlist  * pl);
#endif

//ll
