
/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.
    Copyright 2000-2009 the Rosegarden development team.

    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

#ifndef _RG_ROSEGARDENGUIVIEW_H_
#define _RG_ROSEGARDENGUIVIEW_H_

#include "base/Event.h"
#include "base/MidiProgram.h"
#include "base/Selection.h"
#include "base/Track.h"
#include "sound/AudioFile.h"
#include "gui/editors/segment/TrackEditor.h"
#include <qstring.h>
#include <qvbox.h>


class QWidget;
class QObject;
class LevelInfo;
class KCommand;


namespace Rosegarden
{

class TrackParameterBox;
class TrackEditor;
class SimpleRulerScale;
class SequencerMapper;
class SegmentParameterBox;
class Segment;
class RosegardenGUIDoc;
class RealTime;
class NotationView;
class MultiViewCommandHistory;
class MatrixView;
class MappedEvent;
class InstrumentParameterBox;
class EventView;
class Composition;
class LevelInfo;

/**
 * The RosegardenGUIView class provides the view widget for the
 * RosegardenGUIApp instance.  The View instance inherits QWidget as a
 * base class and represents the view object of a KTMainWindow. As
 * RosegardenGUIView is part of the docuement-view model, it needs a
 * reference to the document object connected with it by the
 * RosegardenGUIApp class to manipulate and display the document
 * structure provided by the RosegardenGUIDoc class.
 *      
 * @author Source Framework Automatically Generated by KDevelop, (c) The KDevelop Team.
 * @version KDevelop version 0.4 code generation
 */
class RosegardenGUIView : public QVBox
{
    Q_OBJECT
public:

    /**p
     * Constructor for the main view
     */
    RosegardenGUIView(bool showTrackLabels,
                      SegmentParameterBox*,
                      InstrumentParameterBox*,
                      TrackParameterBox*,
                      QWidget *parent = 0,
                      const char *name=0);

    /**
     * Destructor for the main view
     */
    ~RosegardenGUIView();

    /**
     * returns a pointer to the document connected to the view
     * instance. Mind that this method requires a RosegardenGUIApp
     * instance as a parent widget to get to the window document
     * pointer by calling the RosegardenGUIApp::getDocument() method.
     *
     * @see RosegardenGUIApp#getDocument
     */
    RosegardenGUIDoc* getDocument() const;

    /**
     * Command history
     */
    MultiViewCommandHistory* getCommandHistory();

    TrackEditor* getTrackEditor() { return m_trackEditor; }
    
    /**
     * contains the implementation for printing functionality
     */
    void print(Composition*, bool previewOnly = false);

    // the following aren't slots because they're called from
    // RosegardenGUIApp

    /**
     * Select a tool at the SegmentCanvas
     */
    void selectTool(QString toolName);

    /**
     * Show output levels
     */
    void showVisuals(const MappedEvent *mE);

    void updateMeters(SequencerMapper *mapper);
    void updateMonitorMeters(SequencerMapper *mapper);

    /**
     * Change zoom size -- set the RulerScale's units-per-pixel to size
     */
    void setZoomSize(double size);

    void initChordNameRuler();
    
    bool haveSelection();
    SegmentSelection getSelection();
    void updateSelectionContents();

    static bool isMainWindowLastActive(const QWidget *w) {
        return w == m_lastActiveMainWindow;
    }

    TrackParameterBox *getTrackParameterBox() {
        return m_trackParameterBox;
    }

public slots:
    void slotEditSegment(Segment*);
    void slotEditSegmentNotation(Segment*);
    void slotEditSegmentsNotation(std::vector<Segment*>);
    void slotEditSegmentMatrix(Segment*);
    void slotEditSegmentsMatrix(std::vector<Segment*>);
    void slotEditSegmentPercussionMatrix(Segment*);
    void slotEditSegmentsPercussionMatrix(std::vector<Segment*>);
    void slotEditSegmentEventList(Segment*);
    void slotEditSegmentsEventList(std::vector<Segment*>);
    void slotEditTriggerSegment(int);
    void slotEditSegmentAudio(Segment*);
    void slotSegmentAutoSplit(Segment*);
    void slotEditRepeat(Segment*, timeT);
/* hjj: WHAT DO DO WITH THIS ?
    void slotEditMetadata(QString);
*/

    /**
     * Highlight all the Segments on a Track because the Track has
     * been selected * We have to ensure we create a Selector object
     * before we can highlight * these tracks.
     *
     * Called by signal from Track selection routine to highlight
     * all available Segments on a Track
     */
    void slotSelectTrackSegments(int);

    void slotSelectAllSegments();

    void slotUpdateInstrumentParameterBox(int id);

    // This is called from the canvas (actually the selector tool) moving out
    //
    void slotSelectedSegments(const SegmentSelection &segments);

    // And this one from the user interface going down
    //
    void slotPropagateSegmentSelection(const SegmentSelection &segments);

    void slotShowRulers(bool);

    void slotShowTempoRuler(bool);

    void slotShowChordNameRuler(bool);

    void slotShowPreviews(bool);

    void slotShowSegmentLabels(bool);

    void slotAddTracks(unsigned int count, InstrumentId instrument, int position);

    void slotDeleteTracks(std::vector<TrackId> tracks);

    void slotAddAudioSegmentCurrentPosition(AudioFileId,
                                            const RealTime &startTime,
                                            const RealTime &endTime);

    void slotAddAudioSegmentDefaultPosition(AudioFileId,
                                            const RealTime &startTime,
                                            const RealTime &endTime);

    void slotAddAudioSegment(AudioFileId audioId,
                             TrackId trackId,
                             timeT position,
                             const RealTime &startTime,
                             const RealTime &endTime);

    void slotDroppedAudio(QString audioDesc);
    void slotDroppedNewAudio(QString audioDesc);

    /*
     * Commands
     *
     */
    void slotAddCommandToHistory(KCommand *command);

    /*
     * Change the Instrument Label
     */
    void slotChangeInstrumentLabel(InstrumentId id, QString label);

    /*
     * Change the Track Label
     */
    void slotChangeTrackLabel(TrackId id, QString label);

    /*
     * Set the mute button on the track buttons and on the instrument
     * parameter box
     */
    void slotSetMuteButton(TrackId track, bool value);

    /*
     * Set mute, record and solo by instrument id
     */
    void slotSetMute(InstrumentId, bool);
    void slotSetRecord(InstrumentId, bool);
    void slotSetSolo(InstrumentId, bool);

    /**
     * To indicate that we should track the recording segment (despite
     * no commands being issued on it)
     */
    void slotUpdateRecordingSegment(Segment *segment,
                                    timeT updatedFrom);

    /**
     * A manual fudgy way of creating a view update for certain
     * semi-static data (devices/instrument labels mainly)
     */
    void slotSynchroniseWithComposition();

    /**
     * To indicate that an edit view, mixer, etc (something that might
     * want to receive MIDI input) has become active.  We only send
     * inputs such as MIDI to a single one of these, in most cases,
     * and it's whichever was most recently made active.  (It doesn't
     * have to still _be_ active -- we want to allow moving focus to
     * another application entirely but still receiving MIDI etc in
     * Rosegarden.)
     */
    void slotActiveMainWindowChanged(const QWidget *);
    void slotActiveMainWindowChanged(); // uses sender()

    /**
     * An event has been received from a device connected to the
     * external controller port.
     */
    void slotControllerDeviceEventReceived(MappedEvent *);
    void slotControllerDeviceEventReceived(MappedEvent *, const void *);

signals:
    void activateTool(QString toolName);

    void stateChange(QString, bool);

    // Inform that we've got a SegmentSelection
    //
    void segmentsSelected(const SegmentSelection&);

    void toggleSolo(bool);

    /**
     * Current used to dispatch things like track select changes, solo, etc...
     * to edit views
     */
    void compositionStateUpdate();
    

    /**
     * This signal is used to dispatch a notification for a request to
     * set the step-by-step-editing target window to all candidate targets,
     * so that they can either know that their request has been granted
     * (if they match the QObject passed) or else deactivate any step-by-
     * step editing currently active in their own window (otherwise).
     */
    void stepByStepTargetRequested(QObject *);

    /*
     * Add an audio file at the sequencer - when we drop a new file
     * on the segment canvas.
     */
    void addAudioFile(AudioFileId);

    void checkTrackAssignments();

    void instrumentLevelsChanged(InstrumentId,
                                 const LevelInfo &);

    void controllerDeviceEventReceived(MappedEvent *,
                                       const void *);

    void instrumentParametersChanged(InstrumentId);

protected:
    NotationView *createNotationView(std::vector<Segment *>);
    MatrixView   *createMatrixView  (std::vector<Segment *>, bool drumMode);
    EventView    *createEventView   (std::vector<Segment *>);

    virtual void windowActivationChange(bool);

    //--------------- Data members ---------------------------------

    SimpleRulerScale  *m_rulerScale;
    TrackEditor                   *m_trackEditor;

    SegmentParameterBox           *m_segmentParameterBox;
    InstrumentParameterBox        *m_instrumentParameterBox;
    TrackParameterBox             *m_trackParameterBox;

    static const QWidget          *m_lastActiveMainWindow;
};


}

#endif
