static int grbs_arc_is_outmost(grbs_t *grbs, grbs_arc_t *arc)
{
	int conc_segi;

	if (arc->link_point.next != NULL)
		return 0;

	/* if there is a cocave above, this is not the outermost */
	for(conc_segi = 0; conc_segi < GRBS_MAX_SEG; conc_segi++) {
		grbs_arc_t *conc_sentinel = gdl_first(&arc->parent_pt->arcs[1][conc_segi]);
		if ((conc_sentinel != NULL) && (grbs_arc_overlaps_arc(arc, conc_sentinel)))
			return 0;
	}


	return 1;
}

static void coll_report(grbs_t *grbs, grbs_2net_t *tn, grbs_2net_t *coll_tn, grbs_arc_t *coll_arc)
{
	if ((grbs->coll_report_cb != NULL) && (coll_tn != NULL))
		grbs->coll_report_cb(grbs, tn, coll_tn, coll_arc);
}


static void coll_report_arc(grbs_t *grbs, grbs_2net_t *tn, grbs_arc_t *coll_arc)
{
	coll_report(grbs, tn, grbs_arc_parent_2net(coll_arc), coll_arc);
}

static void coll_report_tn(grbs_t *grbs, grbs_2net_t *tn, grbs_2net_t *coll_tn)
{
	if (grbs->coll_report_cb != NULL)
		grbs->coll_report_cb(grbs, tn, coll_tn, NULL);
}


#define REPORT(coll_tn) coll_report_tn(grbs, tn, coll_tn)

/* Returns 1 if arc has collided */
static int coll_check_arc(grbs_t *grbs, grbs_2net_t *tn, grbs_arc_t *arc, int new)
{
	grbs_line_t *coll_line;
	grbs_arc_t *coll_arc;
	grbs_point_t *coll_point;
	double da;

	if (!grbs_arc_is_outmost(grbs, arc)) /* do not check collision on inner arcs */
		return 0;

	coll_arc = grbs_arc_arc_collision(grbs, tn, arc, new);
	if (coll_arc != NULL) {
		REPORT(grbs_arc_parent_2net(coll_arc));
		return 1;
	}

	coll_point = grbs_arc_point_collision(grbs, tn, arc, new);
	if (coll_point != NULL)
		return 1;

	da = new ? arc->new_da : arc->da;
	if (da != 0) {
		coll_line = grbs_arc_line_collision(grbs, tn, arc, new);
		if (coll_line != NULL) {
			if (grbs_force_attach_line_to_pt(grbs, coll_line, arc->parent_pt, arc->r + tn->copper, tn->clearance, arc->segi) != 0) {
				REPORT(grbs_arc_parent_2net(coll_line->a1));
				return 1;
			}
		}
	}

	if (grbs->coll_check_arc != NULL) {
		grbs_2net_t *ctn = grbs->coll_check_arc(grbs, tn, arc, new);
		if (ctn != NULL) {
			REPORT(ctn);
			return 1;
		}
	}

	return 0;
}

/* Returns 1 if line has collided */
static int coll_check_line(grbs_t *grbs, grbs_2net_t *tn, grbs_point_t *pt1, double x1, double y1, grbs_point_t *pt2, double x2, double y2, int concave, int incident)
{
	grbs_line_t *line;
	grbs_arc_t *arc;
	grbs_point_t *pt;
	long n;

	grbs->collobjs.used = 0;
	grbs_line_line_collisions(grbs, tn, x1, y1, x2, y2, tn->copper, tn->clearance);
	for(n = 0; n < grbs->collobjs.used; n++) {
		grbs_line_t *line = grbs->collobjs.array[n];

		/* special case: a new concave may collide with an existing concave */
		if (((pt1 == line->a1->parent_pt) && (pt2 == line->a2->parent_pt)) || ((pt1 == line->a2->parent_pt) && (pt2 == line->a1->parent_pt))) {
			if (((line->a1->concave || (line->a1->r == 0)) && (line->a2->concave || (line->a2->r == 0))) && concave)
				continue; /* a convex or a concave line crosses a concave-concave (or inc-inc) line */
			else if (incident && ((line->a2->concave) || (line->a1->concave)))
				continue; /* an incident line crosses a concave-concave line */
		}

		REPORT(grbs_arc_parent_2net(line->a1));
		return 1;
	}

	grbs->collobjs.used = 0;
	grbs_line_arc_collisions(grbs, tn, x1, y1, x2, y2, tn->copper, tn->clearance);
	for(n = 0; n < grbs->collobjs.used; n++) {
		grbs_arc_t *arc = grbs->collobjs.array[n];

		/* special case: line vs. arc crosses around points are checked locally using angles */
		if ((pt1 == arc->parent_pt) || (pt2 == arc->parent_pt))
			continue;

		REPORT(grbs_arc_parent_2net(arc));
		return 1;
	}

	pt = grbs_line_point_collision(grbs, tn, x1, y1, x2, y2, tn->copper, tn->clearance, pt1, pt2);
	if (pt != NULL)
		return 1;


	if (grbs->coll_check_line != NULL) {
		grbs_2net_t *ctn = grbs->coll_check_line(grbs, tn, pt1, x1, y1, pt2, x2, y2);
		if (ctn != NULL) {
			REPORT(ctn);
			return 1;
		}
	}

	return 0;
}

int grbs_is_target_pt_routable(grbs_t *grbs, grbs_2net_t *tn, grbs_point_t *tpt)
{
	grbs_point_t *cpt;
	grbs_arc_t *carc;
	int res = 1;

	cpt = grbs_endcap_point_collision(grbs, tn, tpt);
	if (cpt != NULL) {
		/* can not report collision as points are fixed objects */
		return 0; /* return before reporting any arc - no reason to blame it on arcs and remove arcs if we can't route it at the end */
	}

	carc = grbs_endcap_arc_collision(grbs, tn, tpt);
	if (cpt != NULL) {
		REPORT(grbs_arc_parent_2net(carc));
		res = 0;
	}


	return res;
}

#undef REPORT


/* rtree based collision handling (centerline only) */

static int grbs_arc_is_sentinel(grbs_arc_t *arc)
{
	return (arc->link_point.prev == NULL);
}

static void CHG_PRE(grbs_t *grbs, grbs_arc_t *arc)
{
	if (grbs_arc_is_sentinel(arc))
		return;
	assert(arc->old_in_use == 0);
	arc->old_in_use = 1;
	arc->old_r = arc->r;
	arc->old_sa = arc->sa;
	arc->old_da = arc->da;
}

static void CHG_POST(grbs_t *grbs, grbs_arc_t *arc)
{
	int chg_start = 0, chg_end = 0;

	if (grbs_arc_is_sentinel(arc))
		return;

	assert(arc->old_in_use == 1);
	arc->old_in_use = 0;
	if (arc->r != arc->old_r) {
		chg_start = 1;
		chg_end = 1;
	}
	else {
		if (arc->sa != arc->old_sa)
			chg_start = 1;
		if (arc->sa + arc->da != arc->old_sa + arc->old_da)
			chg_end = 1;
	}

	if (chg_start && (arc->sline != NULL)) { /* sline is NULL in start incident arc */
		grbs_line_unreg(grbs, arc->sline);
		grbs_line_attach(grbs, arc->sline, arc, 2);
		grbs_line_bbox(arc->sline);
		grbs_line_reg(grbs, arc->sline);
	}
	if (chg_end && (arc->eline != NULL)) { /* eline is NULL during initial realize of a net before the next arc is realized */
		grbs_line_unreg(grbs, arc->eline);
		grbs_line_attach(grbs, arc->eline, arc, 1);
		grbs_line_bbox(arc->eline);
		grbs_line_reg(grbs, arc->eline);
	}

	if (chg_start || chg_end) {
		if (arc->registered)
			grbs_arc_unreg(grbs, arc);
		grbs_arc_bbox(arc);
		grbs_arc_reg(grbs, arc);
	}
}

static void CHG_NEW(grbs_t *grbs, grbs_arc_t *arc)
{
	grbs_arc_bbox(arc);
	grbs_arc_reg(grbs, arc);
	arc->old_in_use = 0;
}

