/*
    roxterm - GTK+ 2.0 terminal emulator with tabs
    Copyright (C) 2004 Tony Houghton <h@realh.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "defns.h"

#include <locale.h>

#include "dlg.h"
#include "globalopts.h"
#include "roxterm.h"
#include "rtdbus.h"

/* List of all terminal windows managed by this app */
static GList *main_all_windows = NULL;

static gboolean main_quit_handler(gpointer data)
{
	(void) data;

	/* FIXME: Destroy any windows left? */

	return FALSE;
}

#define ROXTERM_DBUS_NAME RTDBUS_NAME ".term"
#define ROXTERM_DBUS_OBJECT_PATH RTDBUS_OBJECT_PATH "/term"
#define ROXTERM_DBUS_INTERFACE RTDBUS_INTERFACE
#define ROXTERM_DBUS_METHOD_NAME "NewTerminal"

static DBusMessage *create_dbus_message(int argc, char **argv)
{
	DBusMessage *message;
	int n;

	message = rtdbus_method_new(ROXTERM_DBUS_NAME, ROXTERM_DBUS_OBJECT_PATH,
			ROXTERM_DBUS_INTERFACE,
			ROXTERM_DBUS_METHOD_NAME, DBUS_TYPE_INVALID);
	for (n = 0; message && n < argc; ++n)
	{
		message = rtdbus_append_args(message,
				DBUS_TYPE_STRING, RTDBUS_ARG(argv[n]),
				DBUS_TYPE_INVALID);
	}
	return message;
}

static gboolean run_via_dbus(DBusMessage *message)
{
	gboolean result = TRUE;

	if (!message)
		return FALSE;
	/* New roxterm command may have been run in a different directory
	 * from original instance */
	if (!global_options_directory)
	{
		char *cwd = GET_CURRENT_DIR();
		const char *d = "-d";

		message = rtdbus_append_args(message,
				DBUS_TYPE_STRING, RTDBUS_ARG(d), 
				DBUS_TYPE_STRING, RTDBUS_ARG(cwd),
				DBUS_TYPE_INVALID);
		if (!message)
			result = FALSE;
		FREE_CURRENT_DIR(cwd);
	}
	if (result && global_options_commandv)
	{
		int n;
		const char *e = "-e";

		message = rtdbus_append_args(message,
				DBUS_TYPE_STRING, RTDBUS_ARG(e), DBUS_TYPE_INVALID);
		if (!message)
			return FALSE;
		for (n = 0; global_options_commandv[n]; ++n)
		{
			message = rtdbus_append_args(message,
					DBUS_TYPE_STRING, RTDBUS_ARG(global_options_commandv[n]),
					DBUS_TYPE_INVALID);
			if (!message)
				return FALSE;
		}
	}
	if (result)
		result = rtdbus_send_message(message);
	return result;
}

static DBusHandlerResult new_term_listener(DBusConnection *connection,
		DBusMessage *message, void *user_data)
{
	DBusMessage *reply = NULL;
	DBusError derror;
	char **argv;
	int argc;
	int n;

	dbus_error_init(&derror);

	if (!dbus_message_is_method_call(message, ROXTERM_DBUS_INTERFACE,
				ROXTERM_DBUS_METHOD_NAME))
	{
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;	
	}
	argv = rtdbus_get_message_args_as_strings(message);
	for (argc = 0; argv && argv[argc]; ++argc);
	if (argc)
	{
		global_options_preparse_argv_for_execute(&argc, argv, TRUE);
		global_options_init(&argc, &argv, FALSE);
	}
	roxterm_launch();
	if (!dbus_message_get_no_reply(message))
		reply = dbus_message_new_method_return(message);
	if (reply)
		rtdbus_send_message(reply);

	for (n = 0; n < argc; ++n)
		g_free(argv[n]);
	g_free(argv);
	
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;	
}

gboolean listen_for_new_term(void)
{
	return rtdbus_start_service(ROXTERM_DBUS_NAME,
			ROXTERM_DBUS_OBJECT_PATH,
			new_term_listener, global_options_lookup_int("replace") > 0);
}

int main(int argc, char **argv)
{
	gboolean dbus_ok;
	gboolean preparse_ok;
	DBusMessage *message;

	g_set_application_name(PACKAGE);
	preparse_ok = global_options_preparse_argv_for_execute(&argc, argv, FALSE);
	
#if ENABLE_NLS
	global_options_init_appdir(argc, argv);
	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, global_options_appdir ?
			g_strdup_printf("%s/locale", global_options_appdir) : LOCALEDIR);
	textdomain(PACKAGE);
#endif
	
	gtk_init(&argc, &argv);
	if (!preparse_ok)
	{
		/* Only one possible reason for failure */
		dlg_critical(NULL, _("Missing command after -e/--execute option"));
		return 1;
	}
	
	/* Have to create message with args from argv before parsing them */
	dbus_ok = rtdbus_init();
	message = create_dbus_message(argc, argv);
	
	global_options_init(&argc, &argv, TRUE);

	if (dbus_ok)
	{
		dbus_ok = listen_for_new_term();
		/* Only TRUE if another roxterm is providing the service */
	}

	if (global_options_lookup_int("separate") <= 0
			&& dbus_ok && run_via_dbus(message))
	{
		return 0;
	}
	else if (message)
	{
		dbus_message_unref(message);
	}

	gtk_quit_add(0, main_quit_handler, main_all_windows);

	roxterm_init();

	gtk_main();

	return 0;
}

/* vi:set sw=4 ts=4 noet cindent cino= */
