/* scanner.l - configuration file scanner, flex. */
/*
 *  RRDcollect  --  Round-Robin Database Collecting Daemon.
 *  Copyright (C) 2003  Dawid Kuroczko <qnex@knm.org.pl>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

%{
#include "parser.h"
#include "rrdcollect.h"

/* _GNU_SOURCE doesn't work for flex... */
#if (!__USE_GNU && HAVE_STRNDUP)
extern char *strndup(const char *s, size_t n);
#endif

/* Convenience macros. */
#define return_val	yylval.val = atoi(yytext); return
#define return_str	yylval.str = strndup(yytext, yyleng); return
#define return_str_chop	yylval.str = strndup(yytext, yyleng-1); return
#define return_null	yylval.str = NULL; return

%}

%option never-interactive
%option yylineno
%option noyywrap
%option nounput

/* contexts */
%x uri
%s path
%s assign

%x string
%x regex

%s targets

comment		#.*
blanks		[ \t]+

number		[0-9]+
name		[a-zA-Z0-9_]+
variable	\$({name}|\{{name}\})

ip		[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}
dnsname		[A-Za-z-][A-Za-z0-9.-]*
host		{ip}|{dnsname}|\*
port		[0-9]{1,5}

path		(\/[A-Za-z0-9%~_.-]+)+\/?
file		[a-zA-Z0-9_.-]+

rrdfile		[a-zA-Z0-9_.$-]+\.rrd

dsname		[a-zA-Z0-9_]{1,19}
dscreate	(GAUGE|COUNTER|DERIVE|ABSOLUTE):[0-9]+:(-?[0-9]+|U):(-?[0-9]+|U)

%%

{comment}			/* ignore comments */
{blanks}			/* ignore blanks */

				/* URI-s */
^file/:\/\/[^/]			BEGIN(uri); return_null URI_FILE;
^file/:				BEGIN(path); return_null URI_FILE;
^exec/:\/\/[^/]			BEGIN(uri); return_null URI_EXEC;
^exec/:				BEGIN(path); return_null URI_EXEC;
^http/:\/\/[^/]			BEGIN(uri); return_null URI_HTTP;
^rrd/:\/\/[^/]			BEGIN(uri); return_null URI_RRD;

^snmp/:\/\/[^/]			BEGIN(uri); return_null URI_SNMP;
^netlink/:			return_null URI_NETLINK;
^ulog/:				return_null URI_ULOG;
^pcap/:				return_null URI_PCAP;
^ping/:				return_null URI_PING;

<uri>{
	{name}/:		return_str NAME;
	{name}/@		return_str NAME;

	{host}/\/		BEGIN(path); return_str HOST;
	{host}/:		return_str HOST;
	{port}/\/		BEGIN(path); return_val NUMBER;
}

<path>{
	{path}			{
					/* TODO: decode %XX parts... */
					return_str PATH;
				}
}

				/* RRD file target */
{rrdfile}/:			BEGIN(targets); return_str RRDFILE;
<targets>{
	{dsname}		return_str DSNAME;
	{dscreate}		return_str DSCREATE;
	\\\n			/* connect broken lines */ 
}

				/* "strings" */
\"				BEGIN(string);
<string>{
	[^\\\n\"]+		yymore();
	\\.			yymore();
	\"			BEGIN(targets); return_str STRING;
}
				/* /regexes/ */
^\/				BEGIN(regex);
<regex>{
	[^\\\n\/]+		yymore();
	\\.			yymore();
	\/			BEGIN(targets); return_str_chop REGEX;
}
				/* allows using unquoted paths */
<INITIAL>=			BEGIN(path); return yytext[0];
<targets>[=*/+-]		BEGIN(assign); return yytext[0];
,				BEGIN(targets); return yytext[0];

				/* some common types */
{number}			return_val NUMBER;
{name}				return_str NAME;

				/* global rules */
<*>\n				BEGIN(INITIAL); return '\n';
<*>.       		        return yytext[0];

%%

#ifdef STANDALONE_SCANNER
/*
 * This is the code I use for testing scanner...
 */

# define token(t) t: printf(#t "(\"%s\"):%d ", yylval.str, yylineno); break

// :r!awk '/define/ {print "\t\t\tcase token("$3");"}' parser.h 

YYSTYPE yylval;

int main() {
	int type;

	while ((type = yylex())) {
		switch (type) {
			case token(URI_FILE);
			case token(URI_EXEC);
			case token(URI_HTTP);
			case token(URI_PIPE);
			case token(URI_SNMP);
			case token(URI_NETLINK);
			case token(URI_ULOG);
			case token(URI_PCAP);
			case token(URI_PING);
			case token(URI_RRD);
			case token(NAME);
			case token(PATH);
			case token(HOST);
			case token(STRING);
			case token(REGEX);
			case token(RRDFILE);
			case token(DSNAME);
			case token(DSCREATE);

			case NUMBER:	printf("NUMBER(%d) ", yylval.val); break;
			case '\n':	printf("'\\n'\n"); break;
			default:	printf("'%c' ", type); break;
		}
		
	}

	printf("Scanning completed.\n");

	return 0;
}
#endif /* STANDALONE_SCANNER */
