#!/usr/bin/perl -w

# RRDWeather
# Copyright (c) 2005-2006 Sebastien Wains <seb@wains.be>
# Released under the GNU General Public License
# http://www.wains.be/projects/rrdweather/

use RRDs;
use POSIX qw(strftime);

my $VERSION = "0.40";

############################################################
############## EDIT THESE VALUES BELOW #####################
############################################################

my $city = 'Mons, Belgium';     	 # Place monitored
my $zip = 'BEXX0014';		 	 # ZIP/Intl code
my $xpoints = 650;			 # Graphs width
my $ypoints = 250;			 # Graphs height
my $db_dir = '/usr/local/rrdweather'; 	 # DB files directory
my $tmp_dir = '/tmp/rrdweather';	 # Images directory
my $scriptname = 'weather.cgi';		 # Script name

############################################################
### YOU SHOULD NOT HAVE TO EDIT ANYTHING BELOW THIS LINE ###
############################################################

my $date = strftime "%a %b %e %Y %I.%M%p", localtime;
my $points_per_sample = 3;
my $ypoints_err = 96;
my $db_real = "$db_dir/$zip/real.rrd";
my $db_dew = "$db_dir/$zip/dew.rrd";
my $db_felt = "$db_dir/$zip/felt.rrd";
my $db_humidity = "$db_dir/$zip/humidity.rrd";
my $db_wind = "$db_dir/$zip/wind.rrd";
my $db_pressure = "$db_dir/$zip/pressure.rrd";
my $db_uv = "$db_dir/$zip/uv.rrd";

my @graphs = (
	{ title => 'Daily Graphs',   seconds => 3600*24,        },
	{ title => 'Weekly Graphs',  seconds => 3600*24*7,      },
	{ title => 'Monthly Graphs', seconds => 3600*24*31,     },
	{ title => 'Yearly Graphs',  seconds => 3600*24*365, },
);

sub graph_temperature($$$)
{
	my $range = shift;
	my $file = shift;
	my $title = shift;
	my $step = $range*$points_per_sample/$xpoints;

	my ($graphret,$xs,$ys) = RRDs::graph($file,
		'--imgformat', 'PNG',
		'--width', $xpoints,
		'--height', $ypoints,
		'--start', "-$range",
		"--title= Real & felt temperatures in $city",
		'--vertical-label', 'Real & felt temp.',
		'--units-exponent', 0,
		'--lazy',
		'-Y',
		
		"DEF:real_c=$db_real:real:AVERAGE",
		"DEF:felt_c=$db_felt:felt:AVERAGE",
		"DEF:dew_c=$db_dew:dew:AVERAGE",
		"COMMENT:           Min       Max       Ave       Last\\n",
		'LINE2:real_c#DD3F4A:Real', 'GPRINT:real_c:MIN: %5.2lf C',
		'GPRINT:real_c:MAX: %5.2lf C', 'GPRINT:real_c:AVERAGE: %5.2lf C',
		'GPRINT:real_c:LAST: %5.2lf C\\n',
		'LINE1:felt_c#3F4ADD:Felt', , 'GPRINT:felt_c:MIN: %5.2lf C',
                'GPRINT:felt_c:MAX: %5.2lf C', 'GPRINT:felt_c:AVERAGE: %5.2lf C',
                'GPRINT:felt_c:LAST: %5.2lf C\\n',	
		'LINE1:dew_c#4ADD3F:Dew ', , 'GPRINT:dew_c:MIN: %5.2lf C',
                'GPRINT:dew_c:MAX: %5.2lf C', 'GPRINT:dew_c:AVERAGE: %5.2lf C',
                'GPRINT:dew_c:LAST: %5.2lf C\\n',
		"COMMENT:\\n",
		"COMMENT:Generated by RRDWeather on $date",

	);
	my $ERR=RRDs::error;
	die "ERROR: $ERR\n" if $ERR;
}

sub graph_humidity($$$)
{
        my $range = shift;
        my $file = shift;
        my $title = shift;
        my $step = $range*$points_per_sample/$xpoints;

        my ($graphret,$xs,$ys) = RRDs::graph($file,
                '--imgformat', 'PNG',
                '--width', $xpoints,
                '--height', $ypoints,
                '--start', "-$range",
		"--title= Humidity in $city",
                '--vertical-label', 'Humidity',
                '--lower-limit', 0,
		'--upper-limit', 100,
                '--units-exponent', 0,
                '--lazy',
		'-Y',

                "DEF:humidity_pc=$db_humidity:humidity:AVERAGE",
                'COMMENT:        ', "COMMENT:     Min        Max        Ave        Last\\n",
                'LINE1:humidity_pc#4DD34A:Humidity', 'GPRINT:humidity_pc:MIN: %5.2lf pc',
                'GPRINT:humidity_pc:MAX: %5.2lf pc', 'GPRINT:humidity_pc:AVERAGE: %5.2lf pc',
                'GPRINT:humidity_pc:LAST: %5.2lf pc\\n',
		"COMMENT:\\n",
		"COMMENT:Generated by RRDWeather on $date",

	);
        my $ERR=RRDs::error;
        die "ERROR: $ERR\n" if $ERR;
}

sub graph_wind($$$)
{
        my $range = shift;
        my $file = shift;
        my $title = shift;
        my $step = $range*$points_per_sample/$xpoints;

        my ($graphret,$xs,$ys) = RRDs::graph($file,
                '--imgformat', 'PNG',
                '--width', $xpoints,
                '--height', $ypoints,
                '--start', "-$range",
                "--title= Wind in $city",
                '--vertical-label', 'Wind',
                '--units-exponent', 0,
                '--lazy',
                '-Y',

                "DEF:wind_s=$db_wind:wind:AVERAGE",
                "COMMENT:            Min        Max         Ave         Last\\n",
                'LINE2:wind_s#DD3F4A:Wind', 'GPRINT:wind_s:MIN: %5.2lf kph',
                'GPRINT:wind_s:MAX: %5.2lf kph', 'GPRINT:wind_s:AVERAGE: %5.2lf kph',
                'GPRINT:wind_s:LAST: %5.2lf kph\\n',
		"COMMENT:\\n",
		"COMMENT:Generated by RRDWeather on $date",

        );
        my $ERR=RRDs::error;
        die "ERROR: $ERR\n" if $ERR;
}

sub graph_pressure($$$)
{
        my $range = shift;
        my $file = shift;
        my $title = shift;
        my $step = $range*$points_per_sample/$xpoints;

        my ($graphret,$xs,$ys) = RRDs::graph($file,
                '--imgformat', 'PNG',
                '--width', $xpoints,
                '--height', $ypoints,
                '--start', "-$range",
                "--title= Pressure in $city",
                '--vertical-label', 'Pressure',
                '--lower-limit', 1000,
                '--upper-limit', 1040,
		'--units-exponent', 0,
                '--lazy',
		'-Y',

        	"DEF:pressure_hpa=$db_pressure:pressure:AVERAGE",
                'COMMENT:        ', "COMMENT:        Min           Max           Ave          Last\\n",
                'LINE1:pressure_hpa#DD3F4A:Pressure', 'GPRINT:pressure_hpa:MIN: %5.2lf hPa',
                'GPRINT:pressure_hpa:MAX: %5.2lf hPa', 'GPRINT:pressure_hpa:AVERAGE: %5.2lf hPa',
                'GPRINT:pressure_hpa:LAST: %5.2lf hPa\\n',
		"COMMENT:\\n",
		"COMMENT:Generated by RRDWeather on $date",

	);
        my $ERR=RRDs::error;
        die "ERROR: $ERR\n" if $ERR;
}

sub graph_uv($$$)
{
        my $range = shift;
        my $file = shift;
        my $title = shift;
        my $step = $range*$points_per_sample/$xpoints;

        my ($graphret,$xs,$ys) = RRDs::graph($file,
                '--imgformat', 'PNG',
                '--width', $xpoints,
                '--height', $ypoints,
                '--start', "-$range",
                "--title= UV index in $city",
                '--vertical-label', 'UV index',
                '--lower-limit', 0,
	        '--upper-limit', 10,        
		'--units-exponent', 0,
                '--lazy',
		'-y', '1:5',		

                "DEF:uv_index=$db_uv:uv:AVERAGE",
                'COMMENT:        ', "COMMENT:     Min     Max     Ave    Last\\n",
                'AREA:uv_index#DD3F4A:UV index', 'GPRINT:uv_index:MIN: %5.2lf',
                'GPRINT:uv_index:MAX: %5.2lf', 'GPRINT:uv_index:AVERAGE: %5.2lf',
		'GPRINT:uv_index:LAST: %5.2lf\\n',
		"COMMENT:\\n",
		"COMMENT:Generated by RRDWeather on $date",

	);
        my $ERR=RRDs::error;
        die "ERROR: $ERR\n" if $ERR;
}

sub print_html()
{
	print "Content-Type: text/html\n\n";

	print <<HEADER;
<?xml version="1.0" encoding="iso-8859-1"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1//EN" "http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd">
<html>
<head>
<title>Weather in $city - Generated by RRDWeather</title>
<meta http-equiv="pragma" content="no-cache"/>
<meta http-equiv="refresh" content="300"/>
<meta http-equiv="content-type" content="text/html; charset=ISO-8859-1"/>

<style type="text/css">

body
{
	background-color: #ABABAB; 
	color: #737367;	
	font-size: 11px;
	margin: 0px;
	padding: 0px;
	text-align: center;
	font-family: arial, helvetica, sans-serif;
}

a 
{
	text-decoration:none;
	color: #464646; 
	background-color: #ABABAB; 
}

a:visited 
{ 
	color: #59594F; 
	background-color: #ABABAB; 
}

a:hover 
{ 
	color: #FF6633; 
	background-color: #ABABAB; 
}

a:active 
{ 
	color: #444444; 
	background-color: #ABABAB; 
}

.graphtitle
{
	background-color: #dddddd;
	color: #444444; 
	width: 780px;
	margin: 0 auto;
}

</style>

</head>
<body>
HEADER

	print "<h1>Weather in $city</h1>\n";
	print "<h4>According to weather.com</h4>\n";
	for my $n (0..$#graphs) {
		print "<div class=\"graphtitle\"><h2>".$graphs[$n]{title}."</h2></div>\n";
		print "<p><img src=\"$scriptname?${n}-temperature\" alt=\"RRDWeather\"/></p>\n";
		print "<p><img src=\"$scriptname?${n}-humidity\" alt=\"RRDWeather\"/></p>\n";
		print "<p><img src=\"$scriptname?${n}-wind\" alt=\"RRDWeather\"/></p>\n";
		print "<p><img src=\"$scriptname?${n}-pressure\" alt=\"RRDWeather\"/></p>\n";
		print "<p><img src=\"$scriptname?${n}-uv\" alt=\"RRDWeather\"/></p>\n";
	}

	print <<FOOTER;
<table border="0" width="100%" cellpadding="0" cellspacing="0"><tr><td align="center"><a href="http://www.wains.be/projects/rrdweather/" onclick="window.open(this.href); return false;">RRDWeather</a> $VERSION by <a href="http://www.wains.be/" onclick="window.open(this.href); return false;">Sebastien Wains</a></td></tr></table>
FOOTER
}

sub send_image($)
{
	my $file = shift;
	-r $file or do {
		print "Content-type: text/plain\n\nERROR: can't find $file\n";
		exit 1;
	};

	print "Content-type: image/png\n";
	print "Content-length: ".((stat($file))[7])."\n";
	print "\n";
	open(IMG, $file) or die;
	my $data;
	print $data while read IMG, $data, 1024;
}

sub main()
{
	mkdir $tmp_dir, 0777 unless -d $tmp_dir;
	mkdir "$tmp_dir/$zip", 0777 unless -d "$tmp_dir/$zip";

	my $img = $ENV{QUERY_STRING};
	if(defined $img and $img =~ /\S/) {
		if($img =~ /^(\d+)-temperature$/) {
			my $file = "$tmp_dir/$zip/RRDWeather_$1_temperature.png";
			graph_temperature($graphs[$1]{seconds}, $file, $graphs[$1]{title});
			send_image($file);
		}
		elsif($img =~ /^(\d+)-humidity$/) {
			my $file = "$tmp_dir/$zip/RRDWeather_$1_humidity.png";
			graph_humidity($graphs[$1]{seconds}, $file, $graphs[$1]{title});
			send_image($file);
		}
		elsif($img =~ /^(\d+)-wind$/) {
                        my $file = "$tmp_dir/$zip/RRDWeather_$1_wind.png";
                        graph_wind($graphs[$1]{seconds}, $file, $graphs[$1]{title});
                        send_image($file);
                }
		elsif($img =~ /^(\d+)-pressure$/) {
                        my $file = "$tmp_dir/$zip/RRDWeather_$1_pressure.png";
                        graph_pressure($graphs[$1]{seconds}, $file, $graphs[$1]{title});
                        send_image($file);
                }
		elsif($img =~ /^(\d+)-uv$/) {
                        my $file = "$tmp_dir/$zip/RRDWeather_$1_uv.png";
                        graph_uv($graphs[$1]{seconds}, $file, $graphs[$1]{title});
                        send_image($file);
                }
                else {			
			die "ERROR: invalid argument\n";
			}
		}
		else {
			print_html;
		}
	}

main;
