/*
 * RSBAC MS glue module for f-protd
 *
 * Author and (c) 2002 Amon Ott <ao@rsbac.org>
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/fs.h>
#include <linux/in.h>
#include <asm/uaccess.h>
#include <rsbac/types.h>
#include <rsbac/reg.h>
#include <rsbac/adf.h>
#include <rsbac/aci.h>
#include <rsbac/getname.h>
#include <rsbac/net_getname.h>
#include <rsbac/error.h>
#include <rsbac/proc_fs.h>

static u_long nr_calls = 0;

MODULE_AUTHOR("Amon Ott");
MODULE_DESCRIPTION("RSBAC MS glue module for f-protd");

MODULE_PARM(level, "l");
static u_int level = RSBAC_MS_LEVEL + 1;

MODULE_PARM(debug, "l");
static int debug = 0;

static u_int old_level = 0;
rsbac_ms_do_scan_t * old_do_scan = NULL;

/*** Network settings ***/
#define LOCAL_PORT 0
#define LOCAL_ADDR "127.0.0.1"
#define REMOTE_PORT_MIN 10200
#define REMOTE_PORT_MAX 10204
#define REMOTE_ADDR "127.0.0.1"

/* declare net functions */
long sys_socket(int family, int type, int protocol);
long sys_bind(int fd, struct sockaddr *umyaddr, int addrlen);
long sys_connect(int fd, struct sockaddr *uservaddr, int addrlen);

/* PROC functions */

#if defined(CONFIG_RSBAC_PROC)
#define PROC_NAME "ms-f-protd"
static struct proc_dir_entry * proc_info_p;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)
static int
ms_proc_info(char *buffer, char **start, off_t offset, int length, int dummy)
#else
static int
ms_proc_info(char *buffer, char **start, off_t offset, int length)
#endif
{
  int len = 0;
  off_t pos   = 0;
  off_t begin = 0;

  union rsbac_target_id_t       rsbac_target_id;
  union rsbac_attribute_value_t rsbac_attribute_value;

  if (!rsbac_is_initialized())
    return (-ENOSYS);

  rsbac_target_id.scd = ST_rsbac;
  rsbac_attribute_value.dummy = 0;
  if (!rsbac_adf_request(R_GET_STATUS_DATA,
                         current->pid,
                         T_SCD,
                         rsbac_target_id,
                         A_none,
                         rsbac_attribute_value))
    {
      return -EPERM;
    }
  len += sprintf(buffer, "RSBAC MS glue module for f-protd\n----------------------------------\n");
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

  len += sprintf(buffer + len, "%lu calls to do_scan function,\n",
                 nr_calls);
  len += sprintf(buffer + len, "scan level is %u.\n",
                 level);
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

out:
  *start = buffer + (offset - begin);
  len -= (offset - begin);
  
  if (len > length)
    len = length;
  return len;
}
#endif /* CONFIG_RSBAC_PROC */

/**** Scanning Function ****/

static int do_scan(struct dentry * dentry_p)
  {
    u_int i;
    mm_segment_t oldfs;
    int sock_fd;
    struct sockaddr_in addr;
    int err;

    nr_calls++;

    /* create a socket */
    sock_fd = sys_socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sock_fd < 0)
      {
        printk(KERN_WARNING
               "RSBAC MS glue module for F-Protd: creating local log socket failed with error %u, exiting!\n",
               sock_fd);
        return -RSBAC_EWRITEFAILED;
      }
    /* bind local address */
    addr.sin_family = PF_INET;
    addr.sin_port = htons(LOCAL_PORT);
    err = rsbac_net_str_to_inet(LOCAL_ADDR,
                                &addr.sin_addr.s_addr);
    if(err < 0)
      {
        printk(KERN_WARNING
               "RSBAC MS glue module for F-Protd: converting local socket address %s failed with error %u, exiting!\n",
               LOCAL_ADDR,
               err);
        sys_close(sock_fd);
        return -RSBAC_EINVALIDVALUE;
      }
    /* change data segment - sys_bind reads address from user space */
    oldfs = get_fs();
    set_fs(KERNEL_DS);
    err = sys_bind(sock_fd, (struct sockaddr *)&addr, sizeof(addr));
    set_fs(oldfs);
    if(err < 0)
      {
        printk(KERN_WARNING
               "RSBAC MS glue module for F-Protd: binding local socket address %u.%u.%u.%u:%u failed with error %u, exiting!\n",
               NIPQUAD(addr.sin_addr.s_addr),
               LOCAL_PORT,
               err);
        sys_close(sock_fd);
        return -RSBAC_EWRITEFAILED;
      }

    /* convert remote address */
    addr.sin_family = PF_INET;
    err = rsbac_net_str_to_inet(REMOTE_ADDR,
                                &addr.sin_addr.s_addr);
    if(err < 0)
      {
        printk(KERN_WARNING
               "RSBAC MS glue module for F-Protd: converting remote socket address %s failed with error %u, exiting!\n",
               REMOTE_ADDR,
               err);
        sys_close(sock_fd);
        return -RSBAC_EINVALIDVALUE;
      }

    for(i = REMOTE_PORT_MIN; i <= REMOTE_PORT_MAX ; i++)
      {
        addr.sin_port = htons(i);
        err = sys_connect(sock_fd,
                          (struct sockaddr *)&addr,
                          sizeof(addr));
        if(err >=0)
          {
            if(debug)
              printk(KERN_DEBUG
                     "RSBAC MS glue module for F-Protd: successfully connected to scanner, closing now.\n");
            sys_close(sock_fd);
            return level;
          }
      }

    printk(KERN_WARNING
           "RSBAC MS glue module for F-Protd: connecting to scanner failed!\n");
    return -RSBAC_EWRITEFAILED;
  }

/**** Init Function ****/

int init_module(void)
{
  printk(KERN_INFO "RSBAC MS glue module for F-Protd: Initializing with level %u.\n",
         level);

  if(level <= RSBAC_MS_LEVEL)
    {
      printk(KERN_WARNING "RSBAC MS glue module for F-Protd: Need a scan level > %u.\n",
             RSBAC_MS_LEVEL);
      return -ENOEXEC;
    }

  #if defined(CONFIG_RSBAC_PROC)
  proc_info_p = create_proc_entry(PROC_NAME,
                                  S_IFREG | S_IRUGO,
                                  proc_rsbac_root_p);
  if(!proc_info_p)
    {
      printk(KERN_WARNING "RSBAC MS glue module for F-Protd: Not loaded due to failed proc entry registering.\n");
      return -ENOEXEC;
    }
  proc_info_p->get_info = ms_proc_info;
  #endif 
  
  old_level = rsbac_ms_scan_level;
  rsbac_ms_scan_level = level;

  old_do_scan = rsbac_ms_do_scan;
  rsbac_ms_do_scan = do_scan;

  printk(KERN_INFO "RSBAC MS glue module for F-Protd: Loaded.\n");

  return 0;
}

void cleanup_module(void)
{
  printk(KERN_INFO "RSBAC MS glue module for F-Protd: Unloading.\n");
  #if defined(CONFIG_RSBAC_PROC)
  remove_proc_entry(PROC_NAME, proc_rsbac_root_p);
  #endif 
  proc_info_p->get_info = NULL;
  rsbac_ms_scan_level = old_level;
  rsbac_ms_do_scan = old_do_scan;
  printk(KERN_INFO "RSBAC MS glue module for F-Protd: Unloaded.\n");
}
