#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <pwd.h>
#include <rsbac/types.h>
#include <rsbac/aci_data_structures.h>
#include <rsbac/getname.h>
#include <rsbac/acl_getname.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include <rsbac/helpers.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#define NR_ENTRIES 50
#define GRANT_PROG "acl_grant"

int verbose=0;
int recurse=0;
int printall=0;
int backup=0;
u_int scripting=0;
union rsbac_attribute_value_t value;
enum rsbac_target_t target;
char * target_n;
enum rsbac_attribute_t attr;
char * progname;

int process(char * name)
  {
    int res = 0;
    char tmp1[RSBAC_MAXNAMELEN], tmp2[RSBAC_MAXNAMELEN];
    struct stat buf;
    char * i_name;
    struct rsbac_acl_entry_t entry_array[NR_ENTRIES];
    rsbac_time_t ttl_array[NR_ENTRIES];
    union rsbac_target_id_t tid;

    if(!strcmp(name,":DEFAULT:"))
      {
        switch(target)
          {
            case T_FILE:
            case T_DIR:
            case T_FIFO:
            case T_SYMLINK:
            case T_FD:
            case T_DEV:
              i_name = NULL;
              break;

            case T_IPC:
              tid.ipc.type = I_none;
              break;
            case T_SCD:
              tid.scd = AST_none;
              break;
            case T_USER:
              tid.user = RSBAC_NO_USER;
              break;
            case T_PROCESS:
              tid.process = 0;
              break;
            case T_NETDEV:
              tid.netdev[0] = 0;
              break;
            case T_NETTEMP_NT:
              tid.nettemp = 0;
              break;
            case T_NETOBJ:
              tid.netobj.sock_p = NULL;
              tid.netobj.local_addr = NULL;
              tid.netobj.local_len = 0;
              tid.netobj.remote_addr = NULL;
              tid.netobj.remote_len = 0;
              break;

            default:
              fprintf(stderr, gettext("Invalid target %u for %s, skipped!\n"),
                      target, name);
              return(1);
          }
      }
    else
      {
        switch(target)
          {
            case T_FILE:
            case T_DIR:
            case T_FIFO:
            case T_SYMLINK:
            case T_FD:
            case T_DEV:
              i_name = name;
              break;

            case T_SCD:
              tid.scd = get_acl_scd_type_nr(name);
              if((tid.scd == ST_none) || (tid.scd == AST_none))
                {
                  fprintf(stderr, gettext("%s is no valid SCD name, skipped\n"),
                           name);
                  return(1);
                }
              break;
            case T_NETDEV:
              strncpy(tid.netdev, name, RSBAC_IFNAMSIZ);
              tid.netdev[RSBAC_IFNAMSIZ] = 0;
              break;
            case T_NETTEMP:
            case T_NETTEMP_NT:
              tid.nettemp = strtoul(name, 0, 10);
              break;
            case T_NETOBJ:
              tid.netobj.sock_p = (void *) strtoul(name, 0, 0);
              tid.netobj.remote_addr = NULL;
              tid.netobj.remote_len = 0;
              break;

            default:
              fprintf(stderr, gettext("Invalid target %u for %s, skipped!\n"),
                      target, name);
              return(1);
          }
      }
    if(verbose)
      printf(gettext("Processing %s '%s'\n"),
             target_n,
             name);

    switch(target)
      {
        case T_FILE:
        case T_DIR:
        case T_FIFO:
        case T_SYMLINK:
        case T_FD:
        case T_DEV:
          res = rsbac_acl_get_tlist_n(target, i_name, entry_array, ttl_array, NR_ENTRIES);
          break;
        default:
          res = rsbac_acl_get_tlist(target, &tid, entry_array, ttl_array, NR_ENTRIES);
      }
    if(res<0)
      {
        if(   verbose
           || (res != -RSBAC_EINVALIDTARGET)
          )
          {
            get_error_name(tmp1,res);
            fprintf(stderr, gettext("%s: error: %s\n"), name, tmp1);
          }
      }
    else
      {
        rsbac_time_t now = time(NULL);
        int j;

        if(backup)
          {
            if(printall)
              {
                int i;

                for(j=0; j<res; j++)
                  {
                    if(ttl_array[j])
                      printf("%s -V %u -vsT %u %s %u",
                             GRANT_PROG,
                             RSBAC_VERSION_NR,
                             ttl_array[j] + now,
                             get_acl_subject_type_name(tmp1, entry_array[j].subj_type),
                             entry_array[j].subj_id);
                    else
                      printf("%s -V %u -vs %s %u",
                             GRANT_PROG,
                             RSBAC_VERSION_NR,
                             get_acl_subject_type_name(tmp1, entry_array[j].subj_type),
                             entry_array[j].subj_id);
                    for (i=0; i<R_NONE; i++)
                      if(entry_array[j].rights & ((rsbac_acl_rights_vector_t) 1 << i))
                        printf(" %s", get_request_name(tmp1,i));
                    for (i=RSBAC_ACL_SPECIAL_RIGHT_BASE; i<ACLR_NONE; i++)
                      if(entry_array[j].rights & ((rsbac_acl_rights_vector_t) 1 << i))
                        printf(" %s", get_acl_special_right_name(tmp1,i));
                    printf(" %s \"%s\"\n", target_n, name);
                  }
              }
            else
              {
                for(j=0; j<res; j++)
                  {
                    if(ttl_array[j])
                      printf("%s -V %u -vsbt %u %s %u %s %s \"%s\"\n",
                             GRANT_PROG,
                             RSBAC_VERSION_NR,
                             ttl_array[j],
                             get_acl_subject_type_name(tmp1, entry_array[j].subj_type),
                             entry_array[j].subj_id,
                             u64tostracl(tmp2, entry_array[j].rights),
                             target_n,
                             name);
                    else
                      printf("%s -V %u -vsb %s %u %s %s \"%s\"\n",
                             GRANT_PROG,
                             RSBAC_VERSION_NR,
                             get_acl_subject_type_name(tmp1, entry_array[j].subj_type),
                             entry_array[j].subj_id,
                             u64tostracl(tmp2, entry_array[j].rights),
                             target_n,
                             name);
                  }
              }
          }
        else /* no backup */
          {
            if(scripting)
              {
                for(j=0; j<res; j++)
                  {
                    if(ttl_array[j])
                      printf("%s %u(ttl:%us)\n",
                             get_acl_subject_type_name(tmp1, entry_array[j].subj_type),
                             entry_array[j].subj_id,
                             ttl_array[j]);
                    else
                      printf("%s %u\n",
                             get_acl_subject_type_name(tmp1, entry_array[j].subj_type),
                             entry_array[j].subj_id);
                    if(printall)
                      {
                        int i;

                        for (i=0; i<R_NONE; i++)
                          if(entry_array[j].rights & ((rsbac_acl_rights_vector_t) 1 << i))
                            printf("%s\n", get_request_name(tmp1,i));
                        for (i=RSBAC_ACL_SPECIAL_RIGHT_BASE; i<ACLR_NONE; i++)
                          if(entry_array[j].rights & ((rsbac_acl_rights_vector_t) 1 << i))
                            printf("%s\n", get_acl_special_right_name(tmp1,i));
                      }
                  }
              }
            else
              {
                printf(gettext("%s: %i entries\n"),
                       name, res);
                for(j=0; j<res; j++)
                  {
                    if(ttl_array[j])
                      printf("  %s %u:\t%s (ttl: %us)\n",
                             get_acl_subject_type_name(tmp1, entry_array[j].subj_type),
                             entry_array[j].subj_id,
                             u64tostracl(tmp2, entry_array[j].rights),
                             ttl_array[j]);
                    else
                      printf("  %s %u:\t%s\n",
                             get_acl_subject_type_name(tmp1, entry_array[j].subj_type),
                             entry_array[j].subj_id,
                             u64tostracl(tmp2, entry_array[j].rights));
                    if(printall)
                      {
                        int i;

                        for (i=0; i<R_NONE; i++)
                          if(entry_array[j].rights & ((rsbac_acl_rights_vector_t) 1 << i))
                            printf("    %s\n", get_request_name(tmp1,i));
                        for (i=RSBAC_ACL_SPECIAL_RIGHT_BASE; i<ACLR_NONE; i++)
                          if(entry_array[j].rights & ((rsbac_acl_rights_vector_t) 1 << i))
                            printf("    %s\n", get_acl_special_right_name(tmp1,i));
                      }
                  }
              }
          }
      }

do_recurse:
    if(   !lstat(name,&buf)
       && S_ISDIR(buf.st_mode)
       && recurse)
      {
        DIR * dir_stream_p;
        struct dirent * dirent_p;
        char name2[PATH_MAX];

        if(S_ISLNK(buf.st_mode))
          return(0);
        if(!(dir_stream_p = opendir(name)))
          {
            fprintf(stderr, gettext("opendir for dir %s returned error: %s\n"),
                   name,
                   strerror(errno));
            return(-2);
          }
        while((dirent_p = readdir(dir_stream_p)))
          {
            if(   (strcmp(".",dirent_p->d_name))
               && (strcmp("..",dirent_p->d_name)) )
              {
                strcpy(name2,name);
                strcat(name2,"/");
                strcat(name2,dirent_p->d_name);
                process(name2);
              }
          }
        closedir(dir_stream_p);
      }
    return(0);
  }

int main(int argc, char ** argv)
{
  int res = 0;
  int i;
  char none_name[] = "FD";

  locale_init();
  
  progname = argv[0];
  while((argc > 1) && (argv[1][0] == '-'))
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case 'v':
                verbose=1;
                break;
              case 'r':
                recurse=1;
                break;
              case 'p':
                printall=1;
                break;
              case 'b':
                backup=1;
                break;
              case 's':
                scripting=1;
                break;
              case 'n':
                {
                  char tmp[80];
                  for(i=0; i<ST_none; i++)
                    printf("%s\n", get_scd_type_name(tmp, i));
                  for(i=AST_min; i<AST_none; i++)
                    printf("%s\n", get_acl_scd_type_name(tmp, i));
                  exit(0);
                }
              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
            }
          pos++;
        }
      argv++;
      argc--;
    }
  if (argc > 1)
    {
      target = get_target_nr(argv[1]);
      target_n = argv[1];
      if(target == T_NONE)
        {
          if(verbose)
            printf(gettext("%s: %i targets\n\n"), progname, argc - 1);
          fprintf(stderr, gettext("%s: No target type given, assuming FD\n"), progname);
          target = T_FD;
          target_n = none_name;
          if(argc < 1)
            process(".");
          else
            for (i=1;i < (argc);i++)
              {
                process(argv[i]);
              }
        }
      else
        {
          if(argc > 2)
            {
              if(verbose)
                printf(gettext("%s: %i targets\n\n"), progname, argc - 2);
              for (i=1;i < (argc-1);i++)
                {
                  process(argv[i+1]);
                }
            }
          else
            process(".");
        }
    }
  else
    {
      printf(gettext("%s (RSBAC %s)\n***\n"), progname, VERSION);
      printf(gettext("Use: %s [switches] target-type file/dir/scdname(s)\n"), progname);  
      printf(gettext(" -v = verbose, -r = recurse into subdirs,\n"));
      printf(gettext(" -p = print right names, -b = backup mode\n"));
      printf(gettext(" -n = list valid SCD names,\n"));
      printf(gettext(" -s = scripting mode\n"));
      printf(gettext("  target-type = FILE, DIR, FIFO, SYMLINK, DEV, IPC, SCD, USER, PROCESS, NETDEV,\n"));
      printf(gettext("                NETTEMP_NT, NETTEMP, NETOBJ or FD\n"));
      printf(gettext("  (FD: let %s decide between FILE, DIR, FIFO and SYMLINK, no DEV),\n"), progname);
      printf(gettext("  (IPC, USER, PROCESS: only :DEFAULT:\n"));
      printf(gettext("  (NETTEMP: no :DEFAULT:\n"));
      printf(gettext("- Use name :DEFAULT: for default ACL\n"));
    }
  return (res);
}

