#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <rsbac/types.h>
#include <rsbac/getname.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include <rsbac/helpers.h>
#include <rsbac/aci_data_structures.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

const char   set_prog[] = "attr_set_user";
__s64 attr_list[RSBAC_USER_NR_ATTRIBUTES] = RSBAC_USER_ATTR_LIST;
int alluser = 0;
int verbose = 0;
FILE * tfile;
char * filename = NULL;

int def_attr[RSBAC_USER_NR_ATTRIBUTES] = {
      0, /* pseudo */
      0, /* log_user_based */
      SL_unclassified, /* security_level */
      SL_unclassified, /* initial_security_level */
      SL_unclassified, /* min_security_level */
      RSBAC_MAC_DEF_CAT_VECTOR, /* mac_categories */
      RSBAC_MAC_DEF_CAT_VECTOR, /* mac_initial_categories */
      RSBAC_MAC_MIN_CAT_VECTOR, /* mac_min_categories */
      SR_user, /* mac_role */
      RSBAC_MAC_DEF_U_FLAGS, /* mac_user_flags */
      SR_user, /* fc_role */
      SR_user, /* sim_role */
      SR_user, /* ms_role */
      SR_user, /* ff_role */
      SR_user, /* auth_role */
      0, /* pm_task_set */
      PR_user, /* pm_role */
      RSBAC_RC_GENERAL_ROLE, /* rc_def_role */
      0, /* min_caps */
      (rsbac_cap_vector_t) -1, /* max_caps */
      SR_user, /* cap_role */
      SR_user, /* jail_role */
      SR_user /* res_role */
  };

int process(struct passwd * user_info_p)
  {
    int res = 0;
    char tmp1[120],tmp2[120];
    int j;
    union rsbac_target_id_t       tid;
    union rsbac_attribute_value_t value;

    tid.user = user_info_p->pw_uid;
    if(verbose)
      printf(gettext("Processing user %s\n"), user_info_p->pw_name);
    for (j=0;j < RSBAC_USER_NR_ATTRIBUTES;j++)
      {
        value.dummy = -1;
        res = rsbac_get_attr(get_attr_module(attr_list[j]), T_USER, &tid, attr_list[j], &value, 0);
        if(res)
          {
            if(   (res != -RSBAC_EINVALIDMODULE)
               && (   verbose
                   || (res != -RSBAC_EINVALIDTARGET)
                  )
              )
              {
                get_error_name(tmp1,res);
                get_attribute_name(tmp2,attr_list[j]);
                fprintf(stderr, "%s (%s): %s\n",
                        user_info_p->pw_name, tmp2, tmp1);
              }
          }
        else
          {
            switch(attr_list[j])
              {
                case A_rc_def_role:
                  if(value.rc_role != def_attr[j])
                    fprintf(tfile,
                            "%s -V %u %s %s %u\n",
                            set_prog,
                            RSBAC_VERSION_NR,
                            user_info_p->pw_name,
                            get_attribute_name(tmp1,attr_list[j]),
                            value.rc_role);
                  break;
                case A_security_level:
                case A_initial_security_level:
                case A_min_security_level:
                case A_mac_role:
                case A_mac_user_flags:
                case A_fc_role:
                case A_sim_role:
                case A_pm_role:
                case A_ms_role:
                case A_ff_role:
                case A_auth_role:
                case A_cap_role:
                case A_jail_role:
                case A_res_role:
                  if(value.u_char_dummy != def_attr[j])
                    fprintf(tfile,
                            "%s -V %u %s %s %u\n",
                            set_prog,
                            RSBAC_VERSION_NR,
                            user_info_p->pw_name,
                            get_attribute_name(tmp1,attr_list[j]),
                            value.u_char_dummy);
                  break;
                case A_log_user_based:
                  if (value.log_user_based & RSBAC_ALL_REQUEST_VECTOR)
                    fprintf(tfile,
                            "%s -V %u %s %s %s\n",
                            set_prog,
                            RSBAC_VERSION_NR,
                            user_info_p->pw_name,
                            get_attribute_name(tmp1,attr_list[j]),
                            u64tostrlog(tmp2,value.log_user_based));
                  break;
              case A_max_caps:
              case A_min_caps:
                  if (value.max_caps != def_attr[j])
                    fprintf(tfile,
                            "%s -V %u %s %s %s\n",
                            set_prog,
                            RSBAC_VERSION_NR,
                            user_info_p->pw_name,
                            get_attribute_name(tmp1,attr_list[j]),
                            u32tostrcap(tmp2,value.max_caps));
                break;
              case A_mac_categories:
              case A_mac_initial_categories:
                if (value.mac_categories != RSBAC_MAC_DEF_CAT_VECTOR)
                    fprintf(tfile,
                            "%s -V %u %s %s %s\n",
                            set_prog,
                            RSBAC_VERSION_NR,
                            user_info_p->pw_name,
                            get_attribute_name(tmp1,attr_list[j]),
                            u64tostrmac(tmp2,value.mac_categories));
                break;
              case A_mac_min_categories:
                if (value.mac_categories != RSBAC_MAC_MIN_CAT_VECTOR)
                    fprintf(tfile,
                            "%s -V %u %s %s %s\n",
                            set_prog,
                            RSBAC_VERSION_NR,
                            user_info_p->pw_name,
                            get_attribute_name(tmp1,attr_list[j]),
                            u64tostrmac(tmp2,value.mac_categories));
                break;

              default:
                  if(value.dummy != def_attr[j])
                    fprintf(tfile,
                            "%s -V %u %s %s %i\n",
                            set_prog,
                            RSBAC_VERSION_NR,
                            user_info_p->pw_name,
                            get_attribute_name(tmp1,attr_list[j]),
                            value.dummy);
              }
          }
      }
    return(res);
  }

int main(int argc, char ** argv)
{
  int res = 0;
  char tmp1[RSBAC_MAXNAMELEN],tmp2[RSBAC_MAXNAMELEN],tmp3[RSBAC_MAXNAMELEN];
  int i,j;
  char * progname;
  struct passwd * user_info_p;

  locale_init();

  progname = argv[0];
  while((argc > 1) && (argv[1][0] == '-'))
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case 'v':
                verbose++;
                break;
              case 'a':
                alluser=1;
                break;
              case 'o':
                if(argc > 2)
                  {
                    filename = argv[2];
                    argv++;
                    argc--;
                  }
                else
                  fprintf(stderr, gettext("%s: missing filename for parameter o\n"), progname);
                break;
              case 'A':
                printf(gettext("- attributes and values in backup = see following list:\n"));
                for (j=0;j<RSBAC_USER_NR_ATTRIBUTES;j++)
                  {
                    get_switch_target_name(tmp1, get_attr_module(attr_list[j]));
                    get_attribute_name(tmp2,attr_list[j]);
                    get_attribute_param(tmp3,attr_list[j]);
                    printf("[%-4s] %s\n\t%s\n",tmp1,tmp2,tmp3);
                  }
                exit(0);
              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
            }
          pos++;
        }
      argv++;
      argc--;
    }

  if (   (argc > 1)
      || (alluser)
     )
    {
      if(!filename)
        tfile = stdout;
      else
        {
          if (!(tfile=fopen(filename,"w")))
            {
              fprintf(stderr, gettext("opening target file returned error: %s\n"),
                              strerror(errno));
            }
        }
      if(alluser)
        {
          if(verbose)
            printf(gettext("%s: processing all users\n"), progname);
          while ((user_info_p = getpwent()))
            {
              process(user_info_p);
            }
          endpwent();
        }
      else
        {
          if(verbose)
            printf(gettext("%s: %i targets\n"), progname, argc - 2);
          for (i=1;i < argc;i++)
            {
              if(   (!(user_info_p = getpwnam(argv[i])))
                 && (   (!(user_info_p = getpwuid(strtol(argv[i],0,10))))
                     || (!(user_info_p->pw_uid) && strcmp("0",argv[i])) )
                )
                fprintf(stderr, gettext("Unknown user %s - skipped!\n"), argv[i]);
              else
                process(user_info_p);
            }
        }
      if(tfile != stdout)
        fclose(tfile);
    }
  else
    {
      printf(gettext("%s (RSBAC %s)\n***\n"), argv[0], VERSION);
      printf(gettext("Use: %s [-a] [-v] [-o target-file] [username(s)]\n"), progname);  
      printf(gettext("- should be called by root with all rsbac modules switched off,\n"));
      printf(gettext("- -a = process all users, -v = verbose,\n"));
      printf(gettext("- -o target-file = write to file, not stdout,\n"));
      printf(gettext("- -A = list attributes and values\n"));
    }
  return (res);
}
