#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <linux/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <rsbac/types.h>
#include <rsbac/getname.h>
#include <rsbac/net_getname.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include <rsbac/helpers.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#define SETPROG "net_temp"
#define LISTROOM 10

int verbose=0;
int backup=0;
rsbac_version_t version=RSBAC_VERSION_NR;

char * get_temp_name(rsbac_net_temp_id_t temp, char * name)
  {
    union  rsbac_net_temp_syscall_data_t data;

    if(!rsbac_net_template(NTS_get_name, temp, &data))
      {
        strcpy(name, data.name);
      }
    else
      {
        strcpy(name, gettext("*unknown*"));
      }
    return name;
  }

void arg_exit(char * call)
  {
    fprintf(stderr, "Too few arguments for call %s\n", call);
    exit(1);
  }

void list_template(rsbac_net_temp_id_t id)
  {
    union rsbac_net_temp_syscall_data_t data;
    union rsbac_net_temp_syscall_data_t data2;
    __u32 mask;
    char tmp[RSBAC_MAXNAMELEN];

    if(verbose)
      printf("\nGetting data of template %u\n",
             id);
    else
      printf("\n");
    error_exit(rsbac_net_template(NTS_get_name, id, &data));
    if(backup)
      printf("%s -V %u new_template %u \"%s\"\n",
             SETPROG,
             RSBAC_VERSION_NR,
             id, data.name);
    else
      printf("ID:\t\t%u\nName:\t\t%s\n", id, data.name);
    error_exit(rsbac_net_template(NTS_get_address_family, id, &data2));
    if(backup)
      printf("%s -V %u set_address_family %u %s\n",
             SETPROG,
             RSBAC_VERSION_NR,
             id,
             rsbac_get_net_family_name(tmp, data2.address_family));
    else
      printf("Family:\t\t%s\n",
            rsbac_get_net_family_name(tmp, data2.address_family));
    error_exit(rsbac_net_template(NTS_get_type, id, &data));
    if(backup)
      printf("%s -V %u set_type %u %s\n",
             SETPROG,
             RSBAC_VERSION_NR,
             id,
             rsbac_get_net_type_name(tmp, data.type));
    else
      printf("Socket type:\t%s\n",
             rsbac_get_net_type_name(tmp, data.type));
    error_exit(rsbac_net_template(NTS_get_address, id, &data));
    switch(data2.address_family)
      {
        case AF_INET:
          if(backup)
            printf("%s -V %u set_address %u %u.%u.%u.%u\n",
                   SETPROG,
                   RSBAC_VERSION_NR,
                   id,
                   NIPQUAD(*(__u32*) data.address));
          else
            printf("Address:\t%u.%u.%u.%u\n",
                   NIPQUAD(*(__u32*) data.address));
          break;
        case AF_UNIX:
          if(backup)
            printf("%s -V %u -u set_address %u \"%s\"\n",
                   SETPROG,
                   RSBAC_VERSION_NR,
                   id, data.address);
          else
            printf("Address:\t%s\n", data.address);
          break;
        default:
          if(!backup)
            printf("Address:\t(address family not supported)\n",
                 rsbac_get_net_family_name(tmp, data2.address_family));
      }
    error_exit(rsbac_net_template(NTS_get_valid_len, id, &data));
    if(backup)
      printf("%s -V %u set_valid_len %u %u\n",
             SETPROG,
             RSBAC_VERSION_NR,
             id, data.valid_len);
    else
      {
        if(data2.address_family == AF_INET)
          {
            mask = rsbac_net_make_mask_u32(data.valid_len);
            printf("Valid len:\t%u (mask %u.%u.%u.%u)\n",
                   data.valid_len,
                   NIPQUAD(mask));
          }
        else
          {
            printf("Valid len:\t%u\n",
                   data.valid_len);
          }
      }
    error_exit(rsbac_net_template(NTS_get_protocol, id, &data));
    if(backup)
      printf("%s -V %u set_protocol %u %s\n",
             SETPROG,
             RSBAC_VERSION_NR,
             id,
             rsbac_get_net_protocol_name(tmp, data.protocol));
    else
      printf("Protocol:\t%s\n",
             rsbac_get_net_protocol_name(tmp, data.protocol));
    error_exit(rsbac_net_template(NTS_get_netdev, id, &data));
    if(backup)
      printf("%s -V %u set_netdev %u \"%s\"\n",
             SETPROG,
             RSBAC_VERSION_NR,
             id, data.netdev);
    else
      printf("Netdev:\t\t%s\n", data.netdev);
    error_exit(rsbac_net_template(NTS_get_min_port, id, &data));
    error_exit(rsbac_net_template(NTS_get_max_port, id, &data2));
    if(backup)
      {
        printf("%s -V %u set_min_port %u %u\n",
               SETPROG,
               RSBAC_VERSION_NR,
               id, data.min_port);
        printf("%s -V %u set_max_port %u %u\n",
               SETPROG,
               RSBAC_VERSION_NR,
               id, data2.max_port);
      }
    else
      printf("Ports:\t\t%u:%u\n", data.min_port, data2.max_port);
    return;
  }

int main(int argc, char ** argv)
{
  int res = 0;
  int i;
  int scripting=0;
  int numerical=0;
  int afunix=0;
  int dnsname=0;
  int listall=0;
  char * progname;

  locale_init();

  progname = argv[0];
  while((argc > 1) && (argv[1][0] == '-'))
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case 'v':
                verbose++;
                break;
              case 'b':
                backup=1;
                break;
              case 'n':
                numerical=1;
                break;
              case 'u':
                afunix=1;
                break;
              case 'd':
                dnsname=1;
                break;
              case 's':
                scripting=1;
                break;
              case 'l':
                {
                  char tmp[RSBAC_MAXNAMELEN];

                  for(i=0; i< NTS_none ; i++)
                    printf("%s\n",
                           rsbac_get_net_temp_syscall_name(tmp, i));
                  exit(0);
                }
              case 'a':
                listall=1;
                break;
              case 'V':
                if(argc < 2)
                  {
                    fprintf(stderr, gettext("%s: no version number for switch V\n"), progname);
                    exit(1);
                  }
                version = strtol(argv[2],0,10);
                argv++;
                argc--;
                break;
              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
            }
          pos++;
        }
      argv++;
      argc--;
    }
  if(   (argc == 2)
     && (   !strcmp(argv[1], "list_temp_names")
         || !strcmp(argv[1], "list_temp_nr")
        )
    )
    {
      rsbac_net_temp_id_t * temp_array;
      union rsbac_net_temp_syscall_data_t data;
      long count;
      int show_names = 0;
      int j;

      if(!strcmp(argv[1], "list_temp_names"))
        show_names = 1;
      count = rsbac_net_list_all_template(NULL, 0);
      error_exit(count);
      count += LISTROOM;
      temp_array = malloc(count * sizeof(*temp_array));
      if(!temp_array)
        error_exit(-ENOMEM);
      count = rsbac_net_list_all_template(temp_array, count);
      for(i = 0; i< count ; i++)
        {
          if(show_names)
            {
              res = rsbac_net_template(NTS_get_name, temp_array[i], &data);
              if(!res)
                {
                  for(j=0; j<strlen(data.name); j++)
                    if(data.name[j] == ' ')
                      data.name[j] = '_';
                  printf("%u %s\n", temp_array[i], data.name);
                }
            }
          else
            printf("%u\n", temp_array[i]);
        }
      free(temp_array);
    }
  else
  if(   (   (argc >= 3)
         && !strcmp(argv[1], "list_template")
        )
     || (listall)
    )
    {
      if(listall)
        {
          rsbac_net_temp_id_t * temp_array;
          long count;

          count = rsbac_net_list_all_template(NULL, 0);
          error_exit(count);
          count += LISTROOM;
          temp_array = malloc(count * sizeof(*temp_array));
          if(!temp_array)
            error_exit(-ENOMEM);
          count = rsbac_net_list_all_template(temp_array, count);
          for(i = 0; i< count ; i++)
            list_template(temp_array[i]);
          free(temp_array);
        }
      else
        {
          rsbac_net_temp_id_t id;

          for(i=2 ; i< argc ; i++)
            {
              id = strtoul(argv[i],0,10);
              if(id)
                list_template(id);
              else
                show_error(-RSBAC_EINVALIDTARGET);
            }
        }
      exit(0);
    }
  else
  if(argc > 2)
    {
      enum  rsbac_net_temp_syscall_t call;
            rsbac_net_temp_id_t id;
      union rsbac_net_temp_syscall_data_t data;
      char tmp[RSBAC_MAXNAMELEN];

      call = rsbac_get_net_temp_syscall_nr(argv[1]);
      id = strtoul(argv[2],0,10);
      if(!id)
        error_exit(-RSBAC_EINVALIDTARGET);
      switch(call)
        {
          case NTS_set_address:
            if(argc > 3)
              {
                if(afunix)
                  {
                    strncpy(data.address, argv[3], RSBAC_NET_MAX_ADDRESS_LEN-1);
                    data.address[RSBAC_NET_MAX_ADDRESS_LEN-1] = 0;
                    if(verbose)
                      printf("Setting address for template %u to string %s\n",
                             id, data.address);
                  }
                else
                if(numerical)
                  {
                    *(__u32*)data.address = strtoul(argv[3],0,10);
                    if(verbose)
                      printf("Setting address for template %u to IP %u.%u.%u.%u\n",
                             id, NIPQUAD(*(__u32*) data.address) );
                  }
                else
                if(dnsname)
                  {
                    struct hostent * host;
                    struct in_addr * addr_p;

                    host = gethostbyname(argv[3]);
                    if(!host)
                      {
                        fprintf(stderr, "DNS lookup of address %s failed!\n", argv[3]);
                        error_exit(-RSBAC_ENOTFOUND);
                      }
                    addr_p = (struct in_addr *) host->h_addr;
                    *(__u32 *) data.address = addr_p->s_addr;
                    if(verbose)
                      printf("Setting address for template %u to IP %u.%u.%u.%u\n",
                             id, NIPQUAD(*(__u32*) data.address) );
                  }
                else
                  {
                    struct in_addr addr;

                    error_exit(inet_aton(argv[3], &addr));
                    *(__u32 *) data.address = addr.s_addr;
                    if(verbose)
                      printf("Setting address for template %u to IP %u.%u.%u.%u\n",
                             id, NIPQUAD(*(__u32*) data.address) );
                  }
              }
            else
              arg_exit(argv[1]);
            error_exit(rsbac_net_template(call, id, &data));
            exit(0);
          case NTS_set_address_family:
            if(argc > 3)
              {
                data.address_family = rsbac_get_net_family_nr(argv[3]);
                if(data.address_family == AF_MAX)
                  data.address_family = strtoul(argv[3],0,10);
              }
            else
              arg_exit(argv[1]);
            if(verbose)
              printf("Setting address_family for template %u to %u (%s)\n",
                     id,
                     data.address_family,
                     rsbac_get_net_family_name(tmp, data.address_family));
            error_exit(rsbac_net_template(call, id, &data));
            exit(0);
          case NTS_set_valid_len:
            if(argc > 3)
              data.valid_len = strtoul(argv[3],0,10);
            else
              arg_exit(argv[1]);
            if(verbose)
              printf("Setting valid_len for template %u to %u\n",
                     id, data.valid_len);
            error_exit(rsbac_net_template(call, id, &data));
            exit(0);
          case NTS_set_type:
            if(argc > 3)
              {
                data.type = rsbac_get_net_type_nr(argv[3]);
                if(data.type == RSBAC_NET_TYPE_MAX)
                  data.type = strtoul(argv[3],0,10);
              }
            else
              arg_exit(argv[1]);
            if(verbose)
              printf("Setting socket type for template %u to %u (%s)\n",
                     id,
                     data.type,
                     rsbac_get_net_type_name(tmp, data.type));
            error_exit(rsbac_net_template(call, id, &data));
            exit(0);
          case NTS_set_protocol:
            if(argc > 3)
              {
                u_int proto;
                proto = rsbac_get_net_protocol_nr(argv[3]);
                if(proto == RSBAC_NET_PROTO_MAX)
                  proto = strtoul(argv[3],0,10);
                if(proto >= RSBAC_NET_PROTO_MAX)
                  error_exit(-RSBAC_EINVALIDVALUE);
                data.protocol = proto;
              }
            else
              arg_exit(argv[1]);
            if(verbose)
              printf("Setting protocol for template %u to %u (%s)\n",
                     id,
                     data.protocol,
                     rsbac_get_net_protocol_name(tmp, data.protocol));
            error_exit(rsbac_net_template(call, id, &data));
            exit(0);
          case NTS_set_netdev:
            if(argc > 3)
              {
                strncpy(data.netdev, argv[3], RSBAC_IFNAMSIZ);
                data.netdev[RSBAC_IFNAMSIZ] = 0;
              }
            else
              arg_exit(argv[1]);
            if(verbose)
              printf("Setting netdev for template %u to %s\n",
                     id, data.netdev);
            error_exit(rsbac_net_template(call, id, &data));
            exit(0);
          case NTS_set_min_port:
            if(argc > 3)
              data.min_port = strtoul(argv[3],0,10);
            else
              arg_exit(argv[1]);
            if(verbose)
              printf("Setting min_port for template %u to %u\n",
                     id, data.min_port);
            error_exit(rsbac_net_template(call, id, &data));
            exit(0);
          case NTS_set_max_port:
            if(argc > 3)
              data.max_port = strtoul(argv[3],0,10);
            else
              arg_exit(argv[1]);
            if(verbose)
              printf("Setting max_port for template %u to %u\n",
                     id, data.max_port);
            error_exit(rsbac_net_template(call, id, &data));
            exit(0);
          case NTS_set_name:
            if(argc > 3)
              {
                strncpy(data.name, argv[3], RSBAC_NET_TEMP_NAMELEN-1);
                data.name[RSBAC_NET_TEMP_NAMELEN-1] = 0;
              }
            else
              arg_exit(argv[1]);
            if(verbose)
              printf("Setting name for template %u to %s\n",
                     id, data.name);
            error_exit(rsbac_net_template(call, id, &data));
            exit(0);
          case NTS_new_template:
            if(argc > 3)
              {
                strncpy(data.name, argv[3], RSBAC_NET_TEMP_NAMELEN-1);
                data.name[RSBAC_NET_TEMP_NAMELEN-1] = 0;
              }
            else
              strcpy(data.name, "(unknown)");
            if(verbose)
              printf("Creating template %u with name %s\n",
                     id, data.name);
            error_exit(rsbac_net_template(call, id, &data));
            exit(0);
          case NTS_copy_template:
            if(argc > 3)
              data.id = strtoul(argv[3],0,10);
            else
              arg_exit(argv[1]);
            if(verbose)
              printf("Copying template %u from template %u\n",
                     id, data.id);
            error_exit(rsbac_net_template(call, id, &data));
            exit(0);
          case NTS_delete_template:
            if(verbose)
              printf("Deleting template %u\n",
                     id);
            error_exit(rsbac_net_template(call, id, &data));
            exit(0);
          case NTS_check_id:
            if(verbose)
              printf("Checking for template %u\n",
                     id);
            error_exit(rsbac_net_template(call, id, &data));
            printf("%u\n", data.id);
            exit(0);
          case NTS_get_address:
            {
              union rsbac_net_temp_syscall_data_t data2;

              if(verbose)
                printf("Getting address of template %u\n",
                       id);
              error_exit(rsbac_net_template(call, id, &data));
              error_exit(rsbac_net_template(NTS_get_address_family, id, &data2));
              switch(data2.address_family)
                {
                  case AF_INET:
                    printf("%u.%u.%u.%u\n", NIPQUAD(*(__u32*) data.address));
                    break;
                  case AF_UNIX:
                    printf("%s\n", data.address);
                    break;
                  default:
                    printf("(address family not supported)\n");
                }
              exit(0);
            }
          case NTS_get_address_family:
            if(verbose)
              printf("Getting address_family of template %u\n",
                     id);
            error_exit(rsbac_net_template(call, id, &data));
            printf("%s\n",
                   rsbac_get_net_family_name(tmp, data.address_family));
            exit(0);
          case NTS_get_valid_len:
            if(verbose)
              printf("Getting valid_len of template %u\n",
                     id);
            error_exit(rsbac_net_template(call, id, &data));
            printf("%u\n", data.valid_len);
            exit(0);
          case NTS_get_type:
            if(verbose)
              printf("Getting socket type of template %u\n",
                     id);
            error_exit(rsbac_net_template(call, id, &data));
            printf("%s\n",
                   rsbac_get_net_type_name(tmp, data.type));
            exit(0);
          case NTS_get_protocol:
            if(verbose)
              printf("Getting protocol of template %u\n",
                     id);
            error_exit(rsbac_net_template(call, id, &data));
            printf("%s\n",
                   rsbac_get_net_protocol_name(tmp, data.protocol));
            exit(0);
          case NTS_get_netdev:
            if(verbose)
              printf("Getting netdev of template %u\n",
                     id);
            error_exit(rsbac_net_template(call, id, &data));
            printf("%s\n", data.netdev);
            exit(0);
          case NTS_get_min_port:
            if(verbose)
              printf("Getting min_port of template %u\n",
                     id);
            error_exit(rsbac_net_template(call, id, &data));
            printf("%u\n", data.min_port);
            exit(0);
          case NTS_get_max_port:
            if(verbose)
              printf("Getting max_port of template %u\n",
                     id);
            error_exit(rsbac_net_template(call, id, &data));
            printf("%u\n", data.max_port);
            exit(0);
          case NTS_get_name:
            if(verbose)
              printf("Getting name of template %u\n",
                     id);
            error_exit(rsbac_net_template(call, id, &data));
            printf("%s\n", data.name);
            exit(0);

          default:
            fprintf(stderr, "Invalid call %s!\n", argv[1]);
            exit(1);
        }
      exit(0);
    }
  else
    {
      printf(gettext("%s (RSBAC %s)\n***\n"), progname, VERSION);
      printf(gettext("Use: %s [switches] function id [set-param]\n"), progname);  
      printf("     %s [switches] list_temp_{names|nr}\n", progname);  
      printf("     %s [switches] list_template id\n", progname);  
      printf(gettext("- -v = verbose, -l = list functions\n"));
      printf(gettext("- -b = backup mode, -s = scripting mode,\n"));
      printf(gettext("- -n = take number as address, -u = take string as address,\n"));
      printf(gettext("- -d = take DNS name as address and convert to IP address,\n"));
      printf(gettext("- -a = list all templates in detail\n"));
      printf(gettext("  -V version = supply RSBAC integer version number for upgrading\n"));
    }
  exit(0);
}

