/************************************* */
/* Rule Set Based Access Control       */
/* Author and (c) 1999-2003:           */
/*   Amon Ott <ao@rsbac.org>           */
/* Syscall wrapper functions for all   */
/* admin tools                         */
/* Last modified: 05/Jun/2003          */
/************************************* */

#include <rsbac/types.h>
#include <rsbac/getname.h>
#include <rsbac/helpers.h>
#include <rsbac/error.h>
#include <rsbac/syscall_rsbac.h>
#include <rsbac/syscalls.h>

#include "nls.h"

int rsbac_stats(void)
  {
    return rsbac(RSBAC_VERSION_NR, RSYS_stats, NULL);
  }

int rsbac_check(int correct, int check_inode)
  {
    union rsbac_syscall_arg_t arg;

    arg.check.correct = correct;
    arg.check.check_inode = check_inode;
    return rsbac(RSBAC_VERSION_NR, RSYS_check, &arg);
  }

int rsbac_write(void)
  {
    return rsbac(RSBAC_VERSION_NR, RSYS_write, NULL);
  }

int rsbac_get_attr(
                   enum rsbac_switch_target_t module,
                   enum rsbac_target_t target,
                   union rsbac_target_id_t * tid,
                   enum rsbac_attribute_t attr,
                   union rsbac_attribute_value_t * value,
                   int inherit)
  { 
    union rsbac_syscall_arg_t arg;

    arg.get_attr.module = module;
    arg.get_attr.target = target;
    arg.get_attr.tid = tid;
    arg.get_attr.attr = attr;
    arg.get_attr.value = value;
    arg.get_attr.inherit = inherit;
    return rsbac(RSBAC_VERSION_NR, RSYS_get_attr, &arg);
  }

int rsbac_get_attr_n(
                   enum rsbac_switch_target_t module,
                   enum rsbac_target_t target,
                   char * t_name,
                   enum rsbac_attribute_t attr,
                   union rsbac_attribute_value_t * value,
                   int inherit)
  { 
    union rsbac_syscall_arg_t arg;

    arg.get_attr_n.module = module;
    arg.get_attr_n.target = target;
    arg.get_attr_n.t_name = t_name;
    arg.get_attr_n.attr = attr;
    arg.get_attr_n.value = value;
    arg.get_attr_n.inherit = inherit;
    return rsbac(RSBAC_VERSION_NR, RSYS_get_attr_n, &arg);
  }

int rsbac_set_attr(
                   enum rsbac_switch_target_t module,
                   enum rsbac_target_t target,
                   union rsbac_target_id_t * tid,
                   enum rsbac_attribute_t attr,
                   union rsbac_attribute_value_t * value)

  { 
    union rsbac_syscall_arg_t arg;

    arg.set_attr.module = module;
    arg.set_attr.target = target;
    arg.set_attr.tid = tid;
    arg.set_attr.attr = attr;
    arg.set_attr.value = value;
    return rsbac(RSBAC_VERSION_NR, RSYS_set_attr, &arg);
  }

int rsbac_set_attr_n(
                   enum rsbac_switch_target_t module,
                   enum rsbac_target_t target,
                   char * t_name,
                   enum rsbac_attribute_t attr,
                   union rsbac_attribute_value_t * value)

  {
    union rsbac_syscall_arg_t arg;

    arg.set_attr_n.module = module;
    arg.set_attr_n.target = target;
    arg.set_attr_n.t_name = t_name;
    arg.set_attr_n.attr = attr;
    arg.set_attr_n.value = value;
    return rsbac(RSBAC_VERSION_NR, RSYS_set_attr_n, &arg);
  }

int rsbac_remove_target(
        enum rsbac_target_t target,
        union rsbac_target_id_t * tid)
  { 
    union rsbac_syscall_arg_t arg;

    arg.remove_target.target = target;
    arg.remove_target.tid = tid;
    return rsbac(RSBAC_VERSION_NR, RSYS_remove_target, &arg);
  }

int rsbac_remove_target_n(enum rsbac_target_t target,
                            char * t_name)
  { 
    union rsbac_syscall_arg_t arg;

    arg.remove_target_n.target = target;
    arg.remove_target_n.t_name = t_name;
    return rsbac(RSBAC_VERSION_NR, RSYS_remove_target_n, &arg);
  }

int rsbac_net_list_all_netdev(rsbac_netdev_id_t * id_p, u_long maxnum)
  {
    union rsbac_syscall_arg_t arg;

    arg.net_list_all_netdev.id_p = id_p;
    arg.net_list_all_netdev.maxnum = maxnum;
    return rsbac(RSBAC_VERSION_NR, RSYS_net_list_all_netdev, &arg);
  }

int rsbac_net_template(enum rsbac_net_temp_syscall_t call,
                           rsbac_net_temp_id_t id,
                           union rsbac_net_temp_syscall_data_t * data_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.net_template.call = call;
    arg.net_template.id = id;
    arg.net_template.data_p = data_p;
    return rsbac(RSBAC_VERSION_NR, RSYS_net_template, &arg);
  }

int rsbac_net_list_all_template(rsbac_net_temp_id_t * id_p, u_long maxnum)
  {
    union rsbac_syscall_arg_t arg;

    arg.net_list_all_template.id_p = id_p;
    arg.net_list_all_template.maxnum = maxnum;
    return rsbac(RSBAC_VERSION_NR, RSYS_net_list_all_template, &arg);
  }

int rsbac_switch(enum rsbac_switch_target_t target, int value)
  {
    union rsbac_syscall_arg_t arg;

    arg.switch_module.target = target;
    arg.switch_module.value = value;
    return rsbac(RSBAC_VERSION_NR, RSYS_switch, &arg);
  }

/************** MAC ***************/

int rsbac_mac_set_curr_level(rsbac_security_level_t level,
                             rsbac_mac_category_vector_t * categories_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.mac_set_curr_level.level = level;
    arg.mac_set_curr_level.categories_p = categories_p;
    return rsbac(RSBAC_VERSION_NR, RSYS_mac_set_curr_level, &arg);
  }

int rsbac_mac_get_curr_level(rsbac_security_level_t      * level_p,
                             rsbac_mac_category_vector_t * categories_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.mac_get_curr_level.level_p = level_p;
    arg.mac_get_curr_level.categories_p = categories_p;
    return rsbac(RSBAC_VERSION_NR, RSYS_mac_get_curr_level, &arg);
  }

int rsbac_mac_get_max_level(rsbac_security_level_t      * level_p,
                            rsbac_mac_category_vector_t * categories_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.mac_get_max_level.level_p = level_p;
    arg.mac_get_max_level.categories_p = categories_p;
    return rsbac(RSBAC_VERSION_NR, RSYS_mac_get_max_level, &arg);
  }

int rsbac_mac_get_min_level(rsbac_security_level_t      * level_p,
                            rsbac_mac_category_vector_t * categories_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.mac_get_curr_level.level_p = level_p;
    arg.mac_get_curr_level.categories_p = categories_p;
    return rsbac(RSBAC_VERSION_NR, RSYS_mac_get_min_level, &arg);
  }

/************** PM ***************/

int rsbac_stats_pm(void)
  {
    return rsbac(RSBAC_VERSION_NR, RSYS_stats_pm, NULL);
  }

int rsbac_pm(enum  rsbac_pm_function_type_t    function,
                 union rsbac_pm_function_param_t * param_p,
                       rsbac_pm_tkt_id_t           ticket)
  {
    union rsbac_syscall_arg_t arg;

    arg.pm.function = function;
    arg.pm.param_p = param_p;
    arg.pm.ticket = ticket;
    return rsbac(RSBAC_VERSION_NR, RSYS_pm, &arg);
  }

int rsbac_pm_change_current_task(rsbac_pm_task_id_t task)
  {
    union rsbac_syscall_arg_t arg;

    arg.pm_change_current_task.task = task;
    return rsbac(RSBAC_VERSION_NR, RSYS_pm_change_current_task, &arg);
  }

int rsbac_pm_create_file(const char * filename,
                             int mode,
                             rsbac_pm_object_class_id_t object_class)
  {
    union rsbac_syscall_arg_t arg;

    arg.pm_create_file.filename = filename;
    arg.pm_create_file.mode = mode;
    arg.pm_create_file.object_class = object_class;
    return rsbac(RSBAC_VERSION_NR, RSYS_pm_create_file, &arg);
  }

/************** MS ***************/

int rsbac_ms_get_copied(int fd)
  {
    union rsbac_syscall_arg_t arg;

    arg.ms_get_copied.fd = fd;
    return rsbac(RSBAC_VERSION_NR, RSYS_ms_get_copied, &arg);
  }

/************** RC ***************/

int rsbac_rc_copy_role (rsbac_rc_role_id_t from_role,
                            rsbac_rc_role_id_t to_role)
  {
    union rsbac_syscall_arg_t arg;

    arg.rc_copy_role.from_role = from_role;
    arg.rc_copy_role.to_role = to_role;
    return rsbac(RSBAC_VERSION_NR, RSYS_rc_copy_role, &arg);
  }

int rsbac_rc_get_item (enum  rsbac_rc_target_t       target,
                           union rsbac_rc_target_id_t  * tid_p,
                           union rsbac_rc_target_id_t  * subtid_p,
                           enum  rsbac_rc_item_t         item,
                           union rsbac_rc_item_value_t * value_p,
                           rsbac_time_t          * ttl_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.rc_get_item.target = target;
    arg.rc_get_item.tid_p = tid_p;
    arg.rc_get_item.subtid_p = subtid_p;
    arg.rc_get_item.item = item;
    arg.rc_get_item.value_p = value_p;
    arg.rc_get_item.ttl_p = ttl_p;
    return rsbac(RSBAC_VERSION_NR, RSYS_rc_get_item, &arg);
  }

/* Setting values */
int rsbac_rc_set_item (enum  rsbac_rc_target_t       target,
                           union rsbac_rc_target_id_t  * tid_p,
                           union rsbac_rc_target_id_t  * subtid_p,
                           enum  rsbac_rc_item_t         item,
                           union rsbac_rc_item_value_t * value_p,
                           rsbac_time_t            ttl)
  {
    union rsbac_syscall_arg_t arg;

    arg.rc_set_item.target = target;
    arg.rc_set_item.tid_p = tid_p;
    arg.rc_set_item.subtid_p = subtid_p;
    arg.rc_set_item.item = item;
    arg.rc_set_item.value_p = value_p;
    arg.rc_set_item.ttl = ttl;
    return rsbac(RSBAC_VERSION_NR, RSYS_rc_set_item, &arg);
  }

int rsbac_rc_get_list(enum  rsbac_rc_target_t       target,
                          union rsbac_rc_target_id_t  * tid_p,
                          enum  rsbac_rc_item_t         item,
                          u_int maxnum,
                          __u32  * array_p,
                          rsbac_time_t * ttl_array_p)

  {
    union rsbac_syscall_arg_t arg;

    arg.rc_get_list.target = target;
    arg.rc_get_list.tid_p = tid_p;
    arg.rc_get_list.item = item;
    arg.rc_get_list.maxnum = maxnum;
    arg.rc_get_list.array_p = array_p;
    arg.rc_get_list.ttl_array_p = ttl_array_p;
    return rsbac(RSBAC_VERSION_NR, RSYS_rc_get_list, &arg);
  }

int rsbac_rc_change_role (rsbac_rc_role_id_t role)
  {
    union rsbac_syscall_arg_t arg;

    arg.rc_change_role.role = role;
    return rsbac(RSBAC_VERSION_NR, RSYS_rc_change_role, &arg);
  }

int rsbac_rc_get_eff_rights_n(enum  rsbac_target_t   target,
                                  char           * t_name,
                                  rsbac_rc_request_vector_t * request_vector_p,
                                  rsbac_time_t          * ttl_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.rc_get_eff_rights_n.target = target;
    arg.rc_get_eff_rights_n.t_name = t_name;
    arg.rc_get_eff_rights_n.request_vector_p = request_vector_p;
    arg.rc_get_eff_rights_n.ttl_p = ttl_p;
    return rsbac(RSBAC_VERSION_NR, RSYS_rc_get_eff_rights_n, &arg);
  }

/************** AUTH ***************/

/* Provide means for adding and removing of capabilities */
int rsbac_auth_add_p_cap(rsbac_pid_t pid,
                         enum rsbac_auth_cap_type_t cap_type,
                         struct rsbac_auth_cap_range_t cap_range,
                         rsbac_time_t ttl)
  {
    union rsbac_syscall_arg_t arg;

    arg.auth_add_p_cap.pid = pid;
    arg.auth_add_p_cap.cap_type = cap_type;
    arg.auth_add_p_cap.cap_range = cap_range;
    arg.auth_add_p_cap.ttl = ttl;
    return rsbac(RSBAC_VERSION_NR, RSYS_auth_add_p_cap, &arg);
  }

int rsbac_auth_remove_p_cap(rsbac_pid_t pid,
                            enum rsbac_auth_cap_type_t cap_type,
                            struct rsbac_auth_cap_range_t cap_range)
  {
    union rsbac_syscall_arg_t arg;

    arg.auth_remove_p_cap.pid = pid;
    arg.auth_remove_p_cap.cap_type = cap_type;
    arg.auth_remove_p_cap.cap_range = cap_range;
    return rsbac(RSBAC_VERSION_NR, RSYS_auth_remove_p_cap, &arg);
  }

int rsbac_auth_add_f_cap(char * filename,
                         enum rsbac_auth_cap_type_t cap_type,
                         struct rsbac_auth_cap_range_t cap_range,
                         rsbac_time_t ttl)
  {
    union rsbac_syscall_arg_t arg;

    arg.auth_add_f_cap.filename = filename;
    arg.auth_add_f_cap.cap_type = cap_type;
    arg.auth_add_f_cap.cap_range = cap_range;
    arg.auth_add_f_cap.ttl = ttl;
    return rsbac(RSBAC_VERSION_NR, RSYS_auth_add_f_cap, &arg);
  }

int rsbac_auth_remove_f_cap(char * filename,
                            enum rsbac_auth_cap_type_t cap_type,
                            struct rsbac_auth_cap_range_t cap_range)
  {
    union rsbac_syscall_arg_t arg;

    arg.auth_remove_f_cap.filename = filename;
    arg.auth_remove_f_cap.cap_type = cap_type;
    arg.auth_remove_f_cap.cap_range = cap_range;
    return rsbac(RSBAC_VERSION_NR, RSYS_auth_remove_f_cap, &arg);
  }

/* caplist must have space for maxnum cap_range entries - first and last each! */
int rsbac_auth_get_f_caplist(char * filename,
                             enum rsbac_auth_cap_type_t cap_type,
                             struct rsbac_auth_cap_range_t caplist[],
                             rsbac_time_t ttllist[],
                             u_int maxnum)
  {
    union rsbac_syscall_arg_t arg;

    arg.auth_get_f_caplist.filename = filename;
    arg.auth_get_f_caplist.cap_type = cap_type;
    arg.auth_get_f_caplist.caplist = caplist;
    arg.auth_get_f_caplist.ttllist = ttllist;
    arg.auth_get_f_caplist.maxnum = maxnum;
    return rsbac(RSBAC_VERSION_NR, RSYS_auth_get_f_caplist, &arg);
  }

/**********************************/
/************** REG ***************/

int rsbac_reg(rsbac_reg_handle_t handle,
                  void * arg)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.reg.handle = handle;
    s_arg.reg.arg = arg;
    return rsbac(RSBAC_VERSION_NR, RSYS_reg, &s_arg);
  }


/**********************************/
/************** ACL ***************/

int rsbac_acl(enum   rsbac_acl_syscall_type_t call,
                  struct rsbac_acl_syscall_arg_t * arg)
  { 
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl.call = call;
    s_arg.acl.arg = arg;
    return rsbac(RSBAC_VERSION_NR, RSYS_acl, &s_arg);
  }

int rsbac_acl_n(enum   rsbac_acl_syscall_type_t call,
                    struct rsbac_acl_syscall_n_arg_t * arg)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_n.call = call;
    s_arg.acl_n.arg = arg;
    return rsbac(RSBAC_VERSION_NR, RSYS_acl_n, &s_arg);
  }

int rsbac_acl_get_rights (struct rsbac_acl_syscall_arg_t   * arg,
                              rsbac_acl_rights_vector_t * rights_p,
                              u_int                     effective)
  { 
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_get_rights.arg = arg;
    s_arg.acl_get_rights.rights_p = rights_p;
    s_arg.acl_get_rights.effective = effective;
    return rsbac(RSBAC_VERSION_NR, RSYS_acl_get_rights, &s_arg);
  }


int rsbac_acl_get_rights_n(struct rsbac_acl_syscall_n_arg_t * arg,
                               rsbac_acl_rights_vector_t * rights_p,
                               u_int                     effective)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_get_rights_n.arg = arg;
    s_arg.acl_get_rights_n.rights_p = rights_p;
    s_arg.acl_get_rights_n.effective = effective;
    return rsbac(RSBAC_VERSION_NR, RSYS_acl_get_rights_n, &s_arg);
  }

int rsbac_acl_get_tlist (enum   rsbac_target_t              target,
                             union  rsbac_target_id_t         * tid,
                             struct rsbac_acl_entry_t           entry_array[],
                                    rsbac_time_t                ttl_array[],
                                    u_int                       maxnum)
  { 
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_get_tlist.target = target;
    s_arg.acl_get_tlist.tid = tid;
    s_arg.acl_get_tlist.entry_array = entry_array;
    s_arg.acl_get_tlist.ttl_array = ttl_array;
    s_arg.acl_get_tlist.maxnum = maxnum;
    return rsbac(RSBAC_VERSION_NR, RSYS_acl_get_tlist, &s_arg);
  }

int rsbac_acl_get_tlist_n(enum   rsbac_target_t      target,
                              char                     * t_name,
                              struct rsbac_acl_entry_t   entry_array[],
                              rsbac_time_t               ttl_array[],
                              u_int               maxnum)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_get_tlist_n.target = target;
    s_arg.acl_get_tlist_n.t_name = t_name;
    s_arg.acl_get_tlist_n.entry_array = entry_array;
    s_arg.acl_get_tlist_n.ttl_array = ttl_array;
    s_arg.acl_get_tlist_n.maxnum = maxnum;
    return rsbac(RSBAC_VERSION_NR, RSYS_acl_get_tlist_n, &s_arg);
  }

int rsbac_acl_get_mask (enum   rsbac_target_t              target,
                            union  rsbac_target_id_t         * tid,
                                   rsbac_acl_rights_vector_t * mask_p)
  { 
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_get_mask.target = target;
    s_arg.acl_get_mask.tid = tid;
    s_arg.acl_get_mask.mask_p = mask_p;
    return rsbac(RSBAC_VERSION_NR, RSYS_acl_get_mask, &s_arg);
  }

int rsbac_acl_get_mask_n(enum rsbac_target_t              target,
                             char                      * t_name,
                             rsbac_acl_rights_vector_t * mask_p)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_get_mask_n.target = target;
    s_arg.acl_get_mask_n.t_name = t_name;
    s_arg.acl_get_mask_n.mask_p = mask_p;
    return rsbac(RSBAC_VERSION_NR, RSYS_acl_get_mask_n, &s_arg);
  }

/********  ACL groups *********/

int rsbac_acl_group(enum  rsbac_acl_group_syscall_type_t call,
                        union rsbac_acl_group_syscall_arg_t * arg_p)
  { 
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_group.call = call;
    s_arg.acl_group.arg_p = arg_p;
    return rsbac(RSBAC_VERSION_NR, RSYS_acl_group, &s_arg);
  }

/**********************************/
/************** JAIL **************/

int rsbac_jail(rsbac_version_t version,
               char * path,
               rsbac_jail_ip_t ip,
               rsbac_jail_flags_t flags)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.jail.version = version;
    s_arg.jail.path = path;
    s_arg.jail.ip = ip;
    s_arg.jail.flags = flags;
    return rsbac(RSBAC_VERSION_NR, RSYS_jail, &s_arg);
  }

/************************************************* */
/*             DEBUG/LOG functions                 */
/************************************************* */

int rsbac_adf_log_switch(enum rsbac_adf_request_t request,
                         enum rsbac_target_t      target,
                         u_int value)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.adf_log_switch.request = request;
    s_arg.adf_log_switch.target = target;
    s_arg.adf_log_switch.value = value;
    return rsbac(RSBAC_VERSION_NR, RSYS_adf_log_switch, &s_arg);
  }

int rsbac_get_adf_log(enum rsbac_adf_request_t request,
                      enum rsbac_target_t      target,
                      u_int * value_p)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.get_adf_log.request = request;
    s_arg.get_adf_log.target = target;
    s_arg.get_adf_log.value_p = value_p;
    return rsbac(RSBAC_VERSION_NR, RSYS_get_adf_log, &s_arg);
  }

/*
 * Commands to rsbac_log:
 *
 * 	0 -- Close the log.  Currently a NOP.
 * 	1 -- Open the log. Currently a NOP.
 * 	2 -- Read from the log.
 * 	3 -- Read up to the last 4k of messages in the ring buffer.
 * 	4 -- Read and clear last 4k of messages in the ring buffer
 * 	5 -- Clear ring buffer.
 */
int rsbac_log(int type,
                  char * buf,
                  int len)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.log.type = type;
    s_arg.log.buf = buf;
    s_arg.log.len = len;
    return rsbac(RSBAC_VERSION_NR, RSYS_log, &s_arg);
  }

int rsbac_init(char * root_dev)
  {
    union rsbac_syscall_arg_t arg;

    arg.init.root_dev = root_dev;
    return rsbac(RSBAC_VERSION_NR, RSYS_init, &arg);
  }


void locale_init(){
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
}
