/***************************************************************************
 * RT2x00 SourceForge Project - http://rt2x00.serialmonkey.com              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 *                                                                         *
 *   Licensed under the GNU GPL                                            *
 *   Original code supplied under license from RaLink Inc, 2004.           *
 ***************************************************************************/

/***************************************************************************
 *	Module Name:	rtusb_bulk.c
 *
 *	Abstract:
 *
 *	Revision History:
 *	Who		When		What
 *	--------	----------	-------------------------------
 *	Name		Date		Modification logs
 *	Jan Lee		2005-06-01	Release
 *	RobinC		02-06-2005	rtusb_kill_urb fixes for kernels =>2.6.7
 ***************************************************************************/

#include	"rt_config.h"
#ifndef URB_ASYNC_UNLINK
#define URB_ASYNC_UNLINK	0
#endif
//typedef VOID (*STATE_MACHINE_FUNC)(VOID *Adaptor, MLME_QUEUE_ELEM *Elem);

void RTusb_fill_bulk_urb (struct urb *pUrb,
	struct usb_device *usb,
	unsigned int bulkpipe,
	void *TransferBuf,
	int BufSize,
	usb_complete_t Complete,
	void *Context)
{

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
usb_fill_bulk_urb(pUrb, usb, bulkpipe, TransferBuf, BufSize, Complete, Context);


#else
FILL_BULK_URB(pUrb, usb, bulkpipe, TransferBuf, BufSize, Complete, Context);
#endif

}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSBBulkOutDataPacket(
	IN	PRT2570ADAPTER	pAdapter,
	IN	UCHAR	Index)
{
	PTX_CONTEXT pTxContext;
	unsigned long	flags;	// For "Ndis" spin lock
	int				i, ret = 0;

	NdisAcquireSpinLock(&pAdapter->BulkOutLock);
	if (pAdapter->BulkOutPending == TRUE)
	{
		NdisReleaseSpinLock(&pAdapter->BulkOutLock);
		return;
	}
	pAdapter->BulkOutPending = TRUE;
	NdisReleaseSpinLock(&pAdapter->BulkOutLock);
	pTxContext = &(pAdapter->TxContext[Index]);

	// Increase Total transmit byte counter
	pAdapter->RalinkCounters.TransmittedByteCount +=  pTxContext->BulkOutSize;

	// Clear Data flag
	RTUSB_CLEAR_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_DATA_FRAG);
	RTUSB_CLEAR_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_DATA_NORMAL);
	if (pTxContext->InUse != TRUE)
	{
		DBGPRINT(RT_DEBUG_ERROR, "RTUSBBulkOutDataPacket failed, pTxContext->InUse != TRUE, Index %d, NextBulkOutIndex %d\n", Index, pAdapter->NextBulkOutIndex);
		NdisAcquireSpinLock(&pAdapter->BulkOutLock);
		pAdapter->BulkOutPending = FALSE;
		NdisReleaseSpinLock(&pAdapter->BulkOutLock);
		return;
	}
	else if (pAdapter->MediaState == NdisMediaStateDisconnected &&
		  !(pAdapter->PortCfg.BssType == BSS_MONITOR &&
			 pAdapter->PortCfg.MallowRFMONTx == TRUE))
	{
		//
		// Since there is no connection, we need to empty the Tx Bulk out Ring.
		//
		DBGPRINT(RT_DEBUG_ERROR, "RTUSBBulkOutDataPacket failed, since NdisMediaStateDisconnected discard NextBulkOutIndex %d, NextIndex = %d\n", pAdapter->NextBulkOutIndex, pAdapter->NextTxIndex);
		while (pTxContext->LastOne == FALSE) {
			FREE_TX_RING(pAdapter, pTxContext);
			pTxContext = &(pAdapter->TxContext[pAdapter->NextBulkOutIndex]);
		} while (pTxContext->LastOne == FALSE);
		FREE_TX_RING(pAdapter, pTxContext);

		NdisAcquireSpinLock(&pAdapter->BulkOutLock);
		pAdapter->BulkOutPending = FALSE;
		NdisReleaseSpinLock(&pAdapter->BulkOutLock);

		return;
	}
	i = Index;
	do {
		PURB	pUrb = pTxContext->pUrb;
		BOOLEAN	LastOne = pTxContext->LastOne;	// allow for sw timing - bb

		RTusb_fill_bulk_urb(pUrb,
			pAdapter->usb,
			usb_sndbulkpipe(pAdapter->usb, 1),
			pTxContext->TransferBuffer,
			pTxContext->BulkOutSize,
			RTUSBBulkOutDataPacketComplete,
			pTxContext);

		pTxContext->IRPPending = TRUE;	// s/w timing - bb
		if((ret = rtusb_submit_urb(pUrb)) == 0) {
			atomic_inc(&pAdapter->PendingTx);
		}
		else {
			DBGPRINT(RT_DEBUG_ERROR,"-  %s: Submit Tx URB failed %d\n",
					__FUNCTION__, ret);
			break;
		}
		if (LastOne == TRUE) break;

		if (++i >= TX_RING_SIZE) i = 0;
		pTxContext = &(pAdapter->TxContext[i]);
	} while (1);

	DBGPRINT(RT_DEBUG_TRACE,"<-- %s: Size=%d ret=%d\n",
			__FUNCTION__, pTxContext->BulkOutSize, ret);

	if (ret) {
		int j = pAdapter->NextTxIndex;

		pAdapter->NextTxIndex = i;
		do {
			pTxContext->InUse      = FALSE;
			pTxContext->LastOne    = FALSE;
			pTxContext->IRPPending = FALSE;
			pTxContext->BulkOutSize= 0;
			if (++i >= TX_RING_SIZE) i = 0;
			pTxContext = &(pAdapter->TxContext[i]);
		} while (i != j);
	}
	return;

} /* End RTUSBBulkOutDataPacket () */

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSBBulkOutNullFrame(
	IN	PRT2570ADAPTER	pAdapter)
{
	PTX_CONTEXT	pNullContext = &(pAdapter->NullContext);
	PURB			pUrb;
	unsigned long	flags;	// For "Ndis" spin lock
	int ret = 0;

	NdisAcquireSpinLock(&pAdapter->BulkOutLock);
	if (pAdapter->BulkOutPending == TRUE)
	{
		NdisReleaseSpinLock(&pAdapter->BulkOutLock);
		return;
	}
	pAdapter->BulkOutPending = TRUE;
	NdisReleaseSpinLock(&pAdapter->BulkOutLock);
	// Increase Total transmit byte counter
	pAdapter->RalinkCounters.TransmittedByteCount +=  pNullContext->BulkOutSize;

	DBGPRINT(RT_DEBUG_TRACE, "--->RTUSBBulkOutNullFrame \n");

	pUrb = pNullContext->pUrb;
	RTusb_fill_bulk_urb(pUrb,
		pAdapter->usb,
		usb_sndbulkpipe(pAdapter->usb, 1),
		pNullContext->TransferBuffer,
		pNullContext->BulkOutSize,
		RTUSBBulkOutNullFrameComplete,
		pNullContext);

	if((ret = rtusb_submit_urb(pUrb)) == 0)
	{
		atomic_inc(&pAdapter->PendingTx);
	}
	else {
		DBGPRINT(RT_DEBUG_ERROR,"-  %s: Submit Tx URB failed %d\n",
				__FUNCTION__, ret);
	}
	DBGPRINT(RT_DEBUG_TRACE,"<-- %s: Size=%d ret=%d\n",
			__FUNCTION__, pNullContext->BulkOutSize, ret);
	return;
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSBBulkOutMLMEPacket(
	IN	PRT2570ADAPTER	pAdapter,
	IN	UCHAR			Index)
{
	PTX_CONTEXT		pMLMEContext;
	PURB			pUrb;
	unsigned long	flags;	// For "Ndis" spin lock
	int ret = 0;

	pMLMEContext = &pAdapter->MLMEContext[Index];
	NdisAcquireSpinLock(&pAdapter->BulkOutLock);
	if (pAdapter->BulkOutPending == TRUE)
	{
		NdisReleaseSpinLock(&pAdapter->BulkOutLock);
		return;
	}
	pAdapter->BulkOutPending = TRUE;
	NdisReleaseSpinLock(&pAdapter->BulkOutLock);

	// Increase Total transmit byte counter
	pAdapter->RalinkCounters.TransmittedByteCount +=  pMLMEContext->BulkOutSize;

	// Clear MLME bulk flag
	RTUSB_CLEAR_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_MLME);

#if 0
	DBGPRINT(RT_DEBUG_INFO, "RTUSBBulkOutMLMEPacket::PrioRingFirstIndex = %d, PrioRingTxCnt = %d, PopMgmtIndex = %d, PushMgmtIndex = %d, NextMLMEIndex = %d\n",
			pAdapter->PrioRingFirstIndex,
			pAdapter->PrioRingTxCnt, pAdapter->PopMgmtIndex, pAdapter->PushMgmtIndex, pAdapter->NextMLMEIndex);
#endif

	pUrb = pMLMEContext->pUrb;

	RTusb_fill_bulk_urb(pUrb,
		pAdapter->usb,
		usb_sndbulkpipe(pAdapter->usb, 1),
		pMLMEContext->TransferBuffer,
		pMLMEContext->BulkOutSize,
		RTUSBBulkOutMLMEPacketComplete,
		pMLMEContext);

	if((ret = rtusb_submit_urb(pUrb))!=0)
	{
		pMLMEContext->InUse 		= FALSE;
		pAdapter->PrioRingTxCnt--;			// sync to PendingTx
	}
	else {
		pMLMEContext->IRPPending = TRUE;
		atomic_inc(&pAdapter->PendingTx);
	}

	DBGPRINT(RT_DEBUG_TRACE,"<-- %s: Size=%d ret=%d\n",
			__FUNCTION__, pMLMEContext->BulkOutSize, ret);
	return;
}

void RTUSBBulkOutDataPacketComplete(purbb_t pUrb)
{
	PTX_CONTEXT 	pTxContext;
	PRT2570ADAPTER	pAdapter;
	unsigned long	flags;	// For "Ndis" spin lock
	NTSTATUS		status;

	pTxContext= (PTX_CONTEXT)pUrb->context;
	pAdapter = pTxContext->pAdapter;
	status = pUrb->status;
	atomic_dec(&pAdapter->PendingTx);

	DBGPRINT(RT_DEBUG_TRACE,
			"--->%s status=%d len=%d PendingTx=%d LastOne=%d\n",
			__FUNCTION__, status, pUrb->actual_length,
			atomic_read(&pAdapter->PendingTx), pTxContext->LastOne);

	FREE_CTX_ENTRY(pAdapter, pTxContext);
	switch (status) {
		case 0:					// OK
			if (pTxContext->LastOne == TRUE) {
				pTxContext->LastOne = FALSE;
				pAdapter->Counters.GoodTransmits++;

				NdisAcquireSpinLock(&pAdapter->BulkOutLock);
				pAdapter->BulkOutPending = FALSE;
				NdisReleaseSpinLock(&pAdapter->BulkOutLock);

				// Don't worry about an empty queue, this function checks itself
				//RTUSBDeQueuePacket(pAdapter);
				//RTUSBKickBulkOut(pAdapter);
				if (!LOCAL_TX_RING_EMPTY(pAdapter)) {
					RTUSB_SET_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_DATA_NORMAL);
					//RTUSBKickBulkOut(pAdapter);
				}
			}
			RTUSBMlmeUp(pAdapter, &pAdapter->mlme_semaphore);
			return;

		case -ECONNRESET:		// async unlink via call to usb_unlink_urb()
		case -ENOENT:			// stopped by call to rtusb_kill_urb
		case -ESHUTDOWN:		// hardware gone = -108
		case -EPROTO:			// unplugged = -71
			DBGPRINT(RT_DEBUG_ERROR,"=== %s shutdown status=%d\n",
					__FUNCTION__, status);
			break;

		default:
			DBGPRINT(RT_DEBUG_ERROR,"=== %s error status=%d\n",
					__FUNCTION__, status);
			while (pTxContext->LastOne != TRUE) {
				pTxContext = &(pAdapter->TxContext[pAdapter->NextBulkOutIndex]);
				FREE_CTX_ENTRY(pAdapter, pTxContext);
				atomic_dec(&pAdapter->PendingTx);
			}
			FREE_TX_RING(pAdapter, pTxContext);

			if ((!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_RESET_IN_PROGRESS)) &&
				(!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_HALT_IN_PROGRESS)) &&
				(!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_REMOVE_IN_PROGRESS)) &&
				(!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_BULKOUT_RESET)))
			{
				DBGPRINT(RT_DEBUG_ERROR, "Bulk Out Data Packet Failed\n");
				RTMP_SET_FLAG(pAdapter, fRTMP_ADAPTER_BULKOUT_RESET);
				RTUSBEnqueueInternalCmd(pAdapter, RT_OID_USB_RESET_BULK_OUT);
			}
			break;
 	}
	NdisAcquireSpinLock(&pAdapter->BulkOutLock);
	pAdapter->BulkOutPending = FALSE;
	NdisReleaseSpinLock(&pAdapter->BulkOutLock);
}

void  RTUSBBulkOutNullFrameComplete(purbb_t pUrb)
{
	PRT2570ADAPTER	pAdapter;
	PTX_CONTEXT	pNullContext;
	unsigned long	flags;	// For "Ndis" spin lock
	NTSTATUS		status;

	pNullContext= (PTX_CONTEXT)pUrb->context;
	pAdapter = pNullContext->pAdapter;

	DBGPRINT(RT_DEBUG_TRACE, "--->RTUSBBulkOutNullFrameComplete\n");

	// Reset Null frame context flags
	pNullContext->IRPPending = FALSE;
	pNullContext->InUse = FALSE;
	// Clear Null frame bulk flag
	RTUSB_CLEAR_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_DATA_NULL);
	status = pUrb->status;
	atomic_dec(&pAdapter->PendingTx);

	DBGPRINT(RT_DEBUG_TRACE, "--->%s status=%d PendingTx=%d\n",
			__FUNCTION__, status, atomic_read(&pAdapter->PendingTx));

	NdisAcquireSpinLock(&pAdapter->BulkOutLock);
	pAdapter->BulkOutPending = FALSE;
	NdisReleaseSpinLock(&pAdapter->BulkOutLock);

	switch (status) {
		case 0:					// OK
			// Don't worry about an empty queue, this function checks itself
			//RTUSBDeQueuePacket(pAdapter);
			//RTUSBKickBulkOut(pAdapter);
			RTUSBMlmeUp(pAdapter, &pAdapter->mlme_semaphore);
			break;

		case -ECONNRESET:		// async unlink via call to usb_unlink_urb()
		case -ENOENT:			// stopped by call to rtusb_kill_urb
		case -ESHUTDOWN:		// hardware gone = -108
		case -EPROTO:			// unplugged = -71
			DBGPRINT(RT_DEBUG_ERROR,"=== %s shutdown status=%d\n",
					__FUNCTION__, status);
			break;

		default:
			if ((!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_RESET_IN_PROGRESS)) &&
				(!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_HALT_IN_PROGRESS)) &&
				(!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_REMOVE_IN_PROGRESS)) &&
				(!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_BULKOUT_RESET)))
			{
				DBGPRINT(RT_DEBUG_ERROR,"=== %s Error code = %d: issuing reset\n",
						__FUNCTION__, status);
				RTMP_SET_FLAG(pAdapter, fRTMP_ADAPTER_BULKOUT_RESET);
				RTUSBEnqueueInternalCmd(pAdapter, RT_OID_USB_RESET_BULK_OUT);
			}
			break;
	}
	DBGPRINT(RT_DEBUG_TRACE, "<---RTUSBBulkOutNullFrameComplete\n");

}

void RTUSBBulkOutMLMEPacketComplete(purbb_t pUrb)
{
	PTX_CONTEXT			pMLMEContext;
	PRT2570ADAPTER		pAdapter;
	NTSTATUS		status;
	unsigned long	flags;	// For "Ndis" spin lock

	pMLMEContext= (PTX_CONTEXT)pUrb->context;
	pAdapter = pMLMEContext->pAdapter;
	status = pUrb->status;
	atomic_dec(&pAdapter->PendingTx);

	DBGPRINT(RT_DEBUG_TRACE, "--->%s status=%d PendingTx=%d\n",
			__FUNCTION__, status, atomic_read(&pAdapter->PendingTx));

	pAdapter->PrioRingTxCnt--;			// sync to PendingTx
	pAdapter->PrioRingFirstIndex++;
	if (pAdapter->PrioRingFirstIndex >= PRIO_RING_SIZE)
	{
		pAdapter->PrioRingFirstIndex = 0;
	}

	DBGPRINT(RT_DEBUG_INFO, "- %s:: First=%d, TxCnt=%d, Pop=%d, Push=%d, Next=%d\n",
			__FUNCTION__,
			pAdapter->PrioRingFirstIndex,
			pAdapter->PrioRingTxCnt, pAdapter->PopMgmtIndex, pAdapter->PushMgmtIndex, pAdapter->NextMLMEIndex);

	// Reset MLME context flags
	pMLMEContext->IRPPending 	= FALSE;
	pMLMEContext->InUse 		= FALSE;

	NdisAcquireSpinLock(&pAdapter->BulkOutLock);
	pAdapter->BulkOutPending = FALSE;
	NdisReleaseSpinLock(&pAdapter->BulkOutLock);

	switch (status) {
		case 0:					// OK
			// Don't worry about an empty queue, this function checks itself
			//RTUSBDeQueuePacket(pAdapter);
			//RTUSBKickBulkOut(pAdapter);
			if (pAdapter->PrioRingTxCnt > 0) {
				RTUSB_SET_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_MLME);
				//RTUSBKickBulkOut(pAdapter);
			}
			RTUSBMlmeUp(pAdapter, &pAdapter->mlme_semaphore);
			break;

		case -ECONNRESET:		// async unlink via call to usb_unlink_urb()
		case -ENOENT:			// stopped by call to rtusb_kill_urb
		case -ESHUTDOWN:		// hardware gone = -108
		case -EPROTO:			// unplugged = -71
			DBGPRINT(RT_DEBUG_ERROR,"=== %s shutdown status=%d\n",
					__FUNCTION__, status);
			break;

		default:
			if ((!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_RESET_IN_PROGRESS)) &&
				(!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_HALT_IN_PROGRESS)) &&
				(!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_REMOVE_IN_PROGRESS)) &&
				(!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_BULKOUT_RESET)))
			{
				DBGPRINT(RT_DEBUG_ERROR,"=== %s Error code = %d: issuing reset\n",
						__FUNCTION__, status);
				RTMP_SET_FLAG(pAdapter, fRTMP_ADAPTER_BULKOUT_RESET);
				RTUSBEnqueueInternalCmd(pAdapter, RT_OID_USB_RESET_BULK_OUT);
			}
			break;
	}
	DBGPRINT(RT_DEBUG_TRACE, "<---RTUSBBulkOutMLMEPacketComplete\n");

}

/*
	========================================================================

	Routine Description:
		This routine starts a bulk receive operation.

	Arguments:
		pAdapter	Pointer to our adapter.
		pRxContext	Pointer to current USB rcv context

	Return Value:
		== 0: rcv successfully started.
		!= 0: rcv not successfully started.

	Note:
		Advances NextRxBulkInIndex on success.
	========================================================================
*/
static inline int requestBufferedRcv(
	IN		PRT2570ADAPTER	pAdapter,
	IN OUT	PRX_CONTEXT		pRxContext)
{
	PURB		pUrb = pRxContext->pUrb;
	int			ret;

	memset(pRxContext->TransferBuffer, 0, BUFFER_SIZE);
	RTusb_fill_bulk_urb(pUrb,
			pAdapter->usb,
			usb_rcvbulkpipe(pAdapter->usb, 1),
			pRxContext->TransferBuffer,
			BUFFER_SIZE,
			RTUSBBulkRxComplete,
			pRxContext);

	atomic_set(&pRxContext->IrpLock, IRPLOCK_CANCELABLE);	// s/w timing
	if((ret = rtusb_submit_urb(pUrb)) == 0) {
		pRxContext->InUse = TRUE;
		atomic_add(1, &pAdapter->PendingRx);
		pAdapter->NextRxBulkInIndex =
					(pAdapter->NextRxBulkInIndex + 1) % RX_RING_SIZE;
	}
	else {
		atomic_set(&pRxContext->IrpLock, IRPLOCK_COMPLETED);
	}
	DBGPRINT(RT_DEBUG_TRACE,"<-- %s: Submit Rx URB ret=%d\n",
				__FUNCTION__, ret);
	return ret;

} /* End rqstBufferedRcv () */

/*
	========================================================================

	Routine Description:
		This routine process Rx Irp and call rx complete function.

	Arguments:
		DeviceObject	Pointer to the device object for next lower
						device. DeviceObject passed in here belongs to
						the next lower driver in the stack because we
						were invoked via IoCallDriver in USB_RxPacket
						AND it is not OUR device object
	  Irp				Ptr to completed IRP
	  Context			Ptr to our Adapter object (context specified
						in IoSetCompletionRoutine

	Return Value:
		Always returns STATUS_MORE_PROCESSING_REQUIRED

	Note:
		Always returns STATUS_MORE_PROCESSING_REQUIRED
	========================================================================
*/
void RTUSBBulkRxComplete(purbb_t pUrb)
{

	PRX_CONTEXT 	pRxContext;
	PRT2570ADAPTER	pAdapter;
	NTSTATUS		status;

	pRxContext= (PRX_CONTEXT)pUrb->context;
	pAdapter = pRxContext->pAdapter;

	//
	// We have a number of cases:
	//		1) The USB read timed out and we received no data.
	//		2) The USB read timed out and we received some data.
	//		3) The USB read was successful and fully filled our irp buffer.
	//		4) The irp was cancelled.
	//		5) Some other failure from the USB device object.
	//
	status = pUrb->status;
	atomic_set(&pRxContext->IrpLock, IRPLOCK_COMPLETED);
	atomic_dec(&pAdapter->PendingRx);

	DBGPRINT(RT_DEBUG_TRACE, "--->%s status=%d len=%d PendingRx=%d\n",
			__FUNCTION__, status, pUrb->actual_length,
			atomic_read(&pAdapter->PendingRx));

	switch (status)
	{
		case 0:

			RTUSBMlmeUp(pAdapter, &pAdapter->mlme_semaphore);
			break;

		case -ECONNRESET:		// async unlink via call to usb_unlink_urb()
		case -ENOENT:			// stopped by call to rtusb_kill_urb
		case -ESHUTDOWN:		// hardware gone = -108
		case -EPROTO:			// unplugged = -71
			DBGPRINT(RT_DEBUG_ERROR,"=== %s - shutdown status=%d\n",
					__FUNCTION__, status);
		default:
			break;
	}
}

void RTUSBBulkOutPsPollComplete(purbb_t pUrb)
{
	PTX_CONTEXT	pPsPollContext;
	PRT2570ADAPTER	pAdapter;
	unsigned long	flags;	// For "Ndis" spin lock
	NTSTATUS		status;

	pPsPollContext= (PTX_CONTEXT)pUrb->context;
	pAdapter = pPsPollContext->pAdapter;
	pPsPollContext->IRPPending 	= FALSE;
	pPsPollContext->InUse 		= FALSE;
	status = pUrb->status;
	atomic_dec(&pAdapter->PendingTx);

	DBGPRINT(RT_DEBUG_TRACE, "--->%s status=%d PendingTx=%d\n",
			__FUNCTION__, status, atomic_read(&pAdapter->PendingTx));

	NdisAcquireSpinLock(&pAdapter->BulkOutLock);
	pAdapter->BulkOutPending = FALSE;
	NdisReleaseSpinLock(&pAdapter->BulkOutLock);

	switch (status) {
		case 0:					// OK
			// Don't worry about an empty queue, this function checks itself
			//RTUSBDeQueuePacket(pAdapter);
			//RTUSBKickBulkOut(pAdapter);
			RTUSBMlmeUp(pAdapter, &pAdapter->mlme_semaphore);
			break;

		case -ECONNRESET:		// async unlink via call to usb_unlink_urb()
		case -ENOENT:			// stopped by call to rtusb_kill_urb
		case -ESHUTDOWN:		// hardware gone = -108
		case -EPROTO:			// unplugged = -71
			DBGPRINT(RT_DEBUG_ERROR,"=== %s shutdown status=%d\n",
					__FUNCTION__, status);
			break;

		default:
			if ((!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_RESET_IN_PROGRESS)) &&
				(!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_HALT_IN_PROGRESS)) &&
				(!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_REMOVE_IN_PROGRESS)) &&
				(!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_BULKOUT_RESET)))
			{
				DBGPRINT(RT_DEBUG_ERROR,"=== %s Error code = %d: issuing reset\n",
						__FUNCTION__, status);
				RTMP_SET_FLAG(pAdapter, fRTMP_ADAPTER_BULKOUT_RESET);
				RTUSBEnqueueInternalCmd(pAdapter, RT_OID_USB_RESET_BULK_OUT);
			}
			break;
	}
	DBGPRINT(RT_DEBUG_TRACE, "<---RTUSBBulkOutNullFrameComplete\n");

}
void RTUSBBulkOutBeaconComplete(purbb_t pUrb)
{
	PRT2570ADAPTER	pAdapter;
	PTX_CONTEXT	pBeaconContext;
	unsigned long	flags;	// For "Ndis" spin lock

	pBeaconContext= (PTX_CONTEXT)pUrb->context;
	pAdapter = pBeaconContext->pAdapter;

	// Reset Null frame context flags
	pBeaconContext->IRPPending = FALSE;
	pBeaconContext->InUse = FALSE;
	atomic_dec(&pAdapter->PendingTx);

	NdisAcquireSpinLock(&pAdapter->BulkOutLock);
	pAdapter->BulkOutPending = FALSE;
	NdisReleaseSpinLock(&pAdapter->BulkOutLock);

	//RTUSBDeQueuePacket(pAdapter);
	//RTUSBKickBulkOut(pAdapter);
	RTUSBMlmeUp(pAdapter, &pAdapter->mlme_semaphore);
	DBGPRINT(RT_DEBUG_TRACE, "<-- %s\n", __FUNCTION__);

}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID RTUSBBulkOutBeacon(
	IN PRT2570ADAPTER	pAdapter,
	IN INT 			BeaconIndex)
{
	PTX_CONTEXT pBeaconContext;
	PURB			pUrb;
	unsigned long	flags;	// For "Ndis" spin lock
	int ret = 0;

	NdisAcquireSpinLock(&pAdapter->BulkOutLock);
	if (pAdapter->BulkOutPending == TRUE)
	{
		NdisReleaseSpinLock(&pAdapter->BulkOutLock);
		return;
	}
	pAdapter->BulkOutPending = TRUE;
	NdisReleaseSpinLock(&pAdapter->BulkOutLock);


	// Clear Beacon 0, 1 flag and set beacon 1 flag if required
	if (RTUSB_TEST_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_BEACON_0))
	{
		// Clear beacon 0 flag
		RTUSB_CLEAR_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_BEACON_0);
		// Set beacon 1 flag
		RTUSB_SET_BULK_FLAG (pAdapter, fRTUSB_BULK_OUT_BEACON_1);
	}
	else
	{
		// Clear beacon 1 flag
		RTUSB_CLEAR_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_BEACON_1);
	}

	pBeaconContext  = &pAdapter->BeaconContext[BeaconIndex];
	pUrb = pBeaconContext->pUrb;
	RTusb_fill_bulk_urb(pUrb,
		pAdapter->usb,
		usb_sndbulkpipe(pAdapter->usb, 1),
		pBeaconContext->TransferBuffer,
		pBeaconContext->BulkOutSize,
		RTUSBBulkOutBeaconComplete,
		pBeaconContext);

	if((ret = rtusb_submit_urb(pUrb))!=0)
	{
		DBGPRINT(RT_DEBUG_ERROR,"-  %s: Submit Tx URB failed %d\n",
				__FUNCTION__, ret);
		return;
	}
	else {
		atomic_inc(&pAdapter->PendingTx);
	}
	DBGPRINT(RT_DEBUG_TRACE,"<-- %s: Size=%d ret=%d\n",
			__FUNCTION__, pBeaconContext->BulkOutSize, ret);
}


/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSBBulkOutPsPoll(
	IN	PRT2570ADAPTER	pAdapter)
{
	PTX_CONTEXT	pPsPollContext = &(pAdapter->PsPollContext);
	PURB			pUrb;
	unsigned long	flags;	// For "Ndis" spin lock
	int ret = 0;

	NdisAcquireSpinLock(&pAdapter->BulkOutLock);
	if (pAdapter->BulkOutPending == TRUE)
	{
		NdisReleaseSpinLock(&pAdapter->BulkOutLock);
		return;
	}
	pAdapter->BulkOutPending = TRUE;
	NdisReleaseSpinLock(&pAdapter->BulkOutLock);

	// Clear PS-Poll bulk flag
	RTUSB_CLEAR_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_PSPOLL);

	DBGPRINT(RT_DEBUG_TRACE, "--->RTUSBBulkOutPsPollFrame \n");

	pUrb = pPsPollContext->pUrb;
	RTusb_fill_bulk_urb(pUrb,
		pAdapter->usb,
		usb_sndbulkpipe(pAdapter->usb, 1),
		pPsPollContext->TransferBuffer,
		pPsPollContext->BulkOutSize,
		RTUSBBulkOutPsPollComplete,
		pPsPollContext);

	if((ret = rtusb_submit_urb(pUrb))!=0)
	{
		DBGPRINT(RT_DEBUG_ERROR,"-  %s: Submit Tx URB failed %d\n",
				__FUNCTION__, ret);
		return;
	}
	else {
		atomic_inc(&pAdapter->PendingTx);
	}
	DBGPRINT(RT_DEBUG_TRACE,"<-- %s: Size=%d ret=%d\n",
			__FUNCTION__, pPsPollContext->BulkOutSize, ret);

	return;


}


/*
	========================================================================

	Routine Description:
	USB_RxPacket initializes a URB and uses the Rx IRP to submit it
	to USB. It checks if an Rx Descriptor is available and passes the
	the coresponding buffer to be filled. If no descriptor is available
	fails the request. When setting the completion routine we pass our
	Adapter Object as Context.

	Arguments:

	Return Value:
		TRUE			found matched tuple cache
		FALSE			no matched found

	Note:

	========================================================================
*/
VOID	RTUSBBulkReceive(
	IN	PRT2570ADAPTER	pAdapter)
{
	PRX_CONTEXT pRxContext;

	if ((RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_REMOVE_IN_PROGRESS))||
		(RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_RADIO_OFF))||
		(RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_RESET_PIPE_IN_PROGRESS)))
	{
		DBGPRINT(RT_DEBUG_INFO,"RTUSBBulkReceive:: can't start\n");
		return;
	}
	DBGPRINT(RT_DEBUG_TRACE,
		 "--> RTUSBBulkReceive, pAdapter->NextRxBulkInIndex = %d\n",
		 pAdapter->NextRxBulkInIndex);

	pRxContext = &(pAdapter->RxContext[pAdapter->NextRxBulkInIndex]);
	while (pRxContext->InUse == FALSE) {
		requestBufferedRcv(pAdapter, pRxContext);
		pRxContext = &(pAdapter->RxContext[pAdapter->NextRxBulkInIndex]);
	}
	DBGPRINT(RT_DEBUG_TRACE,"<-- %s: PendingRx=%d of %d, NextRxIndex=%d\n",
			__FUNCTION__, atomic_read(&pAdapter->PendingRx),
			RX_RING_SIZE, pAdapter->NextRxBulkInIndex);
	return;
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSBKickBulkOut(
	IN	PRT2570ADAPTER pAdapter)
{
	int	RoundOver = 0;
	do
	{
		// greedy to bulk out. protection are in BulkOut function using InUse parameter
		if (++RoundOver > 2)
			break;

		if (!(RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_REMOVE_IN_PROGRESS)) &&
			!(RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_HALT_IN_PROGRESS)) &&
			!(RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_RESET_IN_PROGRESS)) &&
			!(RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_BULKOUT_RESET)) &&
			!(RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_RADIO_OFF)))
		{
			// Start aribritrating Bulk out candidates

			// 0. Check if no flags set, we will do a dequeue from MLME and Data
			//if (pAdapter->BulkFlags == 0x0)
			//{
			//	RTUSBDequeueMLMEPacket(pAdapter);
			//	RTUSBDeQueuePacket(pAdapter);
			//}

			// 1. Data Fragment has highest priority
			if (RTUSB_TEST_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_DATA_FRAG))
			{
				RTUSBBulkOutDataPacket(pAdapter, pAdapter->NextBulkOutIndex);
			}

			// 2. PS-Poll frame is next
			else if (RTUSB_TEST_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_PSPOLL))
			{
				RTUSBBulkOutPsPoll(pAdapter);
			}

			// 3. Beacon 0, guarding beacon frame is next
			else if (RTUSB_TEST_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_BEACON_0))
			{
				RTUSBBulkOutBeacon(pAdapter, 0);
			}

			// 4. Beacon 1, beacon frame body is next
			else if (RTUSB_TEST_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_BEACON_1))
			{
				RTUSBBulkOutBeacon(pAdapter, 1);
			}

			// 5. Mlme frame is next
			else if (RTUSB_TEST_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_MLME))
			{
				RTUSBBulkOutMLMEPacket(pAdapter, pAdapter->PrioRingFirstIndex);
			}

			// 6. Data frame normal is next
			else if (RTUSB_TEST_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_DATA_NORMAL))
			{
				if ((!LOCAL_TX_RING_EMPTY(pAdapter)) &&
					((!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_BSS_SCAN_IN_PROGRESS)) &&
					(pAdapter->MediaState == NdisMediaStateConnected)))
				{
					RTUSBBulkOutDataPacket(pAdapter, pAdapter->NextBulkOutIndex);
				}
			}

			// 7. Null frame is the last
			else if (RTUSB_TEST_BULK_FLAG(pAdapter, fRTUSB_BULK_OUT_DATA_NULL))
			{
				if (!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_BSS_SCAN_IN_PROGRESS))
				{
					RTUSBBulkOutNullFrame(pAdapter);
				}
			}

			// 8. No data avaliable
			else
			{
				// Do nothing, or dequeue MLME and Data
				//RTUSBDequeueMLMEPacket(pAdapter);
				//RTUSBDeQueuePacket(pAdapter);
			}
		}
	}while(TRUE);
	DBGPRINT(RT_DEBUG_TRACE,"<---RTUSBKickBulkOut\n");
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSBCleanUpDataBulkOutQueue(
	IN	PRT2570ADAPTER	pAdapter)
{
	PTX_CONTEXT pTxContext;
	unsigned long	flags;	// For "Ndis" spin lock

	DBGPRINT(RT_DEBUG_TRACE, "--->CleanUpDataBulkOutQueue\n");

	while (!LOCAL_TX_RING_EMPTY(pAdapter))
	{
		pTxContext 					= &(pAdapter->TxContext[pAdapter->NextBulkOutIndex]);
		pTxContext->LastOne 		= FALSE;
		pTxContext->InUse 			= FALSE;
		pAdapter->NextBulkOutIndex 	= (pAdapter->NextBulkOutIndex + 1) % TX_RING_SIZE;
	}

	NdisAcquireSpinLock(&pAdapter->BulkOutLock);
	pAdapter->BulkOutPending = FALSE;
	NdisReleaseSpinLock(&pAdapter->BulkOutLock);

	DBGPRINT(RT_DEBUG_TRACE, "<---CleanUpDataBulkOutQueue\n");
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSBCleanUpMLMEBulkOutQueue(
	IN	PRT2570ADAPTER	pAdapter)
{
	unsigned long	flags;	// For "Ndis" spin lock

	DBGPRINT(RT_DEBUG_TRACE, "--->%s\n", __FUNCTION__);

	NdisAcquireSpinLock(&pAdapter->MLMEQLock);
	while (pAdapter->PrioRingTxCnt > 0)
	{
		pAdapter->MLMEContext[pAdapter->PrioRingFirstIndex].InUse = FALSE;

		pAdapter->PrioRingFirstIndex++;
		if (pAdapter->PrioRingFirstIndex >= PRIO_RING_SIZE)
		{
			pAdapter->PrioRingFirstIndex = 0;
		}

		pAdapter->PrioRingTxCnt--;
	}
	NdisReleaseSpinLock(&pAdapter->MLMEQLock);

	DBGPRINT(RT_DEBUG_TRACE, "<---%s\n", __FUNCTION__);
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSBCancelPendingIRPs(
	IN	PRT2570ADAPTER	pAdapter)
{
	RTUSBCancelPendingBulkInIRP(pAdapter);
	RTUSBCancelPendingBulkOutIRP(pAdapter);
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:
		Must be called in process context.

	========================================================================
*/
VOID	RTUSBCancelPendingBulkInIRP(
	IN	PRT2570ADAPTER	pAdapter)
{
	PRX_CONTEXT	pRxContext;
	//UINT		i;
	int			i = pAdapter->CurRxBulkInIndex;

	DBGPRINT(RT_DEBUG_TRACE, "--> %s: %d PendingRx left\n",
			__FUNCTION__, atomic_read(&pAdapter->PendingRx));
#if 0
	for ( i = 0; i < RX_RING_SIZE; i++)
	{
		pRxContext = &(pAdapter->RxContext[i]);
		if(atomic_read(&pRxContext->IrpLock) == IRPLOCK_CANCELABLE)
		{
			atomic_set(&pRxContext->IrpLock, IRPLOCK_CANCE_START);
			rtusb_kill_urb(pRxContext->pUrb);
		}
	}
#else
	// Cancel till we've caught up with newly issued recieves - bb
	do {
		pRxContext = &pAdapter->RxContext[i];
		if(atomic_read(&pRxContext->IrpLock) == IRPLOCK_CANCELABLE)
		{
			rtusb_kill_urb(pRxContext->pUrb);
		}
		if (++i >= RX_RING_SIZE) i = 0;
	} while (i != pAdapter->NextRxBulkInIndex);
#endif

	// maybe wait for cancellations to finish.
	for (i = 0; atomic_read(&pAdapter->PendingRx) > 0 && i < 25; i++) {
#if LINUX_VERSION_CODE >KERNEL_VERSION(2,6,9)

		msleep(UNLINK_TIMEOUT_MS);
#endif
	}
	pAdapter->CurRxBulkInIndex = pAdapter->NextRxBulkInIndex = 0;
	DBGPRINT(RT_DEBUG_TRACE, "<-- %s: %d PendingRx left\n",
			__FUNCTION__, atomic_read(&pAdapter->PendingRx));
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSBCancelPendingBulkOutIRP(
	IN	PRT2570ADAPTER	pAdapter)
{
	PTX_CONTEXT		pTxContext;
	PTX_CONTEXT		pMLMEContext;
	PTX_CONTEXT		pBeaconContext;
	PTX_CONTEXT		pNullContext;
	PTX_CONTEXT		pPsPollContext;
	unsigned long	flags;	// For "Ndis" spin lock
	UINT		i;

	DBGPRINT(RT_DEBUG_TRACE, "--> %s: %d PendingTx left\n",
			__FUNCTION__, atomic_read(&pAdapter->PendingTx));

	// TODO: We would like to let active writes complete.
	for ( i = 0; i < TX_RING_SIZE; i++)
	{
		pTxContext = &(pAdapter->TxContext[i]);
		if (pTxContext->IRPPending == TRUE)
		{
			rtusb_kill_urb(pTxContext->pUrb);
		}
	}

	for (i = 0; i < PRIO_RING_SIZE; i++)
	{
		pMLMEContext = &(pAdapter->MLMEContext[i]);

		if(pMLMEContext->IRPPending == TRUE)
		{

			// Get the USB_CONTEXT and cancel it's IRP; the completion routine will itself
			// remove it from the HeadPendingSendList and NULL out HeadPendingSendList
			//  when the last IRP on the list has been  cancelled; that's how we exit this loop
			//

			rtusb_kill_urb(pMLMEContext->pUrb);

			// Sleep 200 microsecs to give cancellation time to work
			//NdisMSleep(200);
		}
	}

	for (i = 0; i < BEACON_RING_SIZE; i++)
	{
		pBeaconContext = &(pAdapter->BeaconContext[i]);

		if(pBeaconContext->IRPPending == TRUE)
		{

			// Get the USB_CONTEXT and cancel it's IRP; the completion routine will itself
			// remove it from the HeadPendingSendList and NULL out HeadPendingSendList
			//  when the last IRP on the list has been  cancelled; that's how we exit this loop
			//

			rtusb_kill_urb(pBeaconContext->pUrb);

			// Sleep 200 microsecs to give cancellation time to work
			//NdisMSleep(200);
		}
	}

	pNullContext = &(pAdapter->NullContext);
	if (pNullContext->IRPPending == TRUE)
	rtusb_kill_urb(pNullContext->pUrb);

	pPsPollContext = &(pAdapter->PsPollContext);
	if (pPsPollContext->IRPPending == TRUE)
	rtusb_kill_urb(pPsPollContext->pUrb);

	NdisAcquireSpinLock(&pAdapter->BulkOutLock);
	pAdapter->BulkOutPending = FALSE;
	NdisReleaseSpinLock(&pAdapter->BulkOutLock);

	// maybe wait for cancellations to finish.
	for (i = 0; atomic_read(&pAdapter->PendingTx) > 0 && i < 25; i++) {
#if LINUX_VERSION_CODE >KERNEL_VERSION(2,6,9)

		msleep(UNLINK_TIMEOUT_MS);
#endif
	}
	DBGPRINT(RT_DEBUG_TRACE, "<-- %s: %d PendingTx left\n",
			__FUNCTION__, atomic_read(&pAdapter->PendingTx));
}

