/*
	Copyright (C) 2006 Ivo van Doorn

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	RF button support
	Laptops are quite often equiped with a RF key to enable or
	disable the radio of the wireless device attached to that key.
	This wireless device usually is an integrated wireless network device,
	infrared or bluetooth device.
	Some of these devices will disable the radio automatically when the
	RF key has been pressed, while other devices need to be polled
	for the RF key status, and leave the action to be taken up to the
	driver for that particular device.
	But in all cases the only interface that will have its radio disabled
	will be the device that has the RF key attached to it. It could however
	be desired that userspace performs this disabling of the radios in case
	more things than just disabling a single radio is desired.

	The rfkill driver will contain a list of all devices with a RF button,
	and hardware drivers need to register their hardware to the rfkill
	interface. Rfkill will then take care of everything. If the RF key
	requires polling to obtain the status this will be handled by rfkill.
	If the RF key does not require polling but sends for example interrupts,
	the hardware driver can report the change of status to rfkill, without
	having to do any other action.
	Once the status of the key has changed and the hardware does not
	automatically enable or disable the radio rfkill provides the
	interface to do this correctly.

	For each registered hardware button an input device will be created.
	If this input device has been opened by the user, rfkill will send a
	signal to userspace instead of the hardware about the new button
	status. This will allow userpace to perform the correct steps
	in order to bring down all interfaces.
 */

#ifndef RFKILL_H
#define RFKILL_H

#include <linux/device.h>

#define RFKILL_POLL_DELAY	( HZ / 10 )

enum key_type {
	KEY_TYPE_WIFI = 0,
	KEY_TYPE_BlUETOOTH = 1,
	KEY_TYPE_IRDA = 2,
	KEY_TYPE_MAX = 3,
};

/**
 * struct rfkill - rfkill button control structure.
 *
 * @dev_name: Name of the interface. This will become the name
 * 	of the input device which will be created for this button.
 * @dev: Pointer to the device structure to which this button belongs to.
 * @data: Pointer to the RF button drivers private data which will be
 * 	passed along with the radio and polling handlers.
 * @poll(void *data): Optional handler which will frequently be
 * 	called to determine the current status of the RF button.
 * @enable_radio(void *data): Optional handler to enable the radio
 * 	once the RF button has been pressed and the hardware does enable
 * 	the radio automaticly.
 * @disable_radio(void *data): Optional handler to disable the radio
 * 	once the RF button has been pressed and the hardware does disable
 * 	the radio automaticly.
 * @key_type: Radio type which the button controls, the value stored
 * 	here should be a value from enum key_type.
 * @key: Internal pointer that should not be touched by key driver.
 *
 * This structure can be used by a key driver to register the key
 * to the rfkill driver in order to take control of the reporting
 * to userspace or handling of radio status.
 */
struct rfkill {
	const char *dev_name;

	struct device *dev;

	void *data;
	int (*poll)(void *data);
	void (*enable_radio)(void *data);
	void (*disable_radio)(void *data);

	unsigned int key_type;

	struct rfkill_key *key;
};

/**
 * rfkill_register_key - Deregister a previously registered rfkill structre.
 * @rfkill: rfkill structure to be deregistered
 * @init_status: initial status of the key at the time this function is called
 *
 * This function should be called by the key driver when the rfkill structure
 * needs to be registered. Immediately from registration the key driver
 * should be able to receive calls through the poll, enable_radio and
 * disable_radio handlers if those were registered.
 */
int rfkill_register_key(struct rfkill *rfkill, int init_status);

/**
 * rfkill_deregister_key - Deregister a previously registered rfkill structre.
 * @rfkill: rfkill structure to be deregistered
 *
 * This function should be called by the key driver when the rfkill structure
 * needs to be deregistered. This function may only be called if it was
 * previously registered with rfkill_register_key.
 */
int rfkill_deregister_key(struct rfkill *rfkill);

/**
 * rfkill_report_event - Report change in key status to rfkill handler.
 * @rfkill: rfkill structure registered by key driver
 * @new_status: new key status
 *
 * This function should be called by the key driver if it has not provided
 * a poll handler with rfkill. As soon as the key driver has determined
 * the status of the key has changed it should report the new status
 * through this function.
 */
void rfkill_report_event(struct rfkill *rfkill, int new_status);

#endif /* RFKILL_H */
