/*
	Copyright (C) 2004 - 2006 rt2x00 SourceForge Project
	<http://rt2x00.serialmonkey.com>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	Module: rt2500pci
	Abstract: rt2500pci device specific routines.
	Supported chipsets: RT2560.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/dma-mapping.h>
#include <linux/delay.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/wireless.h>
#include <linux/ethtool.h>

#include <net/iw_handler.h>
#include <net/d80211.h>

#include <asm/io.h>

/*
 * Set enviroment defines for rt2x00.h
 */
#define DRV_NAME "rt2500pci"

#ifdef CONFIG_RT2500PCI_DEBUG
#define CONFIG_RT2X00_DEBUG
#endif /* CONFIG_RT2500PCI_DEBUG */

#ifdef CONFIG_RT2500PCI_BUTTON
#define CONFIG_RT2X00_BUTTON
#endif /* CONFIG_RT2500PCI_BUTTON */

#include "rt2x00.h"
#include "rt2x00pci.h"
#include "rt2500pci.h"
#include "eeprom_93cx6.h"

/*
 * Register access.
 * All access to the CSR registers will go through the methods
 * rt2x00_register_read and rt2x00_register_write.
 * BBP and RF register require indirect register access,
 * and use the CSR registers BBPCSR and RFCSR to achieve this.
 * These indirect registers work with busy bits,
 * and we will try maximal REGISTER_BUSY_COUNT times to access
 * the register while taking a REGISTER_BUSY_DELAY us delay
 * between each attampt. When the busy bit is still set at that time,
 * the access attempt is considered to have failed,
 * and we will print an error.
 */
static inline void rt2x00_register_read(
	const struct rt2x00_dev *rt2x00dev,
	const unsigned long offset, u32 *value)
{
	__le32 reg = (__force __le32)readl(rt2x00dev->csr_addr + offset);
	*value = le32_to_cpu(reg);
}

static inline void rt2x00_register_multiread(
	const struct rt2x00_dev *rt2x00dev,
	const unsigned long offset, u32 *value, const u16 length)
{
	memcpy_fromio(value, rt2x00dev->csr_addr + offset, length);
}

static inline void rt2x00_register_write(
	const struct rt2x00_dev *rt2x00dev,
	const unsigned long offset, u32 value)
{
	__le32 reg = cpu_to_le32(value);
	writel((__force u32)reg, rt2x00dev->csr_addr + offset);
}

static inline void rt2x00_register_multiwrite(
	const struct rt2x00_dev *rt2x00dev,
	const unsigned long offset, u32 *value, const u16 length)
{
	memcpy_toio(rt2x00dev->csr_addr + offset, value, length);
}

static u32 rt2x00_bbp_check(const struct rt2x00_dev *rt2x00dev)
{
	u32 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, BBPCSR, &reg);
		if (!rt2x00_get_field32(reg, BBPCSR_BUSY))
			return reg;
		udelay(REGISTER_BUSY_DELAY);
	}

	return 0xffff;
}

static void rt2x00_bbp_write(const struct rt2x00_dev *rt2x00dev,
	const u8 reg_id, const u8 value)
{
	u32 reg;

	/*
	 *  Wait untill the BBP becomes ready.
	 */
	if (rt2x00_bbp_check(rt2x00dev) == 0xffff) {
		ERROR("BBPCSR register busy. Write failed.\n");
		return;
	}

	/*
	 * Write the data into the BBP.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, BBPCSR_VALUE, value);
	rt2x00_set_field32(&reg, BBPCSR_REGNUM, reg_id);
	rt2x00_set_field32(&reg, BBPCSR_BUSY, 1);
	rt2x00_set_field32(&reg, BBPCSR_WRITE_CONTROL, 1);

	rt2x00_register_write(rt2x00dev, BBPCSR, reg);
}

static void rt2x00_bbp_read(const struct rt2x00_dev *rt2x00dev,
	const u8 reg_id, u8 *value)
{
	u32 reg;

	/*
	 *  Wait untill the BBP becomes ready.
	 */
	if (rt2x00_bbp_check(rt2x00dev) == 0xffff) {
		ERROR("BBPCSR register busy. Read failed.\n");
		return;
	}

	/*
	 * Write the request into the BBP.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, BBPCSR_REGNUM, reg_id);
	rt2x00_set_field32(&reg, BBPCSR_BUSY, 1);
	rt2x00_set_field32(&reg, BBPCSR_WRITE_CONTROL, 0);

	rt2x00_register_write(rt2x00dev, BBPCSR, reg);

	/*
	 *  Wait untill the BBP becomes ready.
	 */
	reg = rt2x00_bbp_check(rt2x00dev);
	if (reg == 0xffff)
		ERROR("BBPCSR register busy. Read failed.\n");

	*value = rt2x00_get_field32(reg, BBPCSR_VALUE);
}

static void rt2x00_rf_write(const struct rt2x00_dev *rt2x00dev,
	const u32 value)
{
	u32 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, RFCSR, &reg);
		if (!rt2x00_get_field32(reg, RFCSR_BUSY))
			goto rf_write;
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("RFCSR register busy. Write failed.\n");
	return;

rf_write:
	reg = 0;
	rt2x00_set_field32(&reg, RFCSR_VALUE, value);
	rt2x00_set_field32(&reg, RFCSR_NUMBER_OF_BITS, 20);
	rt2x00_set_field32(&reg, RFCSR_IF_SELECT, 0);
	rt2x00_set_field32(&reg, RFCSR_BUSY, 1);

	rt2x00_register_write(rt2x00dev, RFCSR, reg);
}

/*
 * EEPROM access.
 * The EEPROM is being accessed by word index.
 */
static void* rt2x00_eeprom_addr(const struct rt2x00_dev *rt2x00dev,
	const u8 word)
{
	return (void*)&rt2x00dev->eeprom[word];
}

static void rt2x00_eeprom_read(const struct rt2x00_dev *rt2x00dev,
	const u8 word, u16 *data)
{
	*data = le16_to_cpu(rt2x00dev->eeprom[word]);
}

static void rt2x00_eeprom_write(const struct rt2x00_dev *rt2x00dev,
	const u8 word, u16 data)
{
	rt2x00dev->eeprom[word] = cpu_to_le16(data);
}

/*
 * TX/RX Descriptor access functions.
 */
static inline void* rt2x00_data_addr(struct data_entry *entry)
{
	return entry->data_addr;
}

static inline struct data_desc* rt2x00_desc_addr(struct data_entry *entry)
{
	return entry->priv;
}

static inline void rt2x00_desc_read(struct data_desc *desc,
	const u8 word, u32 *value)
{
	*value = le32_to_cpu(desc->word[word]);
}

static inline void rt2x00_desc_write(struct data_desc *desc,
	const u8 word, const u32 value)
{
	desc->word[word] = cpu_to_le32(value);
}

/*
 * Ethtool handlers.
 */
static void rt2500pci_get_drvinfo(struct net_device *net_dev,
	struct ethtool_drvinfo *drvinfo)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_get_hw(net_dev)->priv;

	strcpy(drvinfo->driver, DRV_NAME);
	strcpy(drvinfo->version, DRV_VERSION);
	strcpy(drvinfo->fw_version, "N/A");
	strcpy(drvinfo->bus_info, pci_name(rt2x00dev_pci(rt2x00dev)));
	drvinfo->eedump_len = EEPROM_SIZE;
	drvinfo->regdump_len = CSR_REG_SIZE;
}

static int rt2500pci_get_regs_len(struct net_device *net_dev)
{
	return CSR_REG_SIZE;
}

static void rt2500pci_get_regs(struct net_device *net_dev,
	struct ethtool_regs *regs, void *data)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_get_hw(net_dev)->priv;

	rt2x00_register_multiread(rt2x00dev, CSR_REG_BASE, data, CSR_REG_SIZE);
	regs->version = rt2x00_rev(&rt2x00dev->chip);
	regs->len = CSR_REG_SIZE;
}

#ifdef CONFIG_RT2500PCI_DEBUG
static u32 rt2500pci_get_msglevel(struct net_device *net_dev)
{
	return rt2x00_debug_level;
}

static void rt2500pci_set_msglevel(struct net_device *net_dev, u32 msg)
{
	rt2x00_debug_level = !!msg;
}
#endif /* CONFIG_RT2500PCI_DEBUG */

static int rt2500pci_get_eeprom_len(struct net_device *net_dev)
{
	return EEPROM_SIZE;
}

static int rt2500pci_get_eeprom(struct net_device *net_dev,
	struct ethtool_eeprom *eeprom, u8 *data)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_get_hw(net_dev)->priv;

	eeprom->magic = 0xbad;

	memcpy(data,
		rt2x00_eeprom_addr(rt2x00dev, eeprom->offset / sizeof(u16)),
		eeprom->len);

	return 0;
}

static int rt2500pci_set_eeprom(struct net_device *net_dev,
	struct ethtool_eeprom *eeprom, u8 *data)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_get_hw(net_dev)->priv;
	unsigned int i;
	unsigned int words;

	if (eeprom->magic != 0xbad)
		return -EINVAL;

	words = eeprom->len / sizeof(u16);
	for (i = eeprom->offset / sizeof(u16); i < words; i++)
		rt2x00_eeprom_write(rt2x00dev, i, ((u16*)data)[i]);

	return 0;
}

static struct ethtool_ops rt2500pci_ethtool_ops = {
	.get_drvinfo	= rt2500pci_get_drvinfo,
	.get_regs_len	= rt2500pci_get_regs_len,
	.get_regs	= rt2500pci_get_regs,
#ifdef CONFIG_RT2500PCI_DEBUG
	.get_msglevel	= rt2500pci_get_msglevel,
	.set_msglevel	= rt2500pci_set_msglevel,
#endif /* CONFIG_RT2500PCI_DEBUG */
	.get_link	= ethtool_op_get_link,
	.get_eeprom_len	= rt2500pci_get_eeprom_len,
	.get_eeprom	= rt2500pci_get_eeprom,
	.set_eeprom	= rt2500pci_set_eeprom,
	.get_perm_addr	= ethtool_op_get_perm_addr,
};

#ifdef CONFIG_RT2500PCI_BUTTON
static int rt2500pci_button_poll(void *data)
{
	struct rt2x00_dev *rt2x00dev = data;
	u32 reg;

	rt2x00_register_read(rt2x00dev, GPIOCSR, &reg);
	return rt2x00_get_field32(reg, GPIOCSR_BIT0);
}

static void rt2500pci_button_enable_radio(void *data)
{
	struct rt2x00_dev* rt2x00dev = data;

	if (unlikely(!rt2x00dev))
		return;

	SET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO_HW);

	/*
	 * Only continue if we have an active interface,
	 * either monitor or non-monitor should be present.
	 */
	if (!GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
	    !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		return;

	NOTICE("Hardware button pressed, enabling radio.\n");

	rt2500pci_enable_radio(rt2x00dev);
}

static void rt2500pci_button_disable_radio(void *data)
{
	struct rt2x00_dev* rt2x00dev = data;

	if (unlikely(!rt2x00dev))
		return;

	CLEAR_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO_HW);

	/*
	 * Only continue if we have an active interface,
	 * either monitor or non-monitor should be present.
	 */
	if (!GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
	    !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		return;

	NOTICE("Hardware button pressed, disabling radio.\n");

	rt2500pci_disable_radio(rt2x00dev);
}

static void rt2500pci_button_start(struct rt2x00_dev *rt2x00dev)
{
	struct rfkill *rfkill = &rt2x00dev->rfkill;
	unsigned int key_status;

	/*
	 * Only start the button polling when
	 * the hardware button is present.
	 */
	if (!GET_FLAG(rt2x00dev, DEVICE_SUPPORT_HW_BUTTON))
		return;

	/*
	 * Enable hardware button status reading.
	 */
	rt2x00_register_write(rt2x00dev, TESTCSR, 0x000000f0);

	rfkill->dev_name	= DRV_NAME;
	rfkill->dev		= &rt2x00dev_pci(rt2x00dev)->dev;
	rfkill->data		= rt2x00dev;
	rfkill->poll		= rt2500pci_button_poll;
	rfkill->enable_radio	= rt2500pci_button_enable_radio;
	rfkill->disable_radio	= rt2500pci_button_disable_radio;
	rfkill->key_type	= KEY_TYPE_WIFI;

	key_status = !!rt2500pci_button_poll(rfkill->data);

	if (key_status)
		SET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO_HW);

	if (rfkill_register_key(rfkill, key_status))
		ERROR("Failed to register button handler.\n");
}

static void rt2500pci_button_stop(struct rt2x00_dev *rt2x00dev)
{
	if (!GET_FLAG(rt2x00dev, DEVICE_SUPPORT_HW_BUTTON))
		return;

	rfkill_deregister_key(&rt2x00dev->rfkill);
}
#else /* CONFIG_RT2500PCI_BUTTON */
static inline void rt2500pci_button_start(struct rt2x00_dev *rt2x00dev)
{
	SET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO_HW);
}

static inline void rt2500pci_button_stop(struct rt2x00_dev *rt2x00dev){}
#endif /* CONFIG_RT2500PCI_BUTTON */

/*
 * Configuration handlers.
 */
static void rt2500pci_config_bssid(struct rt2x00_dev *rt2x00dev, u8 *bssid)
{
	u32 reg[2] = { 0, 0 };

	/*
	 * The BSSID is passed to us as an array of bytes,
	 * that array is little endian, so no need for byte ordering.
	 */
	memcpy(&reg, bssid, ETH_ALEN);
	rt2x00_register_multiwrite(rt2x00dev, CSR5, &reg[0], sizeof(reg));
}

static void rt2500pci_config_promisc(struct rt2x00_dev *rt2x00dev, int promisc)
{
	u32 reg;

	rt2x00_register_read(rt2x00dev, RXCSR0, &reg);

	if (promisc) {
		rt2x00_set_field32(&reg, RXCSR0_DROP_NOT_TO_ME, 0);
		SET_FLAG(rt2x00dev, INTERFACE_ENABLED_PROMISC);
	} else {
		rt2x00_set_field32(&reg, RXCSR0_DROP_NOT_TO_ME, 1);
		CLEAR_FLAG(rt2x00dev, INTERFACE_ENABLED_PROMISC);
	}

	rt2x00_register_write(rt2x00dev, RXCSR0, reg);
}

static void rt2500pci_config_type(struct rt2x00_dev *rt2x00dev, int type)
{
	u32 reg;

	/*
	 * Only continue when there is something to be done.
	 */
	if (!(GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED) ^
	     GET_FLAG(rt2x00dev, INTERFACE_ENABLED)) &&
	    !(GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) ^
	      GET_FLAG(rt2x00dev, INTERFACE_ENABLED_MONITOR)))
		return;

	rt2x00_register_write(rt2x00dev, CSR14, 0);

	/*
	 * Apply hardware packet filter.
	 */
	rt2x00_register_read(rt2x00dev, RXCSR0, &reg);

	if (!GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
	    (type == IEEE80211_IF_TYPE_IBSS || type == IEEE80211_IF_TYPE_STA))
		rt2x00_set_field32(&reg, RXCSR0_DROP_TODS, 1);
	else
		rt2x00_set_field32(&reg, RXCSR0_DROP_TODS, 0);

	rt2x00_set_field32(&reg, RXCSR0_DROP_CRC, 1);
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR)) {
		rt2x00_set_field32(&reg, RXCSR0_DROP_PHYSICAL, 0);
		rt2x00_set_field32(&reg, RXCSR0_DROP_CONTROL, 0);
		rt2x00_set_field32(&reg, RXCSR0_DROP_VERSION_ERROR, 0);
	} else {
		rt2x00_set_field32(&reg, RXCSR0_DROP_PHYSICAL, 1);
		rt2x00_set_field32(&reg, RXCSR0_DROP_CONTROL, 1);
		rt2x00_set_field32(&reg, RXCSR0_DROP_VERSION_ERROR, 1);
	}

	rt2x00_set_field32(&reg, RXCSR0_DROP_MCAST, 0);
	rt2x00_set_field32(&reg, RXCSR0_DROP_BCAST, 0);

	rt2x00_register_write(rt2x00dev, RXCSR0, reg);

	/*
	 * Enable promisc mode when in monitor mode.
	 */
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR))
		rt2500pci_config_promisc(rt2x00dev, 1);

	/*
	 * Enable beacon config
	 */
	rt2x00_register_read(rt2x00dev, BCNCSR1, &reg);
	rt2x00_set_field32(&reg, BCNCSR1_PRELOAD,
		PREAMBLE + get_duration(IEEE80211_HEADER, 2));
	rt2x00_set_field32(&reg, BCNCSR1_BEACON_CWMIN,
		rt2x00dev->ring[RING_BEACON].tx_params.cw_min);
	rt2x00_register_write(rt2x00dev, BCNCSR1, reg);

	/*
	 * Enable synchronisation.
	 */
	rt2x00_register_read(rt2x00dev, CSR12, &reg);
	rt2x00_set_field32(&reg, CSR12_BEACON_INTERVAL, 100 * 16);
	rt2x00_set_field32(&reg, CSR12_CFPMAX_DURATION, 100 * 16);
	rt2x00_register_write(rt2x00dev, CSR12, reg);

	rt2x00_register_read(rt2x00dev, CSR14, &reg);
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED)) {
		rt2x00_set_field32(&reg, CSR14_TSF_COUNT, 1);
		rt2x00_set_field32(&reg, CSR14_TBCN, 1);
	}

	if (type == IEEE80211_IF_TYPE_IBSS) {
		rt2x00_set_field32(&reg, CSR14_TSF_SYNC, 2);
		rt2x00_set_field32(&reg, CSR14_BEACON_GEN, 1);
	} else if (type == IEEE80211_IF_TYPE_STA) {
		rt2x00_set_field32(&reg, CSR14_TSF_SYNC, 1);
		rt2x00_set_field32(&reg, CSR14_BEACON_GEN, 0);
	} else if (type == IEEE80211_IF_TYPE_AP) {
		rt2x00_set_field32(&reg, CSR14_TSF_SYNC, 0);
		rt2x00_set_field32(&reg, CSR14_BEACON_GEN, 1);
	} else if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
		   !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED)) {
		rt2x00_set_field32(&reg, CSR14_TSF_SYNC, 0);
		rt2x00_set_field32(&reg, CSR14_BEACON_GEN, 0);
	}
	rt2x00_register_write(rt2x00dev, CSR14, reg);

	/*
	 * Change flags of enabled interfaces.
	 */
	if (type != IEEE80211_IF_TYPE_MNTR) {
		if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
			SET_FLAG(rt2x00dev, INTERFACE_ENABLED);
		else
			CLEAR_FLAG(rt2x00dev, INTERFACE_ENABLED);
	} else {
		if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR))
			SET_FLAG(rt2x00dev, INTERFACE_ENABLED_MONITOR);
		else
			CLEAR_FLAG(rt2x00dev, INTERFACE_ENABLED_MONITOR);
	}
}

static void rt2500pci_config_channel(struct rt2x00_dev *rt2x00dev,
	int rf2, int channel, int freq, int txpower)
{
	u32 rf1 = rt2x00dev->rf1;
	u32 rf3 = rt2x00dev->rf3;
	u32 rf4 = rt2x00dev->rf4;

	/*
	 * Only continue when there is something to be done.
	 */
	if (channel == rt2x00dev->rx_params.channel)
		return;

	if (txpower == 0xff)
		txpower = rt2x00dev->tx_power;
	txpower = TXPOWER_TO_DEV(txpower);

	if (rt2x00_rf(&rt2x00dev->chip, RF2525E) && channel == 14)
		rf4 |= 0x00000010;

	if (rt2x00_rf(&rt2x00dev->chip, RF5222)) {
		if (channel < 14) {
			rf1 = 0x00022020;
			rf4 = 0x00000a0b;
		} else if (channel == 14) {
			rf1 = 0x00022010;
			rf4 = 0x00000a1b;
		} else if (channel < 64) {
			rf1 = 0x00022010;
			rf4 = 0x00000a1f;
		} else if (channel < 140) {
			rf1 = 0x00022010;
			rf4 = 0x00000a0f;
		} else if (channel < 161) {
			rf1 = 0x00022020;
			rf4 = 0x00000a07;
		}
	}

	/*
	 * Set TXpower.
	 */
	rt2x00_set_field32(&rf3, RF3_TXPOWER, txpower);

	INFO("Switching channel. RF1: 0x%08x, RF2: 0x%08x, RF3: 0x%08x, "
		"RF4: 0x%08x.\n", rf1, rf2, rf3, rf4);

	/*
	 * For RT2525 we should first set the channel to half band higher.
	 */
	if (rt2x00_rf(&rt2x00dev->chip, RF2525)) {
		static const u32 vals[] = {
			0x00080cbe, 0x00080d02, 0x00080d06, 0x00080d0a,
			0x00080d0e, 0x00080d12, 0x00080d16, 0x00080d1a,
			0x00080d1e, 0x00080d22, 0x00080d26, 0x00080d2a,
			0x00080d2e, 0x00080d3a
		};

		rt2x00_rf_write(rt2x00dev, rf1);
		rt2x00_rf_write(rt2x00dev, vals[channel - 1]);
		rt2x00_rf_write(rt2x00dev, rf3);
		if (rf4)
			rt2x00_rf_write(rt2x00dev, rf4);
	}

	rt2x00_rf_write(rt2x00dev, rf1);
	rt2x00_rf_write(rt2x00dev, rf2);
	rt2x00_rf_write(rt2x00dev, rf3);
	if (rf4)
		rt2x00_rf_write(rt2x00dev, rf4);

	/*
	 * Channel 14 requires the Japan filter bit to be set.
	 */
	rt2x00_bbp_write(rt2x00dev, 70, (channel == 14) ? 0x4e : 0x46);

	msleep(1);

	/*
	 * Switch off tuning bits.
	 * For RT2523 devices we do not need to update the R1 register.
	 */
	rt2x00_set_field32(&rf1, RF1_TUNER, 0);
	rt2x00_set_field32(&rf3, RF3_TUNER, 0);


	if (!rt2x00_rf(&rt2x00dev->chip, RF2523))
		rt2x00_rf_write(rt2x00dev, rf1);

	rt2x00_rf_write(rt2x00dev, rf3);

	/*
	 * Update active info for RX.
	 */
	rt2x00dev->rx_params.freq = freq;
	rt2x00dev->rx_params.channel = channel;

	/*
	 * Update rf fields
	 */
	rt2x00dev->rf1 = rf1;
	rt2x00dev->rf2 = rf2;
	rt2x00dev->rf3 = rf3;
	rt2x00dev->rf4 = rf4;

	rt2x00dev->tx_power = txpower;

	/*
	 * Clear false CRC during channel switch.
	 */
	rt2x00_register_read(rt2x00dev, CNT0, &rf1);
}

static void rt2500pci_config_txpower(struct rt2x00_dev *rt2x00dev, int txpower)
{
	txpower = TXPOWER_TO_DEV(txpower);

	/*
	 * Only continue when there is something to be done.
	 */
	if (txpower == rt2x00dev->tx_power)
		return;

	rt2x00_set_field32(&rt2x00dev->rf3, RF3_TXPOWER, txpower);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf3);

	rt2x00dev->tx_power = txpower;
}

static void rt2500pci_config_antenna(struct rt2x00_dev *rt2x00dev, int antenna)
{
	u32 reg;
	u8 reg_rx;
	u8 reg_tx;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00dev->rx_params.antenna == antenna)
		return;

	rt2x00_register_read(rt2x00dev, BBPCSR1, &reg);
	rt2x00_bbp_read(rt2x00dev, 14, &reg_rx);
	rt2x00_bbp_read(rt2x00dev, 2, &reg_tx);

	/*
	 * Clear current config antenna bits.
	 */
	reg_rx &= ~0x03;
	reg_tx &= ~0x03;

	/*
	 * Ralink devices have have antenna options for both TX as RX.
	 * The ieee80211 stack currently only provide the user to set
	 * 1 antenna, by default this is considered to be the TX antenna.
	 */
	if (antenna == 0) {
		/* Diversity. */
		reg_rx |= 0x02;
		reg_tx |= 0x02;
		rt2x00_set_field32(&reg, BBPCSR1_CCK, 2);
		rt2x00_set_field32(&reg, BBPCSR1_OFDM, 2);
	} else if (antenna == 1) {
		/* RX: Antenna B */
		reg_rx |= 0x02;
		/* TX: Antenna A */
		reg_tx |= 0x00;
		rt2x00_set_field32(&reg, BBPCSR1_CCK, 0);
		rt2x00_set_field32(&reg, BBPCSR1_OFDM, 0);
	} else if (antenna == 2) {
		/* RX: Antenna A */
		reg_rx |= 0x00;
		/* TX: Antenna B */
		reg_tx |= 0x02;
		rt2x00_set_field32(&reg, BBPCSR1_CCK, 2);
		rt2x00_set_field32(&reg, BBPCSR1_OFDM, 2);
	}

	/*
	 * RT2525E and RT5222 need to flip TX I/Q
	 */
	if (rt2x00_rf(&rt2x00dev->chip, RF2525E) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5222)) {
		reg_tx |= 0x04;
		rt2x00_set_field32(&reg, BBPCSR1_CCK_FLIP, 1);
		rt2x00_set_field32(&reg, BBPCSR1_OFDM_FLIP, 1);

		/*
		 * RT2525E does not need RX I/Q Flip.
		 */
		if (rt2x00_rf(&rt2x00dev->chip, RF2525E))
			reg_rx &= ~0x04;
	} else {
		rt2x00_set_field32(&reg, BBPCSR1_CCK_FLIP, 0);
		rt2x00_set_field32(&reg, BBPCSR1_OFDM_FLIP, 0);
	}

	rt2x00_register_write(rt2x00dev, BBPCSR1, reg);
	rt2x00_bbp_write(rt2x00dev, 14, reg_rx);
	rt2x00_bbp_write(rt2x00dev, 2, reg_tx);

	/*
	 * Update active info for RX.
	 */
	rt2x00dev->rx_params.antenna = antenna;
}

static void rt2500pci_config_duration(struct rt2x00_dev *rt2x00dev,
	int short_slot_time)
{
	u32 reg;

	rt2x00_register_read(rt2x00dev, CSR11, &reg);
	rt2x00_set_field32(&reg, CSR11_SLOT_TIME,
		short_slot_time ? SHORT_SLOT_TIME : SLOT_TIME);
	rt2x00_register_write(rt2x00dev, CSR11, reg);

	rt2x00_register_read(rt2x00dev, CSR18, &reg);
	rt2x00_set_field32(&reg, CSR18_SIFS, SIFS);
	rt2x00_set_field32(&reg, CSR18_PIFS,
		short_slot_time ? SHORT_PIFS : PIFS);
	rt2x00_register_write(rt2x00dev, CSR18, reg);

	rt2x00_register_read(rt2x00dev, CSR19, &reg);
	rt2x00_set_field32(&reg, CSR19_DIFS,
		short_slot_time ? SHORT_DIFS : DIFS);
	rt2x00_set_field32(&reg, CSR19_EIFS, EIFS);
	rt2x00_register_write(rt2x00dev, CSR19, reg);

	rt2x00_register_read(rt2x00dev, TXCSR1, &reg);
	rt2x00_set_field32(&reg, TXCSR1_TSF_OFFSET, IEEE80211_HEADER);
	rt2x00_set_field32(&reg, TXCSR1_AUTORESPONDER, 1);
	rt2x00_register_write(rt2x00dev, TXCSR1, reg);
}

static void rt2500pci_config_rate(struct rt2x00_dev *rt2x00dev, const int rate)
{
	struct ieee80211_conf *conf = &rt2x00dev->hw->conf;
	u32 reg;
	u32 value;
	u32 preamble;

	preamble = DEVICE_GET_RATE_FIELD(rate, PREAMBLE)
		? SHORT_PREAMBLE : PREAMBLE;

	reg = DEVICE_GET_RATE_FIELD(rate, RATEMASK) & DEV_BASIC_RATE;
	rt2x00_register_write(rt2x00dev, ARCSR1, reg);

	rt2x00_register_read(rt2x00dev, TXCSR1, &reg);
	value = ((conf->flags & IEEE80211_CONF_SHORT_SLOT_TIME) ?
		 SHORT_DIFS :  DIFS) +
		PLCP + preamble + get_duration(ACK_SIZE, 10);
	rt2x00_set_field32(&reg, TXCSR1_ACK_TIMEOUT, value);
	value = SIFS + PLCP + preamble + get_duration(ACK_SIZE, 10);
	rt2x00_set_field32(&reg, TXCSR1_ACK_CONSUME_TIME, value);
	rt2x00_register_write(rt2x00dev, TXCSR1, reg);

	preamble = DEVICE_GET_RATE_FIELD(rate, PREAMBLE) ? 0x08 : 0x00;

	rt2x00_register_write(rt2x00dev, ARCSR2, 0x00700400 | preamble);
	rt2x00_register_write(rt2x00dev, ARCSR3, 0x00380401 | preamble);
	rt2x00_register_write(rt2x00dev, ARCSR4, 0x00150402 | preamble);
	rt2x00_register_write(rt2x00dev, ARCSR5, 0x000b8403 | preamble);
}

static void rt2500pci_config_phymode(struct rt2x00_dev *rt2x00dev,
	const int phymode)
{
	struct ieee80211_rate *rate;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00dev->rx_params.phymode == phymode)
		return;

	if (phymode == MODE_IEEE80211A &&
	    rt2x00_rf(&rt2x00dev->chip, RF5222))
		rate = &rt2x00dev->hw->modes[2].rates[
			rt2x00dev->hw->modes[2].num_rates - 1];
	else if (phymode == MODE_IEEE80211B)
		rate = &rt2x00dev->hw->modes[1].rates[
			rt2x00dev->hw->modes[1].num_rates - 1];
	else
		rate = &rt2x00dev->hw->modes[0].rates[
			rt2x00dev->hw->modes[0].num_rates - 1];

	rt2500pci_config_rate(rt2x00dev, rate->val2);

	/*
	 * Update physical mode for rx ring.
	 */
	rt2x00dev->rx_params.phymode = phymode;
}

static void rt2500pci_config_mac_address(struct rt2x00_dev *rt2x00dev,
	void *addr)
{
	u32 reg[2] = { 0, 0 };

	/*
	 * The MAC address is passed to us as an array of bytes,
	 * that array is little endian, so no need for byte ordering.
	 */
	memcpy(&reg, addr, ETH_ALEN);
	rt2x00_register_multiwrite(rt2x00dev, CSR3, &reg[0], sizeof(reg));
}

/*
 * Link tuning
 */
static void rt2500pci_link_tuner(void *data)
{
	struct rt2x00_dev *rt2x00dev = data;
	u32 reg;
	u32 rssi;
	u8 reg_r17;

	/*
	 * Don't perform any tuning when it is disabled
	 * in the EEPROM.
	 */
	if (GET_FLAG(rt2x00dev, CONFIG_DISABLE_LINK_TUNING))
		return;

	/*
	 * Retreive link quality.
	 */
	rssi = rt2x00_get_link_rssi(&rt2x00dev->link);
	if (!rssi)
		goto exit;

	rt2x00_register_read(rt2x00dev, CSR0, &reg);
	rt2x00_bbp_read(rt2x00dev, 17, &reg_r17);

	if (reg < RT2560_VERSION_D)
		goto dynamic_cca_tune;

	if (rssi < 40) {
		if (reg_r17 >= 0x41)
			rt2x00_bbp_write(rt2x00dev, 17, reg_r17);
		goto exit;
	} else if (rssi >= 62) {
		if (reg_r17 != 0x50)
			rt2x00_bbp_write(rt2x00dev, 17, 0x50);
		goto exit;
	} else if (reg_r17 >= 0x41) {
		rt2x00_bbp_write(rt2x00dev, 17, reg_r17);
		goto exit;
	}

dynamic_cca_tune:
	rt2x00_register_read(rt2x00dev, CNT3, &reg);

	reg = rt2x00_get_field32(reg, CNT3_FALSE_CCA);

	if (reg > 512 && reg_r17 < 0x40)
		rt2x00_bbp_write(rt2x00dev, 17, ++reg_r17);
	else if (reg < 100 && reg_r17 > 0x32)
		rt2x00_bbp_write(rt2x00dev, 17, --reg_r17);

exit:
	/*
	 * Update noise statistics.
	 */
	if (reg_r17)
		rt2x00_update_link_noise(&rt2x00dev->link, reg_r17);

	queue_delayed_work(rt2x00dev->workqueue, &rt2x00dev->link.work,
		LINK_TUNE_INTERVAL);
}

/*
 * LED functions.
 */
static void rt2500pci_enable_led(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;

	rt2x00_register_read(rt2x00dev, LEDCSR, &reg);

	rt2x00_set_field32(&reg, LEDCSR_ON_PERIOD, 70);
	rt2x00_set_field32(&reg, LEDCSR_OFF_PERIOD, 30);

	if (rt2x00dev->led_mode == LED_MODE_TXRX_ACTIVITY) {
		rt2x00_set_field32(&reg, LEDCSR_LINK, 1);
		rt2x00_set_field32(&reg, LEDCSR_ACTIVITY, 0);
	} else if (rt2x00dev->led_mode == LED_MODE_ASUS) {
		rt2x00_set_field32(&reg, LEDCSR_LINK, 0);
		rt2x00_set_field32(&reg, LEDCSR_ACTIVITY, 1);
	} else {
		rt2x00_set_field32(&reg, LEDCSR_LINK, 1);
		rt2x00_set_field32(&reg, LEDCSR_ACTIVITY, 1);
	}

	rt2x00_register_write(rt2x00dev, LEDCSR, reg);
}

static void rt2500pci_disable_led(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;

	rt2x00_register_read(rt2x00dev, LEDCSR, &reg);
	rt2x00_set_field32(&reg, LEDCSR_LINK, 0);
	rt2x00_set_field32(&reg, LEDCSR_ACTIVITY, 0);
	rt2x00_register_write(rt2x00dev, LEDCSR, reg);
}

static void rt2500pci_activity_led(struct rt2x00_dev *rt2x00dev, char activity)
{
	u32 reg;

	if (rt2x00dev->led_mode != LED_MODE_TXRX_ACTIVITY)
		return;

	rt2x00_register_read(rt2x00dev, LEDCSR, &reg);
	rt2x00_set_field32(&reg, LEDCSR_ACTIVITY, activity);
	rt2x00_register_write(rt2x00dev, LEDCSR, reg);
}

/*
 * Device state switch.
 * This will put the device to sleep, or awake it.
 */
static int rt2500pci_set_state(struct rt2x00_dev *rt2x00dev,
	enum dev_state state)
{
	u32 reg;
	unsigned int i;
	char put_to_sleep;
	char bbp_state;
	char rf_state;

	put_to_sleep = (state != STATE_AWAKE);

	rt2x00_register_read(rt2x00dev, PWRCSR1, &reg);
	rt2x00_set_field32(&reg, PWRCSR1_SET_STATE, 1);
	rt2x00_set_field32(&reg, PWRCSR1_BBP_DESIRE_STATE, state);
	rt2x00_set_field32(&reg, PWRCSR1_RF_DESIRE_STATE, state);
	rt2x00_set_field32(&reg, PWRCSR1_PUT_TO_SLEEP, put_to_sleep);
	rt2x00_register_write(rt2x00dev, PWRCSR1, reg);

	/*
	 * Device is not guarenteed to be in the requested state yet.
	 * We must wait untill the register indicates that the
	 * device has entered the correct state.
	 */
	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, PWRCSR1, &reg);
		bbp_state = rt2x00_get_field32(reg, PWRCSR1_BBP_CURR_STATE);
		rf_state = rt2x00_get_field32(reg, PWRCSR1_RF_CURR_STATE);
		if (bbp_state == state && rf_state == state)
			return 0;
		msleep(10);
	}

	NOTICE("Device failed to enter state %d, "
		"current device state: bbp %d and rf %d.\n",
		state, bbp_state, rf_state);

	return -EBUSY;
}

/*
 * Initialization functions.
 */
static int rt2500pci_alloc_dma_ring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type, void (*handler)(void *),
	const u16 max_entries, const u16 data_size, const u16 desc_size)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	unsigned int i;

	/*
	 * Initialize work structure for deferred work.
	 */
	INIT_WORK(&ring->irq_work, handler, ring);

	ring->stats.limit = max_entries;
	ring->data_size = data_size;
	ring->desc_size = desc_size;

	/*
	 * Allocate all ring entries.
	 */
	ring->entry = kzalloc(ring->stats.limit * sizeof(struct data_entry),
		GFP_KERNEL);
	if (!ring->entry)
		return -ENOMEM;

	/*
	 * Allocate DMA memory for descriptor and buffer.
	 */
	ring->data_addr = pci_alloc_consistent(rt2x00dev_pci(rt2x00dev),
		rt2x00_get_ring_size(ring), &ring->data_dma);
	if (!ring->data_addr) {
		kfree(ring->entry);
		return -ENOMEM;
	}

	/*
	 * Initialize all ring entries to contain valid
	 * addresses.
	 */
	for (i = 0; i < ring->stats.limit; i++) {
		ring->entry[i].flags = 0;
		ring->entry[i].ring = ring;
		ring->entry[i].skb = NULL;
		ring->entry[i].priv = ring->data_addr
			+ (i * ring->desc_size);
		ring->entry[i].data_addr = ring->data_addr
			+ (ring->stats.limit * ring->desc_size)
			+ (i * ring->data_size);
		ring->entry[i].data_dma = ring->data_dma
			+ (ring->stats.limit * ring->desc_size)
			+ (i * ring->data_size);
	}

	return 0;
}

static void rt2500pci_free_ring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	if (ring->data_addr)
		pci_free_consistent(rt2x00dev_pci(rt2x00dev),
			rt2x00_get_ring_size(ring),
			ring->data_addr, ring->data_dma);
	ring->data_addr = NULL;

	kfree(ring->entry);
	ring->entry = NULL;
}

static int rt2500pci_allocate_dma_rings(struct rt2x00_dev *rt2x00dev)
{
	if (rt2500pci_alloc_dma_ring(rt2x00dev, RING_RX,
		rt2500pci_rxdone, RX_ENTRIES, DATA_FRAME_SIZE,
		RXD_DESC_SIZE) ||
	    rt2500pci_alloc_dma_ring(rt2x00dev, RING_TX,
		rt2500pci_txdone, TX_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt2500pci_alloc_dma_ring(rt2x00dev, RING_ATIM,
		rt2500pci_txdone, ATIM_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt2500pci_alloc_dma_ring(rt2x00dev, RING_PRIO,
		rt2500pci_txdone, TX_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt2500pci_alloc_dma_ring(rt2x00dev, RING_BEACON,
		rt2500pci_beacondone, BEACON_ENTRIES, MGMT_FRAME_SIZE,
		TXD_DESC_SIZE)) {
		return -ENOMEM;
	}

	return 0;
}

static void rt2500pci_free_rings(struct rt2x00_dev *rt2x00dev)
{
	rt2500pci_free_ring(rt2x00dev, RING_RX);
	rt2500pci_free_ring(rt2x00dev, RING_TX);
	rt2500pci_free_ring(rt2x00dev, RING_ATIM);
	rt2500pci_free_ring(rt2x00dev, RING_PRIO);
	rt2500pci_free_ring(rt2x00dev, RING_BEACON);
}

static void rt2500pci_init_rxring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	struct data_desc *rxd;
	unsigned int i;
	u32 word;

	memset(ring->data_addr, 0x00, rt2x00_get_ring_size(ring));

	ring->type = ring_type;

	for (i = 0; i < ring->stats.limit; i++) {
		rxd = rt2x00_desc_addr(&ring->entry[i]);

		rt2x00_desc_read(rxd, 1, &word);
		rt2x00_set_field32(&word, RXD_W1_BUFFER_ADDRESS,
			ring->entry[i].data_dma);
		rt2x00_desc_write(rxd, 1, word);

		rt2x00_desc_read(rxd, 0, &word);
		rt2x00_set_field32(&word, RXD_W0_OWNER_NIC, 1);
		rt2x00_desc_write(rxd, 0, word);
	}

	rt2x00_ring_index_clear(ring);
}

static void rt2500pci_init_txring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	struct data_desc *txd;
	unsigned int i;
	u32 word;

	memset(ring->data_addr, 0x00, rt2x00_get_ring_size(ring));

	ring->type = ring_type;

	for (i = 0; i < ring->stats.limit; i++) {
		txd = rt2x00_desc_addr(&ring->entry[i]);

		rt2x00_desc_read(txd, 1, &word);
		rt2x00_set_field32(&word, TXD_W1_BUFFER_ADDRESS,
			ring->entry[i].data_dma);
		rt2x00_desc_write(txd, 1, word);

		rt2x00_desc_read(txd, 0, &word);
		rt2x00_set_field32(&word, TXD_W0_VALID, 0);
		rt2x00_set_field32(&word, TXD_W0_OWNER_NIC, 0);
		rt2x00_desc_write(txd, 0, word);
	}

	rt2x00_ring_index_clear(ring);
}

static int rt2500pci_init_rings(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;

	/*
	 * Initialize rings.
	 */
	rt2500pci_init_rxring(rt2x00dev, RING_RX);
	rt2500pci_init_txring(rt2x00dev, RING_TX);
	rt2500pci_init_txring(rt2x00dev, RING_ATIM);
	rt2500pci_init_txring(rt2x00dev, RING_PRIO);
	rt2500pci_init_txring(rt2x00dev, RING_BEACON);

	/*
	 * Initialize registers.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, TXCSR2_TXD_SIZE,
		rt2x00dev->ring[RING_TX].desc_size);
	rt2x00_set_field32(&reg, TXCSR2_NUM_TXD,
		rt2x00dev->ring[RING_TX].stats.limit);
	rt2x00_set_field32(&reg, TXCSR2_NUM_ATIM,
		rt2x00dev->ring[RING_ATIM].stats.limit);
	rt2x00_set_field32(&reg, TXCSR2_NUM_PRIO,
		rt2x00dev->ring[RING_PRIO].stats.limit);
	rt2x00_register_write(rt2x00dev, TXCSR2, reg);

	reg = 0;
	rt2x00_set_field32(&reg, TXCSR3_TX_RING_REGISTER,
		rt2x00dev->ring[RING_TX].data_dma);
	rt2x00_register_write(rt2x00dev, TXCSR3, reg);

	reg = 0;
	rt2x00_set_field32(&reg, TXCSR5_PRIO_RING_REGISTER,
		rt2x00dev->ring[RING_PRIO].data_dma);
	rt2x00_register_write(rt2x00dev, TXCSR5, reg);

	reg = 0;
	rt2x00_set_field32(&reg, TXCSR4_ATIM_RING_REGISTER,
		rt2x00dev->ring[RING_ATIM].data_dma);
	rt2x00_register_write(rt2x00dev, TXCSR4, reg);

	reg = 0;
	rt2x00_set_field32(&reg, TXCSR6_BEACON_RING_REGISTER,
		rt2x00dev->ring[RING_BEACON].data_dma);
	rt2x00_register_write(rt2x00dev, TXCSR6, reg);

	reg = 0;
	rt2x00_set_field32(&reg, RXCSR1_RXD_SIZE,
		rt2x00dev->ring[RING_RX].desc_size);
	rt2x00_set_field32(&reg, RXCSR1_NUM_RXD,
		rt2x00dev->ring[RING_RX].stats.limit);
	rt2x00_register_write(rt2x00dev, RXCSR1, reg);

	reg = 0;
	rt2x00_set_field32(&reg, RXCSR2_RX_RING_REGISTER,
		rt2x00dev->ring[RING_RX].data_dma);
	rt2x00_register_write(rt2x00dev, RXCSR2, reg);

	return 0;
}

static int rt2500pci_init_registers(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;

	if (rt2500pci_set_state(rt2x00dev, STATE_AWAKE))
		return -EBUSY;

	rt2x00_register_write(rt2x00dev, PWRCSR0, 0x3f3b3100);
	rt2x00_register_write(rt2x00dev, PCICSR, 0x000003b8);

	rt2x00_register_write(rt2x00dev, PSCSR0, 0x00020002);
	rt2x00_register_write(rt2x00dev, PSCSR1, 0x00000002);
	rt2x00_register_write(rt2x00dev, PSCSR2, 0x00020002);
	rt2x00_register_write(rt2x00dev, PSCSR3, 0x00000002);

	rt2x00_register_read(rt2x00dev, TIMECSR, &reg);
	rt2x00_set_field32(&reg, TIMECSR_US_COUNT, 33);
	rt2x00_set_field32(&reg, TIMECSR_US_64_COUNT, 63);
	rt2x00_set_field32(&reg, TIMECSR_BEACON_EXPECT, 0);
	rt2x00_register_write(rt2x00dev, TIMECSR, reg);

	rt2x00_register_read(rt2x00dev, CSR9, &reg);
	rt2x00_set_field32(&reg, CSR9_MAX_FRAME_UNIT,
		(rt2x00dev->ring[RING_RX].data_size / 128));
	rt2x00_register_write(rt2x00dev, CSR9, reg);

	rt2x00_register_write(rt2x00dev, CNT3, 0);

	rt2x00_register_write(rt2x00dev, GPIOCSR, 0x0000ff00);
	rt2x00_register_write(rt2x00dev, TESTCSR, 0x000000f0);

	rt2x00_register_write(rt2x00dev, MACCSR0, 0x00213223);
	rt2x00_register_write(rt2x00dev, MACCSR1, 0x00235518);

	rt2x00_register_read(rt2x00dev, MACCSR2, &reg);
	rt2x00_set_field32(&reg, MACCSR2_DELAY, 64);
	rt2x00_register_write(rt2x00dev, MACCSR2, reg);

	/*
	 * Always use CWmin and CWmax set in descriptor.
	 */
	rt2x00_register_read(rt2x00dev, CSR11, &reg);
	rt2x00_set_field32(&reg, CSR11_CW_SELECT, 0);
	rt2x00_register_write(rt2x00dev, CSR11, reg);

	rt2x00_register_read(rt2x00dev, RXCSR3, &reg);
	/*
	 * Signal.
	 */
	rt2x00_set_field32(&reg, RXCSR3_BBP_ID0, 47);
	rt2x00_set_field32(&reg, RXCSR3_BBP_ID0_VALID, 1);
	/*
	 * Rssi.
	 */
	rt2x00_set_field32(&reg, RXCSR3_BBP_ID1, 51);
	rt2x00_set_field32(&reg, RXCSR3_BBP_ID1_VALID, 1);
	/*
	 * OFDM Rate.
	 */
	rt2x00_set_field32(&reg, RXCSR3_BBP_ID2, 42);
	rt2x00_set_field32(&reg, RXCSR3_BBP_ID2_VALID, 1);
	/*
	 * OFDM.
	 */
	rt2x00_set_field32(&reg, RXCSR3_BBP_ID3, 51);
	rt2x00_set_field32(&reg, RXCSR3_BBP_ID3_VALID, 1);
	rt2x00_register_write(rt2x00dev, RXCSR3, reg);

	rt2x00_register_read(rt2x00dev, RALINKCSR, &reg);
	rt2x00_set_field32(&reg, RALINKCSR_AR_BBP_DATA0, 17);
	rt2x00_set_field32(&reg, RALINKCSR_AR_BBP_ID0, 26);
	rt2x00_set_field32(&reg, RALINKCSR_AR_BBP_VALID0, 1);
	rt2x00_set_field32(&reg, RALINKCSR_AR_BBP_DATA1, 0);
	rt2x00_set_field32(&reg, RALINKCSR_AR_BBP_ID1, 26);
	rt2x00_set_field32(&reg, RALINKCSR_AR_BBP_VALID1, 1);
	rt2x00_register_write(rt2x00dev, RALINKCSR, reg);

	rt2x00_register_write(rt2x00dev, BBPCSR1, 0x82188200);

	rt2x00_register_write(rt2x00dev, TXACKCSR0, 0x00000020);

	rt2x00_register_write(rt2x00dev, ARTCSR0, 0x7038140a);
	rt2x00_register_write(rt2x00dev, ARTCSR1, 0x1d21252d);
	rt2x00_register_write(rt2x00dev, ARTCSR2, 0x1919191d);

	rt2x00_register_read(rt2x00dev, CSR1, &reg);
	rt2x00_set_field32(&reg, CSR1_SOFT_RESET, 1);
	rt2x00_set_field32(&reg, CSR1_BBP_RESET, 0);
	rt2x00_set_field32(&reg, CSR1_HOST_READY, 0);
	rt2x00_register_write(rt2x00dev, CSR1, reg);

	rt2x00_register_read(rt2x00dev, CSR1, &reg);
	rt2x00_set_field32(&reg, CSR1_SOFT_RESET, 0);
	rt2x00_set_field32(&reg, CSR1_HOST_READY, 1);
	rt2x00_register_write(rt2x00dev, CSR1, reg);

	/*
	 * We must clear the FCS and FIFO error count.
	 * These registers are cleared on read,
	 * so we may pass a useless variable to store the value.
	 */
	rt2x00_register_read(rt2x00dev, CNT0, &reg);
	rt2x00_register_read(rt2x00dev, CNT4, &reg);

	return 0;
}

static int rt2500pci_init_bbp(struct rt2x00_dev *rt2x00dev)
{
	unsigned int i;
	u16 eeprom;
	u8 reg_id;
	u8 value;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_bbp_read(rt2x00dev, 0, &value);
		if ((value != 0xff) && (value != 0x00))
			goto continue_csr_init;
		NOTICE("Waiting for BBP register.\n");
	}

	ERROR("BBP register access failed, aborting.\n");
	return -EACCES;

continue_csr_init:
	rt2x00_bbp_write(rt2x00dev, 3, 0x02);
	rt2x00_bbp_write(rt2x00dev, 4, 0x19);
	rt2x00_bbp_write(rt2x00dev, 14, 0x1c);
	rt2x00_bbp_write(rt2x00dev, 15, 0x30);
	rt2x00_bbp_write(rt2x00dev, 16, 0xac);
	rt2x00_bbp_write(rt2x00dev, 17, 0x48);
	rt2x00_bbp_write(rt2x00dev, 18, 0x18);
	rt2x00_bbp_write(rt2x00dev, 19, 0xff);
	rt2x00_bbp_write(rt2x00dev, 20, 0x1e);
	rt2x00_bbp_write(rt2x00dev, 21, 0x08);
	rt2x00_bbp_write(rt2x00dev, 22, 0x08);
	rt2x00_bbp_write(rt2x00dev, 23, 0x08);
	rt2x00_bbp_write(rt2x00dev, 24, 0x70);
	rt2x00_bbp_write(rt2x00dev, 25, 0x40);
	rt2x00_bbp_write(rt2x00dev, 26, 0x08);
	rt2x00_bbp_write(rt2x00dev, 27, 0x23);
	rt2x00_bbp_write(rt2x00dev, 30, 0x10);
	rt2x00_bbp_write(rt2x00dev, 31, 0x2b);
	rt2x00_bbp_write(rt2x00dev, 32, 0xb9);
	rt2x00_bbp_write(rt2x00dev, 34, 0x12);
	rt2x00_bbp_write(rt2x00dev, 35, 0x50);
	rt2x00_bbp_write(rt2x00dev, 39, 0xc4);
	rt2x00_bbp_write(rt2x00dev, 40, 0x02);
	rt2x00_bbp_write(rt2x00dev, 41, 0x60);
	rt2x00_bbp_write(rt2x00dev, 53, 0x10);
	rt2x00_bbp_write(rt2x00dev, 54, 0x18);
	rt2x00_bbp_write(rt2x00dev, 56, 0x08);
	rt2x00_bbp_write(rt2x00dev, 57, 0x10);
	rt2x00_bbp_write(rt2x00dev, 58, 0x08);
	rt2x00_bbp_write(rt2x00dev, 61, 0x6d);
	rt2x00_bbp_write(rt2x00dev, 62, 0x10);

	DEBUG("Start initialization from EEPROM...\n");
	for (i = 0; i < EEPROM_BBP_SIZE; i++) {
		rt2x00_eeprom_read(rt2x00dev, EEPROM_BBP_START + i, &eeprom);

		if (eeprom != 0xffff && eeprom != 0x0000) {
			reg_id = rt2x00_get_field16(eeprom, EEPROM_BBP_REG_ID);
			value = rt2x00_get_field16(eeprom, EEPROM_BBP_VALUE);
			DEBUG("BBP: 0x%02x, value: 0x%02x.\n", reg_id, value);
			rt2x00_bbp_write(rt2x00dev, reg_id, value);
		}
	}
	DEBUG("...End initialization from EEPROM.\n");

	return 0;
}

static int rt2500pci_init_channel_time(struct rt2x00_dev *rt2x00dev)
{
	unsigned long jiffies_start;
	unsigned long jiffies_end;

	/*
	 * Only initialize the channel_change_time
	 * if it has not been set previously.
	 */
	if (rt2x00dev->hw->channel_change_time)
		return 0;

	/*
	 * Invalidate the rx_params.channel value to make sure
	 * the config channel will be correctly executed.
	 */
	rt2x00dev->rx_params.channel = 0;

	/*
	 * Determine channel_change_time
	 * by measuring the time it takes
	 * to switch the channel.
	 */
	jiffies_start = jiffies;
	rt2500pci_config_channel(rt2x00dev,
		rt2x00dev->hw->modes[0].channels[0].val,
		rt2x00dev->hw->modes[0].channels[0].chan,
		rt2x00dev->hw->modes[0].channels[0].freq,
		rt2x00dev->hw->modes[0].channels[0].power_level);
	jiffies_end = jiffies;

	rt2x00dev->hw->channel_change_time =
		jiffies_to_usecs((long)jiffies_end - (long)jiffies_start);

	NOTICE("Channel change time has been set to %d.\n",
		rt2x00dev->hw->channel_change_time);

	return 0;
}

/*
 * Device initialization functions.
 */
static int rt2500pci_initialize(struct rt2x00_dev *rt2x00dev)
{
	if (GET_FLAG(rt2x00dev, DEVICE_INITIALIZED))
		return 0;

	/*
	 * Allocate all data rings.
	 */
	if (rt2500pci_allocate_dma_rings(rt2x00dev)) {
		ERROR("DMA allocation failed.\n");
		goto exit_fail;
	}

	/*
	 * Reset the channel_change_time value
	 * to make sure it will be correctly initialized
	 * after the radio has been enabled.
	 */
	rt2x00dev->hw->channel_change_time = 0;

	/*
	 * Register interrupt handler.
	 */
	if (request_irq(rt2x00dev_pci(rt2x00dev)->irq, rt2500pci_interrupt,
		IRQF_SHARED, DRV_NAME, rt2x00dev)) {
		ERROR("IRQ %d allocation failed.\n",
			rt2x00dev_pci(rt2x00dev)->irq);
		goto exit_fail;
	}

	SET_FLAG(rt2x00dev, DEVICE_INITIALIZED);

	return 0;

exit_fail:
	rt2500pci_free_rings(rt2x00dev);

	return -EIO;
}

static void rt2500pci_uninitialize(struct rt2x00_dev *rt2x00dev)
{
	if (!GET_FLAG(rt2x00dev, DEVICE_INITIALIZED))
		return;

	/*
	 * Cancel scanning.
	 */
	if (rt2x00dev->scan)
		rt2x00_signal_scan(rt2x00dev->scan, SCANNING_CANCELLED);

	/*
	 * Flush out all pending work.
	 */
	flush_workqueue(rt2x00dev->workqueue);

	/*
	 * Free DMA rings.
	 */
	rt2500pci_free_rings(rt2x00dev);

	/*
	 * Free irq line.
	 */
	free_irq(rt2x00dev_pci(rt2x00dev)->irq, rt2x00dev);

	CLEAR_FLAG(rt2x00dev, DEVICE_INITIALIZED);
}

/*
 * Radio control functions.
 */
static int rt2500pci_enable_radio(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;

	/*
	 * Don't enable the radio twice,
	 * or if the hardware button has been disabled.
	 */
	if (GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO) ||
	    !GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO_HW))
		return 0;

	/*
	 * Initialize all registers.
	 */
	if (rt2500pci_init_rings(rt2x00dev) ||
	    rt2500pci_init_registers(rt2x00dev) ||
	    rt2500pci_init_bbp(rt2x00dev)) {
		ERROR("Register initialization failed.\n");
		goto exit_fail;
	}

	/*
	 * Determine channel change time.
	 */
	if (rt2500pci_init_channel_time(rt2x00dev))
		goto exit_fail;

	/*
	 * Clear interrupts.
	 */
	rt2x00_register_read(rt2x00dev, CSR7, &reg);
	rt2x00_register_write(rt2x00dev, CSR7, reg);

	SET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO);

	/*
	 * Enable interrupts.
	 */
	rt2x00_register_read(rt2x00dev, CSR8, &reg);
	rt2x00_set_field32(&reg, CSR8_TBCN_EXPIRE, 0);
	rt2x00_set_field32(&reg, CSR8_TXDONE_TXRING, 0);
	rt2x00_set_field32(&reg, CSR8_TXDONE_ATIMRING, 0);
	rt2x00_set_field32(&reg, CSR8_TXDONE_PRIORING, 0);
	rt2x00_set_field32(&reg, CSR8_RXDONE, 0);
	rt2x00_register_write(rt2x00dev, CSR8, reg);

	/*
	 * Enable RX.
	 */
	rt2x00_register_read(rt2x00dev, RXCSR0, &reg);
	rt2x00_set_field32(&reg, RXCSR0_DISABLE_RX, 0);
	rt2x00_register_write(rt2x00dev, RXCSR0, reg);

	/*
	 * Enable LED
	 */
	rt2500pci_enable_led(rt2x00dev);

	ieee80211_start_queues(rt2x00dev->hw);
	ieee80211_netif_oper(rt2x00dev->hw, NETIF_WAKE);

	return 0;

exit_fail:
	rt2500pci_uninitialize(rt2x00dev);
	return -EIO;
}

static void rt2500pci_disable_radio(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;

	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return;

	ieee80211_netif_oper(rt2x00dev->hw, NETIF_STOP);
	ieee80211_stop_queues(rt2x00dev->hw);

	/*
	 * Disable LED
	 */
	rt2500pci_disable_led(rt2x00dev);

	CLEAR_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO);

	rt2x00_register_write(rt2x00dev, PWRCSR0, 0);

	/*
	 * Disable synchronisation.
	 */
	rt2x00_register_write(rt2x00dev, CSR14, 0);

	/*
	 * Cancel RX and TX.
	 */
	rt2x00_register_read(rt2x00dev, TXCSR0, &reg);
	rt2x00_set_field32(&reg, TXCSR0_ABORT, 1);
	rt2x00_register_write(rt2x00dev, TXCSR0, reg);

	rt2x00_register_read(rt2x00dev, RXCSR0, &reg);
	rt2x00_set_field32(&reg, RXCSR0_DISABLE_RX, 1);
	rt2x00_register_write(rt2x00dev, RXCSR0, reg);

	/*
	 * Disable interrupts.
	 */
	rt2x00_register_read(rt2x00dev, CSR8, &reg);
	rt2x00_set_field32(&reg, CSR8_TBCN_EXPIRE, 1);
	rt2x00_set_field32(&reg, CSR8_TXDONE_TXRING, 1);
	rt2x00_set_field32(&reg, CSR8_TXDONE_ATIMRING, 1);
	rt2x00_set_field32(&reg, CSR8_TXDONE_PRIORING, 1);
	rt2x00_set_field32(&reg, CSR8_RXDONE, 1);
	rt2x00_register_write(rt2x00dev, CSR8, reg);
}

/*
 * RTS frame creation.
 */
static struct sk_buff* rt2500pci_create_rts(struct rt2x00_dev *rt2x00dev,
	struct ieee80211_hdr *hdr, unsigned short duration)
{
	struct ieee80211_hdr *ieee80211hdr;
	struct sk_buff *skb;
	u16 frame_control;

	skb = dev_alloc_skb(IEEE80211_HEADER);
	if (!skb)
		return NULL;

	/*
	 * Copy the entire header over to RTS frame.
	 */
	memcpy(skb_put(skb, IEEE80211_HEADER), hdr, IEEE80211_HEADER);
	ieee80211hdr = (struct ieee80211_hdr*)skb->data;

	frame_control = IEEE80211_FTYPE_CTL | IEEE80211_STYPE_RTS;
	ieee80211hdr->frame_control = cpu_to_le16(frame_control);

	ieee80211hdr->duration_id = cpu_to_le16(duration);

	ieee80211hdr->seq_ctrl = 0;

	return skb;
}

/*
 * TX descriptor initialization
 */
static void rt2500pci_write_tx_desc(struct rt2x00_dev *rt2x00dev,
	struct data_desc *txd, struct sk_buff *skb,
	struct ieee80211_tx_control *control)
{
	struct ieee80211_hdr *ieee80211hdr = (struct ieee80211_hdr*)skb->data;
	struct data_ring *ring;
	int tx_rate;
	u32 word;
	u32 length;
	u32 residual;
	u16 length_high;
	u16 length_low;
	u16 frame_control;
	u16 seq_ctrl;
	char rts_frame;
	char ofdm_rate;
	char req_timestamp;
	char more_frag;
	char ifs;
	u8 signal;
	u8 service;
	u8 bitrate;

	/*
	 * We require the ring structure this packet is being send to.
	 */
	ring = rt2x00_get_ring(rt2x00dev, control->queue);
	if (unlikely(!ring))
		return;

	/*
	 * Read required fields from ieee80211 header.
	 */
	frame_control = le16_to_cpu(ieee80211hdr->frame_control);
	seq_ctrl = le16_to_cpu(ieee80211hdr->seq_ctrl);

	/*
	 * Check if this frame is a RTS frame.
	 */
	rts_frame = is_rts_frame(frame_control);

	/*
	 * Check which rate should be used for this frame.
	 */
	if (rts_frame && control->rts_cts_rate)
		tx_rate = control->rts_cts_rate;
	else
		tx_rate = control->tx_rate;

	/*
	 * Are we working with OFDM rates.
	 */
	ofdm_rate = !!(DEVICE_GET_RATE_FIELD(tx_rate, RATEMASK) &
		       DEV_OFDM_RATE);

	/*
	 * Check if more fragments will follow this frame.
	 */
	more_frag = !!(ieee80211_get_morefrag(ieee80211hdr));

	/*
	 * Beacons and probe responses require the tsf timestamp
	 * to be inserted into the frame.
	 */
	req_timestamp = !!(control->queue == IEEE80211_TX_QUEUE_BEACON ||
		  	   control->pkt_type == PKT_PROBE_RESP);

	/*
	 * Determine with what IFS priority this frame should be send.
	 * Set ifs to IFS_SIFS when the this is not the first fragment,
	 * or this fragment came after RTS/CTS.
	 */
	if (((seq_ctrl & IEEE80211_SCTL_FRAG) > 0) || rts_frame)
		ifs = IFS_SIFS;
	else
		ifs = IFS_BACKOFF;

	/*
	 * Add 4 bytes for FCS.
	 */
	length = skb->len + FCS_LEN;

	/*
	 * How the length should be processed depends
	 * on if we are working with OFDM rates or not.
	 */
	if (ofdm_rate) {
		residual = 0;
		length_high = (length >> 6) & 0x3f;
		length_low = (length & 0x3f);

	} else {
		bitrate = DEVICE_GET_RATE_FIELD(tx_rate, RATE);

		/*
		 * Convert length to microseconds.
		 */
		residual = get_duration_res(length, bitrate);
		length = get_duration(length, bitrate);

		if (residual != 0)
			length++;

		length_high = length >> 8;
		length_low = length & 0xff;
	}

	/*
	 * Create the signal and service values.
	 */
	signal = DEVICE_GET_RATE_FIELD(tx_rate, PLCP);
	if (DEVICE_GET_RATE_FIELD(tx_rate, PREAMBLE))
		signal |= 0x08;

	service = 0x04;
	if (residual <= (8 % 11))
		service |= 0x80;

	/*
	 * Start writing the descriptor words.
	 */
	rt2x00_desc_read(txd, 2, &word);
	rt2x00_set_field32(&word, TXD_W2_IV_OFFSET, IEEE80211_HEADER);
	rt2x00_set_field32(&word, TXD_W2_AIFS, ring->tx_params.aifs);
	rt2x00_set_field32(&word, TXD_W2_CWMIN, ring->tx_params.cw_min);
	rt2x00_set_field32(&word, TXD_W2_CWMAX, ring->tx_params.cw_max);
	rt2x00_desc_write(txd, 2, word);

	rt2x00_desc_read(txd, 3, &word);
	rt2x00_set_field32(&word, TXD_W3_PLCP_SIGNAL, signal);
	rt2x00_set_field32(&word, TXD_W3_PLCP_SERVICE, service);
	rt2x00_set_field32(&word, TXD_W3_PLCP_LENGTH_LOW, length_low);
	rt2x00_set_field32(&word, TXD_W3_PLCP_LENGTH_HIGH, length_high);
	rt2x00_desc_write(txd, 3, word);

	rt2x00_desc_read(txd, 10, &word);
	rt2x00_set_field32(&word, TXD_W10_RTS, rts_frame);
	rt2x00_desc_write(txd, 10, word);

	rt2x00_desc_read(txd, 0, &word);
	rt2x00_set_field32(&word, TXD_W0_OWNER_NIC, 1);
	rt2x00_set_field32(&word, TXD_W0_VALID, 1);
	rt2x00_set_field32(&word, TXD_W0_MORE_FRAG, more_frag);
	rt2x00_set_field32(&word, TXD_W0_ACK,
		!(control->flags & IEEE80211_TXCTL_NO_ACK));
	rt2x00_set_field32(&word, TXD_W0_TIMESTAMP, req_timestamp);
	rt2x00_set_field32(&word, TXD_W0_OFDM, ofdm_rate);
	rt2x00_set_field32(&word, TXD_W0_CIPHER_OWNER, 1);
	rt2x00_set_field32(&word, TXD_W0_IFS, ifs);
	rt2x00_set_field32(&word, TXD_W0_RETRY_MODE, 0);
	rt2x00_set_field32(&word, TXD_W0_DATABYTE_COUNT, skb->len);
	rt2x00_set_field32(&word, TXD_W0_CIPHER_ALG, CIPHER_NONE);
	rt2x00_desc_write(txd, 0, word);
}

/*
 * Interrupt functions.
 */
static void rt2500pci_beacondone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_dev *rt2x00dev = ring->rt2x00dev;
	struct data_entry *entry = rt2x00_get_data_entry(
		&rt2x00dev->ring[RING_BEACON]);
	struct sk_buff *skb;

	skb = ieee80211_beacon_get(rt2x00dev->hw,
		rt2x00dev->interface.id, &entry->tx_status.control);
	if (!skb)
		return;

	rt2500pci_beacon_update(rt2x00dev->hw, skb, &entry->tx_status.control);

	dev_kfree_skb_any(skb);
}

static void rt2500pci_rxdone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_dev *rt2x00dev = ring->rt2x00dev;
	struct data_entry *entry;
	struct sk_buff *skb;
	struct data_desc *rxd;
	u32 word0;
	u32 word2;
	u16 size;

	while (1) {
		entry = rt2x00_get_data_entry(ring);
		rxd = rt2x00_desc_addr(entry);
		rt2x00_desc_read(rxd, 0, &word0);
		rt2x00_desc_read(rxd, 2, &word2);

		if (rt2x00_get_field32(word0, RXD_W0_OWNER_NIC))
			break;

		size = rt2x00_get_field32(word0, RXD_W0_DATABYTE_COUNT);

		/*
		 * TODO: Don't we need to keep statistics
		 * updated about events like CRC and physical errors?
		 */
		if (!rt2x00_get_field32(word0, RXD_W0_CRC) &&
		    !rt2x00_get_field32(word0, RXD_W0_PHYSICAL_ERROR)) {
			skb = dev_alloc_skb(size + NET_IP_ALIGN);
			if (!skb)
				break;

			skb_reserve(skb, NET_IP_ALIGN);

			memcpy(skb_put(skb, size), rt2x00_data_addr(entry),
				size);

			rt2x00dev->rx_params.rate = device_signal_to_rate(
				&rt2x00dev->hw->modes[1],
				rt2x00_get_field32(word2, RXD_W2_SIGNAL),
				rt2x00_get_field32(word0, RXD_W0_OFDM));

			rt2x00dev->rx_params.ssi =
				rt2x00_get_field32(word2, RXD_W2_RSSI);

			/*
			 * The following set of assignments supply
			 * additional data for wireless statistics. The
			 * necessary quantities are (1) the "noise" value
			 * in dBm, and (2) the "signal" value in dBm.
			 */
			rt2x00dev->rx_params.noise =
				rt2x00_get_link_noise(&rt2x00dev->link);
			rt2x00dev->rx_params.signal = rt2x00dev->rx_params.ssi
				- rt2x00dev->hw->maxssi;

			__ieee80211_rx(rt2x00dev->hw,
				skb, &rt2x00dev->rx_params);

			/*
			 * Update link statistics
			 */
			rt2x00_update_link_rssi(&rt2x00dev->link,
				rt2x00dev->rx_params.ssi);
		}
		rt2x00_set_field32(&word0, RXD_W0_OWNER_NIC, 1);
		rt2x00_desc_write(rxd, 0, word0);
		rt2x00_ring_index_inc(ring);
	}

	/*
	 * Update LED.
	 */
	rt2500pci_activity_led(rt2x00dev, 0);
}

static void rt2500pci_txdone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_dev *rt2x00dev = ring->rt2x00dev;
	struct data_entry *entry;
	struct data_desc *txd;
	u32 word;
	int tx_status;
	int ack;
	int rts;

	while (!rt2x00_ring_empty(ring)) {
		entry = rt2x00_get_data_entry_done(ring);
		txd = rt2x00_desc_addr(entry);
		rt2x00_desc_read(txd, 0, &word);

		if (rt2x00_get_field32(word, TXD_W0_OWNER_NIC) ||
		    !rt2x00_get_field32(word, TXD_W0_VALID))
			break;

		entry->tx_status.flags = 0;
		entry->tx_status.queue_length = ring->stats.limit;
		entry->tx_status.queue_number = entry->tx_status.control.queue;

		/*
		 * The TXD_W0_RESULT field will only be set when
		 * we had requested an ACK. So we have received an
		 * ACK response when ACK was requested and status
		 * was succesfull.
		 */
		ack = rt2x00_get_field32(word, TXD_W0_ACK);
		rts = GET_FLAG(entry, ENTRY_RTS_FRAME);
		tx_status = rt2x00_get_field32(word, TXD_W0_RESULT);
		rt2x00_update_tx_stats(rt2x00dev, &entry->tx_status, tx_status,
				       ack, rts);

		rt2x00_bbp_read(rt2x00dev, 32,
			(u8*)&entry->tx_status.ack_signal);

		entry->tx_status.retry_count = rt2x00_get_field32(
			word, TXD_W0_RETRY_COUNT);

		/*
		 * If this is not an RTS frame send the tx_status to d80211,
		 * that method also cleans up the skb structure. When this
		 * is a RTS frame, that it is our job to clean this structure up.
		 */
		if (!rts)
			ieee80211_tx_status(rt2x00dev->hw,
				entry->skb, &entry->tx_status);
		else
			dev_kfree_skb(entry->skb);

		rt2x00_set_field32(&word, TXD_W0_VALID, 0);
		rt2x00_desc_write(txd, 0, word);
		CLEAR_FLAG(entry, ENTRY_RTS_FRAME);
		entry->skb = NULL;

		rt2x00_ring_index_done_inc(ring);
	}

	/*
	 * Check if we are waiting on an empty queue
	 * to start scanning.
	 */
	if (rt2x00dev->scan &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_TX]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_ATIM]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_PRIO]))
		rt2x00_signal_scan(rt2x00dev->scan, SCANNING_READY);

	/*
	 * If the data ring was full before the txdone handler
	 * we must make sure the packet queue in the d80211 stack
	 * is reenabled when the txdone handler has finished.
	 */
	entry = ring->entry;
	if (!rt2x00_ring_full(ring))
		ieee80211_wake_queue(rt2x00dev->hw,
			entry->tx_status.control.queue);
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 19)
static irqreturn_t rt2500pci_interrupt(int irq, void *dev_instance)
#else
static irqreturn_t rt2500pci_interrupt(int irq, void *dev_instance,
	struct pt_regs *regs)
#endif
{
	struct rt2x00_dev *rt2x00dev = dev_instance;
	u32 reg;

	/*
	 * Get the interrupt sources & saved to local variable.
	 * Write register value back to clear pending interrupts.
	 */
	rt2x00_register_read(rt2x00dev, CSR7, &reg);
	rt2x00_register_write(rt2x00dev, CSR7, reg);

	if (!reg)
		return IRQ_NONE;

	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return IRQ_HANDLED;

	/*
	 * Handle interrupts, walk through all bits
	 * and run the tasks, the bits are checked in order of
	 * priority.
	 */

	/*
	 * 1 - Beacon timer expired interrupt.
	 */
	if (rt2x00_get_field32(reg, CSR7_TBCN_EXPIRE))
		queue_work(rt2x00dev->workqueue,
			&rt2x00dev->ring[RING_BEACON].irq_work);

	/*
	 * 2 - Rx ring done interrupt.
	 * Enable the TXRX activity led.
	 */
	if (rt2x00_get_field32(reg, CSR7_RXDONE)) {
		queue_work(rt2x00dev->workqueue,
			&rt2x00dev->ring[RING_RX].irq_work);
		rt2500pci_activity_led(rt2x00dev, 1);
	}

	/*
	 * 3 - Atim ring transmit done interrupt.
	 */
	if (rt2x00_get_field32(reg, CSR7_TXDONE_ATIMRING))
		queue_work(rt2x00dev->workqueue,
			&rt2x00dev->ring[RING_ATIM].irq_work);

	/*
	 * 4 - Priority ring transmit done interrupt.
	 */
	if (rt2x00_get_field32(reg, CSR7_TXDONE_PRIORING))
		queue_work(rt2x00dev->workqueue,
			&rt2x00dev->ring[RING_PRIO].irq_work);

	/*
	 * 5 - Tx ring transmit done interrupt.
	 */
	if (rt2x00_get_field32(reg, CSR7_TXDONE_TXRING))
		queue_work(rt2x00dev->workqueue,
			&rt2x00dev->ring[RING_TX].irq_work);

	return IRQ_HANDLED;
}

/*
 * IEEE80211 stack callback functions.
 */
static int rt2500pci_tx(struct ieee80211_hw *hw, struct sk_buff *skb,
	struct ieee80211_tx_control *control)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	struct ieee80211_hdr *ieee80211hdr = (struct ieee80211_hdr*)skb->data;
	struct data_ring *ring;
	struct data_entry *entry;
	struct data_desc *txd;
	struct sk_buff *skb_rts;
	u16 frame_control;
	u32 reg;
	int res;

	/*
	 * Determine which ring to put packet on.
	 */
	ring = rt2x00_get_ring(rt2x00dev, control->queue);
	if (unlikely(!ring)) {
		ERROR("Attempt to send packet over invalid queue %d.\n"
			"Please file bug report to %s.\n",
			control->queue, DRV_PROJECT);
		dev_kfree_skb_any(skb);
		return NETDEV_TX_OK;
	}

	if (rt2x00_ring_full(ring)) {
		ieee80211_stop_queue(hw, control->queue);
		return NETDEV_TX_BUSY;
	}

	/*
	 * If RTS is required. and this frame is not RTS,
	 * create and queue that frame first.
	 */
	frame_control = le16_to_cpu(ieee80211hdr->frame_control);
	if ((control->flags & IEEE80211_TXCTL_USE_RTS_CTS) &&
	    !is_rts_frame(frame_control)) {
		skb_rts = rt2500pci_create_rts(rt2x00dev,
				ieee80211hdr, control->rts_cts_duration);
		if (!skb_rts) {
			WARNING("Failed to create RTS frame.\n");
			return NETDEV_TX_BUSY;
		}

		res = rt2500pci_tx(hw, skb_rts, control);
		if (res) {
			WARNING("Failed to send RTS frame.\n");
			return res;
		}
	}

	entry = rt2x00_get_data_entry(ring);
	txd = rt2x00_desc_addr(entry);
	rt2x00_desc_read(txd, 0, &reg);

	if (rt2x00_get_field32(reg, TXD_W0_OWNER_NIC) ||
	    rt2x00_get_field32(reg, TXD_W0_VALID)) {
		ERROR("Arrived at non-free entry in the non-full queue %d.\n"
			"Please file bug report to %s.\n",
			control->queue, DRV_PROJECT);
		ieee80211_stop_queue(hw, control->queue);
		return NETDEV_TX_BUSY;
	}

	if (entry->flags)
		ERROR("Arrived at broken entry (flags 0x%08x) in queue %d.\n"
			"Please file bug report to %s.\n",
			control->queue, entry->flags, DRV_PROJECT);

	/*
	 * Set the software sequence number.
	 */
	rt2x00_set_sequence(skb, &rt2x00dev->interface.sequence);

	memcpy(rt2x00_data_addr(entry), skb->data, skb->len);
	rt2500pci_write_tx_desc(rt2x00dev, txd, skb, control);
	memcpy(&entry->tx_status.control, control, sizeof(*control));
	if (is_rts_frame(frame_control))
		SET_FLAG(entry, ENTRY_RTS_FRAME);
	entry->skb = skb;

	rt2x00_ring_index_inc(ring);

	if (rt2x00_ring_full(ring))
		ieee80211_stop_queue(hw, control->queue);

	rt2x00_register_read(rt2x00dev, TXCSR0, &reg);
	if (control->queue == IEEE80211_TX_QUEUE_DATA0)
		rt2x00_set_field32(&reg, TXCSR0_KICK_PRIO, 1);
	else if (control->queue == IEEE80211_TX_QUEUE_DATA1)
		rt2x00_set_field32(&reg, TXCSR0_KICK_TX, 1);
	else if (control->queue == IEEE80211_TX_QUEUE_AFTER_BEACON)
		rt2x00_set_field32(&reg, TXCSR0_KICK_ATIM, 1);
	rt2x00_register_write(rt2x00dev, TXCSR0, reg);

	return NETDEV_TX_OK;
}

static int rt2500pci_reset(struct ieee80211_hw *hw)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	rt2500pci_disable_radio(rt2x00dev);
	return rt2500pci_enable_radio(rt2x00dev);
}

static int rt2500pci_add_interface(struct ieee80211_hw *hw,
	struct ieee80211_if_init_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	int status;

	/*
	 * We only support 1 non-monitor interface.
	 */
	if (conf->type != IEEE80211_IF_TYPE_MNTR &&
	    GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		return -ENOBUFS;

	/*
	 * Add the new interface.
	 */
	rt2x00_add_interface(&rt2x00dev->interface, conf);

	/*
	 * Initialize interface, and enable the radio when this
	 * is the first interface that is brought up.
	 */
	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO)) {
		/*
		 * Before doing anything else, the MAC address
		 * of this device should be initialized correctly.
		 */
		rt2500pci_config_mac_address(rt2x00dev, conf->mac_addr);

		/*
		 * Initialize the device.
		 */
		status = rt2500pci_initialize(rt2x00dev);
		if (status)
			return status;

		/*
		 * Enable radio.
		 */
		status = rt2500pci_enable_radio(rt2x00dev);
		if (status)
			return status;

		/*
		 * Enable periodic link tuning if this is a non-monitor
		 * interface. Also set the INTERFACE_INITIALIZED FLAG
		 * to prevent new non-monitor interfaces to be added.
		 */
		if (conf->type != IEEE80211_IF_TYPE_MNTR) {
			queue_delayed_work(rt2x00dev->workqueue,
				&rt2x00dev->link.work, LINK_TUNE_INTERVAL);
			SET_FLAG(rt2x00dev, INTERFACE_INITIALIZED);
		} else
			SET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR);
	}

	return 0;
}

static void rt2500pci_remove_interface(struct ieee80211_hw *hw,
	struct ieee80211_if_init_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	/*
	 * We only support 1 non-monitor interface.
	 */
	if (conf->type != IEEE80211_IF_TYPE_MNTR &&
	    !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		return;

	/*
	 * Remove the interface.
	 */
	rt2x00_remove_interface(&rt2x00dev->interface, conf);

	/*
	 * When this is a non-monitor mode,
	 * stop the periodic link tuning,
	 * and clear the INTERFACE_INITIALIZED FLAG to allow
	 * new non-monitor interfaces to be added.
	 */
	if (conf->type != IEEE80211_IF_TYPE_MNTR) {
		cancel_rearming_delayed_workqueue(rt2x00dev->workqueue,
			&rt2x00dev->link.work);
		CLEAR_FLAG(rt2x00dev, INTERFACE_INITIALIZED);
	}

	/*
	 * Disable radio if this was the last interface
	 * that was working with this device.
	 */
	if (!GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
	    !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		rt2500pci_disable_radio(rt2x00dev);

	/*
	 * Check if we still have 1 non-monitor or a monitor
	 * interface enabled. In that case we should update the
	 * registers.
	 */
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) ^
	    GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED)) {
		if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
			rt2500pci_config_type(rt2x00dev,
				rt2x00dev->interface.type);
		else
			rt2500pci_config_type(rt2x00dev,
				IEEE80211_IF_TYPE_MNTR);
	}
}

static int rt2500pci_config(struct ieee80211_hw *hw,
	struct ieee80211_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	u32 reg;

	/*
	 * Check if we need to disable the radio,
	 * if this is not the case, at least the RX must be disabled.
	 */
	if (GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO)) {
		if (!conf->radio_enabled)
			rt2500pci_disable_radio(rt2x00dev);
		else {
			rt2x00_register_read(rt2x00dev, RXCSR0, &reg);
			rt2x00_set_field32(&reg, RXCSR0_DISABLE_RX, 1);
			rt2x00_register_write(rt2x00dev, RXCSR0, reg);
		}
	}

	rt2500pci_config_channel(rt2x00dev,
		conf->channel_val, conf->channel, conf->freq,
		conf->power_level);
	rt2500pci_config_txpower(rt2x00dev, conf->power_level);
	rt2500pci_config_antenna(rt2x00dev, conf->antenna_sel);
	rt2500pci_config_duration(rt2x00dev,
		(conf->flags & IEEE80211_CONF_SHORT_SLOT_TIME));
	rt2500pci_config_phymode(rt2x00dev, conf->phymode);

	/*
	 * Reenable RX only if the radio should be on.
	 */
	if (GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO)) {
		rt2x00_register_read(rt2x00dev, RXCSR0, &reg);
		rt2x00_set_field32(&reg, RXCSR0_DISABLE_RX, 0);
		rt2x00_register_write(rt2x00dev, RXCSR0, reg);
	} else if (conf->radio_enabled)
		return rt2500pci_enable_radio(rt2x00dev);

	return 0;
}

static int rt2500pci_config_interface(struct ieee80211_hw *hw, int if_id,
	struct ieee80211_if_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	/*
	 * Monitor mode does not need configuring.
	 * If the given type does not match the configured type,
	 * there has been a problem.
	 */
	if (conf->type == IEEE80211_IF_TYPE_MNTR)
		return 0;
	else if (conf->type != rt2x00dev->interface.type)
		return -EINVAL;

	/*
	 * If the interface does not work in master mode,
	 * then the bssid value in the interface structure
	 * should now be set.
	 */
	if (conf->type != IEEE80211_IF_TYPE_AP)
		memcpy(&rt2x00dev->interface.bssid, conf->bssid, ETH_ALEN);

	/*
	 * Enable configuration.
	 */
	rt2500pci_config_type(rt2x00dev, conf->type);
	rt2500pci_config_bssid(rt2x00dev, &rt2x00dev->interface.bssid[0]);

	return 0;
}

static void rt2500pci_set_multicast_list(struct ieee80211_hw *hw,
	unsigned short flags, int mc_count)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	int update = 0;

	if (GET_FLAG(rt2x00dev, INTERFACE_ENABLED_PROMISC)) {
		if (!(flags & IFF_PROMISC)) {
			rt2x00dev->interface.promisc = 0;
			update = 1;
		}
	} else {
		if (flags & IFF_PROMISC) {
			rt2x00dev->interface.promisc = 1;
			update = 1;
		}
	}

	/*
	 * Monitor mode works with PROMISC mode forced on,
	 * so there is nothing to be done here.
	 */
	if (update && !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR))
		rt2500pci_config_promisc(rt2x00dev,
			rt2x00dev->interface.promisc);
}

static void rt2500pci_scan(void *data)
{
	struct rt2x00_dev *rt2x00dev = data;

	if (unlikely(!rt2x00dev->scan))
		return;

	/*
	 * Before we can start switch the channel for scanning
	 * we need to wait untill all TX rings are empty to
	 * guarentee that all frames are send on the correct channel.
	 */
	if (rt2x00_wait_scan(rt2x00dev->scan))
		goto exit;

	/*
	 * Switch channel and update active info for RX.
	 */
	if (rt2x00dev->scan->state == IEEE80211_SCAN_START) {
		rt2500pci_config_phymode(rt2x00dev,
			rt2x00dev->scan->conf.scan_phymode);

		rt2500pci_config_channel(rt2x00dev,
			rt2x00dev->scan->conf.scan_channel_val,
			rt2x00dev->scan->conf.scan_channel,
			rt2x00dev->scan->conf.scan_freq,
			rt2x00dev->scan->conf.scan_power_level);
	} else {
		rt2500pci_config_phymode(rt2x00dev,
			rt2x00dev->scan->conf.running_phymode);

		rt2500pci_config_channel(rt2x00dev,
			rt2x00dev->scan->conf.running_channel_val,
			rt2x00dev->scan->conf.running_channel,
			rt2x00dev->scan->conf.running_freq,
			rt2x00dev->scan->conf.scan_power_level);
	}

exit:
	kfree(rt2x00dev->scan);
	rt2x00dev->scan = NULL;
}

static int rt2500pci_passive_scan(struct ieee80211_hw *hw,
	int state, struct ieee80211_scan_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	/*
	 * Check if we are not busy with the previous
	 * passive scan request.
	 */
	if (rt2x00dev->scan)
		return -EBUSY;

	/*
	 * Check if the radio is enabled.
	 */
	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return -EIO;

	/*
	 * Allocate scanning structure to store scanning info.
	 */
	rt2x00dev->scan = kzalloc(sizeof(struct scanning), GFP_ATOMIC);
	if (!rt2x00dev->scan)
		return -ENOMEM;

	/*
	 * Check if we have to send a packet before the
	 * channel switch.
	 */
	if (conf->skb) {
		if (rt2500pci_tx(hw, conf->skb, conf->tx_control))
			goto exit;
	}

	/*
	 * Initialize Scanning structure.
	 */
	rt2x00_start_scan(rt2x00dev->scan, conf, state);

	/*
	 * Queue work.
	 */
	INIT_WORK(&rt2x00dev->scan->work, rt2500pci_scan, rt2x00dev);
	if (!queue_work(rt2x00dev->workqueue, &rt2x00dev->scan->work))
		goto exit;

	return 0;

exit:
	kfree(rt2x00dev->scan);
	rt2x00dev->scan = NULL;

	return -EIO;
}

static int rt2500pci_get_stats(struct ieee80211_hw *hw,
	struct ieee80211_low_level_stats *stats)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	u32 reg;

	/*
	 * Update FCS error count from register.
	 * The dot11ACKFailureCount, dot11RTSFailureCount and
	 * dot11RTSSuccessCount are updated in interrupt time.
	 */
	rt2x00_register_read(rt2x00dev, CNT0, &reg);
	rt2x00dev->low_level_stats.dot11FCSErrorCount +=
		rt2x00_get_field32(reg, CNT0_FCS_ERROR);

	memcpy(stats, &rt2x00dev->low_level_stats, sizeof(*stats));

	return 0;
}

static int rt2500pci_set_retry_limit(struct ieee80211_hw *hw,
	u32 short_retry, u32 long_retry)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	u32 reg;

	rt2x00_register_read(rt2x00dev, CSR11, &reg);
	rt2x00_set_field32(&reg, CSR11_LONG_RETRY, long_retry);
	rt2x00_set_field32(&reg, CSR11_SHORT_RETRY, short_retry);
	rt2x00_register_write(rt2x00dev, CSR11, reg);

	return 0;
}

static int rt2500pci_conf_tx(struct ieee80211_hw *hw,
	int queue, const struct ieee80211_tx_queue_params *params)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	struct data_ring *ring;

	ring = rt2x00_get_ring(rt2x00dev, queue);
	if (unlikely(!ring))
		return -EINVAL;

	memcpy(&ring->tx_params, params, sizeof(*params));

	/*
	 * The passed variables are stored as real value ((2^n)-1).
	 * RT2500 registers require to know the bit number 'n'.
	 */
	if (params->cw_min)
		ring->tx_params.cw_min = HIGHEST_BIT16(params->cw_min) + 1;
	else
		ring->tx_params.cw_min = 5; /* cw_min: 2^5 = 32. */

	if (params->cw_max)
		ring->tx_params.cw_max = HIGHEST_BIT16(params->cw_max) + 1;
	else
		ring->tx_params.cw_max = 10; /* cw_min: 2^10 = 1024. */

	if (!params->aifs)
		ring->tx_params.aifs = 2;

	INFO("Configured TX ring %d - CWmin: %d, CWmax: %d, Aifs: %d.\n",
		queue, ring->tx_params.cw_min, ring->tx_params.cw_max,
		ring->tx_params.aifs);

	return 0;
}

static int rt2500pci_get_tx_stats(struct ieee80211_hw *hw,
	struct ieee80211_tx_queue_stats *stats)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA0],
		&rt2x00dev->ring[RING_PRIO].stats,
		sizeof(rt2x00dev->ring[RING_PRIO].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA1],
		&rt2x00dev->ring[RING_TX].stats,
		sizeof(rt2x00dev->ring[RING_TX].stats));

	return 0;
}

static u64 rt2500pci_get_tsf(struct ieee80211_hw *hw)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	u64 tsf;
	u32 reg;

	rt2x00_register_read(rt2x00dev, CSR17, &reg);
	tsf = (u64)rt2x00_get_field32(reg, CSR17_HIGH_TSFTIMER) << 32;
	rt2x00_register_read(rt2x00dev, CSR16, &reg);
	tsf |= rt2x00_get_field32(reg, CSR16_LOW_TSFTIMER);

	return tsf;
}

static void rt2500pci_reset_tsf(struct ieee80211_hw *hw)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	rt2x00_register_write(rt2x00dev, CSR16, 0);
	rt2x00_register_write(rt2x00dev, CSR17, 0);
}

static int rt2500pci_beacon_update(struct ieee80211_hw *hw,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	struct data_entry *entry;

	entry = rt2x00_get_data_entry(&rt2x00dev->ring[RING_BEACON]);

	/*
	 * Just in case the ieee80211 doesn't set this,
	 * but we need this queue set for the descriptor
	 * initialization.
	 */
	control->queue = IEEE80211_TX_QUEUE_BEACON;

	/*
	 * Set the software sequence number.
	 */
	rt2x00_set_sequence(skb, &rt2x00dev->interface.sequence);

	/*
	 * Update the beacon entry.
	 */
	memcpy(rt2x00_data_addr(entry), skb->data, skb->len);
	rt2500pci_write_tx_desc(rt2x00dev, rt2x00_desc_addr(entry),
		skb, control);

	return 0;
}

static int rt2500pci_tx_last_beacon(struct ieee80211_hw *hw)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	u32 reg;

	rt2x00_register_read(rt2x00dev, CSR15, &reg);
	return rt2x00_get_field32(reg, CSR15_BEACON_SENT);
}

static struct ieee80211_ops rt2500pci_d80211_ops = {
	.tx			= rt2500pci_tx,
	.reset			= rt2500pci_reset,
	.add_interface		= rt2500pci_add_interface,
	.remove_interface	= rt2500pci_remove_interface,
	.config			= rt2500pci_config,
	.config_interface	= rt2500pci_config_interface,
	.set_multicast_list	= rt2500pci_set_multicast_list,
	.passive_scan		= rt2500pci_passive_scan,
	.get_stats		= rt2500pci_get_stats,
	.set_retry_limit	= rt2500pci_set_retry_limit,
	.conf_tx		= rt2500pci_conf_tx,
	.get_tx_stats		= rt2500pci_get_tx_stats,
	.get_tsf		= rt2500pci_get_tsf,
	.reset_tsf		= rt2500pci_reset_tsf,
	.beacon_update		= rt2500pci_beacon_update,
	.tx_last_beacon		= rt2500pci_tx_last_beacon,
	.ethtool		= &rt2500pci_ethtool_ops,
};

/*
 * Device initialization functions.
 */
static void rt2500pci_register_read(struct eeprom_93cx6 *eeprom)
{
	struct rt2x00_dev *rt2x00dev = eeprom->data;
	u32 reg;

	rt2x00_register_read(rt2x00dev, CSR21, &reg);

	eeprom->reg_data_in = !!rt2x00_get_field32(reg,
		CSR21_EEPROM_DATA_IN);
	eeprom->reg_data_out = !!rt2x00_get_field32(reg,
		CSR21_EEPROM_DATA_OUT);
	eeprom->reg_data_clock = !!rt2x00_get_field32(reg,
		CSR21_EEPROM_DATA_CLOCK);
	eeprom->reg_chip_select = !!rt2x00_get_field32(reg,
		CSR21_EEPROM_CHIP_SELECT);
}

static void rt2500pci_register_write(struct eeprom_93cx6 *eeprom)
{
	struct rt2x00_dev *rt2x00dev = eeprom->data;
	u32 reg = 0;

	rt2x00_set_field32(&reg, CSR21_EEPROM_DATA_IN,
		!!eeprom->reg_data_in);
	rt2x00_set_field32(&reg, CSR21_EEPROM_DATA_OUT,
		!!eeprom->reg_data_out);
	rt2x00_set_field32(&reg, CSR21_EEPROM_DATA_CLOCK,
		!!eeprom->reg_data_clock);
	rt2x00_set_field32(&reg, CSR21_EEPROM_CHIP_SELECT,
		!!eeprom->reg_chip_select);

	rt2x00_register_write(rt2x00dev, CSR21, reg);
}

static int rt2500pci_alloc_eeprom(struct rt2x00_dev *rt2x00dev)
{
	struct eeprom_93cx6 eeprom;
	u32 reg;

	/*
	 * Allocate the eeprom memory, check the eeprom width
	 * and copy the entire eeprom into this allocated memory.
	 */
	rt2x00dev->eeprom = kzalloc(EEPROM_SIZE, GFP_KERNEL);
	if (!rt2x00dev->eeprom)
		return -ENOMEM;

	rt2x00_register_read(rt2x00dev, CSR21, &reg);

	eeprom.data = rt2x00dev;
	eeprom.register_read = rt2500pci_register_read;
	eeprom.register_write = rt2500pci_register_write;
	eeprom.width = rt2x00_get_field32(reg, CSR21_TYPE_93C46) ?
		PCI_EEPROM_WIDTH_93C46 : PCI_EEPROM_WIDTH_93C66;
	eeprom.reg_data_in = 0;
	eeprom.reg_data_out = 0;
	eeprom.reg_data_clock = 0;
	eeprom.reg_chip_select = 0;

	eeprom_93cx6_multiread(&eeprom, EEPROM_BASE, rt2x00dev->eeprom,
		EEPROM_SIZE);

	return 0;
}

static int rt2500pci_alloc_rings(struct rt2x00_dev *rt2x00dev)
{
	unsigned int i;

	rt2x00dev->ring = kzalloc(
		sizeof(struct data_ring) * RING_NUM, GFP_KERNEL);
	if (!rt2x00dev->ring) {
		ERROR("Ring allocation failed.\n");
		return -ENOMEM;
	}

	SET_FLAG(rt2x00dev, DEVICE_SUPPORT_ATIM);

	for (i = 0; i < RING_NUM; i++) {
		rt2x00dev->ring[i].rt2x00dev = rt2x00dev;

		/*
		 * Initialize ring parameters.
		 * cw_min: 2^5 = 32.
		 * cw_max: 2^10 = 1024.
		 */
		rt2x00dev->ring[i].tx_params.aifs = 2;
		rt2x00dev->ring[i].tx_params.cw_min = 5;
		rt2x00dev->ring[i].tx_params.cw_max = 10;
	}

	return 0;
}

static int rt2500pci_init_eeprom(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;
	u16 value;
	u16 eeprom;

	/*
	 * Read EEPROM word for configuration.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_ANTENNA, &eeprom);

	/*
	 * Identify RF chipset.
	 */
	value = rt2x00_get_field16(eeprom, EEPROM_ANTENNA_RF_TYPE);
	rt2x00_register_read(rt2x00dev, CSR0, &reg);
	rt2x00_set_chip(&rt2x00dev->chip, RT2560, value, reg);

	if (!rt2x00_rf(&rt2x00dev->chip, RF2522) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF2523) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF2524) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF2525) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF2525E) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF5222))
		return -ENODEV;

	/*
	 * Identify default antenna configuration.
	 * Ralink devices have have antenna options for both TX as RX.
	 * The ieee80211 stack currently only provide the user to set
	 * 1 antenna, by default this is considered to be the TX antenna.
	 */
	rt2x00dev->hw->conf.antenna_sel = rt2x00_get_field16(eeprom,
		EEPROM_ANTENNA_TX_DEFAULT);

	/*
	 * Store led mode, for correct led behaviour.
	 */
	rt2x00dev->led_mode = rt2x00_get_field16(eeprom,
		EEPROM_ANTENNA_LED_MODE);

	/*
	 * Detect if this device has an hardware controlled radio.
	 */
	if (rt2x00_get_field16(eeprom, EEPROM_ANTENNA_HARDWARE_RADIO))
		SET_FLAG(rt2x00dev, DEVICE_SUPPORT_HW_BUTTON);

	/*
	 * Check if the BBP tuning should be enabled.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_NIC, &eeprom);
	if (eeprom == 0xffff)
		eeprom = 0;
	if (rt2x00_get_field16(eeprom, EEPROM_NIC_DYN_BBP_TUNE))
		SET_FLAG(rt2x00dev, CONFIG_DISABLE_LINK_TUNING);

	/*
	 * Read the RSSI <-> dBm offset information.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_CALIBRATE_OFFSET, &eeprom);
	rt2x00dev->hw->maxssi =
		rt2x00_get_field16(eeprom, EEPROM_CALIBRATE_OFFSET_RSSI);
	if (rt2x00dev->hw->maxssi == 0x00 || rt2x00dev->hw->maxssi == 0xff)
		rt2x00dev->hw->maxssi = MAX_RX_SSI;

	return 0;
}

static int rt2500pci_init_hw_mac(struct rt2x00_dev *rt2x00dev)
{
	u8 *addr;

	/*
	 * Get the pointer to the MAC address in the EEPROM.
	 */
	addr = rt2x00_eeprom_addr(rt2x00dev, EEPROM_MAC_ADDR_0);

	/*
	 * Check if a valid MAC address is present.
	 */
	if (!is_valid_ether_addr(addr)) {
		ERROR("Invalid MAC address: " MAC_FMT ".\n", MAC_ARG(addr));
		return -EINVAL;
	}

	/*
	 * Write MAC address to register.
	 */
	rt2500pci_config_mac_address(rt2x00dev, addr);

	/*
	 * Copy MAC address to the hw structure.
	 */
	memcpy(&rt2x00dev->hw->perm_addr, addr, ETH_ALEN);

	return 0;
}

static void rt2500pci_init_hw_channels(struct rt2x00_dev *rt2x00dev,
	struct ieee80211_channel *channels)
{
	unsigned int i;
	u32 rf2_base;
	u16 eeprom;
	static const struct {
		unsigned int chip;
		u32 val[3];
	} rf[] = {
		{ RF2522,	{ 0x00002050, 0x00000101, 0x00000000 } },
		{ RF2523,	{ 0x00022010, 0x000e0111, 0x00000a1b } },
		{ RF2524,	{ 0x00032020, 0x00000101, 0x00000a1b } },
		{ RF2525,	{ 0x00022020, 0x00060111, 0x00000a1b } },
		{ RF2525E,	{ 0x00022020, 0x00060111, 0x00000a0b } },
		{ RF5222,	{ 0x00000000, 0x00000101, 0x00000000 } }
	};

	/*
	 * Channel initialization.
	 * First we set the basic variables.
	 */
	for (i = 0; i < 13; i++) {
		channels[i].chan = i + 1;
		channels[i].freq = 2407 + ((i + 1) * 5);
		channels[i].flag = IEEE80211_CHAN_W_IBSS |
			IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
		channels[i].antenna_max = 0xff;
	}

	channels[13].chan = 14;
	channels[13].freq = 2484;
	channels[13].flag = IEEE80211_CHAN_W_IBSS |
		IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
	channels[13].antenna_max = 0xff;

	if (rt2x00_rf(&rt2x00dev->chip, RF5222)) {
		for (i = 14; i < 37; i++) {
			if (i < 22)
				channels[i].chan = 36;
			else if (i < 33)
				channels[i].chan = 100;
			else
				channels[i].chan = 149;
			channels[i].chan += ((i - 14) * 4);
			channels[i].freq = ((i - 13) + 1000) * 5;
			channels[i].flag = IEEE80211_CHAN_W_IBSS |
				IEEE80211_CHAN_W_ACTIVE_SCAN |
				IEEE80211_CHAN_W_SCAN;
			channels[i].power_level = DEFAULT_TXPOWER;
			channels[i].antenna_max = 0xff;
		}
	}

	/*
	 * Set device specific value.
	 */
	rf2_base = 0;
	if (rt2x00_rf(&rt2x00dev->chip, RF2525) ||
	    rt2x00_rf(&rt2x00dev->chip, RF2525E))
		rf2_base = 0x00080000;

	if (rt2x00_rf(&rt2x00dev->chip, RF2522)) {
		static const u32 vals[] = {
			0x000c1fda, 0x000c1fee, 0x000c2002, 0x000c2016,
			0x000c202a, 0x000c203e, 0x000c2052, 0x000c2066,
			0x000c207a, 0x000c208e, 0x000c20a2, 0x000c20b6,
			0x000c20ca, 0x000c20fa
		};

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			channels[i].val = vals[i];
	} else if (rt2x00_rf(&rt2x00dev->chip, RF2523) ||
		   rt2x00_rf(&rt2x00dev->chip, RF2524) ||
		   rt2x00_rf(&rt2x00dev->chip, RF2525)) {
		static const u32 vals[] = {
			0x00000c9e, 0x00000ca2, 0x00000ca6, 0x00000caa,
			0x00000cae, 0x00000cb2, 0x00000cb6, 0x00000cba,
			0x00000cbe, 0x00000d02, 0x00000d06, 0x00000d0a,
			0x00000d0e, 0x00000d1a
		};

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			channels[i].val = vals[i] | rf2_base;
	} else if (rt2x00_rf(&rt2x00dev->chip, RF2525E) ||
		   rt2x00_rf(&rt2x00dev->chip, RF5222)) {
		static const u32 vals[] = {
			0x00001136, 0x0000113a, 0x0000113e, 0x00001182,
			0x00001186, 0x0000118a, 0x0000118e, 0x00001192,
			0x00001196, 0x0000119a, 0x0000119e, 0x000011a2,
			0x000011a6, 0x000011ae
		};

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			channels[i].val = vals[i] | rf2_base;
	}
	if (rt2x00_rf(&rt2x00dev->chip, RF5222)) {
		static const u32 vals[] = {
			0x00018896, 0x0001889a, 0x0001889e, 0x000188a2,
			0x000188a6, 0x000188aa, 0x000188ae, 0x000188b2,
			0x00008802, 0x00008806, 0x0000880a, 0x0000880e,
			0x00008812, 0x00008816, 0x0000881a, 0x0000881e,
			0x00008822, 0x00008826, 0x0000882a, 0x000090a6,
			0x000090ae, 0x000090b6, 0x000090be
		};

		struct ieee80211_channel *chan = channels + 14;

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			(chan++)->val = vals[i];
	}

	/*
	 * Set TX power, each EEPROM TXpower entry
	 * contains the TXpower value for 2 channels.
	 */
	for (i = 0; i < EEPROM_TXPOWER_SIZE; i++) {
		rt2x00_eeprom_read(rt2x00dev,
			EEPROM_TXPOWER_START + i, &eeprom);

		channels[(i * 2)].power_level = TXPOWER_FROM_DEV(
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_1));

		channels[(i * 2) + 1].power_level = TXPOWER_FROM_DEV(
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_2));
	}

	/*
	 * Set device specific, but channel independent RF values.
	 */
	for (i = 0; i < ARRAY_SIZE(rf); i++) {
		if (rt2x00_rf(&rt2x00dev->chip, rf[i].chip)) {
			rt2x00dev->rf1 = rf[i].val[0];
			rt2x00dev->rf3 = rf[i].val[1];
			rt2x00dev->rf4 = rf[i].val[2];
		}
	}
}

static void rt2500pci_init_hw_rates(struct rt2x00_dev *rt2x00dev,
	struct ieee80211_rate *rates)
{
	/*
	 * Rates initialization.
	 */
	device_rate_entry(&rates[0], 10, 0x001, 0x00, IEEE80211_RATE_CCK);
	device_rate_entry(&rates[1], 20, 0x003, 0x01, IEEE80211_RATE_CCK_2);
	device_rate_entry(&rates[2], 55, 0x007, 0x02, IEEE80211_RATE_CCK_2);
	device_rate_entry(&rates[3], 110, 0x00f, 0x03, IEEE80211_RATE_CCK_2);
	device_rate_entry(&rates[4], 60, 0x01f, 0x0b, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[5], 90, 0x03f, 0x0f, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[6], 120, 0x07f, 0x0a, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[7], 180, 0x0ff, 0x0e, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[8], 240, 0x1ff, 0x09, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[9], 360, 0x3ff, 0x0d, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[10], 480, 0x7ff, 0x08, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[11], 540, 0xfff, 0x0c, IEEE80211_RATE_OFDM);
}

static int rt2500pci_init_hw_modes(struct rt2x00_dev *rt2x00dev)
{
	struct ieee80211_hw *hw = rt2x00dev->hw;
	int num_modes;
	int num_channels;

	/*
	 * RT2500 only supports 802.11b & 802.11g,
	 * so we should allocate 14 OFDM channels, 4 CCK rates
	 * and 8 OFDM rates.
	 * RF5222 also supports 802.11a, so allocate an
	 * additional 23 5.2GHz channels.
	 */
	num_modes = 2;
	num_channels = 14;
	if (rt2x00_rf(&rt2x00dev->chip, RF5222)) {
		num_modes = 3;
		num_channels = 37;
	}

	hw->num_modes = num_modes;
	hw->modes =
		kzalloc((sizeof(struct ieee80211_hw_modes) * num_modes),
		GFP_KERNEL);
	if (!hw->modes)
		goto exit;

	hw->modes->channels =
		kzalloc((sizeof(struct ieee80211_channel) * num_channels),
		GFP_KERNEL);
	if (!hw->modes->channels)
		goto exit_free_modes;

	hw->modes->rates =
		kzalloc((sizeof(struct ieee80211_rate) * 12),
		GFP_KERNEL);
	if (!hw->modes->rates)
		goto exit_free_channels;

	/*
	 * Intitialize 802.11g
	 * Rates: CCK, OFDM.
	 * Channels: OFDM.
	 */
	hw->modes[0].mode = MODE_IEEE80211G;
	hw->modes[0].num_channels = 14;
	hw->modes[0].num_rates = 12;

	/*
	 * Intitialize 802.11b
	 * Rates: CCK.
	 * Channels: OFDM.
	 */
	hw->modes[1].mode = MODE_IEEE80211B;
	hw->modes[1].num_channels = 14;
	hw->modes[1].num_rates = 4;
	hw->modes[1].channels = hw->modes[0].channels;
	hw->modes[1].rates = hw->modes[0].rates;

	/*
	 * Intitialize 802.11a
	 * Rates: OFDM.
	 * Channels: OFDM, UNII, HiperLAN2.
	 */
	if (rt2x00_rf(&rt2x00dev->chip, RF5222)) {
		hw->modes[2].mode = MODE_IEEE80211A;
		hw->modes[2].num_channels = 37;
		hw->modes[2].num_rates = 8;
		hw->modes[2].channels = &hw->modes[0].channels[14];
		hw->modes[2].rates = &hw->modes[0].rates[4];
	}

	rt2500pci_init_hw_channels(rt2x00dev, hw->modes[0].channels);
	rt2500pci_init_hw_rates(rt2x00dev, hw->modes[0].rates);

	return ieee80211_update_hw(hw);

exit_free_channels:
	kfree(hw->modes->channels);
	hw->modes->channels = NULL;

exit_free_modes:
	kfree(hw->modes);
	hw->modes = NULL;

exit:
	ERROR("Allocation ieee80211 modes failed.\n");
	return -ENOMEM;
}

static int rt2500pci_init_hw(struct rt2x00_dev *rt2x00dev)
{
	int status;

	if (GET_FLAG(rt2x00dev, DEVICE_INITIALIZED_HW))
		return 0;

	rt2x00dev->hw->dev = &rt2x00dev_pci(rt2x00dev)->dev;

	/*
	 * Read MAC address from EEPROM.
	 */
	status = rt2500pci_init_hw_mac(rt2x00dev);
	if (status)
		return status;

	/*
	 * Initialize all hw fields.
	 */
	rt2x00dev->hw->flags = IEEE80211_HW_HOST_GEN_BEACON |
		IEEE80211_HW_HOST_BROADCAST_PS_BUFFERING |
		IEEE80211_HW_WEP_INCLUDE_IV |
		IEEE80211_HW_DATA_NULLFUNC_ACK |
		IEEE80211_HW_NO_TKIP_WMM_HWACCEL |
		IEEE80211_HW_MONITOR_DURING_OPER;
	rt2x00dev->hw->extra_tx_headroom = 0;
	rt2x00dev->hw->maxssi = MAX_RX_SSI;
	rt2x00dev->hw->queues = RING_NUM_TX;

	status = rt2500pci_init_hw_modes(rt2x00dev);
	if (status)
		return status;

	if (ieee80211_register_hw(rt2x00dev->hw))
		return -EIO;

	SET_FLAG(rt2x00dev, DEVICE_INITIALIZED_HW);

	return 0;
}

static void rt2500pci_free_dev(struct rt2x00_dev *rt2x00dev)
{
	/*
	 * Shutdown poll_timer for hardware button.
	 */
	rt2500pci_button_stop(rt2x00dev);

	/*
	 * Free ring structures.
	 */
	kfree(rt2x00dev->ring);
	rt2x00dev->ring = NULL;

	/*
	 * Free workqueue.
	 */
	if (likely(rt2x00dev->workqueue)) {
		destroy_workqueue(rt2x00dev->workqueue);
		rt2x00dev->workqueue = NULL;
	}

	/*
	 * Free EEPROM memory.
	 */
	kfree(rt2x00dev->eeprom);

	/*
	 * Release CSR memory.
	 */
	if (likely(rt2x00dev->csr_addr)) {
		iounmap(rt2x00dev->csr_addr);
		rt2x00dev->csr_addr = NULL;
	}

	/*
	 * Free ieee80211_hw memory.
	 */
	if (likely(rt2x00dev->hw->modes)) {
		kfree(rt2x00dev->hw->modes->channels);
		kfree(rt2x00dev->hw->modes->rates);
		kfree(rt2x00dev->hw->modes);
		rt2x00dev->hw->modes = NULL;
	}
}

static int rt2500pci_alloc_dev(struct pci_dev *pci_dev,
	struct ieee80211_hw *hw)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	rt2x00dev->dev = pci_dev;
	rt2x00dev->hw = hw;

	/*
	 * Allocate the CSR memory.
	 */
	rt2x00dev->csr_addr = ioremap(
		pci_resource_start(rt2x00dev_pci(rt2x00dev), 0),
		pci_resource_len(rt2x00dev_pci(rt2x00dev), 0));
	if (!rt2x00dev->csr_addr) {
		ERROR("Ioremap failed.\n");
		return -ENOMEM;
	}

	/*
	 * Allocate eeprom data.
	 */
	if (rt2500pci_alloc_eeprom(rt2x00dev))
		goto exit;

	/*
	 * Create workqueue.
	 */
	rt2x00dev->workqueue = create_singlethread_workqueue(DRV_NAME);
	if (!rt2x00dev->workqueue)
		goto exit;

	/*
	 * Initialize configuration work.
	 */
	INIT_WORK(&rt2x00dev->link.work, rt2500pci_link_tuner, rt2x00dev);

	/*
	 * Reset current working type.
	 */
	rt2x00dev->interface.type = -EINVAL;

	/*
	 * Intialize scanning attributes.
	 */
	rt2x00dev->scan = NULL;

	/*
	 * Allocate ring array.
	 */
	if (rt2500pci_alloc_rings(rt2x00dev))
		goto exit;

	/*
	 * Initialize hardware.
	 */
	if (rt2500pci_init_eeprom(rt2x00dev) ||
	    rt2500pci_init_hw(rt2x00dev)) {
		ERROR("Failed to initialize device.\n");
		goto exit;
	}

	/*
	 * If required start hardware button polling.
	 */
	rt2500pci_button_start(rt2x00dev);

	return 0;

exit:
	rt2500pci_free_dev(rt2x00dev);

	return -ENODEV;
}

/*
 * PCI driver handlers.
 */
static int rt2500pci_probe(struct pci_dev *pci_dev,
	const struct pci_device_id *id)
{
	struct ieee80211_hw *hw;
	int status;

	status = pci_request_regions(pci_dev, pci_name(pci_dev));
	if (status) {
		ERROR("PCI request regions failed.\n");
		return status;
	}

	status = pci_enable_device(pci_dev);
	if (status) {
		ERROR("Enable device failed.\n");
		goto exit_release_regions;
	}

	pci_set_master(pci_dev);

	if (pci_set_mwi(pci_dev))
		NOTICE("MWI not available.\n");

	if (pci_set_dma_mask(pci_dev, DMA_64BIT_MASK) &&
	    pci_set_dma_mask(pci_dev, DMA_32BIT_MASK)) {
		ERROR("PCI DMA not supported.\n");
		status = -EIO;
		goto exit_disable_device;
	}

	hw = ieee80211_alloc_hw(sizeof(struct rt2x00_dev),
		&rt2500pci_d80211_ops);
	if (!hw) {
		ERROR("Failed to allocate hardware.\n");
		status = -ENOMEM;
		goto exit_disable_device;
	}

	pci_set_drvdata(pci_dev, hw);

	status = rt2500pci_alloc_dev(pci_dev, hw);
	if (status) {
		ERROR("Failed to allocate device.\n");
		goto exit_free_device;
	}

	ieee80211_netif_oper(hw, NETIF_ATTACH);

	return 0;

exit_free_device:
	ieee80211_free_hw(hw);

exit_disable_device:
	if (status != -EBUSY)
		pci_disable_device(pci_dev);

exit_release_regions:
	pci_release_regions(pci_dev);

	pci_set_drvdata(pci_dev, NULL);

	return status;
}

static void rt2500pci_remove(struct pci_dev *pci_dev)
{
	struct ieee80211_hw *hw = pci_get_drvdata(pci_dev);
	struct rt2x00_dev *rt2x00dev = hw->priv;

	ieee80211_netif_oper(hw, NETIF_DETACH);

	/*
	 * Uninitialize and free the rt2500pci driver data.
	 */
	rt2500pci_disable_radio(rt2x00dev);
	rt2500pci_uninitialize(rt2x00dev);
	rt2500pci_free_dev(rt2x00dev);

	/*
	 * Uninitialize and free the 80211 stack data.
	 */
	ieee80211_unregister_hw(hw);
	ieee80211_free_hw(hw);

	/*
	 * Free the PCI device data.
	 */
	pci_set_drvdata(pci_dev, NULL);
	pci_disable_device(pci_dev);
	pci_release_regions(pci_dev);
}

#ifdef CONFIG_PM
static int rt2500pci_suspend(struct pci_dev *pci_dev, pm_message_t state)
{
	struct ieee80211_hw *hw = pci_get_drvdata(pci_dev);
	struct rt2x00_dev *rt2x00dev = hw->priv;
	int status;

	NOTICE("Going to sleep.\n");

	ieee80211_netif_oper(hw, NETIF_DETACH);

	/*
	 * Disable the radio.
	 */
	rt2500pci_disable_radio(rt2x00dev);

	/*
	 * Set device mode to sleep for power management.
	 */
	status = rt2500pci_set_state(rt2x00dev, STATE_SLEEP);
	if (status)
		return status;

	/*
	 * Uninitialize device and hardware.
	 */
	rt2500pci_uninitialize(rt2x00dev);
	rt2500pci_free_dev(rt2x00dev);

	/*
	 * Disable PCI.
	 */
	pci_save_state(pci_dev);
	pci_disable_device(pci_dev);
	return pci_set_power_state(pci_dev, pci_choose_state(pci_dev, state));
}

static int rt2500pci_resume(struct pci_dev *pci_dev)
{
	struct ieee80211_hw *hw = pci_get_drvdata(pci_dev);
	struct rt2x00_dev *rt2x00dev = hw->priv;
	int status;

	NOTICE("Waking up.\n");

	/*
	 * Enable PCI.
	 */
	if (pci_set_power_state(pci_dev, PCI_D0) ||
	    pci_enable_device(pci_dev) ||
	    pci_restore_state(pci_dev)) {
		ERROR("Failed to resume device.\n");
		return -EIO;
	}

	/*
	 * Initialize hardware.
	 */
	status = rt2500pci_alloc_dev(pci_dev, hw);
	if (status) {
		ERROR("Failed to allocate device.\n");
		return status;
	}

	/*
	 * Set device mode to awake for power management.
	 */
	status = rt2500pci_set_state(rt2x00dev, STATE_AWAKE);
	if (status)
		return status;

	ieee80211_netif_oper(hw, NETIF_ATTACH);

	return 0;
}
#endif /* CONFIG_PM */

/*
 * RT2500pci module information.
 */
static char version[] =
	DRV_NAME " - " DRV_VERSION " (" DRV_RELDATE ") by " DRV_PROJECT;

static struct pci_device_id rt2500pci_device_table[] = {
	{ PCI_DEVICE(0x1814, 0x0201) },
	{ 0, }
};

MODULE_AUTHOR(DRV_PROJECT);
MODULE_VERSION(DRV_VERSION);
MODULE_DESCRIPTION("Ralink RT2500 PCI & PCMCIA Wireless LAN driver.");
MODULE_SUPPORTED_DEVICE("Ralink RT2560 PCI & PCMCIA chipset based cards");
MODULE_DEVICE_TABLE(pci, rt2500pci_device_table);
MODULE_LICENSE("GPL");

#ifdef CONFIG_RT2500PCI_DEBUG
module_param_named(debug, rt2x00_debug_level, bool, S_IWUSR | S_IRUGO);
MODULE_PARM_DESC(debug, "Set this parameter to 1 to enable debug output.");
#endif /* CONFIG_RT2500PCI_DEBUG */

static struct pci_driver rt2500pci_driver = {
	.name		= DRV_NAME,
	.id_table	= rt2500pci_device_table,
	.probe		= rt2500pci_probe,
	.remove		= __devexit_p(rt2500pci_remove),
#ifdef CONFIG_PM
	.suspend	= rt2500pci_suspend,
	.resume		= rt2500pci_resume,
#endif /* CONFIG_PM */
};

static int __init rt2500pci_init(void)
{
	printk(KERN_INFO "Loading module: %s.\n", version);
	return pci_register_driver(&rt2500pci_driver);
}

static void __exit rt2500pci_exit(void)
{
	printk(KERN_INFO "Unloading module: %s.\n", version);
	pci_unregister_driver(&rt2500pci_driver);
}

module_init(rt2500pci_init);
module_exit(rt2500pci_exit);
