/*
	Copyright (C) 2004 - 2006 rt2x00 SourceForge Project
	<http://rt2x00.serialmonkey.com>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	Module: rt2x00
	Abstract: rt2x00 global information.
	Supported chipsets: RT2460, RT2560, RT2570,
	rt2561, rt2561s, rt2661 & rt2573.
 */

#ifndef RT2X00_H
#define RT2X00_H

/*
 * Module information.
 */
#ifndef DRV_NAME
#define DRV_NAME	"rt2x00"
#endif /* DRV_NAME */
#define DRV_VERSION	"CVS"
#define DRV_RELDATE	"N/A"
#define DRV_PROJECT	"http://rt2x00.serialmonkey.com"

/*
 * Debug definitions.
 * Debug output has to be enabled during compile time,
 * and should be switched on using the module parameter.
 */
#ifdef CONFIG_RT2X00_DEBUG
/*
 * Module parameter.
 */
static int rt2x00_debug_level = 0;

#define DEBUG_PRINTK(__message...) \
	do { if (rt2x00_debug_level) printk(__message); } while (0)
#else /* CONFIG_RT2X00_DEBUG */
#define DEBUG_PRINTK(__message...) \
	do { } while (0)

#endif /* CONFIG_RT2X00_DEBUG */

/*
 * Hardware button support (through radiobtn)
 * The delay between each poll can be set by the user
 * using the module parameter. Default value is 0,
 * which means polling is disabled.
 */
#ifdef CONFIG_RT2X00_BUTTON
#include <linux/rfkill.h>
#endif /* CONFIG_RT2X00_BUTTON */

/*
 * Various debug levels.
 * The debug levels PANIC and ERROR both indicate serious problems,
 * for this reason they should never be ignored.
 */
#define PANIC(__message, __args...) \
	printk(KERN_PANIC DRV_NAME "->%s: Panic - " __message, \
		__FUNCTION__, ##__args);
#define ERROR(__message, __args...) \
	printk(KERN_ERR DRV_NAME "->%s: Error - " __message, \
		__FUNCTION__, ##__args);
#define WARNING(__message, __args...) \
	DEBUG_PRINTK(KERN_WARNING DRV_NAME "->%s: Warning - " __message, \
		__FUNCTION__, ##__args);
#define NOTICE(__message, __args...) \
	DEBUG_PRINTK(KERN_NOTICE DRV_NAME "->%s: Notice - " __message, \
		__FUNCTION__, ##__args);
#define INFO(__message, __args...) \
	DEBUG_PRINTK(KERN_INFO DRV_NAME "->%s: Info - " __message, \
		__FUNCTION__, ##__args);
#define DEBUG(__message, __args...) \
	DEBUG_PRINTK(KERN_DEBUG DRV_NAME "->%s: Debug - " __message, \
		__FUNCTION__, ##__args);

/*
 * Ring sizes.
 * Ralink PCI devices demand the Frame size to be a multiple of 128 bytes.
 * DATA_FRAME_SIZE is used for TX, RX, ATIM and PRIO rings.
 * MGMT_FRAME_SIZE is used for the BEACON ring.
 */
#define DATA_FRAME_SIZE	2432
#define MGMT_FRAME_SIZE	256

/*
 * Number of entries in a packet ring.
 */
#define RX_ENTRIES	12
#define TX_ENTRIES	12
#define ATIM_ENTRIES	1
#define BEACON_ENTRIES	1

/*
 * Flag handlers
 */
#define SET_FLAG(__dev, __flag)		( (__dev)->flags |= (__flag) )
#define GET_FLAG(__dev, __flag)		( !!((__dev)->flags & (__flag)) )
#define CLEAR_FLAG(__dev, __flag)	( (__dev)->flags &= ~(__flag) )
#define CLEAR_FLAGS(__dev)		( (__dev)->flags = 0 )

/*
 * Standard timing and size defines.
 */
#define ACK_SIZE		14
#define IEEE80211_HEADER	24
#define PLCP			48
#define BEACON			100
#define PREAMBLE		144
#define SHORT_PREAMBLE		72
#define SLOT_TIME		20
#define SHORT_SLOT_TIME		9
#define SIFS			10
#define PIFS			( SIFS + SLOT_TIME )
#define SHORT_PIFS		( SIFS + SHORT_SLOT_TIME )
#define DIFS			( PIFS + SLOT_TIME )
#define SHORT_DIFS		( SHORT_PIFS + SHORT_SLOT_TIME )
#define EIFS			( SIFS + (8 * (IEEE80211_HEADER + ACK_SIZE)) )

/*
 * IEEE802.11 header defines
 */
#define is_rts_frame(__fc) \
	( !!((((__fc) &  IEEE80211_FCTL_FTYPE) == IEEE80211_FTYPE_CTL) && \
	     (((__fc) &  IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_RTS)) )

/*
 * TX result flags.
 */
enum TX_STATUS {
	TX_SUCCESS = 0,
	TX_SUCCESS_RETRY = 1,
	TX_FAIL_RETRY = 2,
	TX_FAIL_INVALID = 3,
	TX_FAIL_OTHER = 4,
};

/*
 * Led mode values.
 */
enum led_mode {
	LED_MODE_DEFAULT = 0,
	LED_MODE_TXRX_ACTIVITY = 1,
	LED_MODE_SIGNAL_STRENGTH = 2,
	LED_MODE_ASUS = 3,
	LED_MODE_ALPHA = 4,
};

/*
 * Device states
 */
enum dev_state {
	STATE_DEEP_SLEEP = 0,
	STATE_SLEEP = 1,
	STATE_STANDBY = 2,
	STATE_AWAKE = 3,
};

/*
 * IFS backoff values
 */
enum ifs {
	IFS_BACKOFF = 0,
	IFS_SIFS = 1,
	IFS_NEW_BACKOFF = 2,
	IFS_NONE = 3,
};

/*
 * Cipher types for hardware encryption
 */
enum cipher {
	CIPHER_NONE = 0,
	CIPHER_WEP64 = 1,
	CIPHER_WEP128 = 2,
	CIPHER_TKIP = 3,
	CIPHER_AES = 4,
/* The following fields were added by rt61pci and rt73usb. */
	CIPHER_CKIP64 = 5,
	CIPHER_CKIP128 = 6,
	CIPHER_TKIP_NO_MIC = 7,
};

/*
 * Macros for determining which is the lowest or highest bit
 * set in a 16 or 32 bit variable.
 */
#define BIT_SET(__val, __bit)	(__val & (1 << __bit))

#define BIT_OK(__val, __bit, __low, __high) \
	(__bit < __low ? 1 : \
	(__bit > __high ? 1 : \
	BIT_SET(__val, __bit) ? 1 : 0))

#define LOWEST_BIT16(__val) \
	(BIT_SET(__val, 0) ? 0 : (BIT_SET(__val, 1) ? 1 : \
	(BIT_SET(__val, 2) ? 2 : (BIT_SET(__val, 3) ? 3 : \
	(BIT_SET(__val, 4) ? 4 : (BIT_SET(__val, 5) ? 5 : \
	(BIT_SET(__val, 6) ? 6 : (BIT_SET(__val, 7) ? 7 : \
	(BIT_SET(__val, 8) ? 8 : (BIT_SET(__val, 9) ? 9 : \
	(BIT_SET(__val, 10) ? 10 : (BIT_SET(__val, 11) ? 11 : \
	(BIT_SET(__val, 12) ? 12 : (BIT_SET(__val, 13) ? 13 : \
	(BIT_SET(__val, 14) ? 14 : (BIT_SET(__val, 15) ? 15 : \
	-EINVAL))))))))))))))))

#define LOWEST_BIT32(__val) \
	(BIT_SET(__val, 0) ? 0 : (BIT_SET(__val, 1) ? 1 : \
	(BIT_SET(__val, 2) ? 2 : (BIT_SET(__val, 3) ? 3 : \
	(BIT_SET(__val, 4) ? 4 : (BIT_SET(__val, 5) ? 5 : \
	(BIT_SET(__val, 6) ? 6 : (BIT_SET(__val, 7) ? 7 : \
	(BIT_SET(__val, 8) ? 8 : (BIT_SET(__val, 9) ? 9 : \
	(BIT_SET(__val, 10) ? 10 : (BIT_SET(__val, 11) ? 11 : \
	(BIT_SET(__val, 12) ? 12 : (BIT_SET(__val, 13) ? 13 : \
	(BIT_SET(__val, 14) ? 14 : (BIT_SET(__val, 15) ? 15 : \
	(BIT_SET(__val, 16) ? 16 : (BIT_SET(__val, 17) ? 17 : \
	(BIT_SET(__val, 18) ? 18 : (BIT_SET(__val, 19) ? 19 : \
	(BIT_SET(__val, 20) ? 20 : (BIT_SET(__val, 21) ? 21 : \
	(BIT_SET(__val, 22) ? 22 : (BIT_SET(__val, 23) ? 23 : \
	(BIT_SET(__val, 24) ? 24 : (BIT_SET(__val, 25) ? 25 : \
	(BIT_SET(__val, 26) ? 27 : (BIT_SET(__val, 27) ? 27 : \
	(BIT_SET(__val, 28) ? 28 : (BIT_SET(__val, 29) ? 29 : \
	(BIT_SET(__val, 30) ? 30 : (BIT_SET(__val, 31) ? 31 : \
	-EINVAL))))))))))))))))))))))))))))))))

#define HIGHEST_BIT16(__val) \
	(BIT_SET(__val, 15) ? 15 : (BIT_SET(__val, 14) ? 14 : \
	(BIT_SET(__val, 13) ? 13 : (BIT_SET(__val, 12) ? 12 : \
	(BIT_SET(__val, 11) ? 11 : (BIT_SET(__val, 10) ? 10 : \
	(BIT_SET(__val, 9) ? 9 : (BIT_SET(__val, 8) ? 8 : \
	(BIT_SET(__val, 7) ? 7 : (BIT_SET(__val, 6) ? 6 : \
	(BIT_SET(__val, 5) ? 5 : (BIT_SET(__val, 4) ? 4 : \
	(BIT_SET(__val, 3) ? 3 : (BIT_SET(__val, 2) ? 2 : \
	(BIT_SET(__val, 1) ? 1 : (BIT_SET(__val, 0) ? 0 : \
	-EINVAL))))))))))))))))

#define HIGHEST_BIT32(__val) \
	(BIT_SET(__val, 31) ? 31 : (BIT_SET(__val, 30) ? 30 : \
	(BIT_SET(__val, 29) ? 29 : (BIT_SET(__val, 28) ? 28 : \
	(BIT_SET(__val, 27) ? 27 : (BIT_SET(__val, 26) ? 26 : \
	(BIT_SET(__val, 25) ? 25 : (BIT_SET(__val, 24) ? 24 : \
	(BIT_SET(__val, 23) ? 23 : (BIT_SET(__val, 22) ? 22 : \
	(BIT_SET(__val, 21) ? 21 : (BIT_SET(__val, 20) ? 20 : \
	(BIT_SET(__val, 19) ? 19 : (BIT_SET(__val, 18) ? 18 : \
	(BIT_SET(__val, 17) ? 17 : (BIT_SET(__val, 16) ? 16 : \
	(BIT_SET(__val, 15) ? 15 : (BIT_SET(__val, 14) ? 14 : \
	(BIT_SET(__val, 13) ? 13 : (BIT_SET(__val, 12) ? 12 : \
	(BIT_SET(__val, 11) ? 11 : (BIT_SET(__val, 10) ? 10 : \
	(BIT_SET(__val, 9) ? 9 : (BIT_SET(__val, 8) ? 8 : \
	(BIT_SET(__val, 7) ? 7 : (BIT_SET(__val, 6) ? 6 : \
	(BIT_SET(__val, 5) ? 5 : (BIT_SET(__val, 4) ? 4 : \
	(BIT_SET(__val, 3) ? 3 : (BIT_SET(__val, 2) ? 2 : \
	(BIT_SET(__val, 1) ? 1 : (BIT_SET(__val, 0) ? 0 : \
	-EINVAL))))))))))))))))))))))))))))))))

#define BITRANGE_OK16(__val, __low, __high) \
	((BIT_OK(__val, 0, __low, __high) && \
	  BIT_OK(__val, 1, __low, __high) && \
	  BIT_OK(__val, 2, __low, __high) && \
	  BIT_OK(__val, 3, __low, __high) && \
	  BIT_OK(__val, 4, __low, __high) && \
	  BIT_OK(__val, 5, __low, __high) && \
	  BIT_OK(__val, 6, __low, __high) && \
	  BIT_OK(__val, 7, __low, __high) && \
	  BIT_OK(__val, 8, __low, __high) && \
	  BIT_OK(__val, 9, __low, __high) && \
	  BIT_OK(__val, 10, __low, __high) && \
	  BIT_OK(__val, 11, __low, __high) && \
	  BIT_OK(__val, 12, __low, __high) && \
	  BIT_OK(__val, 13, __low, __high) && \
	  BIT_OK(__val, 14, __low, __high) && \
	  BIT_OK(__val, 15, __low, __high)) ? \
	0 : -EINVAL)

#define BITRANGE_OK32(__val, __low, __high) \
	((BIT_OK(__val, 0, __low, __high) && \
	  BIT_OK(__val, 1, __low, __high) && \
	  BIT_OK(__val, 2, __low, __high) && \
	  BIT_OK(__val, 3, __low, __high) && \
	  BIT_OK(__val, 4, __low, __high) && \
	  BIT_OK(__val, 5, __low, __high) && \
	  BIT_OK(__val, 6, __low, __high) && \
	  BIT_OK(__val, 7, __low, __high) && \
	  BIT_OK(__val, 8, __low, __high) && \
	  BIT_OK(__val, 9, __low, __high) && \
	  BIT_OK(__val, 10, __low, __high) && \
	  BIT_OK(__val, 11, __low, __high) && \
	  BIT_OK(__val, 12, __low, __high) && \
	  BIT_OK(__val, 13, __low, __high) && \
	  BIT_OK(__val, 14, __low, __high) && \
	  BIT_OK(__val, 15, __low, __high) && \
	  BIT_OK(__val, 16, __low, __high) && \
	  BIT_OK(__val, 17, __low, __high) && \
	  BIT_OK(__val, 18, __low, __high) && \
	  BIT_OK(__val, 19, __low, __high) && \
	  BIT_OK(__val, 20, __low, __high) && \
	  BIT_OK(__val, 21, __low, __high) && \
	  BIT_OK(__val, 22, __low, __high) && \
	  BIT_OK(__val, 23, __low, __high) && \
	  BIT_OK(__val, 24, __low, __high) && \
	  BIT_OK(__val, 25, __low, __high) && \
	  BIT_OK(__val, 26, __low, __high) && \
	  BIT_OK(__val, 27, __low, __high) && \
	  BIT_OK(__val, 28, __low, __high) && \
	  BIT_OK(__val, 29, __low, __high) && \
	  BIT_OK(__val, 30, __low, __high) && \
	  BIT_OK(__val, 31, __low, __high)) ? \
	0 : -EINVAL)

extern int error_lowest_bit_not_constant;
extern int error_highest_bit_not_constant;
extern int error_bitrange_not_constant;
extern int error_bitrange_bad;

#define BUILD_LOWEST_BIT16(__val) \
 	(!__builtin_constant_p(__val) ? error_lowest_bit_not_constant : \
	LOWEST_BIT16(__val))

#define BUILD_LOWEST_BIT32(__val) \
 	(!__builtin_constant_p(__val) ? error_lowest_bit_not_constant : \
	LOWEST_BIT32(__val))

#define BUILD_HIGHEST_BIT16(__val) \
 	(!__builtin_constant_p(__val) ? error_highest_bit_not_constant : \
	HIGHEST_BIT16(__val))

#define BUILD_HIGHEST_BIT32(__val) \
 	(!__builtin_constant_p(__val) ? error_highest_bit_not_constant : \
	HIGHEST_BIT32(__val))

#define BUILD_BITRANGE_OK16(__val, __low, __high) \
	((!__builtin_constant_p(__val) || !__builtin_constant_p(__low) || \
	  !__builtin_constant_p(__high)) ? error_bitrange_not_constant : \
	BITRANGE_OK16(__val, __low, __high))

#define BUILD_BITRANGE_OK32(__val, __low, __high) \
	((!__builtin_constant_p(__val) || !__builtin_constant_p(__low) || \
	  !__builtin_constant_p(__high)) ? error_bitrange_not_constant : \
	BITRANGE_OK16(__val, __low, __high))

/*
 * Register handlers.
 * We store the position of a register field inside a field structure,
 * This will simplify the process of setting and reading a certain field
 * inside the register while making sure the process remains byte order safe.
 * Before setting the value into the structure we use macros to determine
 * whether all bits in the field are contineous and valid.
 * These additional checks will be optimized away at compile time,
 * but do have a major impact on compile speed, therefor we only make this
 * check when compiling with debug enabled.
 */
struct rt2x00_field16 {
	u16 bit_offset;
	u16 bit_mask;
};

struct rt2x00_field32 {
	u32 bit_offset;
	u32 bit_mask;
};

/*
 * Before intitializing the rt2x00_field# structures,
 * we will check if the bitmask is correct and does
 * not contain any gaps.
 * This check is only done in debug mode, since it severely
 * impacts compilation speed.
 */
#ifdef CONFIG_RT2X00_DEBUG
#define FIELD16(__mask) \
	( (struct rt2x00_field16) { \
		(BUILD_BITRANGE_OK16(__mask, BUILD_LOWEST_BIT16(__mask), \
			BUILD_HIGHEST_BIT16(__mask)) == 0) ? \
		BUILD_LOWEST_BIT16(__mask) : error_bitrange_bad, \
		(__mask) \
	} )

#define FIELD32(__mask) \
	( (struct rt2x00_field32) { \
		(BUILD_BITRANGE_OK32(__mask, BUILD_LOWEST_BIT32(__mask), \
			BUILD_HIGHEST_BIT32(__mask)) == 0) ? \
		BUILD_LOWEST_BIT32(__mask) : error_bitrange_bad, \
		(__mask) \
	} )
#else /* CONFIG_RT2X00_DEBUG */
#define FIELD16(__mask) \
	( (struct rt2x00_field16) { \
		BUILD_LOWEST_BIT16(__mask), (__mask) \
	} )

#define FIELD32(__mask) \
	( (struct rt2x00_field32) { \
		BUILD_LOWEST_BIT32(__mask), (__mask) \
	} )
#endif /* CONFIG_RT2X00_DEBUG */

static inline void rt2x00_set_field32(u32 *reg,
	const struct rt2x00_field32 field, const u32 value)
{
	*reg &= ~(field.bit_mask);
	*reg |= (value << field.bit_offset) & field.bit_mask;
}

static inline u32 rt2x00_get_field32(const u32 reg,
	const struct rt2x00_field32 field)
{
	return (reg & field.bit_mask) >> field.bit_offset;
}

static inline void rt2x00_set_field16(u16 *reg,
	const struct rt2x00_field16 field, const u16 value)
{
	*reg &= ~(field.bit_mask);
	*reg |= (value << field.bit_offset) & field.bit_mask;
}

static inline u16 rt2x00_get_field16(const u16 reg,
	const struct rt2x00_field16 field)
{
	return (reg & field.bit_mask) >> field.bit_offset;
}

/*
 * Chipset identification
 * The chipset on the device is composed of a RT and RF chip.
 * The chipset combination is important for determining device capabilities.
 */
struct rt2x00_chip {
	u16 rt;
	u16 rf;
	u32 rev;
	u8 fw_h;
	u8 fw_l;
};

/*
 * Set chipset data.
 */
static inline void rt2x00_set_chip(struct rt2x00_chip *chipset,
	const u16 rt, const u16 rf, const u32 rev)
{
	INFO("Chipset detected - rt: %04x, rf: %04x, rev: %08x.\n",
		rt, rf, rev);

	chipset->rt = rt;
	chipset->rf = rf;
	chipset->rev = rev;
}

static inline void rt2x00_set_chip_fw(struct rt2x00_chip *chipset,
	const u8 fw_h, const u8 fw_l)
{
	INFO("Firmware detected - version: %d.%d.\n", fw_h, fw_l);

	chipset->fw_h = fw_h;
	chipset->fw_l = fw_l;
}

static inline char rt2x00_rt(const struct rt2x00_chip *chipset, const u16 chip)
{
	return (chipset->rt == chip);
}

static inline char rt2x00_rf(const struct rt2x00_chip *chipset, const u16 chip)
{
	return (chipset->rf == chip);
}

static inline u16 rt2x00_rev(const struct rt2x00_chip *chipset)
{
	return chipset->rev;
}

static inline char* rt2x00_fw(const struct rt2x00_chip *chipset)
{
	return chipset->fw_h + "." + chipset->fw_l;
}

/*
 * data_desc
 * Each data entry also contains a descriptor which is used by the
 * device to determine what should be done with the packet and
 * what the current status is.
 * This structure is greatly simplified, but the descriptors
 * are basically a list of little endian 32 bit values.
 * Make the array by default 1 word big, this will allow us
 * to use sizeof() correctly.
 */
struct data_desc {
	__le32 word[1];
};

/*
 * data_entry
 * The data ring is a list of data entries.
 * Each entry holds a reference to the descriptor
 * and the data buffer. For TX rings the reference to the
 * sk_buff of the packet being transmitted is also stored here.
 */
struct data_entry {
	/*
	 * Status flags
	 */
	unsigned int flags;
#define ENTRY_OWNER_NIC	0x00000001
#define ENTRY_RTS_FRAME	0x00000002

	/*
	 * Ring we belong to.
	 */
	struct data_ring *ring;

	/*
	 * sk_buff for the packet which is being transmitted
	 * in this entry (Only used with TX related rings).
	 */
	struct sk_buff *skb;

	/*
	 * Store a ieee80211_tx_status structure in each
	 * ring entry, this will optimize the txdone
	 * handler.
	 */
	struct ieee80211_tx_status tx_status;

	/*
	 * private pointer specific to driver.
	 */
	void *priv;

	/*
	 * Data address for this entry.
	 */
	void *data_addr;
	dma_addr_t data_dma;
};

/*
 * data_ring
 * Data rings are used by the device to send and receive packets.
 * The data_addr is the base address of the data memory.
 * To determine at which point in the ring we are,
 * have to use the rt2x00_ring_index_*() functions.
 */
struct data_ring {
	/*
	 * Pointer to main rt2x00dev structure where this
	 * ring belongs to.
	 */
	struct rt2x00_dev *rt2x00dev;

	/*
	 * Work structure for bottom half interrupt handling.
	 */
	struct work_struct irq_work;

	/*
	 * Base address for the device specific data entries.
	 */
	struct data_entry *entry;

	/*
	 * TX queue statistic info.
	 */
	struct ieee80211_tx_queue_stats_data stats;

	/*
	 * TX Queue parameters.
	 */
	struct ieee80211_tx_queue_params tx_params;

	/*
	 * Base address for data ring.
	 */
	dma_addr_t data_dma;
	void *data_addr;

	/*
	 * Index variables.
	 */
	u8 index;
	u8 index_done;

	/*
	 * Ring type.
	 */
	u16 type;

	/*
	 * Size of packet and descriptor in bytes.
	 */
	u16 data_size;
	u16 desc_size;
};

/*
 * Handlers to determine the address of the current device specific
 * data entry, where either index or index_done points to.
 */
static inline struct data_entry* rt2x00_get_data_entry(
	struct data_ring *ring)
{
	return &ring->entry[ring->index];
}

static inline struct data_entry* rt2x00_get_data_entry_done(
	struct data_ring *ring)
{
	return &ring->entry[ring->index_done];
}

/*
 * Total ring memory
 */
static inline int rt2x00_get_ring_size(struct data_ring *ring)
{
	return ring->stats.limit * (ring->desc_size + ring->data_size);
}

/*
 * Ring index manipulation functions.
 */
static inline void rt2x00_ring_index_inc(struct data_ring *ring)
{
	ring->index++;
	if (ring->index >= ring->stats.limit)
		ring->index = 0;
	ring->stats.len++;
}

static inline void rt2x00_ring_index_done_inc(struct data_ring *ring)
{
	ring->index_done++;
	if (ring->index_done >= ring->stats.limit)
		ring->index_done = 0;
	ring->stats.len--;
	ring->stats.count++;
}

static inline void rt2x00_ring_index_clear(struct data_ring *ring)
{
	ring->index = 0;
	ring->index_done = 0;
	ring->stats.len = 0;
	ring->stats.count = 0;
}

static inline int rt2x00_ring_empty(struct data_ring *ring)
{
	return ring->stats.len == 0;
}

static inline int rt2x00_ring_full(struct data_ring *ring)
{
	return ring->stats.len == ring->stats.limit;
}

/*
 * To optimize the quality of the link we need to store
 * the quality of received frames and periodically
 * optimize the link.
 */
struct link {
	/*
	 * RSSI statistics.
	 */
	u32 count_rssi;
	u32 total_rssi;

	/*
	 * Noise statistics.
	 */
	u32 curr_noise;

	/*
	 * Work structure for scheduling periodic link tuning.
	 */
	struct work_struct work;
};

static inline void rt2x00_start_link_tune(struct link *link)
{
	link->count_rssi = 0;
	link->total_rssi = 0;
	link->curr_noise = 0;
}

static inline void rt2x00_update_link_rssi(struct link *link, u32 rssi)
{
	link->count_rssi++;
	link->total_rssi += rssi;
}

static inline void rt2x00_update_link_noise(struct link *link, u32 noise)
{
	link->curr_noise = noise;
}

static inline u32 rt2x00_get_link_rssi(struct link *link)
{
	u32 average = 0;

	if (link->count_rssi && link->total_rssi)
		average = link->total_rssi / link->count_rssi;

	link->count_rssi = 0;
	link->total_rssi = 0;

	return average;
}

static inline u32 rt2x00_get_link_noise(struct link *link)
{
	return link->curr_noise;
}

/*
 * Interface structure
 * Configuration details about the current interface.
 */
struct interface {
	/*
	 * Interface identification. The value is assigned
	 * to us by the 80211 stack, and is used to request
	 * new beacons.
	 */
	int id;

	/*
	 * Current working type (IEEE80211_IF_TYPE_*).
	 * This excludes the type IEEE80211_IF_TYPE_MNTR
	 * since that is counted seperately in the monitor_count
	 * field.
	 */
	int type;

	/*
	 * BBSID of the AP to associate with.
	 */
	u8 bssid[ETH_ALEN];

	/*
	 * Store the promisc mode for the current interface.
	 * monitor mode always forces promisc mode to be enabled,
	 * so we need to store the promisc mode seperately.
	 */
	short promisc;

	/*
	 * Monitor mode count, the number of interfaces
	 * in monitor mode that that have been added.
	 */
	short monitor_count;

	/*
	 * Sequence number for software controled sequence counters.
	 */
	u16 sequence;
};

static inline void rt2x00_add_interface(struct interface *intf,
	struct ieee80211_if_init_conf *conf)
{
	/*
	 * We support muliple monitor mode interfaces.
	 * All we need to do is increase the monitor_count.
	 */
	if (conf->type == IEEE80211_IF_TYPE_MNTR) {
		intf->monitor_count++;
	} else {
		intf->id = conf->if_id;
		intf->type = conf->type;
		if (conf->type == IEEE80211_IF_TYPE_AP)
			memcpy(&intf->bssid, conf->mac_addr, ETH_ALEN);
		intf->promisc = 0;
		intf->sequence = 0;
	}
}

static inline void rt2x00_remove_interface(struct interface *intf,
	struct ieee80211_if_init_conf *conf)
{
	/*
	 * We support muliple monitor mode interfaces.
	 * All we need to do is decrease the monitor_count.
	 */
	if (conf->type == IEEE80211_IF_TYPE_MNTR) {
		intf->monitor_count--;
	} else if (intf->type == conf->type) {
		intf->id = 0;
		intf->type = -EINVAL;
		memset(&intf->bssid, 0x00, ETH_ALEN);
		intf->promisc = 0;
		intf->sequence = 0;
	}
}

/*
 * Scanning structure.
 * Swithing channel during scanning will be put
 * in a workqueue so we will be able to sleep
 * during the switch.
 * We also make use of the completion structure
 * in case a frame must be send before a
 * channel switch.
 */
struct scanning {
	/*
	 * Completion structure if an packet needs to be send.
	 */
	struct completion completion;

	/*
	 * Scanning parameters.
	 */
	struct ieee80211_scan_conf conf;

	/*
	 * Scanning state: IEEE80211_SCAN_START or IEEE80211_SCAN_END.
	 */
	short state;

	/*
	 * Flag to see if this scan has been cancelled.
	 */
	short status;
#define SCANNING_READY		0x0000
#define SCANNING_WAITING	0x0001
#define SCANNING_CANCELLED	0x0002

	/*
	 * Work structure for scheduling the scanning work.
	 */
	struct work_struct work;
};

static inline void rt2x00_start_scan(struct scanning *scan,
	struct ieee80211_scan_conf *conf, int state)
{
	init_completion(&scan->completion);

	memcpy(&scan->conf, conf, sizeof(*conf));

	scan->state = state;

	/*
	 * Set initial status to SCANNING_WAITING to prevent scanning
	 * to begin while there are still TX packets queued.
	 */
	scan->status = SCANNING_WAITING;
}

static inline void rt2x00_signal_scan(struct scanning *scan, short status)
{
	scan->status = status;

	complete_all(&scan->completion);
}

static inline int rt2x00_wait_scan(struct scanning *scan)
{
	/*
	 * Only wait for completion when the status
	 * indicates we should. The SCANNING_READY
	 * and SCANNING_CANCELLED are both states
	 * which indicate complete_all has already
	 * been called.
	 */
	if (scan->status == SCANNING_WAITING)
		wait_for_completion(&scan->completion);

	/*
	 * Status field will have been updated by the handler
	 * that has called complete_all() on our complete structure.
	 */
	return scan->status;
}

/*
 * rt2x00 device structure.
 */
struct rt2x00_dev {
	/*
	 * Device structure.
	 * The structure stored in here depends on the
	 * system bus (PCI or USB).
	 * When accessing this variable, the rt2x00dev_{pci,usb}
	 * macro's should be used for correct typecasting.
	 */
	void *dev;
#define rt2x00dev_pci(__dev)	( (struct pci_dev*)(__dev)->dev )
#define rt2x00dev_usb(__dev)	( (struct usb_interface*)(__dev)->dev )

	/*
	 * IEEE80211 control structure.
	 */
	struct ieee80211_hw *hw;

	/*
	 * Device flags.
	 * In these flags the current status and some
	 * of the device capabilities are stored.
	 */
	unsigned int flags;
#define DEVICE_ENABLED_RADIO		0x00000001
#define DEVICE_ENABLED_RADIO_HW		0x00000002
#define DEVICE_INITIALIZED		0x00000004
#define DEVICE_INITIALIZED_HW		0x00000008

#define FIRMWARE_LOADED			0x00000020
#define FIRMWARE_FAILED			0x00000040
#define INTERFACE_INITIALIZED		0x00000080
#define INTERFACE_INITIALIZED_MONITOR	0x00000100
#define INTERFACE_ENABLED		0x00000200
#define INTERFACE_ENABLED_MONITOR	0x00000400
#define INTERFACE_ENABLED_PROMISC	0x00000800
#define DEVICE_SUPPORT_ATIM		0x00001000
#define DEVICE_SUPPORT_HW_BUTTON	0x00002000
#define CONFIG_FRAME_TYPE		0x00004000
#define CONFIG_RF_SEQUENCE		0x00008000
#define CONFIG_EXTERNAL_LNA		0x00010000
#define CONFIG_EXTERNAL_LNA_A		0x00020000
#define CONFIG_EXTERNAL_LNA_BG		0x00040000
#define CONFIG_DOUBLE_ANTENNA		0x00080000
#define CONFIG_DISABLE_LINK_TUNING	0x00100000

	/*
	 * Chipset identification.
	 */
	struct rt2x00_chip chip;

	/*
	 * Base address of device registers (PCI devices only).
	 */
	void __iomem *csr_addr;

	/*
	 * If enabled, the structure for the
	 * hardware button control.
	 */
#ifdef CONFIG_RT2X00_BUTTON
	struct rfkill rfkill;
#endif /* CONFIG_RT2X00_BUTTON */

	/*
	 * Queue for deferred work.
	 */
	struct workqueue_struct *workqueue;

	/*
	 * Interface configuration.
	 */
	struct interface interface;

	/*
	 * Link quality
	 */
	struct link link;

	/*
	 * EEPROM data.
	 */
	__le16 *eeprom;

	/*
	 * Active RF register values.
	 * These are stored here for easier working
	 * with the rf registers.
	 */
	u32 rf1;
	u32 rf2;
	u32 rf3;
	u32 rf4;

	/*
	 * Current TX power value.
	 */
	u16 tx_power;

	/*
	 * LED register (for rt61pci & rt73usb).
	 */
	u16 led_reg;

	/*
	 * Led mode (LED_MODE_*)
	 */
	u8 led_mode;

	/*
	 * EEPROM bus width (PCI devices only).
	 */
	u8 eeprom_width;

	/*
	 * False CCA counter. (for rt2400pci).
	 */
	u8 false_cca;

	/*
	 * Frequency offset (for rt61pci & rt73usb).
	 */
	u8 freq_offset;

	/*
	 * Low level statistics which will have
	 * to be kept up to date while device is running.
	 */
	struct ieee80211_low_level_stats low_level_stats;

	/*
	 * RX configuration information.
	 */
	struct ieee80211_rx_status rx_params;

	/*
	 * Scanning structure.
	 */
	struct scanning *scan;

	/*
	 * Data rings for both RX and TX.
	 * The first entries must be the normal TX
	 * rings, followed by a possible ATIM ring
	 * (when atim is used atim_available must be set)
	 * after that the beacon and RX ring follow.
	 */
	struct data_ring *ring;
};

static inline struct data_ring* rt2x00_get_ring(
	struct rt2x00_dev *rt2x00dev, const int ring)
{
	int atim;

	atim = GET_FLAG(rt2x00dev, DEVICE_SUPPORT_ATIM);

	/*
	 * Check if the rings have been allocated.
	 */
	if (!rt2x00dev->ring)
		return NULL;

	/*
	 * Check for beacon ring, the beacon ring
	 * is located behing the normal TX and, when available
	 * the atim ring.
	 */
	if (ring == IEEE80211_TX_QUEUE_BEACON)
		return &rt2x00dev->ring[rt2x00dev->hw->queues + atim];

	if (ring == IEEE80211_TX_QUEUE_AFTER_BEACON && atim)
		return &rt2x00dev->ring[rt2x00dev->hw->queues];

	/*
	 * Make sure the requested ring does not exceed
	 * the maximum number of rings.
	 */
	if (ring < rt2x00dev->hw->queues)
		return &rt2x00dev->ring[ring];

	return NULL;
}

static inline void rt2x00_update_tx_stats(struct rt2x00_dev *rt2x00dev,
	struct ieee80211_tx_status *tx_status, const int status,
	const int is_ack, const int is_rts)
{
	struct ieee80211_low_level_stats *stats = &rt2x00dev->low_level_stats;

	if (is_ack)
		tx_status->flags |= IEEE80211_TX_STATUS_ACK;
	tx_status->excessive_retries = (status == TX_FAIL_RETRY);

	if (!is_ack && status == TX_FAIL_RETRY)
		stats->dot11ACKFailureCount++;

	if (is_rts) {
		if (status == TX_SUCCESS || status == TX_SUCCESS_RETRY)
			stats->dot11RTSSuccessCount++;
		else
			stats->dot11RTSFailureCount++;
	}
}

/*
 * Device specific rate value.
 * We will have to create the device specific rate value
 * passed to the ieee80211 kernel. We need to make it a consist of
 * multiple fields because we want to store more then 1 device specific
 * values inside the value.
 *	1 - rate, stored as 100 kbit/s.
 *	2 - preamble, short_preamble enabled flag.
 *	3 - MASK_RATE, which rates are enabled in this mode, this mask
 *	corresponds with the TX register format for the current device.
 *	4 - plcp, 802.11b rates are device specific,
 *	802.11g rates are set according to the ieee802.11a-1999 p.14.
 * The bit to enable preamble is set in a seperate define.
 */
#define DEV_RATE	FIELD32(0x000007ff)
#define DEV_PREAMBLE	FIELD32(0x00000800)
#define DEV_RATEMASK	FIELD32(0x00fff000)
#define DEV_PLCP	FIELD32(0xff000000)

/*
 * Bitmask for MASK_RATE
 */
#define DEV_RATE_1MB	0x00000001
#define DEV_RATE_2MB	0x00000002
#define DEV_RATE_5_5MB	0x00000004
#define DEV_RATE_11MB	0x00000008
#define DEV_RATE_6MB	0x00000010
#define DEV_RATE_9MB	0x00000020
#define DEV_RATE_12MB	0x00000040
#define DEV_RATE_18MB	0x00000080
#define DEV_RATE_24MB	0x00000100
#define DEV_RATE_36MB	0x00000200
#define DEV_RATE_48MB	0x00000400
#define DEV_RATE_54MB	0x00000800

/*
 * Bitmask groups of bitrates
 */
#define DEV_BASIC_RATE \
	( DEV_RATE_1MB | DEV_RATE_2MB | DEV_RATE_5_5MB | DEV_RATE_11MB | \
	  DEV_RATE_6MB | DEV_RATE_12MB | DEV_RATE_24MB )

#define DEV_CCK_RATE \
	( DEV_RATE_1MB | DEV_RATE_2MB | DEV_RATE_5_5MB | DEV_RATE_11MB )

#define DEV_OFDM_RATE \
	( DEV_RATE_6MB | DEV_RATE_9MB | DEV_RATE_12MB | DEV_RATE_18MB | \
	  DEV_RATE_24MB | DEV_RATE_36MB | DEV_RATE_48MB | DEV_RATE_54MB )

/*
 * Macro's to set and get specific fields from the device specific val and val2
 * fields inside the ieee80211_rate entry.
 */
#define DEVICE_SET_RATE_FIELD(__value, __mask) \
	(int)( ((__value) << DEV_##__mask.bit_offset) & DEV_##__mask.bit_mask )

#define DEVICE_GET_RATE_FIELD(__value, __mask) \
	(int)( ((__value) & DEV_##__mask.bit_mask) >> DEV_##__mask.bit_offset )

/*
 * Initialize a ieee80211_entry by filling in all fields and correctly
 * construct the device specific val and val2 fields.
 */
static inline void device_rate_entry(struct ieee80211_rate *entry,
	int rate, int mask, int plcp, int flags)
{
	entry->rate = rate;
	entry->val = DEVICE_SET_RATE_FIELD(rate, RATE) |
		DEVICE_SET_RATE_FIELD(mask, RATEMASK) |
		DEVICE_SET_RATE_FIELD(plcp, PLCP);
	entry->flags = flags;
	entry->val2 = entry->val |
		DEVICE_SET_RATE_FIELD(entry->flags & IEEE80211_RATE_PREAMBLE2,
				      PREAMBLE);
	entry->min_rssi_ack = 0;
	entry->min_rssi_ack_delta = 0;
}

static inline int device_signal_to_rate(struct ieee80211_hw_modes *mode,
	u8 signal, u8 ofdm)
{
	unsigned int i;
	u8 plcp;
	u8 rate;

	for (i = 0; i < mode->num_rates; i++) {
		/*
		 * When frame was received with an OFDM bitrate,
		 * the signal is the PLCP value. If it was received with
		 * a CCK bitrate the signal is the rate in 0.5kbit/s.
		 */
		if (!ofdm) {
			rate = DEVICE_GET_RATE_FIELD(mode->rates[i].val, RATE);
			if (rate == signal)
				return mode->rates[i].val;
		} else {
			plcp = DEVICE_GET_RATE_FIELD(mode->rates[i].val, PLCP);
			if (plcp == signal) {
				/*
				 * Check for preamble bit.
				 */
				if (signal & 0x08)
					return mode->rates[i].val2;
				return mode->rates[i].val;
			}
		}
	}

	return 0;
}

/*
 * Sequence number handlers.
 */
static inline int rt2x00_require_sequence(struct ieee80211_hdr *ieee80211hdr)
{
	/*
	 * Check if the packet has a header
	 * that contains a seq_ctrl field.
	 */
	if (ieee80211_get_hdrlen(
		le16_to_cpu(ieee80211hdr->frame_control)) >= IEEE80211_HEADER)
		return 1;
	return 0;
}

static inline void rt2x00_set_sequence(struct sk_buff *skb, u16 *seq)
{
	struct ieee80211_hdr *ieee80211hdr =
		(struct ieee80211_hdr*)skb->data;

	/*
	 * Only attach the sequence number, when
	 * the buffer requires it.
	 */
	if (!rt2x00_require_sequence(ieee80211hdr))
		return;

	/*
	 * Set the sequence number for this frame,
	 * but do not change the fragment number.
	 */
	ieee80211hdr->seq_ctrl |= cpu_to_le16(*seq & IEEE80211_SCTL_SEQ);

	/*
	 * Since the dscape stack sends all packets and
	 * fragments individually, we need to check if there
	 * will be new fragments following.
	 * If this is the last or only fragment, the sequence
	 * counter should be increased.
	 */
	if (!ieee80211_get_morefrag(ieee80211hdr))
		*seq = (*seq + 0x10) & IEEE80211_SCTL_SEQ;
}

/*
 * Duration calculations
 * The rate variable passed is: 100kbs.
 * To convert from bytes to bits we multiply size with 8,
 * then the size is multiplied with 10 to make the
 * real rate -> rate argument correction.
 */
static inline u16 get_duration(const unsigned int size, const u8 rate)
{
	return ((size * 8 * 10) / rate);
}

static inline u16 get_duration_res(const unsigned int size, const u8 rate)
{
	return ((size * 8 * 10) % rate);
}

#endif /* RT2X00_H */
