/*
	Copyright (C) 2006 Ivo van Doorn

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	RF button support
	Laptops are quite often equipped with an RF key to enable or
	disable the radio of the wireless device attached to that key.
	This wireless device usually is an integrated wireless network device,
	infrared or bluetooth device.
	There are 3 catagories of radio control keys:
	 A) The key directly toggles the hardware radio, and does not send an
	    event to userspace.
	 B) The key does not toggle the hardware radio, and does not send an
	    event to userspace.
	 C) The key does not toggle the hardware radio, but does send an event
	    to userspace.
	Catagory (A) should create an input device themselves and send an
	KEY_RFKILL event over that input device.
	Catagory (B) should register themselves with rkfkill and allow rfkill
	to toggle the radio and report events to userspace.
	Catagory (C) should register with rfkill, who will listen to userspace
	requests to toggle the radio and will send the signal to the driver.

	The rfkill driver will contain a list of all devices with an RF button,
	and hardware drivers need to register their hardware to the rfkill
	interface. Rfkill will then take care of everything. If the RF key
	requires polling to obtain the status this will be handled by rfkill.
	If the RF key does not require polling but sends for example interrupts,
	the hardware driver can report the change of status to rfkill, without
	having to do any other action.

	For each registered hardware button an input device will be created.
	If this input device has been opened by the user, rfkill will send a
	signal to userspace instead of the hardware about the new button
	status. This will allow userpace to perform the correct steps
	in order to bring down all interfaces.

	Through sysfs it is also possible the user requests the toggling of
	the radio, this means that the radio could be toggled even without
	pressing the radio key.
 */

#ifndef RFKILL_H
#define RFKILL_H

#include <linux/device.h>

/*
 * The keys will be periodically checked if the
 * key has been toggled. By default this will be 100ms.
 */
#define RFKILL_POLL_DELAY	( HZ / 10 )

/**
 * enum key_type - Key type for rfkill keys.
 * KEY_TYPE_WLAN: key type for Wireless network devices.
 * KEY_TYPE_BlUETOOTH: key type for bluetooth devices.
 * KEY_TYPE_IRDA: key type for infrared devices.
 */
enum key_type {
	KEY_TYPE_WLAN = 0,
	KEY_TYPE_BlUETOOTH = 1,
	KEY_TYPE_IRDA = 2,
	KEY_TYPE_MAX = 3,
};

/**
 * enum user_claim - Users' claim on rfkill key.
 * @USER_CLAIM_IGNORE: Don't notify user of key events.
 * @USER_CLAIM_NOTIFY: Notify user of key events, but
 * 	still automatically toggle the radio.
 * @USER_CLAIM_SINGLE: Notify user of key events, and
 * 	do not toggle the radio anymore.
 */
enum user_claim {
	USER_CLAIM_IGNORE = 0,
	USER_CLAIM_NOTIFY = 1,
	USER_CLAIM_SINGLE = 2,
	USER_CLAIM_MAX = 3,
};

/**
 * struct rfkill - rfkill button control structure.
 * @dev_name: Name of the interface. This will become the name
 * 	of the input device which will be created for this button.
 * @dev: Pointer to the device structure to which this button belongs to.
 * @data: Pointer to the RF button drivers private data which will be
 * 	passed along with the radio and polling handlers.
 * @poll(void *data): Optional handler which will frequently be
 * 	called to determine the current status of the RF button.
 * @enable_radio(void *data): Optional handler to enable the radio
 * 	once the RF button has been pressed and the hardware does enable
 * 	the radio automaticly.
 * @disable_radio(void *data): Optional handler to disable the radio
 * 	once the RF button has been pressed and the hardware does disable
 * 	the radio automaticly.
 * @key_type: Radio type which the button controls, the value stored
 * 	here should be a value from enum key_type.
 * @key: Internal pointer that should not be touched by key driver.
 *
 * This structure can be used by a key driver to register the key
 * to the rfkill driver in order to take control of the reporting
 * to userspace or handling of radio status.
 */
struct rfkill {
	char *dev_name;

	struct device *dev;

	void *data;
	int (*poll)(void *data);
	void (*enable_radio)(void *data);
	void (*disable_radio)(void *data);

	unsigned int key_type;

	struct rfkill_key *key;
};

int rfkill_register_key(struct rfkill *rfkill, int init_status);
int rfkill_deregister_key(struct rfkill *rfkill);
void rfkill_report_event(struct rfkill *rfkill, int new_status);

#endif /* RFKILL_H */
