# Copyright 2015 Google Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

require 'google/apis/core/base_service'
require 'google/apis/core/json_representation'
require 'google/apis/core/hashable'
require 'google/apis/errors'

module Google
  module Apis
    module PeopleV1
      # Google People API
      #
      # Provides access to information about profiles and contacts.
      #
      # @example
      #    require 'google/apis/people_v1'
      #
      #    People = Google::Apis::PeopleV1 # Alias the module
      #    service = People::PeopleServiceService.new
      #
      # @see https://developers.google.com/people/
      class PeopleServiceService < Google::Apis::Core::BaseService
        # @return [String]
        #  Available to use for quota purposes for server-side applications. Can be any
        #  arbitrary string assigned to a user, but should not exceed 40 characters.
        attr_accessor :quota_user

        # @return [String]
        #  API key. Your API key identifies your project and provides you with API access,
        #  quota, and reports. Required unless you provide an OAuth 2.0 token.
        attr_accessor :key

        def initialize
          super('https://people.googleapis.com/', '')
          @batch_path = 'batch'
        end
        
        # Delete an existing contact group owned by the authenticated user by
        # specifying a contact group resource name.
        # @param [String] resource_name
        #   The resource name of the contact group to delete.
        # @param [Boolean] delete_contacts
        #   Set to true to also delete the contacts in the specified group.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::PeopleV1::Empty] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::PeopleV1::Empty]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_contact_group(resource_name, delete_contacts: nil, quota_user: nil, fields: nil, options: nil, &block)
          command =  make_simple_command(:delete, 'v1/{+resourceName}', options)
          command.response_representation = Google::Apis::PeopleV1::Empty::Representation
          command.response_class = Google::Apis::PeopleV1::Empty
          command.params['resourceName'] = resource_name unless resource_name.nil?
          command.query['deleteContacts'] = delete_contacts unless delete_contacts.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['fields'] = fields unless fields.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Get a list of contact groups owned by the authenticated user by specifying
        # a list of contact group resource names.
        # @param [Array<String>, String] resource_names
        #   The resource names of the contact groups to get.
        # @param [Fixnum] max_members
        #   Specifies the maximum number of members to return for each group.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::PeopleV1::BatchGetContactGroupsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::PeopleV1::BatchGetContactGroupsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def batch_contact_group_get(resource_names: nil, max_members: nil, quota_user: nil, fields: nil, options: nil, &block)
          command =  make_simple_command(:get, 'v1/contactGroups:batchGet', options)
          command.response_representation = Google::Apis::PeopleV1::BatchGetContactGroupsResponse::Representation
          command.response_class = Google::Apis::PeopleV1::BatchGetContactGroupsResponse
          command.query['resourceNames'] = resource_names unless resource_names.nil?
          command.query['maxMembers'] = max_members unless max_members.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['fields'] = fields unless fields.nil?
          execute_or_queue_command(command, &block)
        end
        
        # List all contact groups owned by the authenticated user. Members of the
        # contact groups are not populated.
        # @param [String] sync_token
        #   A sync token, returned by a previous call to `contactgroups.list`.
        #   Only resources changed since the sync token was created will be returned.
        # @param [String] page_token
        #   The next_page_token value returned from a previous call to
        #   [ListContactGroups](/people/api/rest/v1/contactgroups/list).
        #   Requests the next page of resources.
        # @param [Fixnum] page_size
        #   The maximum number of resources to return.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::PeopleV1::ListContactGroupsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::PeopleV1::ListContactGroupsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_contact_groups(sync_token: nil, page_token: nil, page_size: nil, quota_user: nil, fields: nil, options: nil, &block)
          command =  make_simple_command(:get, 'v1/contactGroups', options)
          command.response_representation = Google::Apis::PeopleV1::ListContactGroupsResponse::Representation
          command.response_class = Google::Apis::PeopleV1::ListContactGroupsResponse
          command.query['syncToken'] = sync_token unless sync_token.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['fields'] = fields unless fields.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Create a new contact group owned by the authenticated user.
        # @param [Google::Apis::PeopleV1::CreateContactGroupRequest] create_contact_group_request_object
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::PeopleV1::ContactGroup] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::PeopleV1::ContactGroup]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_contact_group(create_contact_group_request_object = nil, quota_user: nil, fields: nil, options: nil, &block)
          command =  make_simple_command(:post, 'v1/contactGroups', options)
          command.request_representation = Google::Apis::PeopleV1::CreateContactGroupRequest::Representation
          command.request_object = create_contact_group_request_object
          command.response_representation = Google::Apis::PeopleV1::ContactGroup::Representation
          command.response_class = Google::Apis::PeopleV1::ContactGroup
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['fields'] = fields unless fields.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Get a specific contact group owned by the authenticated user by specifying
        # a contact group resource name.
        # @param [String] resource_name
        #   The resource name of the contact group to get.
        # @param [Fixnum] max_members
        #   Specifies the maximum number of members to return.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::PeopleV1::ContactGroup] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::PeopleV1::ContactGroup]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_contact_group(resource_name, max_members: nil, quota_user: nil, fields: nil, options: nil, &block)
          command =  make_simple_command(:get, 'v1/{+resourceName}', options)
          command.response_representation = Google::Apis::PeopleV1::ContactGroup::Representation
          command.response_class = Google::Apis::PeopleV1::ContactGroup
          command.params['resourceName'] = resource_name unless resource_name.nil?
          command.query['maxMembers'] = max_members unless max_members.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['fields'] = fields unless fields.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Update the name of an existing contact group owned by the authenticated
        # user.
        # @param [String] resource_name
        #   The resource name for the contact group, assigned by the server. An ASCII
        #   string, in the form of `contactGroups/<contact_group_id>`.
        # @param [Google::Apis::PeopleV1::UpdateContactGroupRequest] update_contact_group_request_object
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::PeopleV1::ContactGroup] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::PeopleV1::ContactGroup]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_contact_group(resource_name, update_contact_group_request_object = nil, quota_user: nil, fields: nil, options: nil, &block)
          command =  make_simple_command(:put, 'v1/{+resourceName}', options)
          command.request_representation = Google::Apis::PeopleV1::UpdateContactGroupRequest::Representation
          command.request_object = update_contact_group_request_object
          command.response_representation = Google::Apis::PeopleV1::ContactGroup::Representation
          command.response_class = Google::Apis::PeopleV1::ContactGroup
          command.params['resourceName'] = resource_name unless resource_name.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['fields'] = fields unless fields.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Modify the members of a contact group owned by the authenticated user.
        # @param [String] resource_name
        #   The resource name of the contact group to modify.
        # @param [Google::Apis::PeopleV1::ModifyContactGroupMembersRequest] modify_contact_group_members_request_object
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::PeopleV1::ModifyContactGroupMembersResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::PeopleV1::ModifyContactGroupMembersResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def modify_contact_group_members(resource_name, modify_contact_group_members_request_object = nil, quota_user: nil, fields: nil, options: nil, &block)
          command =  make_simple_command(:post, 'v1/{+resourceName}/members:modify', options)
          command.request_representation = Google::Apis::PeopleV1::ModifyContactGroupMembersRequest::Representation
          command.request_object = modify_contact_group_members_request_object
          command.response_representation = Google::Apis::PeopleV1::ModifyContactGroupMembersResponse::Representation
          command.response_class = Google::Apis::PeopleV1::ModifyContactGroupMembersResponse
          command.params['resourceName'] = resource_name unless resource_name.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['fields'] = fields unless fields.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Delete a contact person. Any non-contact data will not be deleted.
        # @param [String] resource_name
        #   The resource name of the contact to delete.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::PeopleV1::Empty] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::PeopleV1::Empty]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_person_contact(resource_name, quota_user: nil, fields: nil, options: nil, &block)
          command =  make_simple_command(:delete, 'v1/{+resourceName}:deleteContact', options)
          command.response_representation = Google::Apis::PeopleV1::Empty::Representation
          command.response_class = Google::Apis::PeopleV1::Empty
          command.params['resourceName'] = resource_name unless resource_name.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['fields'] = fields unless fields.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Provides information about a list of specific people by specifying a list
        # of requested resource names. Use `people/me` to indicate the authenticated
        # user.
        # <br>
        # The request throws a 400 error if 'personFields' is not specified.
        # @param [String] request_mask_include_field
        #   **Required.** Comma-separated list of person fields to be included in the
        #   response. Each path should start with `person.`: for example,
        #   `person.names` or `person.photos`.
        # @param [Array<String>, String] resource_names
        #   The resource name, such as one returned by
        #   [`people.connections.list`](/people/api/rest/v1/people.connections/list),
        #   of one of the people to provide information about. You can include this
        #   parameter up to 50 times in one request.
        # @param [String] person_fields
        #   **Required.** A field mask to restrict which fields on each person are
        #   returned. Valid values are:
        #   * addresses
        #   * ageRanges
        #   * biographies
        #   * birthdays
        #   * braggingRights
        #   * coverPhotos
        #   * emailAddresses
        #   * events
        #   * genders
        #   * imClients
        #   * interests
        #   * locales
        #   * memberships
        #   * metadata
        #   * names
        #   * nicknames
        #   * occupations
        #   * organizations
        #   * phoneNumbers
        #   * photos
        #   * relations
        #   * relationshipInterests
        #   * relationshipStatuses
        #   * residences
        #   * skills
        #   * taglines
        #   * urls
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::PeopleV1::GetPeopleResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::PeopleV1::GetPeopleResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_people(request_mask_include_field: nil, resource_names: nil, person_fields: nil, quota_user: nil, fields: nil, options: nil, &block)
          command =  make_simple_command(:get, 'v1/people:batchGet', options)
          command.response_representation = Google::Apis::PeopleV1::GetPeopleResponse::Representation
          command.response_class = Google::Apis::PeopleV1::GetPeopleResponse
          command.query['requestMask.includeField'] = request_mask_include_field unless request_mask_include_field.nil?
          command.query['resourceNames'] = resource_names unless resource_names.nil?
          command.query['personFields'] = person_fields unless person_fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['fields'] = fields unless fields.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Provides information about a person by specifying a resource name. Use
        # `people/me` to indicate the authenticated user.
        # <br>
        # The request throws a 400 error if 'personFields' is not specified.
        # @param [String] resource_name
        #   The resource name of the person to provide information about.
        #   - To get information about the authenticated user, specify `people/me`.
        #   - To get information about a google account, specify `people/<account_id>`.
        #   - To get information about a contact, specify the resource name that
        #   identifies the contact as returned by
        #   [`people.connections.list`](/people/api/rest/v1/people.connections/list).
        # @param [String] person_fields
        #   **Required.** A field mask to restrict which fields on the person are
        #   returned. Valid values are:
        #   * addresses
        #   * ageRanges
        #   * biographies
        #   * birthdays
        #   * braggingRights
        #   * coverPhotos
        #   * emailAddresses
        #   * events
        #   * genders
        #   * imClients
        #   * interests
        #   * locales
        #   * memberships
        #   * metadata
        #   * names
        #   * nicknames
        #   * occupations
        #   * organizations
        #   * phoneNumbers
        #   * photos
        #   * relations
        #   * relationshipInterests
        #   * relationshipStatuses
        #   * residences
        #   * skills
        #   * taglines
        #   * urls
        # @param [String] request_mask_include_field
        #   **Required.** Comma-separated list of person fields to be included in the
        #   response. Each path should start with `person.`: for example,
        #   `person.names` or `person.photos`.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::PeopleV1::Person] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::PeopleV1::Person]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_person(resource_name, person_fields: nil, request_mask_include_field: nil, quota_user: nil, fields: nil, options: nil, &block)
          command =  make_simple_command(:get, 'v1/{+resourceName}', options)
          command.response_representation = Google::Apis::PeopleV1::Person::Representation
          command.response_class = Google::Apis::PeopleV1::Person
          command.params['resourceName'] = resource_name unless resource_name.nil?
          command.query['personFields'] = person_fields unless person_fields.nil?
          command.query['requestMask.includeField'] = request_mask_include_field unless request_mask_include_field.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['fields'] = fields unless fields.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Update contact data for an existing contact person. Any non-contact data
        # will not be modified.
        # The request throws a 400 error if `updatePersonFields` is not specified.
        # <br>
        # The request throws a 400 error if `person.metadata.sources` is not
        # specified for the contact to be updated.
        # <br>
        # The request throws a 412 error if `person.metadata.sources.etag` is
        # different than the contact's etag, which indicates the contact has changed
        # since its data was read. Clients should get the latest person and re-apply
        # their updates to the latest person.
        # @param [String] resource_name
        #   The resource name for the person, assigned by the server. An ASCII string
        #   with a max length of 27 characters, in the form of `people/<person_id>`.
        # @param [Google::Apis::PeopleV1::Person] person_object
        # @param [String] update_person_fields
        #   **Required.** A field mask to restrict which fields on the person are
        #   updated. Valid values are:
        #   * addresses
        #   * biographies
        #   * birthdays
        #   * braggingRights
        #   * emailAddresses
        #   * events
        #   * genders
        #   * imClients
        #   * interests
        #   * locales
        #   * names
        #   * nicknames
        #   * occupations
        #   * organizations
        #   * phoneNumbers
        #   * relations
        #   * residences
        #   * skills
        #   * urls
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::PeopleV1::Person] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::PeopleV1::Person]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_person_contact(resource_name, person_object = nil, update_person_fields: nil, quota_user: nil, fields: nil, options: nil, &block)
          command =  make_simple_command(:patch, 'v1/{+resourceName}:updateContact', options)
          command.request_representation = Google::Apis::PeopleV1::Person::Representation
          command.request_object = person_object
          command.response_representation = Google::Apis::PeopleV1::Person::Representation
          command.response_class = Google::Apis::PeopleV1::Person
          command.params['resourceName'] = resource_name unless resource_name.nil?
          command.query['updatePersonFields'] = update_person_fields unless update_person_fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['fields'] = fields unless fields.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Create a new contact and return the person resource for that contact.
        # @param [Google::Apis::PeopleV1::Person] person_object
        # @param [String] parent
        #   The resource name of the owning person resource.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::PeopleV1::Person] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::PeopleV1::Person]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_person_contact(person_object = nil, parent: nil, quota_user: nil, fields: nil, options: nil, &block)
          command =  make_simple_command(:post, 'v1/people:createContact', options)
          command.request_representation = Google::Apis::PeopleV1::Person::Representation
          command.request_object = person_object
          command.response_representation = Google::Apis::PeopleV1::Person::Representation
          command.response_class = Google::Apis::PeopleV1::Person
          command.query['parent'] = parent unless parent.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['fields'] = fields unless fields.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Provides a list of the authenticated user's contacts merged with any
        # connected profiles.
        # <br>
        # The request throws a 400 error if 'personFields' is not specified.
        # @param [String] resource_name
        #   The resource name to return connections for. Only `people/me` is valid.
        # @param [String] sync_token
        #   A sync token, returned by a previous call to `people.connections.list`.
        #   Only resources changed since the sync token was created will be returned.
        # @param [String] person_fields
        #   **Required.** A field mask to restrict which fields on each person are
        #   returned. Valid values are:
        #   * addresses
        #   * ageRanges
        #   * biographies
        #   * birthdays
        #   * braggingRights
        #   * coverPhotos
        #   * emailAddresses
        #   * events
        #   * genders
        #   * imClients
        #   * interests
        #   * locales
        #   * memberships
        #   * metadata
        #   * names
        #   * nicknames
        #   * occupations
        #   * organizations
        #   * phoneNumbers
        #   * photos
        #   * relations
        #   * relationshipInterests
        #   * relationshipStatuses
        #   * residences
        #   * skills
        #   * taglines
        #   * urls
        # @param [String] sort_order
        #   The order in which the connections should be sorted. Defaults to
        #   `LAST_MODIFIED_ASCENDING`.
        # @param [Boolean] request_sync_token
        #   Whether the response should include a sync token, which can be used to get
        #   all changes since the last request.
        # @param [String] page_token
        #   The token of the page to be returned.
        # @param [Fixnum] page_size
        #   The number of connections to include in the response. Valid values are
        #   between 1 and 2000, inclusive. Defaults to 100.
        # @param [String] request_mask_include_field
        #   **Required.** Comma-separated list of person fields to be included in the
        #   response. Each path should start with `person.`: for example,
        #   `person.names` or `person.photos`.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::PeopleV1::ListConnectionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::PeopleV1::ListConnectionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_person_connections(resource_name, sync_token: nil, person_fields: nil, sort_order: nil, request_sync_token: nil, page_token: nil, page_size: nil, request_mask_include_field: nil, quota_user: nil, fields: nil, options: nil, &block)
          command =  make_simple_command(:get, 'v1/{+resourceName}/connections', options)
          command.response_representation = Google::Apis::PeopleV1::ListConnectionsResponse::Representation
          command.response_class = Google::Apis::PeopleV1::ListConnectionsResponse
          command.params['resourceName'] = resource_name unless resource_name.nil?
          command.query['syncToken'] = sync_token unless sync_token.nil?
          command.query['personFields'] = person_fields unless person_fields.nil?
          command.query['sortOrder'] = sort_order unless sort_order.nil?
          command.query['requestSyncToken'] = request_sync_token unless request_sync_token.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['pageSize'] = page_size unless page_size.nil?
          command.query['requestMask.includeField'] = request_mask_include_field unless request_mask_include_field.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['fields'] = fields unless fields.nil?
          execute_or_queue_command(command, &block)
        end

        protected

        def apply_command_defaults(command)
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['key'] = key unless key.nil?
        end
      end
    end
  end
end
