=begin

    magick.rb - Ruby interface for ImageMagick

    Copyright (C) 2001 Ryuichi Tamura (tam@kais.kyoto-u.ac.jp)

    $Revision: 1.24 $
    $Date: 2001/07/24 22:46:44 $

=end

require "c_magick.so"

module Magick

  module GeometryParser

    private

    def parse_geometry(geom, default_proc)
      return geom if geom.type == Array
      g = Private::Geometry.new(true, geom, nil, nil)
      default_proc.call(g) if default_proc
      [g.width, g.height, g.x, g.y]
    end

    def parse_image_geometry(geom, default_proc)
      return geom if geom.type == Array
      g = nil
      if /(\%|!|\>|\<|)/ =~ geom
        g = Private::Geometry.new(false, geom, self.get("columns"), self.get("rows"))
      else
        g = Private::Geometry.new(true, geom, nil, nil)
      end
      default_proc.call(g) if default_proc
      [g.width, g.height, g.x, g.y]
    end

  end # module Magick::GeometryParser

=begin

= class Magick::Image

Note some of methods in Magick::Magick_c_Image are declared
private. These use classes under Magick::Private that simply export
members of ImageMagick structures. See 'magick_private.c'

=end



  class Image < Magick_c_Image

    include Magick::GeometryParser

    def initialize(*args)
      super()
      case args.size
      when 0
        return
      else
        do_read_from_files(args)
      end
    end

    def read(*args)
      case args.size
      when 0
        raise ArgumentError, "Wrong # of arguments(0 for 1)"
      else
        do_read_from_files(args)
      end
    end
    alias read_image read
    alias read_images read



    #
    #  annotate
    #
    def annotate(hash)
      rect = parse_geometry(hash["geometry"],nil) if hash["geometry"]
      draw_text = Drawable::Text.new(self, rect[2], rect[3], hash["text"])
      draw_text.set hash
      do_annotate(draw_text.get_drawinfo)
    end

    #
    #  draw
    #
    def draw(*args)
      args.each do |drawable|
        if self.id != drawable.image_id
          raise RuntimeError, "Not the image you are to draw"
        end
        if drawable.is_a? Magick::Drawable::Text
          do_annotate(drawable.get_drawinfo)
        else
          do_draw(drawable.get_drawinfo)
        end
      end
      self
    end

    #
    #  color_floodfill
    #
    def color_floodfill(geom, fill_color, border_color,paint_method=Magick::PaintMethod::Floodfill)
      dinfo = Private::DrawInfo.new(self)
      dinfo.border_color = border_color
      rect = parse_geometry(geom,nil)
      do_color_floodfill(dinfo,fill_color,rect,paint_method)
    end

    #
    #  matte_floodfill
    #
    def matte_floodfill(geom,color,amount=100,paint_method=Magick::PaintMethod::Floodfill)
      rect = parse_geometry(geom,nil)
      do_matte_floodfill(color,amount,rect,paint_method)
    end

    #
    #  montage
    #
    def montage(hash)
      montage_info = Private::MontageInfo.new(self)
      hash.each do |k,v|
        begin
          montage_info.__send__(k+'=',v)
        rescue NameError
          STDERR.print "Invalid key or value: \"#{k}\"=>\"#{v}\". ignoring...\n"
          next
        end
      end
      ret = do_montage(montage_info)
      ret
    end



    #
    #  crop
    #
    def crop(arg)
      rect = parse_geometry(arg, crop_defaults)
      do_crop(rect)
    end

    def crop_defaults
      lambda do |g|
        g.height = g.width unless g.has_x?
      end
    end
    private :crop_defaults

    #
    #  chop
    #
    def chop(arg)
      rect = parse_geometry(arg, chop_defaults)
      do_chop(rect)
    end

    def chop_defaults
      lambda do |g|
        g.height = g.width unless g.has_x?
      end
    end
    private :chop_defaults

    #
    #  resize
    #
    def resize(arg, filter=Magick::FilterType::Lanczos, blur=1.0)
      rect = parse_image_geometry(arg, nil)
      do_resize(rect, filter, blur)
    end

    #
    #  sample
    #
    def sample(arg)
      rect = parse_image_geometry(arg, nil)
      do_sample(rect)
    end

    #
    #  scale
    #
    def scale(arg)
      rect = parse_image_geometry(arg, nil)
      do_scale(rect)
    end

    #
    #  zoom
    #
    def zoom(arg)
      rect = parse_image_geometry(arg, nil)
      do_zoom(rect)
    end



    #
    #  border
    #  (border_width)x(border_height)
    def border(arg, color)
      rect = parse_geometry(arg, border_defaults)
      do_border(rect, color)
    end

    def border_defaults
      lambda do |g|
        g.height = 6 unless g.has_x?
        g.width  = 6 unless g.has_y?
        g.x      = self.get("rows") unless g.has_height?
        g.y      = self.get("columns") unless g.has_width?
      end
    end
    private :border_defaults

    #
    #  raise
    #  (raise_width)x(raise_height)
    def raise_image(arg="", raised=true)
      rect = parse_geometry(arg, raise_defaults)
      do_raise(rect, raised)
    end

    def raise_defaults
      lambda do |g|
        g.height = 6 unless g.has_x?
        g.width  = 6 unless g.has_y?
        g.x      = self.get("rows") unless g.has_height?
        g.y      = self.get("columns") unless g.has_width?
      end
    end
    private :raise_defaults

    #
    # frame
    # (frame_columns)x(frame_rows) :somewhat different from above 2 meths.
    def frame(arg="", inner=nil, outer=nil)
      rect = parse_geometry(arg, frame_defaults)
      inner = if !inner then rect[2] / 2 else inner end
      outer = if !outer then rect[3] / 2 else outer end
      rect[0] += inner + outer
      rect[1] += inner + outer
      do_frame(rect[0], rect[1], rect[2], rect[3], inner, outer)
    end

    def frame_defaults
      lambda do |g|
        g.x      = 20 unless g.has_height?
        g.y      = 20 unless g.has_width?
        g.height = self.get("rows")    + g.x  unless g.has_height?
        g.width  = self.get("columns") + g.y  unless g.has_width?
      end
    end
    private :frame_defaults



    #
    #  quantize image
    #
    def quantize(hash)
      quantize_info = Private::QuantizeInfo.new
      hash.each do |k, v|
        begin
          quantize_info.__send__(k+'=', v)
        rescue NameError
          STDERR.print "Invalid key or value: \"#{k}\"=>\"#{v}\". ignoring...\n"
          next
        end
      end
      if hash['global'] == true
        do_quantize_images(quantize_info)
      else
        do_quantize_image(quantize_info)
      end
    end



    #
    # utility method
    #
    def query_font_metrics(hash)
      draw_text = Drawable::Text.new(self, 0, 0, hash["text"])
      draw_text.set hash
      do_get_font_metrics(draw_text.get_drawinfo)
    end

  end # class Magick::Image





=begin

= module Magick::Drawable

Module that collects the drawable primitives used
in Magick::Image.draw().

= class Magick::Drawable::Base

Abstract class. All the drawable primitives will be
derived from this class. Class method "new" is declared to be
private, so you cannot create an instance.

== Class Methods

none.

== Methods

--- Magick::Drawable::Base#set({"key"=>"value",...})

--- Magick::Drawable::Base#get_drawinfo

== Private Methods

--- Magick::Drawable::Base#set_primitive_params

All the derived class should implement this method.
Specify how to set primitive parameters(defined as
instance variables in each class), and how to set them to
@draw_info.

=end

  module Drawable

    # These classes will be used in Magick::Image#draw.

    class Base

      private_class_method :new
      def initialize(image)
        @image_id = image.id
        @draw_info = Private::DrawInfo.new(image)
      end
      attr_reader :image_id

      def set_primitive_params
        raise NotImplementedError, "subclass resposibility"
      end
      private :set_primitive_params

      def set(hash)
        if hash.has_key? "tile"
          @draw_info.tile = hash["tile"]
        end
        hash.each do |k,v|
          begin
            next if k == "tile"
            @draw_info.__send__(k+'=', v)
          rescue NameError
            STDERR.print "Invalid key or value: \"#{k}\"=>\"#{v}\". ignoring...\n"
            next
          end
        end
      end

      def get_drawinfo
        set_primitive_params
        @draw_info
      end

    end # class Magick::Drawable::Base


    # subclasses call super(image) so that drawinfo object
    # initialize and set to its default value.
    # these classes correspond to enum in PrimitiveType
    # defined in 'classify.h'

    class Point < Base

      public_class_method :new
      def initialize(image, x,y)
        @x = x
        @y = y
        super(image)
      end

      def set_primitive_params
        @draw_info.primitive = sprintf("point %d,%d", @x, @y)
      end
      private :set_primitive_params

    end # class Magick::Drawable::Point

    class Line < Base

      public_class_method :new
      def initialize(image, startx,starty,endx,endy)
        @startx = startx
        @starty = starty
        @endx   = endx
        @endy   = endy
        super(image)
      end

      def set_primitive_params
        @draw_info.primitive = sprintf("line %g,%g,%g,%g",
                                        @startx, @starty,
                                        @endx,   @endy)
      end
      private :set_primitive_params

    end # class Magick::Drawable::Line

    class Rectangle < Base

      public_class_method :new
      def initialize(image, ulX,urY,llX,lrY)
        @upperLeftX  = ulX
        @upperRightY = urY
        @lowerLeftX  = llX
        @lowerRightY = lrY
        super(image)
      end

      def set_primitive_params
        @draw_info.primitive = sprintf("rect %g,%g,%g,%g",
                                        @upperLeftX,
                                        @upperRightY,
                                        @lowerLeftX,
                                        @lowerRightY)
      end
      private :set_primitive_params

    end # class Magick::Drawable::Rectangle

    class RoundRectangle < Base

      public_class_method :new
      def initialize(image, cX,cY,w,h,cW,cH)
        @centerX      = cX
        @centerY      = cY
        @width        = w
        @height       = h
        @cornerWidth  = cW
        @cornerHeight = cH
        super(image)
      end

      def set_primitive_params
        @draw_info.primitive = sprintf("roundrectangle %g,%g,%g,%g,%g,%g",
                                        @centerX, @centerY,
                                        @width, @height,
                                        @cornerWidth, @cornerHeight);
      end
      private :set_primitive_params

    end

    class Arc < Base

      public_class_method :new
      def initialize(image, startX,startY,endX,endY,startDeg=0,endDeg=0)
        @startX       = startX
        @startY       = startY
        @endX         = endX
        @endY         = endY
        @startDegrees = startDeg
        @endDegrees   = endDeg
        super(image)
      end

      def set_primitive_params
        @draw_info.primitive = sprintf("arc %g,%g,%g,%g,%g,%g",
                                       @startX,@startY,@endX,@endY,
                                       @startDegrees,@endDegrees)
      end
      private :set_primitive_params

    end # class Magick::Drawable:: Arc

    class Ellipse < Base

      public_class_method :new
      def initialize(image, origX,origY,w,h,arcStart=0,arcEnd=360)
        @originX  = origX
        @originY  = origY
        @width    = w
        @height   = h
        @arcStart = arcStart
        @arcEnd   = arcEnd
        super(image)
      end

      def set_primitive_params
        @draw_info.primitive = sprintf("ellipse %g,%g,%g,%g,%g,%g",
                                       @originX,@originY,@width,@height,
                                       @arcStart,@argEnd)
      end
      private :set_primitive_params

    end # class Magick::Drawable::Ellipse

    class Circle < Base

      public_class_method :new
      def initialize(image, origX,origY,perimX,perimY)
        @origX  = origX
        @origY  = origY
        @perimX = perimX
        @perimY = perimY
        super(image)
      end

      def set_primitive_params
        @draw_info.primitive = sprintf("circle %g,%g,%g,%g",
                                        @origX,@origY,@perimX,@perimY)
      end
      private :set_primitive_params

    end # class Magick::Drawable::Circle

    class Polyline < Base

      public_class_method :new
      def initialize(image, *args)
        if args.size % 2 != 0
          raise ArgumentError, "Even # of elements required"
        end
        @arg_size = args.size
        @points = args
        super(image)
      end

      def set_primitive_params
        str = "polyline "
        for i in 0 .. (@points.size - 2)
          str << sprintf("%g,",@points[i])
        end
        str << sprintf("%g", @points[-1])
        @draw_info.primitive = str
      end
      private :set_primitive_params

    end # class Magick::Drawable::Polyline

    class Polygon < Base

      public_class_method :new
      def initialize(image, *args)
        if args.size % 2 != 0
          raise ArgumentError, "Even # of elements required"
        end
        @arg_size = args.size
        @points = args
        super(image)
      end

      def set_primitive_params
        str = "polygon "
        for i in 0 .. (@points.size - 2)
          str << sprintf("%g,",@points[i])
        end
        str << sprintf("%g", @points[-1])
        @draw_info.primitive = str
      end
      private :set_primitive_params

    end # class Magick::Drawable::Polygon

    class Bezier < Base

      public_class_method :new
      def initialize(image, *args)
        if args.size % 2 != 0
          raise ArgumentError, "Even # of elements required"
        end
        @arg_size = args.size
        @points = args
        super(image)
      end

      def set_primitive_params
        str = "bezier "
        for i in 0 .. (@points.size - 2)
          str << sprintf("%g,",@points[i])
        end
        str << sprintf("%g", @points[-1])
        @draw_info.primitive = str
      end
      private :set_primitive_params

    end # class Magick::Drawable::Bezier

    class Text < Base

      public_class_method :new
      def initialize(image, x,y,text)
        @x    = x
        @y    = y
        @text = text
        super(image)
      end

      def set_primitive_params
        @draw_info.text = @text
        @draw_info.geometry = sprintf("+%d+%d", @x, @y)
        @draw_info.primitive = sprintf("text %g,%g,\"%s\"",@x,@y,@text)
      end
      private :set_primitive_params

    end # class Magick::Drawable::Text

    class Path < Base

      public_class_method :new
      def initialize(image, str)
        @svg_command = str
        super(image)
      end

      def set_primitive_params
        @draw_info.primitive = sprintf("path '%s'", @svg_command)
      end
      private :set_primitive_params

    end # class Magick::Drawable::Path

    class Image < Base

      public_class_method :new
      def initialize(image, x, y, filename, width=0, height=0)
        @x = x
        @y = y
        @filename = filename
        @width = width
        @height = height

        super(image)
      end

      def set_primitive_params
        @draw_info.primitive = sprintf("image %d,%d,%d,%d,%s",
                                        @x,@y,@width,@height,@filename)
      end

    end # class Magick::Drawable::Image

    class Matte < Base

      public_class_method :new
      def initialize(image, x, y, paint_method)
        @x = x
        @y = y
        @paintMethod = paint_method # see Magick::PaintMethod
      end

      def set_primitive_params
        @drawinfo.primitives = sprintf("matte %d,%d,%d",
                                        @x,@y,@paintMethod)
      end

    end # class Magick::Drawable::Matte

  end # module Magick::Drawable

end # module Magick
