/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file CWE17Module.cxx
    \author Romain BONDUE
    \date 21/08/2005 */
#include <sstream>

#include "CWE17Module.h"
#include "WE17Driver.h"
#include "WE17DriverRoot.h"
#include "WWidgets.h"



namespace
{
    const unsigned TabModeSize (7);
    const nsWireless::Mode_e TabMode [TabModeSize] = {nsWireless::Auto,
                                nsWireless::AdHoc, nsWireless::Managed,
                                nsWireless::Master, nsWireless::Repeater,
                                nsWireless::Secondary,  nsWireless::Monitor};

    const unsigned TabAuthSize (2);
    const nsWireless::AuthType_e TabAuth [TabAuthSize] = {nsWireless::Open,
                                                          nsWireless::Shared};

    const unsigned TabNoneEncryptSize (2);
    const nsWireless::EncryptType_e TabNoneEncrypt [TabNoneEncryptSize] =
                                        {nsWireless::None, nsWireless::WEP};

    const unsigned TabSharedEncryptSize (1);
    const nsWireless::EncryptType_e TabSharedEncrypt [TabSharedEncryptSize] =
                                                            {nsWireless::WEP};

} // anonymous namespace



const char* const nsCore::CWE17Module::NewProfileDefaultName ("New profile");



nsCore::CWE17Module::CWE17Module (const std::string& DeviceName,
                                  const std::string& Path)
                                 throw (nsErrors::CException, std::bad_alloc)
    : m_pDriver (new nsWireless::CWE17Driver (DeviceName)),
      m_Editor (new nsGUI::CSharedConfigDialog (TabMode, TabModeSize,
                                                m_pDriver->GetSupportedFreq(),
                                                TabAuth, TabAuthSize,
                                                TabNoneEncrypt,
                                                TabNoneEncryptSize,
                                                TabSharedEncrypt,
                                                TabSharedEncryptSize, 0, 0)),
      m_ProfilesFile (Path)
{
    m_ProfilesFile.Extract();

} // CWE17Module()


nsCore::CWE17Module::~CWE17Module () throw (nsErrors::CException)
{
    m_ProfilesFile.Record();

} // ~CWE17Module()


nsWireless::IDriver& nsCore::CWE17Module::GetDriver () throw()
{
    return *m_pDriver;

} // GetDriver()


const nsWireless::IDriver& nsCore::CWE17Module::GetDriver () const throw()
{
    return *m_pDriver;

} // GetDriver()


nsUserData::CProfile* nsCore::CWE17Module::EditNewProfile () throw()
{
    if (!m_Editor.Edit (m_ProfilesFile.back()))
    {
        m_ProfilesFile.DeleteLastProfile();
        return 0;
    }
    return &m_ProfilesFile.back();

} // EditNewProfile()


void nsCore::CWE17Module::ApplyProfile (unsigned Pos)
                                                throw (nsErrors::CSystemExc)
{
    m_pDriver->SetSSID (m_ProfilesFile [Pos].GetSSID());  
    m_pDriver->SetMode (m_ProfilesFile [Pos].GetMode());
    try{m_pDriver->SetChannel (m_ProfilesFile [Pos].GetChannel());}
    catch (const nsErrors::CSystemExc&)
    {       // May be fine in managed mode.
        if (m_ProfilesFile [Pos].GetMode() != nsWireless::Managed)
            throw;
    }
    m_pDriver->SetEncryption (m_ProfilesFile [Pos].GetEncryptionD());
    try{m_pDriver->Commit();}
        // May fail because it's not always a necessity.
    catch (const nsErrors::CSystemExc&) {}

} // ApplyProfile()


void nsCore::CWE17Module::Connect (const nsWireless::CCell& Cell)
                                   throw (nsErrors::CSystemExc, std::bad_alloc)
{
        // Only the txrate is not set, but it should be enough without it.
    m_pDriver->SetSSID (Cell.GetSSID());
    const nsWireless::Mode_e Mode (nsWireless::GetMatchingMode
                                                            (Cell.GetMode()));
    m_pDriver->SetMode (Mode);
    try{m_pDriver->SetChannel (Cell.GetChannel());}
    catch (const nsErrors::CSystemExc&)
    {       // May be fine in managed mode.
        if (Mode != nsWireless::Managed)
            throw;
    }
    m_pDriver->SetEncryption (Cell.GetEncryptionD());
        // May fail because it's not always a necessity.
    try {m_pDriver->SetAPMacAddr (Cell.GetAPMacAddress());}
    catch (const nsErrors::CSystemExc&) {}
        // May fail because it's not always a necessity.
    try {m_pDriver->Commit();}
    catch (const nsErrors::CSystemExc&) {}

} // Connect()


void nsCore::CWE17Module::BecomeRoot () throw (nsErrors::CSystemExc,
                                               std::bad_alloc)
{
    m_pDriver.reset (new nsWireless::CWE17DriverRoot
                                                (m_pDriver->GetDeviceName()));

} // BecomeRoot()
