/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file NetTools.cxx
    \author Romain BONDUE
    \date 08/05/2006 */
#include <cstdlib> // abort()
#include <fstream>
#include <limits>
#include <string>
#include <cstring> // memset()

extern "C"{
#include <linux/sockios.h>
#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>
}

#include "NetTools.h"
#include "SystemTools.h"
#include "ErrorsCode.h"
#include "StaticSettings.h"



std::string nsNet::CIPAddress::GetStr () const throw()
{
    ::in_addr Addr = {m_Addr};
    return ::inet_ntoa (Addr);

} // GetStr()


nsNet::CInterface::CInterface (const std::string& IfName) throw()
    : m_IfName (IfName), m_RXBytes (0), m_RXPackets (0), m_RXErrors (0),
      m_RXDropped (0), m_RXFifo (0), m_RXFrame (0), m_RXCompressed (0),
      m_Multicast (0), m_TXBytes (0), m_TXPackets (0), m_TXErrors (0),
      m_TXDropped (0), m_TXFifo (0), m_Collisions (0), m_Carrier (0),
      m_TXCompressed (0) {}


void nsNet::CInterface::RefreshStats () throw (nsErrors::CException)
{
    try
    {
        std::ifstream Is ("/proc/net/dev");
        Is.exceptions (std::ios_base::failbit | std::ios_base::badbit);
        Is.ignore (std::numeric_limits<int>::max(), '\n');
        Is.ignore (std::numeric_limits<int>::max(), '\n');
        while (Is)
        {
            std::string Buffer;
            for (char C ; ; )
            {
                Is >> C;
                if (C == ':')
                    break;
                Buffer += C;
            }
            if (Buffer == m_IfName)
            {
                Is >> m_RXBytes >> m_RXPackets >> m_RXErrors >> m_RXDropped
                   >> m_RXFifo >> m_RXFrame >> m_RXCompressed >> m_Multicast
                   >> m_TXBytes >> m_TXPackets >> m_TXErrors >> m_TXDropped
                   >> m_TXFifo >> m_Collisions >> m_Carrier >> m_TXCompressed;
                return;
            }
            Is.ignore (std::numeric_limits<int>::max(), '\n');
        }
    }
    catch (...) {}
    throw nsErrors::CException ("Can't parse /proc/net/dev",
                                nsErrors::StatsParsing);

} // RefreshStats()


nsNet::CIPAddress nsNet::CInterface::GetIP () const
                                                throw (nsErrors::CSystemExc)
{
    ::ifreq Req;
    std::memset (&Req, 0, sizeof (Req));
    ::sockaddr_in* const pIn (reinterpret_cast< ::sockaddr_in*>
                                                            (&Req.ifr_addr));
    pIn->sin_family = AF_INET;
    m_IfName.copy (Req.ifr_name, IFNAMSIZ - 1);
    Req.ifr_name [m_IfName.size() > IFNAMSIZ - 1 ? IFNAMSIZ - 1
                                                 : m_IfName.size()] = '\0';
    nsSystem::Ioctl (SIOCGIFADDR, &Req, "Can't get IP address.");
    return pIn->sin_addr.s_addr;

} // GetIP()


void nsNet::CInterface::SetIPFromDHCP (const std::string& Data)
                                                throw (nsErrors::CException)
{
    if (nsSystem::Fork())
    {
        const char* Argv [5] = {(nsCore::DhcpScriptPrefix +
                                               nsCore::DhcpScriptName).c_str(),
                                m_IfName.c_str(), Data.c_str(), 0};
        nsSystem::MandatoryExec (Argv);
    }
 
} // SetIPFromDHCP()


void nsNet::CInterface::ClearStats () throw()
{
     m_RXBytes = 0;
     m_RXPackets = 0;
     m_RXErrors = 0;
     m_RXDropped = 0;
     m_RXFifo = 0;
     m_RXFrame = 0;
     m_RXCompressed = 0;
     m_Multicast = 0;
     m_TXBytes = 0;
     m_TXPackets = 0;
     m_TXErrors = 0;
     m_TXDropped = 0;
     m_TXFifo = 0;
     m_Collisions = 0;
     m_Carrier = 0;
     m_TXCompressed = 0;

} // ClearStats()
