/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file RTWMsgHandlers.cxx
    \author Romain BONDUE
    \date 21/02/2005 */
#include <sstream>

#include "RTWMsgHandlers.h"
#include "RTDrivers.h"
#include "RTSuWProtocolCodes.h"
#include "ErrorsCode.h"



namespace
{
    template<typename T>
    T Extract (const std::string& Str) throw (nsErrors::CException)
    {
        std::istringstream Is (Str);
        T Value;
        Is >> Value;
        if (Is || Is.eof()) return Value;
        throw nsErrors::CException ("Invalid data read (from helper).",
                                    nsErrors::InvalidDataFromHelper);

    } // Extract()

} // anonymous namespace


void nsRoot::CRT2400MsgHandler::SetDriver (const std::string& DeviceName)
                                throw (std::bad_alloc, nsErrors::CException)
{
    CWirelessMsgHandler::SetDriver
                                (new nsWireless::CRT2400Driver (DeviceName));

} // SetDriver()


void nsRoot::CRT2500MsgHandler::SetDriver (const std::string& DeviceName)
                                throw (std::bad_alloc, nsErrors::CException)
{
    CWirelessMsgHandler::SetDriver
                                (new nsWireless::CRT2500Driver (DeviceName));

} // SetDriver()


void nsRoot::CRT2570MsgHandler::SetDriver (const std::string& DeviceName)
                                throw (std::bad_alloc, nsErrors::CException)
{
    CWirelessMsgHandler::SetDriver
                                (new nsWireless::CRT2570Driver (DeviceName));

} // SetDriver()


nsRoot::CMsg nsRoot::CRT2500MsgHandler::operator () (const CMsg& Msg)
                                throw (nsErrors::CException, std::bad_alloc)
{
    switch (Msg.GetCode())
    {
      case nsWireless::WModeC :
        GetDriver()->SetWirelessMode (nsWireless::WirelessMode_e (Extract<int>
                                                            (Msg.GetText())));
      break;

      case nsWireless::TxBurstC :
        GetDriver()->SetTxBurst (Extract<bool> (Msg.GetText()));
      break;

      case nsWireless::BGProtectionC :
        GetDriver()->SetBGProtection (nsWireless::BGProtection_e (Extract<int>
                                                            (Msg.GetText())));
      break;

      case nsWireless::TxPreambleC :
        GetDriver()->SetTxPreamble (nsWireless::TxPreamble_e (Extract<int>
                                                            (Msg.GetText())));
      break;

      case nsWireless::TurboRateC :
        GetDriver()->SetTurboRate (Extract<bool> (Msg.GetText()));
      break;

      case nsWireless::AdHocOFDMC :
        GetDriver()->SetAdHocOFDM (Extract<bool> (Msg.GetText()));
      break;

      case nsWireless::RfmontxC :
        GetDriver()->SetRfmontx (Extract<bool> (Msg.GetText()));
      break;

      default :
        return CWirelessMsgHandler::operator () (Msg);
    }
    return CMsg();

} // operator() ()
