/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file UserData.cxx
    \author Romain BONDUE
    \date 03/08/2005 */
#include <fstream>
#include <sstream>
#include <cstdlib> // getenv()
#include <cstring> // strcmp(), strlen()
#include <cctype> // isspace()
#include <cerrno>

extern "C"{
#include <glib.h>
#include <sys/stat.h> // ::chmod()
}

#include "UserData.h"
#include "StaticSettings.h"
#include "ErrorsCode.h"



namespace
{
    const char* const RootMarkupLabel ("Rutilt_Profiles");
    const char* const ProfileName ("Name");
    const char* const ProfileSSID ("SSID");
    const char* const ProfileMode ("Mode");
    const char* const ProfileChannel ("Channel");
    const char* const ProfileEncryption ("Encryption");
    const char* const ProfileEncryptionEncrypt ("Type");
    const char* const ProfileEncryptionKey ("Key");
    const char* const ProfileEncryptionAuth ("AuthenticationType");
    const char* const ProfileIPSettings ("IPSettings");
    const char* const ProfileIPSettingsHow ("How");
        // This variable is a work-around for g++-3.3.
    const std::string HomeDirectory (getenv ("HOME"));
    const std::string ConfigDirectory (HomeDirectory + '/' +
                                       nsCore::UserConfigPrefix);

} // anonymous namespace



const std::string nsUserData::AppDirectory (ConfigDirectory +
                                                    nsCore::UserConfigName);


void nsUserData::MakeAppDirectory () throw (nsErrors::CSystemExc)
{
    if (::mkdir (ConfigDirectory.c_str(), 0755) && errno != EEXIST)
        throw nsErrors::CSystemExc ("Can't create \"" + ConfigDirectory +
                                    "\" directory.");
    if (::mkdir (AppDirectory.c_str(), 0755) && errno != EEXIST)
        throw nsErrors::CSystemExc ("Can't create \"" + AppDirectory +
                                    "\" directory.");

} // MakeAppDirectory()


nsUserData::CProfile::CProfile (const std::string& Name,
                                const nsWireless::CCell& Cell)
                                                        throw (std::bad_alloc)
    : m_Name (Name), m_SSID (Cell.GetSSID()),
      m_Mode (nsWireless::GetMatchingMode (Cell.GetMode())),
      m_Channel (Cell.GetChannel()), m_EncryptionD (Cell.GetEncryptionD()) {}


bool nsUserData::CProfile::Match (const nsWireless::CCell& Cell) const throw()
{
    return m_Mode == Cell.GetMode() && m_Channel == Cell.GetChannel() &&
           m_SSID == Cell.GetSSID() && m_EncryptionD == Cell.GetEncryptionD();

} // Match()


const char* const nsUserData::CProfilesFileBase::Indentation ("    ");


const char* nsUserData::CProfilesFileBase::GetRootMarkupLabel () const throw()
{
    return "Profile";

} // GetRootMarkupLabel()


void nsUserData::CProfilesFileBase::RecordFields (std::ostream& Os,
                                                  unsigned Pos)
                                                throw (std::ios_base::failure)
{
    CProfile& Profile (GetProfile (Pos));
    Os << Markup1B << ProfileSSID << Markup1E
       << ::g_markup_escape_text (Profile.GetSSID().c_str(),
                                  Profile.GetSSID().size())
       << Markup2B << ProfileSSID << Markup2E << Indentation
       << Markup1B << ProfileMode << Markup1E << Profile.GetMode()
       << Markup2B << ProfileMode << Markup2E << Indentation
       << Markup1B << ProfileChannel << Markup1E<< Profile.GetChannel()
       << Markup2B << ProfileChannel << Markup2E;
    if (Profile.GetEncryptionD().GetEncrypt() != nsWireless::None)
    {
        Os << Indentation << Markup1B << ProfileEncryption << Markup2E
           << Indentation << Indentation << Markup1B
           << ProfileEncryptionEncrypt << Markup1E
           << Profile.GetEncryptionD().GetEncrypt() << Markup2B
           << ProfileEncryptionEncrypt << Markup2E << Indentation
           << Indentation;
        for (unsigned I (0) ; I < nsWireless::CEncryptionD::MaxNbKey ; ++I)
            Os << Markup1B << ProfileEncryptionKey << Markup1E
               << Profile.GetEncryptionD().GetKey (I).GetStr() << Markup2B
               << ProfileEncryptionKey << Markup2E << Indentation
               << Indentation;
        Os << Markup1B << ProfileEncryptionAuth << Markup1E
           << Profile.GetEncryptionD().GetAuth()
           << Markup2B << ProfileEncryptionAuth << Markup2E
           << Indentation << Markup2B << ProfileEncryption << Markup2E;
    }
    if (Profile.GetIPSettings().GetSetting() != nsUserData::None)
        Os << Indentation << Markup1B << ProfileIPSettings << Markup2E
           << Indentation << Indentation << Markup1B << ProfileIPSettingsHow
           << Markup1E << Profile.GetIPSettings().GetSetting()
           << Markup2B << ProfileIPSettingsHow << Markup2E << Indentation
           << Markup2B << ProfileIPSettings << Markup2E;

} // RecordFields()


void nsUserData::CProfilesFileBase::Record () throw (nsErrors::CException)
{
    std::ofstream Os (m_FileName.c_str(), std::ios_base::trunc);
    try
    {
        Os.exceptions (std::ios_base::badbit | std::ios_base::failbit);

        Os << XMLHeader << '\n' << Markup1B << RootMarkupLabel << Markup2E;
        const unsigned VecSize (Size());
        for (unsigned i (0) ; i < VecSize ; ++i)
        {
            Os << Markup1B << GetRootMarkupLabel() << ' ' << ProfileName
               << "=\"" << GetProfile (i).GetName() << '"' << Markup2E
               << Indentation;
            RecordFields (Os, i);
            Os << Markup2B << GetRootMarkupLabel() << Markup2E;
        }
        Os << Markup2B << RootMarkupLabel << Markup2E;
    }
    catch (const std::ios_base::failure& Exc)
    {
        throw nsErrors::CException (Exc.what(),
                                    nsErrors::ProfilesRecordFailure);
    }
        // The keys are stored in this file, this is the least I can do...
    if (::chmod (m_FileName.c_str(), 00600))
    {
        std::ostringstream Os;
        Os << "Can't set permission on file : " << m_FileName
           << ".\nerrno = " << errno;
        throw nsErrors::CException (Os.str(),
                                    nsErrors::ProfileFilePermissionError);
    }

} // Record()


void nsUserData::CProfilesFileBase::StartElement (const char* ElementName,
                                                  const char** AttributesNames,
                                              const char** AttributesValues)
                                   throw (std::bad_alloc, nsErrors::CException)
{
    if (!strcmp (ElementName, GetRootMarkupLabel()))
    {
        if (!AttributesNames [0] || strcmp (AttributesNames [0],
                                            ProfileName) ||
                                                !strlen (AttributesValues [0]))
            throw nsErrors::CException ("Invalid or missing profile name.",
                                   nsErrors::ProfilesExtractionMissingName);
        else NewProfile (AttributesValues [0]);
    }

} // StartElement()


void nsUserData::CProfilesFileBase::EndElement (const char* ElementName)
                                                throw (nsErrors::CException)
{     // Here we check if we have everything required.
    if (!strcmp (ElementName, GetRootMarkupLabel()))
    {
        CProfile& Profile (GetProfile (Size() - 1));
        if (Profile.GetMode() == nsWireless::Unknown)
            throw nsErrors::CException ("Missing mode.",
                                   nsErrors::ProfilesExtractionMissingMode);
        else if (!Profile.GetChannel())
            throw nsErrors::CException ("Missing channel.",
                                nsErrors::ProfilesExtractionMissingChannel);
    }

} // EndElement()


void nsUserData::CProfilesFileBase::Text (const char* ElementName,
                                          const std::string& Text)
                                                throw (nsErrors::CException)
{
        // FIXME HORRIBLE... *Shocked*
    static unsigned Cnt (0);

    CProfile& Profile (GetProfile (Size() - 1));
    if (!strcmp (ElementName, ProfileSSID))
        Profile.SetSSID (Text);
    else if (!strcmp (ElementName, ProfileMode))
    {
        nsWireless::Mode_e Mode (nsWireless::Unknown);
        Mode = nsWireless::Mode_e (atoi (Text.c_str()));
            // I could have also check the value.
        if (Mode == nsWireless::Unknown)
            throw nsErrors::CException ("Invalid mode.",
                                   nsErrors::ProfilesExtractionInvalidMode);
        Profile.SetMode (Mode);
    }
    else if (!strcmp (ElementName, ProfileChannel))
    {
        unsigned Channel (0);
        Channel = atoi (Text.c_str());
        if (!Channel || Channel > 14)
            throw nsErrors::CException ("Invalid channel.",
                                nsErrors::ProfilesExtractionInvalidChannel);
        Profile.SetChannel (Channel);
    }
    else if (!strcmp (ElementName, ProfileEncryptionKey))
        try
        {
            nsWireless::CHexaKey Key;
            if (!Text.empty())
                Key.Set (Text);
            Profile.GetEncryptionD().SetKey (Key, Cnt++);
        }
        catch (const nsWireless::CHexaKey::CBadFormatExc&)
        {
            throw nsErrors::CException ("Invalid key.",
                                    nsErrors::ProfilesExtractionInvalidKey);
        }
    else if (!strcmp (ElementName, ProfileEncryptionAuth))
            // No check...
        Profile.GetEncryptionD().SetAuth (nsWireless::AuthType_e
                                                        (atoi (Text.c_str())));
    else if (!strcmp (ElementName, ProfileEncryptionEncrypt))
    {       // No check...
        Profile.GetEncryptionD().SetEncrypt (nsWireless::EncryptType_e
                                                        (atoi (Text.c_str())));
        Cnt = 0;
    }
    else if (!strcmp (ElementName, ProfileIPSettingsHow))
            // No check...
        Profile.GetIPSettings().SetSetting (IPSetting_e (atoi (Text.c_str())));

} // Text()
