/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file WE17Driver.cxx
    \author Romain BONDUE
    \date 05/07/2005
    \note Many functions here could have been "inline", but since they are also
          "virtual", it's almost useless. */
#include <sstream>
#include <cmath> // pow()
#include <cstring> // strncpy()
#include <cerrno>
#include <algorithm> // sort()

#include "WE17Driver.h"
#include "ErrorsCode.h"



namespace
{
    inline nsWireless::CQuality iw_qualityToCQuality (const ::iw_quality& Iw)
                                                                        throw()
    {
            // There's an unusual 8 bits arithmetic, that's why the -256.
        return nsWireless::CQuality (Iw.qual, -256 + Iw.level,
                                     -256 + Iw.noise);

    } // iw_qualityToCQuality()


    inline nsWireless::CEncryptionD MakeCEncryption (const char* Key,
                                            unsigned Length, int Flags) throw()
    {
        return nsWireless::CEncryptionD (Flags & IW_ENCODE_DISABLED
                                        ? nsWireless::CHexaKey()
                                        : nsWireless::CHexaKey (Key, Length),
                            Flags & IW_ENCODE_RESTRICTED ? nsWireless::Shared
                                                         : nsWireless::Open,
                                Flags & IW_ENCODE_DISABLED ? nsWireless::None
                                                           : nsWireless::WEP);

    } // MakeCEncryption()


    std::string CatStringNInt (std::string Str, int Number) throw()
    {
        std::ostringstream Os;
        Os << Str << Number;
        return Os.str();

    } // CatStringNInt()


    inline double iw_freqToDouble (::iw_freq Freq) throw()
    {
        return Freq.m / pow (10, 9 - Freq.e);

    } // iw_freqToDouble()


    const unsigned TxUnit (1000);

} // anonymous namespace



nsWireless::CWE17Driver::CWE17Driver (const std::string& Name)
                                                throw (nsErrors::CSystemExc)
    : m_Data (m_Req.u)
{
        // Starting from here, we just collect informations :
    Name.copy (m_Req.ifr_ifrn.ifrn_name, IFNAMSIZ);
    m_Req.ifr_ifrn.ifrn_name [Name.size() > IFNAMSIZ - 1 ? IFNAMSIZ - 1
                                                         : Name.size()] = '\0';
    Ioctl (SIOCGIWNAME, "Can't get protocol/provider name.");
    m_ProtoName = m_Req.u.name;

    ::iw_range Range;
    m_Req.u.data.pointer = reinterpret_cast< ::caddr_t> (&Range);
    m_Req.u.data.length = sizeof (Range);
    m_Req.u.data.flags = 0;
    try
    {       // Only used to get channels values.
        Ioctl (SIOCGIWRANGE, "Can't get interface informations.");
        for (unsigned i (0) ; i < Range.num_frequency ; ++i)
            m_SupportedFreqVec.push_back (CFreq (Range.freq [i].i,
                                         iw_freqToDouble (Range.freq [i])));
    }
    catch (const nsErrors::CSystemExc& Exc)
    {
        if (Exc.GetCode() != EOPNOTSUPP) throw;
    }
    if (m_SupportedFreqVec.empty())
    {
        for (unsigned i (1) ; i < 14 ; ++i) // I put the whole spectrum.
            m_SupportedFreqVec.push_back (CFreq (i, 2.412 + (i - 1) * 0.005));
        m_SupportedFreqVec.push_back (CFreq (14, 2.484));
    }

} // CWE17Driver()


std::string nsWireless::CWE17Driver::GetSSID () const
                                                throw (nsErrors::CSystemExc)
{
    char Buffer [IW_ESSID_MAX_SIZE];
    m_Req.u.data.pointer = reinterpret_cast< ::caddr_t> (Buffer);
    m_Req.u.data.length = IW_ESSID_MAX_SIZE;
    m_Req.u.data.flags = 0;
    Ioctl (SIOCGIWESSID, "Can't get the SSID.");
    if (!m_Req.u.data.length) return std::string();
    return std::string (Buffer, 0, m_Req.u.data.length);

} // GetSSID()


    // No control on the SSID, however it'll be truncated if too long.
void nsWireless::CWE17Driver::SetSSID (const std::string& SSID)
                                                throw (nsErrors::CSystemExc)
{
    m_Req.u.data.pointer = const_cast< ::caddr_t> (SSID.c_str());
    m_Req.u.data.length = SSID.size() + 1 > IW_ESSID_MAX_SIZE
                                                          ? IW_ESSID_MAX_SIZE  
                             /* + 1 for the '\0'. */      : SSID.size() + 1;
    m_Req.u.data.flags = 1; // Only use *this* SSID.
    Ioctl (SIOCSIWESSID, std::string ("Can't set SSID : ") + SSID);

} // SetSSID()


unsigned nsWireless::CWE17Driver::GetChannel () const
                                                throw (nsErrors::CException)
{
    const double Value (GetDriverFreqValue());
    if (Value > 0) return GetMatchingFreq (0, Value).GetChannel();
    return unsigned (-Value);

} // GetChannel()


void nsWireless::CWE17Driver::SetChannel (unsigned Channel)
                                                throw (nsErrors::CSystemExc)
{
    m_Req.u.freq.m = Channel;
    m_Req.u.freq.e = 0;
#if WIRELESS_EXT > 16
    m_Req.u.freq.flags = IW_FREQ_FIXED;
#endif
    Ioctl (SIOCSIWFREQ, CatStringNInt ("Can't set channel : ", Channel));

} // SetChannel()


double nsWireless::CWE17Driver::GetFrequency () const
                                                throw (nsErrors::CException)
{
    const double Value (GetDriverFreqValue());
    if (Value > 0) return Value;
    return GetMatchingFreq (unsigned (-Value)).GetFrequency();

} // GetFrequency()


void nsWireless::CWE17Driver::SetFrequency (double F)
                                                throw (nsErrors::CSystemExc)
{
    m_Req.u.freq.m = unsigned (F * 1e8);
    m_Req.u.freq.e = 1;
#if WIRELESS_EXT > 16
    m_Req.u.freq.flags = IW_FREQ_FIXED;
#endif
    Ioctl (SIOCSIWFREQ, CatStringNInt ("Can't set frequency : ",
                                       m_Req.u.freq.m));

} // SetFrequency()


nsWireless::CEncryptionD nsWireless::CWE17Driver::GetEncryption ()
                                throw (nsErrors::CSystemExc, std::bad_alloc)
{
        /* ? FIXME ? The key index is not considered and only the first key
                     is reported. */
    char Buffer [IW_ENCODING_TOKEN_MAX];
    m_Req.u.data.pointer = reinterpret_cast< ::caddr_t> (Buffer);
    m_Req.u.data.length = IW_ENCODING_TOKEN_MAX;
    m_Req.u.data.flags = 0;
    Ioctl (SIOCGIWENCODE, "Can't get Wep status.");
    return MakeCEncryption (Buffer, m_Req.u.data.length, m_Req.u.data.flags);

} // GetEncryption()


void nsWireless::CWE17Driver::SetEncryption (const CEncryptionD& Descriptor)
                                throw (nsErrors::CSystemExc, std::bad_alloc)
{
    if (Descriptor.GetEncrypt() == nsWireless::WEP)
    {
        const int AuthFlag (Descriptor.GetAuth() ==
                                    nsWireless::Shared ? IW_ENCODE_RESTRICTED
                                                       : IW_ENCODE_OPEN);
        for (unsigned I (CEncryptionD::MaxNbKey) ; I ; )
        {
            m_Req.u.data.flags = AuthFlag;
            m_Req.u.data.flags |= I;
            if (Descriptor.GetKey (--I).Empty()) // Decremented here.
            {
                m_Req.u.data.pointer = 0;
                m_Req.u.data.length = 0;
                m_Req.u.data.flags |= IW_ENCODE_NOKEY | IW_ENCODE_DISABLED;
            }
            else
            {
                m_Req.u.data.pointer = const_cast< ::caddr_t>
                                                (Descriptor.GetKey (I).Get());
                m_Req.u.data.length = Descriptor.GetKey (I).Size();
            }
            Ioctl (SIOCSIWENCODE,
        CatStringNInt (std::string ("Can't set WEP encryption with key : ") +
                                Descriptor.GetKey (I).GetStr() + " Index : ",
                       I));
        }
        try   // Now activate key 1, it may fail because the previous calls
        {     // already activate it with some drivers. */
            m_Req.u.data.pointer = 0;
            m_Req.u.data.length = 0;
            m_Req.u.data.flags |= IW_ENCODE_NOKEY | 1;
            Ioctl (SIOCSIWENCODE, "Can't activate key 1.");
        }
        catch (...) {}
    }
    else
    {
        m_Req.u.data.flags |= IW_ENCODE_NOKEY | IW_ENCODE_DISABLED;;
        m_Req.u.data.pointer = 0;
        m_Req.u.data.length = 0;
        Ioctl (SIOCSIWENCODE, "Can't disable WEP encryption.");
    }

} // SetEncryption()


nsWireless::Mode_e nsWireless::CWE17Driver::GetMode () const
                                                throw (nsErrors::CSystemExc)
{
    Ioctl (SIOCGIWMODE, "Can't get the mode.");
    return Mode_e (m_Req.u.mode);

} // GetMode()


void nsWireless::CWE17Driver::SetMode (Mode_e Mode)
                                                throw (nsErrors::CSystemExc)
{
    m_Req.u.mode = Mode;
    Ioctl (SIOCSIWMODE,
           std::string ("Can't set mode : ") + GetModeName (Mode));

} // SetMode()


nsWireless::CQuality nsWireless::CWE17Driver::GetQuality () const
                                                throw (nsErrors::CSystemExc)
{
    ::iw_statistics Stat;
    m_Req.u.data.pointer = reinterpret_cast < ::caddr_t> (&Stat);
    m_Req.u.data.length = sizeof (Stat);
    m_Req.u.data.flags = 0;
    Ioctl (SIOCGIWSTATS, "Can't get the signal quality.");
    return iw_qualityToCQuality (Stat.qual);

} // GetQuality()


int nsWireless::CWE17Driver::GetTxRate () const throw (nsErrors::CSystemExc)
{
    Ioctl (SIOCGIWRATE, "Can't get the Tx rate.");
    return m_Req.u.bitrate.value / TxUnit;

} // GetTxRate()


void nsWireless::CWE17Driver::SetTxRate (int TxRate)
                                                throw (nsErrors::CSystemExc)
{
    m_Req.u.bitrate.value = TxRate * TxUnit;
    m_Req.u.bitrate.fixed = 0;  // Hardware should use auto-select.
    m_Req.u.bitrate.disabled = 0;
    m_Req.u.bitrate.flags = 0;
    Ioctl (SIOCSIWRATE, CatStringNInt ("Can't set Tx rate : ", TxRate));

} // SetTxRate()


nsWireless::CMacAddress nsWireless::CWE17Driver::GetAPMacAddr () const
                                                throw (nsErrors::CSystemExc)
{
    try {Ioctl (SIOCGIWAP, "Can't get access point Mac address.");}
    catch (const nsErrors::CSystemExc& Exc)
    {     // I don't consider this as an error.
        if (Exc.GetCode() == ENOTCONN) return CMacAddress();
        throw;
    }
    return CMacAddress (m_Req.u.ap_addr.sa_data);

} // GetAPMacAddr()


void nsWireless::CWE17Driver::SetAPMacAddr (const CMacAddress& Addr)
                                throw (std::bad_alloc, nsErrors::CSystemExc)
{
    strncpy (m_Req.u.ap_addr.sa_data, Addr.Get(), CMacAddress::Size);
    Ioctl (SIOCSIWAP, std::string ("Can't set access point Mac address : ") +
                      Addr.GetStr());

} // SetAPMacAddr()


void nsWireless::CWE17Driver::Scan () throw (nsErrors::CSystemExc)
{
    m_Req.u.data.pointer = 0;
    m_Req.u.data.length = 0;
        // Full scanning.
    m_Req.u.data.flags = IW_SCAN_ALL_ESSID | IW_SCAN_ALL_FREQ |
                         IW_SCAN_ALL_MODE | IW_SCAN_ALL_RATE;
    Ioctl (SIOCSIWSCAN, "Can't trigger scanning.");

} // Scan()


void nsWireless::CWE17Driver::GetScanResult (std::vector<CCell>& CellVec) const
                                throw (nsErrors::CException, std::bad_alloc)
{
    m_Req.u.data.flags = 0;
    unsigned BufferSize (256); // In iwlist, the starting size is 128.
    unsigned char* Buffer;
    for ( ; ; ) // *Never ever* a goto :p (especially in C++).
    {
        Buffer = new unsigned char [BufferSize];
        m_Req.u.data.pointer = reinterpret_cast < ::caddr_t> (Buffer);
        m_Req.u.data.length = BufferSize;
        try
        {
            Ioctl (SIOCGIWSCAN, "Can't get the scan results.");
            break;
        }
        catch (const nsErrors::CSystemExc& Exc)
        {
            if (Exc.GetCode() == E2BIG)
            {       // If the driver gives the expected size.
                if (m_Req.u.data.length > BufferSize)
                    BufferSize = m_Req.u.data.length;
                    // Else we try a bigger size.
                else BufferSize *= 2;
            }
            else
            {
                delete[] Buffer;
                throw;
            }
        }
        delete[] Buffer;
    }
        // Things are MUCH more complicated in iwlib.c...
    try
    {
        for (unsigned Offset (0) ; Offset < m_Req.u.data.length ; )
            Offset += FillCellVec (reinterpret_cast<const ::iw_event*> 
                                                (Buffer + Offset), CellVec);
    }
    catch (...)
    {
        delete[] Buffer;
        throw;
    }
    delete[] Buffer;

} // GetScanResult()


unsigned nsWireless::CWE17Driver::FillCellVec (const ::iw_event* pEvent,
                                               std::vector<CCell>& Vec) const
                                                throw (nsErrors::CException)
{
        // Jean Tourrilhes in iwlib.c warn that data may be unaligned.
    switch (pEvent->cmd)
    {
      case SIOCGIWAP :
        Vec.push_back (CCell (CMacAddress (pEvent->u.ap_addr.sa_data)));
      break;

      case SIOCGIWMODE :
        Vec.back().m_Mode = Mode_e (pEvent->u.mode);
      break;

      case SIOCGIWESSID :
      {
          /* Fix for new WE version until a CWEXXDriver is implemented.
           * From wireless.h (v19) : iw_point events are special.
           * First, the payload (extra data) come at the end of the event,
           * so they are bigger than IW_EV_POINT_LEN.
           * Second, we omit the pointer, so start at an offset. */
#if WIRELESS_EXT > 17
        const unsigned CstLength (reinterpret_cast<const ::iw_point*>
        (reinterpret_cast<const char*> (&pEvent->u) - sizeof (void*))->length);
#else
        const unsigned CstLength (pEvent->u.essid.length);
#endif
        if (CstLength)
            /* From wireless.h (v17) : "Note : in the case of iw_point,
             * the extra data will come at the end of the event". */
            Vec.back().m_SSID = std::string (reinterpret_cast<const char*>
                                                    (pEvent) + IW_EV_POINT_LEN,
                                             0, CstLength);
      }
      break;

      case SIOCGIWENCODE :
      {
          /* Fix for new WE version until a CWEXXDriver is implemented.
           * From wireless.h (v19) : iw_point events are special.
           * First, the payload (extra data) come at the end of the event,
           * so they are bigger than IW_EV_POINT_LEN.
           * Second, we omit the pointer, so start at an offset. */
#if WIRELESS_EXT > 17
        const unsigned CstLength (reinterpret_cast<const ::iw_point*>
        (reinterpret_cast<const char*> (&pEvent->u) - sizeof (void*))->length);
        unsigned Flags (reinterpret_cast<const ::iw_point*>
        (reinterpret_cast<const char*> (&pEvent->u) - sizeof (void*))->flags);
#else
        const unsigned CstLength (pEvent->u.essid.length);
        unsigned Flags (pEvent->u.essid.flags);
#endif
        const char* Data (0);
        if (CstLength)
            Data = reinterpret_cast<const char*> (pEvent) + IW_EV_POINT_LEN;
        else Flags |= IW_ENCODE_NOKEY;
        Vec.back().m_EncDescriptor = MakeCEncryption (Data, CstLength, Flags);
      }
      break;

      case SIOCGIWFREQ :
      {
        if (pEvent->u.freq.m > 1000)
            try
            {
                Vec.back().m_Channel =
                    GetMatchingFreq (0,
                                iw_freqToDouble (pEvent->u.freq)).GetChannel();
            }
            catch (const nsErrors::CException& Exc)
            {
                if (Exc.GetCode() != nsErrors::NoMatchingFreq) throw;
            }
        else Vec.back().m_Channel = pEvent->u.freq.m;
      }
      break;

      case IWEVQUAL :
        Vec.back().m_Quality = iw_qualityToCQuality (pEvent->u.qual);
      break;

      case SIOCGIWRATE :
        Vec.back().m_Rate =  pEvent->u.bitrate.value / TxUnit;
      //break;
    }
    return pEvent->len;

} // FillCellVec()


const nsWireless::CFreq& nsWireless::CWE17Driver::GetMatchingFreq
        (unsigned Channel, double Frequency) const throw (nsErrors::CException)
{
    for (unsigned i (m_SupportedFreqVec.size()) ; i-- ; )
        if (m_SupportedFreqVec [i].GetChannel() == Channel ||
            m_SupportedFreqVec [i].GetFrequency() == Frequency)
            return m_SupportedFreqVec [i];
    throw nsErrors::CException ("Can't find matching Freq.",
                                nsErrors::NoMatchingFreq);

} // GetMatchingFreq()


double nsWireless::CWE17Driver::GetDriverFreqValue () const
                                                throw (nsErrors::CSystemExc)
{
    Ioctl (SIOCGIWFREQ, "Can't get the frequency/channel.");
    if (m_Req.u.freq.m > 1000)
        return iw_freqToDouble (m_Req.u.freq);
    return -m_Req.u.freq.m;

} // GetDriverFreqValue())


unsigned nsWireless::CWE17Driver::GetPrivateIoctls
        (::iw_priv_args* Tab, unsigned TabSize) throw (nsErrors::CSystemExc)
{
    m_Data.data.pointer = reinterpret_cast< ::caddr_t> (Tab);
    m_Data.data.length = TabSize;
    m_Data.data.flags = 0;
    Ioctl (SIOCGIWPRIV, "Can't get private ioctl interface.");

    return m_Data.data.length;

} // GetPrivateIoctl()


void nsWireless::CWE17Driver::Commit () throw (nsErrors::CSystemExc)
{
    m_Data.data.pointer = reinterpret_cast< ::caddr_t> (0);
    m_Data.data.length = 0;
    m_Data.data.flags = 0;
    Ioctl (SIOCSIWCOMMIT, "Can't commit.");

} // Commit()


void nsWireless::CWE17Driver::GetSupportedRates (std::vector<int>& RatesVec)
                                            const throw (nsErrors::CSystemExc)
{
    ::iw_range Range;
    m_Req.u.data.pointer = reinterpret_cast< ::caddr_t> (&Range);
    m_Req.u.data.length = sizeof (Range);
    m_Req.u.data.flags = 0;
    Ioctl (SIOCGIWRANGE, "Can't get supported rates.");
    for (unsigned i (0) ; i < Range.num_bitrates ; ++i)
        RatesVec.push_back (Range.bitrate [i] / TxUnit);
    std::sort (RatesVec.rbegin(), RatesVec.rend());

} // GetSupportedRates()
