/*
 * zgetdump - Tool for copying and converting System z dumps
 *
 * kdump input format
 *
 * Copyright IBM Corp. 2001, 2010
 * Author(s): Michael Holzheu <holzheu@linux.vnet.ibm.com>
 */

#include "zgetdump.h"

struct l_new_utsname {
	char sysname[65];
	char nodename[65];
	char release[65];
	char version[65];
	char machine[65];
	char domainname[65];
};

struct df_kdump_hdr {
	char			signature[8];
	int			header_version;
	struct l_new_utsname	utsname;
	struct timeval		timestamp;
	unsigned int		status;
	int			block_size;
	int			sub_hdr_size;
	unsigned int		bitmap_blocks;
	unsigned int		max_mapnr;
	unsigned int		total_ram_blocks;
	unsigned int		device_blocks;
	unsigned int		written_blocks;
	unsigned int		current_cpu;
	int			nr_cpus;
	void			*tasks[0];
};

struct df_kdump_sub_hdr {
	unsigned long	phys_base;
	int		dump_level;
	int		split;
	unsigned long	start_pfn;
	unsigned long	end_pfn;
	off_t		offset_vmcoreinfo;
	unsigned long	size_vmcoreinfo;
};

/*
 * File local static data
 */
static struct {
	struct df_kdump_hdr	hdr;	/* kdump (diskdump) dump header */
	struct df_kdump_sub_hdr	shdr;	/* kdump subheader */
} l;

#ifdef DEBUG
static void print_header(void)
{
	STDERR("diskdump main header\n");
	STDERR("  signature        : %s\n", l.hdr.signature);
	STDERR("  header_version   : %d\n", l.hdr.header_version);
	STDERR("  status           : %d\n", l.hdr.status);
	STDERR("  block_size       : %d\n", l.hdr.block_size);
	STDERR("  sub_hdr_size     : %d\n", l.hdr.sub_hdr_size);
	STDERR("  bitmap_blocks    : %d\n", l.hdr.bitmap_blocks);
	STDERR("  max_mapnr        : 0x%x\n", l.hdr.max_mapnr);
	STDERR("  total_ram_blocks : %d\n", l.hdr.total_ram_blocks);
	STDERR("  device_blocks    : %d\n", l.hdr.device_blocks);
	STDERR("  written_blocks   : %d\n", l.hdr.written_blocks);
	STDERR("  current_cpu      : %d\n", l.hdr.current_cpu);
	STDERR("  nr_cpus          : %d\n", l.hdr.nr_cpus);
	STDERR("kdump sub header\n");
	STDERR("  phys_base        : 0x%lx\n", l.shdr.phys_base);
	STDERR("  dump_level       : %d\n", l.shdr.dump_level);
	STDERR("  split            : %d\n", l.shdr.split);
	STDERR("  start_pfn        : 0x%lx\n", l.shdr.start_pfn);
	STDERR("  end_pfn          : 0x%lx\n", l.shdr.end_pfn);
}
#endif

/*
 * Read kdump dump header
 */
static int read_kdump_hdr(void)
{
	if ((zg_type(g.fh) == ZG_TYPE_FILE) && (zg_size(g.fh) < sizeof(l.hdr)))
		return -ENODEV;
	zg_read(g.fh, &l.hdr, sizeof(l.hdr), ZG_CHECK);
	if (memcmp(l.hdr.signature, "KDUMP", 5) != 0)
		return -ENODEV;
	zg_seek(g.fh, l.hdr.block_size, ZG_CHECK);
	zg_read(g.fh, &l.shdr, sizeof(l.shdr), ZG_CHECK);
	dfi_attr_version_set(l.hdr.header_version);
	dfi_attr_real_cpu_cnt_set(l.hdr.nr_cpus);
	dfi_arch_set(DFI_ARCH_64);
#ifdef DEBUG
	print_header();
#endif
	return 0;
}

/*
 * Initialize kdump DFI
 */
static int dfi_kdump_init(void)
{
	if (read_kdump_hdr() != 0)
		return -ENODEV;
	dfi_mem_chunk_add(0, l.hdr.max_mapnr * PAGE_SIZE, NULL, NULL);
	return 0;

}

/*
 * S390 DFI operations
 */
struct dfi dfi_kdump = {
	.name		= "kdump",
	.init		= dfi_kdump_init,
	.feat_bits	= 0,
};
