/* ====================================================================
 * ===  Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 * 
 *    Project              : SagCAD
 *    Description          : CAD/CAM
 *    Source               : List_Undo.c
 * 
 *    ----------------------------------
 * 
 *    License              : GNU General Public License (GPL)
 *    Copyright            : (C) 1998-2007 by Yutaka Sagiya
 *    email                : kappa@a6s.highway.ne.jp
 *                         : yutaka@sagiya.com
 *    Begin                : 2001/01/16
 *    Last                 : 2007/11/08
 * ====================================================================
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include "MemoryLeak.h"
#include "List_cad.h"
#include "List_Dimension.h"
#include "List_PolyLine.h"
#include "List_Ellipse.h"
#include "List_Vertex.h"
#include "List_insert.h"
#include "List_Block.h"
#include "toolbar.h"
#define _LIST_UNDO_
#include "List_Undo.h"
#include "List_Select.h"
#include "etc.h"





/* -------------------------------------------------------------------
 * Undo_index をインクリメントする
 * -------------------------------------------------------------------
 * Undo/Redo の途中のときは、新しい図形を書くので
 * それ以降の Redo 情報は削除される
 *	
 * -------------------------------------------------------------------
 */
void undo_index_plus(void)
{
	/* Undo/Redo の途中で新しい図形の操作をしたとき */
	if (Undo_index + 1 != Undo_Now) {
		/* Undo_Now 以降の index を持つ Undo Buffer を削除する */
		Redo_Delete();
		Undo_index = Undo_Now - 1;
	}

	Undo_index++;
	Undo_Now = Undo_index + 1;

	undo_buttun(1);
	redo_buttun(0);
}



/* -------------------------------------------------------------------
 * Undo  CAD or Dimension  Data をリストの最初に追加
 * -------------------------------------------------------------------
 * 外部変数 Undo_index はこの関数の外であらかじめ設定しておくこと
 *	
 * type : 11 : CAD_NEW  : CAD Data の新規作成の記録
 *		: 12 : CAD_DEL  : CAD Data の削除の記録
 *		: 13 : CAD_EDIT : CAD Data の編集の記録
 * 　　 : 21 : DIMENSION_NEW  : Dimension Data の新規作成の記録
 *		: 22 : DIMENSION_DEL  : Dimension Data の削除の記録
 *		: 23 : DIMENSION_EDIT : Dimension Data の編集の記録
 * -------------------------------------------------------------------
 */
int undo_list_add_first(int type, LPDIAGRAM diagram_point, DIAGRAM diagram)
{
	UNDO_LIST *p;
//	int i;


	/* データ (LIST) を書き込むための領域を確保する */
	p_undo_list = (UNDO_LIST *)xmalloc(sizeof(UNDO_LIST));

	/* -----------------------------------------------------
	 * 新規のときは作成したデータを指すアドレス
	 * 
	 * 削除のときは削除したデータを指すアドレス
	 * (実際に削除せずにデータを Undo Buffer に移す)
	 * 
	 * 編集のときは、前のデータを変更せずに新しいデータを作成して
	 * ＣＡＤデータリストを入れ代える。
	 * そして、前のデータをアンドゥリストに入れる。
	 * その際、next には新しいデータのアドレスを入れておく。
	 */

	/* 確保した領域にデータ (undo) を書き込む */
	p_undo_list->type = type;
	p_undo_list->index = Undo_index;
	p_undo_list->next = NULL;


	/* 新規 */
	if (type == CAD_NEW) {
		p_undo_list->diagram_point.cad_point = diagram_point.cad_point;
		p_undo_list->diagram.cad = diagram.cad;
	}
	else if (type == DIMENSION_NEW) {
		p_undo_list->diagram_point.dimension_point = diagram_point.dimension_point;
		p_undo_list->diagram.dimension = diagram.dimension;
	}
	else if (type == POLYLINE_NEW) {
		p_undo_list->diagram_point.polyline_point = diagram_point.polyline_point;
		p_undo_list->diagram.polyline = diagram.polyline;
	}
	else if (type == ELLIPSE_NEW) {
		p_undo_list->diagram_point.ellipse_point = diagram_point.ellipse_point;
		p_undo_list->diagram.ellipse = diagram.ellipse;
	}
	else if (type == INSERT_NEW) {
		p_undo_list->diagram_point.insert_point = diagram_point.insert_point;
		p_undo_list->diagram.insert = diagram.insert;
	}
	else if (type == BLOCK_NEW) {
		p_undo_list->diagram_point.block_point = diagram_point.block_point;
		p_undo_list->diagram.block = diagram.block;
	}


	/* 削除 */
	else if (type == CAD_DEL) {
		p_undo_list->diagram_point.cad_point = diagram_point.cad_point;
		p_undo_list->diagram.cad = diagram.cad;
	}
	else if (type == DIMENSION_DEL) {
		p_undo_list->diagram_point.dimension_point = diagram_point.dimension_point;
		p_undo_list->diagram.dimension = diagram.dimension;
	}
	else if (type == POLYLINE_DEL) {
		p_undo_list->diagram_point.polyline_point = diagram_point.polyline_point;
		p_undo_list->diagram.polyline = diagram.polyline;
	}
	else if (type == ELLIPSE_DEL) {
		p_undo_list->diagram_point.ellipse_point = diagram_point.ellipse_point;
		p_undo_list->diagram.ellipse = diagram.ellipse;
	}
	else if (type == INSERT_DEL) {
		p_undo_list->diagram_point.insert_point = diagram_point.insert_point;
		p_undo_list->diagram.insert = diagram.insert;
	}
	else if (type == BLOCK_DEL) {
		p_undo_list->diagram_point.block_point = diagram_point.block_point;
		p_undo_list->diagram.block = diagram.block;
	}


	/* 編集 */
	else if (type == CAD_EDIT) {
		p_undo_list->diagram_point.cad_point = diagram_point.cad_point;
		p_undo_list->diagram.cad = diagram.cad;
	}
	else if (type == DIMENSION_EDIT) {
		p_undo_list->diagram_point.dimension_point = diagram_point.dimension_point;
		p_undo_list->diagram.dimension = diagram.dimension;
	}
	else if (type == POLYLINE_EDIT) {
		p_undo_list->diagram_point.polyline_point = diagram_point.polyline_point;
		p_undo_list->diagram.polyline = diagram.polyline;
	}
	else if (type == ELLIPSE_EDIT) {
		p_undo_list->diagram_point.ellipse_point = diagram_point.ellipse_point;
		p_undo_list->diagram.ellipse = diagram.ellipse;
	}
	else if (type == INSERT_EDIT) {
		p_undo_list->diagram_point.insert_point = diagram_point.insert_point;
		p_undo_list->diagram.insert = diagram.insert;
	}
	else if (type == BLOCK_EDIT) {
		p_undo_list->diagram_point.block_point = diagram_point.block_point;
		p_undo_list->diagram.block = diagram.block;
	}


	else {
		xfree(p_undo_list);
		return 0;
	}


	/* -----------------------------------------------------
	 * head & tail の両方が NULL のときは、１個もデータがない 
	 */
	if (undo_list_info.head == NULL && undo_list_info.tail == NULL) {
		undo_list_info.head = p_undo_list;
		undo_list_info.tail = p_undo_list;
	}
	/* -----------------------------------------------------
	 * データが１個以上ある
	 */
	else {
		p = undo_list_info.head;
		undo_list_info.head = p_undo_list;
		p_undo_list->next = p;
	}
	return 0;
}





/* -------------------------------------------------------------------
 * Redo を削除する
 * -------------------------------------------------------------------
 * Undo/Redo の途中のときは、新しい図形を書くので
 * それ(Undo_Now)以降の Redo 情報は削除する
 *	
 * -------------------------------------------------------------------
 */
void Redo_Delete(void)
{
	UNDO_LIST *p;


	p = search_list_undo_end(Undo_Now);
	if (p == NULL) {
		return;
	}

	while (undo_list_info.head != NULL && undo_list_info.head->index >= Undo_Now) {
		if (undo_list_free(undo_list_info.head) == 0) {
			break;
		}
	}
}





/* -------------------------------------------------------------------
 * 引数 index と同じ最初の Undo_index を探し,その Undo Buffer のポインタを返す	
 * 
 */
UNDO_LIST *search_list_undo_start(long index)
{
	UNDO_LIST *p;

	p = undo_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while( p != NULL && p->index > index ) {
		p = p->next;	/* ポインタを次のデータに移す */
	}

	if (p != NULL && p->next != NULL) {
		return p;
	}
	else {
		return NULL;
	}
}





/* -------------------------------------------------------------------
 * 引数 index と同じ最後の Undo_index を探し,その Undo Buffer のポインタを返す	
 * 
 */
UNDO_LIST *search_list_undo_end(long index)
{
	UNDO_LIST *p = NULL;
	UNDO_LIST *pb = NULL;


	p = undo_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while( p != NULL && p->index >= index ) {
		pb = p;
		p = p->next;	/* ポインタを次のデータに移す */
	}

	if (pb != NULL) {
		return pb;
	}
	else {
		return NULL;
	}
}





/* -------------------------------------------------------------------
 * Undo
 * -------------------------------------------------------------------
 * アンドゥ実行
 *	
 *	 アンドゥは、まず先に Undo_Now を -1 にして、複数または１個ある Undo_Now と
 * 同じ番号の index に対して処理を実行する。
 *	 １個のデータに対して、実際の処理をするのは、UndoRedo() 関数でこれは、
 * リドゥで処理する関数と同じ。
 *	 アンドゥは、index の中でも、後から入ったデータから処理をする。(リドゥは、逆)
 * -------------------------------------------------------------------
 */
int Undo(void)
{
	UNDO_LIST *p;
	UNDO_LIST *p_start;
	UNDO_LIST *p_end;


	/* Undo は、1 よりも大きくなければ実行できない */
	if (1 < Undo_Now) Undo_Now--;
	else return 0;


	/* index の最初のアドレスを探す */
	p_start = search_list_undo_start(Undo_Now);
	/* index の最後のアドレスを探す */
	p_end = search_list_undo_end(Undo_Now);


	/* index が１個だけのとき */
	if (p_start == p_end) {
		UndoRedo(p_start);
	}

	else if (p_start == NULL && p_end == undo_list_info.tail) {
		UndoRedo(p_end);
	}



	/* index が複数あるとき */
	else {
		p = p_start;
		while( p != NULL && p != p_end->next ) {
			UndoRedo(p);
			p = p->next;	/* ポインタを次のデータに移す */
		}
	}
	return 1;
}



/* -------------------------------------------------------------------
 * Redo
 * -------------------------------------------------------------------
 * リドゥ実行
 *	
 *	 リドゥは、Undo_Now を +1 にするのは、アンドゥとは逆で 処理の後。
 *	 複数または１個ある Undo_Now と同じ番号の index に対して処理を実行するが、
 * index の中でも、先に入った古いデータから処理をする。(アンドゥとは逆)。
 *	 １個のデータに対して、実際の処理をするのは、UndoRedo() 関数でこれは、
 * アンドゥで処理する関数と同じ。
 * -------------------------------------------------------------------
 */
int Redo(void)
{
	UNDO_LIST *p = NULL;
	UNDO_LIST *pb = NULL;
	UNDO_LIST *p_start = NULL;
	UNDO_LIST *p_end = NULL;


	/* Rndo は、(Undo_index + 1) よりも小さくなければ実行できない */
	if (Undo_index + 1 > Undo_Now) {
		/* index の最初のアドレスを探す */
		p_start = search_list_undo_start(Undo_Now);
		/* index の最後のアドレスを探す */
		p_end = search_list_undo_end(Undo_Now);


		if (p_start == NULL && p_end == undo_list_info.tail) {
//			LogFileWrite("Undo run : p_end = tail\n");
			UndoRedo(p_end);
		}

		else {
			/* -------------------------------------------------
			 * p_end から p_start の方向で UndoRedo() を実行する 
			 * index のデータが１個だけのときは、while を抜けて
			 * UndoRedo(p_start); だけを実行する。
			 */
			while (p_start != p_end) {
				p = p_start;
				while (p != NULL && p != p_end) {
					pb = p;
					p = p->next;
				}
				UndoRedo(p);
				p_end = pb;
			}
			UndoRedo(p_start);
		}
		Undo_Now++;
		return 1;
	}
	return 0;
}



/* -------------------------------------------------------------------
 * Undo/Redo の実際の処理をする
 * -------------------------------------------------------------------
 *	 (type = 11 or 21) 新規データを作ったという情報があれば、そのデータを
 * CAD データの List から外して(メモリから開放しない)、(type = 12 or 22) に変更する。
 *	
 *	 (type = 12 or 22) データを削除したという情報があれば、そのデータを
 * CAD データの List に追加して、(type = 11 or 21) に変更する。
 *	
 *	 (type = 13 or 23) データを変更したという情報があれば、Undo List にあるデータと
 * Undo List の next メンバが指している CAD のデータを入れかえる。
 * -------------------------------------------------------------------
 */
void UndoRedo(UNDO_LIST *undo)
{
	CAD dummy_cad;
	DIMENSION dummy_dimension;
	ELLIPSE dummy_ellipse;
	POLYLINE dummy_polyline;
	INSERT dummy_insert;


	if (undo == NULL) return;

	/* -----------------------------------------------------
	 * CAD Data 
	 * -----------------------------------------------------
	 */
	/* New */
	if (undo->type == CAD_NEW) {
		/* CAD のデータを削除して、type = 12 にする */
		cad_list_delete(cad_search_cad(undo->diagram_point.cad_point, &cad_list_info), &cad_list_info);
		undo->type = CAD_DEL;
	}

	/* Delete */
	else if (undo->type == CAD_DEL) {
		/* CAD のデータに追加して、type = 11 にする */
		/* CAD Data をリストの最初に追加 */
		r_cad_list_add_first(undo->diagram_point.cad_point, &cad_list_info);
		undo->type = CAD_NEW;
	}

	/* Edit */
	else if (undo->type == CAD_EDIT) {
		/* CAD_LIST の領域のデータを keep */
		dummy_cad = *undo->diagram_point.cad_point;
		/* CAD_LIST の領域に Undo にに入っていたデータを入れる */
		*undo->diagram_point.cad_point = undo->diagram.cad;
		/* Undo の領域に keep してあった変更前の CAD のデータをいれる */
		undo->diagram.cad = dummy_cad;
	}



	/* -----------------------------------------------------
	 * DIMENSION Data 
	 * -----------------------------------------------------
	 */
	/* New */
	else if (undo->type == DIMENSION_NEW) {
		/* DIMENSION のデータを削除して、type = 22 にする */
		dimension_list_delete(dimension_search_dimension(undo->diagram_point.dimension_point, &dimension_list_info), &dimension_list_info);
		undo->type = DIMENSION_DEL;
	}

	/* Delete */
	else if (undo->type == DIMENSION_DEL) {
		/* DIMENSION のデータに追加して、type = 21 にする */
		/* CAD Data をリストの最初に追加 */
		r_dimension_list_add_first(undo->diagram_point.dimension_point, &dimension_list_info);
		undo->type = DIMENSION_NEW;
	}
	/* Edit */
	else if (undo->type == DIMENSION_EDIT) {
		/* DIMENSION_LIST の領域のデータを keep */
		dummy_dimension = *undo->diagram_point.dimension_point;
		/* DIMENSION_LIST の領域に Undo にに入っていたデータを入れる */
		*undo->diagram_point.dimension_point = undo->diagram.dimension;
		/* Undo の領域に keep してあった変更前の DIMENSION のデータをいれる */
		undo->diagram.dimension = dummy_dimension;
	}



	/* -----------------------------------------------------
	 * ELLIPSE Data 
	 * -----------------------------------------------------
	 */
	/* New */
	else if (undo->type == ELLIPSE_NEW) {
		/* ELLIPSE のデータを削除して、type = ELLIPSE_DEL にする */
		ellipse_list_delete(ellipse_search_ellipse(undo->diagram_point.ellipse_point, &ellipse_list_info), &ellipse_list_info);
		undo->type = ELLIPSE_DEL;
	}

	/* Delete */
	else if (undo->type == ELLIPSE_DEL) {
		/* ELLIPSE のデータに追加して、type = ELLIPSE_NEW にする */
		/* ELLIPSE Data をリストの最初に追加 */
		r_ellipse_list_add_first(undo->diagram_point.ellipse_point, &ellipse_list_info);
		undo->type = ELLIPSE_NEW;
	}

	/* Edit */
	else if (undo->type == ELLIPSE_EDIT) {
		/* ELLIPSE_LIST の領域のデータを keep */
		dummy_ellipse = *undo->diagram_point.ellipse_point;
		/* ELLIPSE_LIST の領域に Undo にに入っていたデータを入れる */
		*undo->diagram_point.ellipse_point = undo->diagram.ellipse;
		/* Undo の領域に keep してあった変更前の ELLIPSE のデータをいれる */
		undo->diagram.ellipse = dummy_ellipse;
	}



	/* -----------------------------------------------------
	 * POLYLINE Data 
	 * -----------------------------------------------------
	 */
	/* New */
	else if (undo->type == POLYLINE_NEW) {
		/* POLYLINE のデータを削除して、type = POLYLINE_DEL にする */
		polyline_list_delete(polyline_search_polyline(undo->diagram_point.polyline_point, &polyline_list_info), &polyline_list_info);
		undo->type = POLYLINE_DEL;
	}

	/* Delete */
	else if (undo->type == POLYLINE_DEL) {
		/* POLYLINE のデータに追加して、type = POLYLINE_NEW にする */
		/* POLYLINE Data をリストの最初に追加 */
		r_polyline_list_add_first(undo->diagram_point.polyline_point, &polyline_list_info);
		undo->type = POLYLINE_NEW;
	}

	/* Edit */
	else if (undo->type == POLYLINE_EDIT) {
		/* ELLIPSE_LIST の領域のデータを keep */
		dummy_polyline = *undo->diagram_point.polyline_point;
		/* ELLIPSE_LIST の領域に Undo にに入っていたデータを入れる */
		*undo->diagram_point.polyline_point = undo->diagram.polyline;
		/* Undo の領域に keep してあった変更前の ELLIPSE のデータをいれる */
		undo->diagram.polyline = dummy_polyline;
	}



	/* -----------------------------------------------------
	 * INSERT Data 
	 * -----------------------------------------------------
	 */
	/* New */
	else if (undo->type == INSERT_NEW) {
		/* INSERT のデータを削除して、type = INSERT_DEL にする */
		insert_list_delete(insert_search_insert(undo->diagram_point.insert_point, &insert_list_info), &insert_list_info);
		undo->type = INSERT_DEL;
	}

	/* Delete */
	else if (undo->type == INSERT_DEL) {
		/* INSERT のデータに追加して、type = INSERT_NEW にする */
		/* INSERT Data をリストの最初に追加 */
		r_insert_list_add_first(undo->diagram_point.insert_point, &insert_list_info);
		undo->type = INSERT_NEW;
	}

	/* Edit */
	else if (undo->type == INSERT_EDIT) {
		/* ELLIPSE_LIST の領域のデータを keep */
		dummy_insert = *undo->diagram_point.insert_point;
		/* ELLIPSE_LIST の領域に Undo にに入っていたデータを入れる */
		*undo->diagram_point.insert_point = undo->diagram.insert;
		/* Undo の領域に keep してあった変更前の ELLIPSE のデータをいれる */
		undo->diagram.insert = dummy_insert;
	}



}



/* ---------------------------------------------------------
 * ポインタが指す Undo Data をリストから削除
 */
int undo_list_free(UNDO_LIST *del_List)
{
char str[256];
	UNDO_LIST *p = NULL;
	UNDO_LIST *pb = NULL;
	long i;

	/* -------------------------------------------
	 * 削除するポインタの前のデータを知りたいので、
	 * head (リストの先頭) から削除するポインタが
	 * 見つかるまで繰り返し、前のデータを探す。
	 */
	p = undo_list_info.head;
	i = 0;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while(p != NULL && p != del_List) {
		i++;
		pb = p;			/* 前のデータのポインタとして保存しておく */
		p = p->next;	/* ポインタを次のデータに移す */
	}


	/* 見つかった場合 (p が NULL ではないとき) */
	if (p != NULL){
		/* 先頭のデータ (i = 0 のとき) */
		if (i == 0) {
			/* 先頭のデータ１個しかない場合(次のデータのポインタが NULL) */
			if (p->next == NULL) {
				undo_list_info.head = NULL;
				undo_list_info.tail = NULL;
			}
			/* ２個以上データがあるとき */
			else 
				undo_list_info.head = p->next;
		}

		/* 最後のデータのとき (p->next = NULL) */
		else if (p->next == NULL) pb->next = NULL;

		/* 中間のデータ (上の条件以外) */
		else pb->next = p->next;


		/* -------------------------------------------------
		 * CAD DATA 
		 */
		if (del_List->type == CAD_DEL/* || del_List->type == 13*/) {
			if (del_List->diagram_point.cad_point != NULL) {
				xfree(del_List->diagram_point.cad_point);
			}
		}


		/* -------------------------------------------------
		 * DIMENSION DATA 
		 */
		/* 削除 */
		else if (del_List->type == DIMENSION_DEL) {
			/* AssistLine */
			if (
				del_List->diagram_point.dimension_point->index > 0 
				&& 
				del_List->diagram_point.dimension_point->AssistLine != NULL
				) 
			{
				xfree(del_List->diagram_point.dimension_point->AssistLine);
			}

			/* diagram_of_char */
			if (
				del_List->diagram_point.dimension_point->diagram_of_char_index > 0 
				&& 
				del_List->diagram_point.dimension_point->diagram_of_char != NULL
				) 
			{
				xfree(del_List->diagram_point.dimension_point->diagram_of_char);
			}

			/* DIMENSION */
			if (del_List->diagram_point.dimension_point != NULL) {
				xfree(del_List->diagram_point.dimension_point);
			}
			else if (del_List->diagram_point.dimension_point == NULL) {
				printf("This place is funny for NULL.");
			}
		}
		/* 編集 */
		else if (del_List->type == DIMENSION_EDIT) {
			/* AssistLine */
			if (
				del_List->diagram.dimension.index > 0 
				&& 
				del_List->diagram.dimension.AssistLine != NULL
				) 
			{
				// ここは、寸法線は、編集操作はしないので、実データだけ開放すればいい。
				//xfree(del_List->diagram.dimension.AssistLine);
			}

			/* diagram_of_char */
			if (
				del_List->diagram.dimension.diagram_of_char_index > 0 
				&& 
				del_List->diagram.dimension.diagram_of_char != NULL
				) 
			{
				xfree(del_List->diagram.dimension.diagram_of_char);
			}
		}


		/* -------------------------------------------------
		 * POLYLINE DATA 
		 */
		if (del_List->type == POLYLINE_DEL) {
			if (del_List->diagram_point.polyline_point->vertex_list_info.head != NULL) {
				vertex_list_all_free(&del_List->diagram_point.polyline_point->vertex_list_info);
			}
			
			if (del_List->diagram_point.polyline_point != NULL) {
				xfree(del_List->diagram_point.polyline_point);
			}
		}


		/* -------------------------------------------------
		 * ELLIPSE DATA 
		 */
		if (del_List->type == ELLIPSE_DEL) {
			if (del_List->diagram_point.ellipse_point != NULL) {
				xfree(del_List->diagram_point.ellipse_point);
			}
		}


		xfree(del_List);
		return 1;
	}
	else {
		sprintf(str, "list_undo.cpp : undo_list_free() : Not Found(0x%x)\n", (int)p);
		OneShotLog(str);
	}

	return 0;
//	xfree(del_List);
}





/* ---------------------------------------------------------
 *	全リストを削除
 */
void undo_list_all_free(void)
{
	while (undo_list_info.head != NULL) {
		undo_list_free(undo_list_info.head);
	}
	Undo_index = 0;
	Undo_Now = 1;
}





/* ---------------------------------------------------------
 * UNDO_LIST のデータ log
 *	
 */
void undo_list_log(void)
{
	UNDO_LIST *p;
	int i = 0;
	char str[256], msg[256];

	sprintf(str, "----- Undo List Start -----\n");
	OneShotLog(str);
	sprintf(str, "Undo_index = %d   Undo_Now = %d\n", (int)Undo_index, (int)Undo_Now);
	OneShotLog(str);

	p = undo_list_info.head;
	while (p != NULL) {
		i++;


		if (p->type == 11) {
			sprintf(msg, "データ作成  POINT は、CAD_LIST 上");
		}
		if (p->type == 12) {
			sprintf(msg, "データ削除  POINT は、CAD_LIST 上にはない。");
		}
		if (p->type == 13) {
			sprintf(msg, "データ編集  POINT は、CAD_LIST 上。CAD は、編集前のデータ。");
		}


		sprintf(str, "%d : 0x%x : index = %d  %s\n", 
				i, (int)p->diagram_point.cad_point, (int) p->index, msg);
		OneShotLog(str);
		if (p->diagram_point.cad_point != NULL) {
			if (p->diagram_point.cad_point != NULL) {
				sprintf (str, "    POINT    : %d (%f,%f)-(%f,%f)  (%f,%f),R%f\n", 
						p->diagram_point.cad_point->code, 
						p->diagram_point.cad_point->sx, p->diagram_point.cad_point->sy, 
						p->diagram_point.cad_point->ex, p->diagram_point.cad_point->ey, 
						p->diagram_point.cad_point->cx, p->diagram_point.cad_point->cy, p->diagram_point.cad_point->r);
				OneShotLog(str);
			}
		}		
		
		if (p->type == 13) {
			sprintf (str, "    OLD DATA : %d (%f,%f)-(%f,%f)  (%f,%f),R%f\n", 
						p->diagram_point.cad_point->code, 
						p->diagram_point.cad_point->sx, p->diagram_point.cad_point->sy, 
						p->diagram_point.cad_point->ex, p->diagram_point.cad_point->ey, 
						p->diagram_point.cad_point->cx, p->diagram_point.cad_point->cy, p->diagram_point.cad_point->r);
			OneShotLog(str);
		}

		p = p->next;
	}

	sprintf(str, "----- Undo List End   -----\n");
	OneShotLog(str);
}





/* ====================================================================
 * ===  Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 *    Project              : SagCAD
 *    Source               : List_Undo.c
 * ====================================================================
 */
