/*
 * Copyright (C) 1993-2004 Robert & Jeremy Lain
 * See AUTHORS file for a full list of contributors.
 *
 * $Id: formmain.cpp,v 1.109 2005/05/15 15:16:04 jeremy_laine Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
  #include "config.h"
#endif

#include "formmain.h"
#include "formdef.h"
#include "formhelp.h"
#include "formmould.h"
#include "formrig.h"

#include "sailcutqt.h"
#include "saildoc.h"
#include "sailpainter.h"
#include "sailreader-xml.h"
#include "sailwriter-carlson.h"
#include "sailwriter-dxf.h"
#include "sailwriter-txt.h"
#include "sailwriter-xml.h"
#include "sailviewer-panel.h"

#include <qdir.h>
#include <qlayout.h>
#include <qmenubar.h>
#include <qstatusbar.h>
#include <qtabwidget.h>



/**
 * Constructs Sailcut CAD's main window.
 *
 * @param pref the user's preferences
 */
CFormMain::CFormMain(CSailApp *myApp, QWidget *parent, bool modal)
        : QDialog(parent, 0, modal), app(myApp), prefs(&myApp->prefs)
{
    setMinimumSize( QSize( 300, 220 ) );

    /* locate Handbook */
    handbook = app->findHandbook(prefs->language.data());
    
    /* create blank menu bar */
    menubar = new QMenuBar(this);

    /* put it all together */
    QGridLayout *layout = new QGridLayout( this, 1, 1, 11, 6 );
    layout->addItem( new QSpacerItem( 20, 30, QSizePolicy::Minimum, QSizePolicy::Fixed ), 0, 0 );

    tabs = new QTabWidget(this);
    layout->addWidget( tabs, 1, 0);

    // create status bar
    statusbar = new QStatusBar(this);
    layout->addWidget( statusbar, 2, 0);

    menuRecent = new QPopupMenu(this);

    CSailViewerPanel *tmp;
#ifdef HAVE_GL

    tmp = new CSailViewerPanel(tabs, SHADED, true);
    panel.push_back(tmp);
#endif

    tmp = new CSailViewerPanel(tabs, WIREFRAME, true);
    panel.push_back(tmp);
    tmp = new CSailViewerPanel(tabs, WIREFRAME, false);
    panel.push_back(tmp);

    for (unsigned int i = 0 ; i < panel.size(); i++)
        tabs->addTab(panel[i],"");

    // set language and size
    languageChange();

    resize( QSize(prefs->sailWindowWidth,prefs->sailWindowHeight).expandedTo(minimumSizeHint()) );
    clearWState( WState_Polished );
}


/**
 * This event is received when the user closes the dialog.
 */
void CFormMain::closeEvent(QCloseEvent *e)
{
    prefs->sailWindowHeight = height();
    prefs->sailWindowWidth = width();
    QDialog::closeEvent(e);
}


/**
 * We received a keypress, we pass it down to the visible tab.
 */
void CFormMain::keyPressEvent ( QKeyEvent * e )
{
    panel[tabs->currentPageIndex()]->keyPressEvent(e);
}


/**
 * This method is called whenever we read or write a sail definition.
 * It updates the Most Recently Used files list correspondingly.
 *
 * @param event The action performed on the file (read, write)
 * @param file The file that was accessed
 */
void CFormMain::fileAccess(QString event, QString file)
{
    statusbar->message(event);
    prefs->mruSaildef.touchEntry(file);
    makeMenuMru();
}


/**
 * Sets the strings of the subwidgets using the current
 * language.
 */
void CFormMain::languageChange()
{
    setCaption( "Sailcut CAD" );

    /* File menu */
    QPopupMenu * menuFile = new QPopupMenu(this);
    menuFile->insertItem( tr("&New"), this, SLOT( slotNew() ) );
    menuFile->insertItem( tr("&Open"), this, SLOT( slotOpen() ) );
    menuFile->insertItem( tr("Open &recent"), menuRecent );
    menuFile->insertSeparator();
    QPopupMenu * menuPrint = new QPopupMenu(this);
    menuPrint->insertItem( tr("data"), this, SLOT( slotPrintData() ) );
    menuPrint->insertItem( tr("drawing"), this, SLOT( slotPrintDwg() ) );
    menuPrint->insertItem( tr("develop"), this, SLOT( slotPrintDev() ) );
    menuFile->insertItem( tr("&Print"), menuPrint );
    menuFile->insertSeparator();
    menuFile->insertItem( tr("&Save"), this, SLOT( slotSave() ) );
    menuFile->insertItem( tr("Save &As"), this, SLOT( slotSaveAs() ) );
    /* export 3d submenu */
    QPopupMenu * menuExport3d = new QPopupMenu(this);
    menuExport3d->insertItem( tr("to &DXF"), this, SLOT( slotExportDXF() ) );
    menuExport3d->insertItem( tr("to &TXT sail"), this, SLOT( slotExportTXT() ) );
    menuExport3d->insertItem( tr("to &XML sail"), this, SLOT( slotExportXML() ) );
    menuFile->insertItem( tr("E&xport 3D sail"), menuExport3d );

    /* export flat submenu */
    QPopupMenu * menuExportFlat = new QPopupMenu(this);
    menuExportFlat->insertItem( tr("to &Carlson plotter"), this, SLOT( slotExportFlatCarlson() ) );
    menuExportFlat->insertItem( tr("to &DXF"), this, SLOT( slotExportFlatDXF() ) );
    menuExportFlat->insertItem( tr("to &TXT sail"), this, SLOT( slotExportFlatTXT() ) );
    menuExportFlat->insertItem( tr("to &XML sail"), this, SLOT( slotExportFlatXML() ) );
    menuFile->insertItem( tr("Export &development"), menuExportFlat );

    menuFile->insertSeparator();
    menuFile->insertItem( tr("&Quit"), this, SLOT( close() ) );

    /* View menu */
    QPopupMenu * menuView = new QPopupMenu(this);
    menuView->insertItem( tr("&Dimensions"), this, SLOT( slotDef() ) );
    menuView->insertItem( tr("&Mould"), this, SLOT ( slotMould() ) );
    menuView->insertItem( tr("&Patches") );
    menuView->insertItem( tr("&Rig"), this, SLOT ( slotRig() ) );
    menuView->insertSeparator();

    /* language submenu */
    QPopupMenu * menuLanguage = new QPopupMenu(this);
    // language text is not to be translated
    menuLanguage->insertItem( "English", this, SLOT ( slotLanguage(int)), 0, 0 );
    menuLanguage->insertItem( "Franais", this, SLOT ( slotLanguage(int)), 0, 1 );
    menuLanguage->insertItem( "Nederlands", this, SLOT ( slotLanguage(int)), 0, 2 );
    menuLanguage->insertItem( "Deutsch", this, SLOT ( slotLanguage(int)), 0, 3 );
    menuLanguage->insertItem( "Italiano", this, SLOT ( slotLanguage(int)), 0, 4 );
    menuLanguage->insertItem( "Norsk", this, SLOT ( slotLanguage(int)), 0, 5 );
    menuLanguage->insertItem( "Dansk", this, SLOT ( slotLanguage(int)), 0, 6 );
    menuLanguage->insertItem( "Svenska", this, SLOT ( slotLanguage(int)), 0, 7 );
    menuLanguage->insertItem("Portugus", this, SLOT ( slotLanguage(int )), 0, 8 );
    menuLanguage->insertItem( "Espaol", this, SLOT ( slotLanguage(int)), 0, 9 );
    menuLanguage->insertItem( "Russian", this, SLOT ( slotLanguage(int)), 0, 10 );

    menuView->insertItem( tr("Language"), menuLanguage );

    /* Help menu */
    QPopupMenu * menuHelp = new QPopupMenu(this);
    if (!handbook.isEmpty())
    {
        menuHelp->insertItem( tr("Sailcut &Handbook"), this, SLOT( slotHandbook() ) );
        menuHelp->insertSeparator();
    }
    menuHelp->insertItem( tr("About &Qt"), this, SLOT( slotAboutQt() ) );
    menuHelp->insertItem( tr("About &Sailcut"), this, SLOT( slotAbout() ) );

    /* insert menus into menu bar */
    menubar->clear();
    menubar->insertItem( tr("&File"), menuFile );
    menubar->insertItem( tr("&View"), menuView );
    menubar->insertItem( tr("&Help"), menuHelp );

    /* send changeLanguage to the tabs */
    for (unsigned int i = 0; i < panel.size(); i++)
        panel[i]->languageChange();

#ifdef HAVE_GL

    tabs->changeTab(panel[0], tr("shaded view"));
    tabs->changeTab(panel[1], tr("wireframe view"));
    tabs->changeTab(panel[2], tr("development"));
#else

    tabs->changeTab(panel[0], tr("wireframe view"));
    tabs->changeTab(panel[1], tr("development"));
#endif

}


/**
 * Creates the "Open Recent" menu from the Most Recently Used files list.
 */
void CFormMain::makeMenuMru()
{
    menuRecent->clear();
    for ( unsigned int i = 0; i < prefs->mruSaildef.size(); i++)
        menuRecent->insertItem( prefs->mruSaildef[i].data(), this, SLOT ( slotOpenRecent(int) ), 0, i );
}


/**
 * Replaces the current sail definition.
 *
 * @param newdef
 */
void CFormMain::setSailDef(const CSailDef newdef)
{
    saildef = newdef;
    sail = CSailWorker(saildef).makeSail(flatsail,dispsail);
    panel[0]->setSail(sail);
#ifdef HAVE_GL

    panel[1]->setSail(sail);
    panel[2]->setSail(dispsail);
#else

    panel[1]->setSail(dispsail);
#endif
}


/**
 * Opens a given sail file
 */
void CFormMain::show(const QString newname)
{
    // load preferences
    makeMenuMru();
    languageChange();

    // load or create sail
    CSailDef newdef;
    filename = newname;
    if (!filename.isNull())
    {
        newdef = CSailDefXmlReader("saildef").read(filename);
        fileAccess(tr("loaded '%1'").arg(filename),filename);
    }
    else
    {
        statusbar->message( tr("created new sail") );
    }
    setSailDef(newdef);
    QDialog::show();
}


/**
 * Displays the "About Sailcut" dialog box
 */
void CFormMain::slotAbout()
{
    QMessageBox::about( this, tr("About Sailcut"),
                        "<h1>Sailcut CAD"
#ifdef VERSION
                        " version : " VERSION
#endif
                        "</h1>"
                        "<p>Sailcut is a software for designing boat sails<br>"
                        "(C) 1993-2005 Robert & Jeremy Lain<br>"
                        "see  <i>http://sailcut.sourceforge.net/</i></p>"
                        "<p>Sailcut is a trademark registered by Robert Lain.</p>"
                        "<p>This program is distributed under the GNU Public License.</p>"
                      );
}


/**
 * Displays the "About Qt" dialog box
 */
void CFormMain::slotAboutQt()
{
    QMessageBox::aboutQt( this );
}



/**
 * Displays the sail CFormDef sail definition dialog.
 */
void CFormMain::slotDef()
{  // we pass the CFormDef a pointer to a copy of the sail definition so
    // that it can update it if necessary
    CSailDef saildefcopy = saildef;

    if ( CFormDef(this, &saildefcopy).exec() )
    {
        // we returned from the dialog with an 'OK',
        setSailDef(saildefcopy);
    }
}


/**
 * Export the 3D sail to a DXF file
 */
void CFormMain::slotExportDXF()
{
    CSailDxfWriter3d(sail, "3Dsail").writeDialog();
}


/**
 * Exports the 3D sail to a TXT file.
 */
void CFormMain::slotExportTXT()
{
    CSailTxtWriter(sail, "3Dsail").writeDialog();
}


/**
 * Exports the 3D sail to an XML file.
 */
void CFormMain::slotExportXML()
{
    CSailXmlWriter(sail, "3Dsail").writeDialog();
}


/**
 * Exports the flat sail to a Carlson plotter file
 */
void CFormMain::slotExportFlatCarlson()
{
    CSailCarlsonWriter(flatsail,"sail").writeDialog();
}


/**
 * Exports the flat sail with panels staggered as displayed to a DXF file
  */
void CFormMain::slotExportFlatDXF()
{
    CSailDxfWriter2d(dispsail,"sail").writeDialog();
}


/**
 * Exports the flat sail to a TXT sail file.
 */
void CFormMain::slotExportFlatTXT()
{
    CSailTxtWriter(flatsail,"sail").writeDialog();
}


/**
 * Exports the flat sail to an XML sail file
 */
void CFormMain::slotExportFlatXML()
{
    CSailXmlWriter(flatsail,"sail").writeDialog();
}


/**
 * Display the Sailcut handbook.
 */
void CFormMain::slotHandbook()
{
    if (!handbook.isEmpty())
    {
        CFormHelp(this, prefs, QDir(handbook).absPath()).exec();
    }
}


/**
 * Switches the current language.
 */
void CFormMain::slotLanguage(int index)
{
    string locale;
    switch (index)
    {
    case 0:
        locale = "en";
        break;
    case 1:
        locale = "fr";
        break;
    case 2:
        locale = "nl";
        break;
    case 3:
        locale = "de";
        break;
    case 4:
        locale = "it";
        break;
    case 5:
        locale = "no";
        break;
    case 6:
        locale = "dk";
        break;
    case 7:
        locale = "se";
        break;
    case 8:
        locale = "pt";
        break;
    case 9:
        locale = "es";
        break;
    case 10:
        locale = "ru";
        break;

    }
    prefs->language = locale;
    app->loadTranslation(locale.data());
    
    // try to locate handbook
    handbook = app->findHandbook(locale.data());
    
    languageChange();
}


/**
 * Displays the CFormMould sail mould definition dialog.
 */
void CFormMain::slotMould()
{  // we pass the CFormMould a pointer to a copy of the sail mould so
    // that it can update it if necessary
    CSailDef saildefcopy = saildef;

    if ( CFormMould(this, &saildefcopy.mould).exec() )
    {
        // we returned from the dialog with an 'OK'
        setSailDef(saildefcopy);
    }
}


/**
 * Creates a new sail
 */
void CFormMain::slotNew()
{
    filename = "";
    setSailDef(CSailDef());
    statusbar->message( tr("created new sail") );
}


/**
 * Displays a dialog box to open an XML sail definition.
 */
void CFormMain::slotOpen()
{
    CSailDef newdef;
    QString newname = CSailDefXmlReader("saildef").readDialog(newdef,filename);
    if (!newname.isNull())
    {
        filename = newname;
        fileAccess(tr("loaded '%1'").arg(filename), filename);
        setSailDef(newdef);
    }
}


/**
 * Opens a recently used sail definition.
 */
void CFormMain::slotOpenRecent(int index)
{
    filename = prefs->mruSaildef[index].data();
    try
    {
        setSailDef(CSailDefXmlReader("saildef").read(filename));
        fileAccess(tr("loaded '%1'").arg(filename), filename);
    }
    catch (CException e)
    {
        prefs->mruSaildef.removeEntry(filename);
        makeMenuMru();
        statusbar->message( tr("error loading '%1'").arg(filename) );
    }
}


/**
 * Print the current sail data.
 */
void CFormMain::slotPrintData()
{  // try printing
    try
    { // Epson printer scale is 96 pixel/inch
        QPrinter myprinter;
        myprinter.setOrientation(QPrinter::Portrait);
        myprinter.setFullPage(FALSE);

        bool ret = myprinter.setup(); // show printing dialog
        if (ret == true)
        {
            QPainter painter(&myprinter);

            QPaintDeviceMetrics pdm(&myprinter);
            //double aspect = (double)pdm.widthMM() / (double)pdm.heightMM();
            unsigned int dpix = pdm.logicalDpiX();
            //unsigned int dpiy = pdm.logicalDpiY();
            real scale = dpix/96;

            unsigned int fontsz1 = 11;
            painter.setFont(QFont("times", fontsz1 ));
            QString text1=" ", text2=" ", text3=" ", text4=" ";
            unsigned int x=0, x1=0, x2=0, x3=0, x4=0, y=0, w=0, h=0;
            x1 = int(scale*96);  // first position at 1 inch
            x2 = x1 + int(scale*16*fontsz1);
            x3 = x2 + int(scale*6*fontsz1);
            x4 = x3 + int(scale*6*fontsz1); // fourth position
            h = int(scale*1.5*fontsz1); // line spacing

            // text of page header
            y = int(1+scale*(1.25*fontsz1));
            x = int(1+scale*2*fontsz1);
            painter.drawText(x,y, tr("Sailcut data sheet"));

            // text of sail identification
            text2 = saildef.sailID;
            if (text2.length()<1)
                text2="*";
            painter.drawText(x2,y, text2);

            // draw boxes around header and ID
            h = int(scale*(1.5*fontsz1));
            w = int(scale*(12*fontsz1));
            x = int(1+scale*fontsz1);
            y = int(1+scale);
            painter.drawRect(x,y, w,h);

            w = int(scale*(30*fontsz1));
            x = x2- int(scale*fontsz1);
            painter.drawRect(x,y, w,h);

            // initialise printing font and position
            y = y + 3*h;
            h = int(scale*1.5*fontsz1); // line spacing

            // sail cut and type
            text1 = tr("Sail type");
            switch (saildef.sailType )
            {
            case MAINSAIL:
                text2 = tr("Mainsail");
                break;
            case JIB:
                text2 = tr("Jib");
                break;
            case WING:
                text2 = tr("Wing")+" @ " + QString::number(saildef.dihedralDeg) + tr("deg");
                break;
            }
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            y = y + h;

            text1 = tr("Sail layout");
            switch ( saildef.sailCut )
            {
            case CROSS:
                text2 = tr("Cross Cut");
                break;
            case HORIZONTAL:
                text2 = tr("Horizontal Cut");
                break;
            case RADIAL:
                text2 = tr("Radial Cut");
                break;
            case TWIST:
                text2 = tr("Twist Foot Cut");
                break;
            case VERTICAL:
                text2 = tr("Vertical Cut");
                break;
            case MITRE:
                text2 = tr("Mitre Cut");
                break;
            }
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            if ( saildef.sailCut == RADIAL )
            {
                text1  = QString::number(saildef.nbSections);
                text1  = text1 + tr(" sections,   ");
                text2 = QString::number(saildef.nbGores);
                text1  = text1 +text2 + tr(" head gores,   ");
                text2 = QString::number(saildef.nbLuffGores);
                text1  = text1 +text2 + tr(" luff gores.");
                painter.drawText(x3,y, text1);
            }
            y = y + h;

            painter.drawText(int(x1/2),y, tr("Rig "));
            y = y + h;

            //* boat data
            text1 = tr("Boat LOA");
            text2 = QString::number(saildef.LOA);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Fore triangle hoist I");
            text2 = QString::number(saildef.foreI);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Fore triangle base J");
            text2 = QString::number(saildef.foreJ);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Tack position X");
            text2 = QString::number(saildef.tackX);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Tack height Y");
            text2 = QString::number(saildef.tackY);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            painter.drawText(int(x1/2),y, tr("Sail dimensions"));
            y = y + h;

            // sides of the sail
            text1 = tr("Luff length");
            text2 = QString::number(saildef.luffL);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Mast/Luff rake");
            text2 = QString::number(saildef.rake);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Gaff angle wrt luff");
            text2 = QString::number(saildef.gaffDeg);
            text3 = "deg.";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Gaff length");
            text2 = QString::number(saildef.gaffL);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Foot length");
            text2 = QString::number(saildef.footL);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Leech length");
            text2 = QString::number(saildef.leechL);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            painter.drawText(int(x1/2), y, tr("Shape of edges"));
            y = y + h;

            // shape of sides
            text1 = tr("Luff round");
            text2 = QString::number(saildef.luffR);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Gaff round");
            text2 = QString::number(saildef.gaffR);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Leech round");
            text2 = QString::number(saildef.leechR);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Foot round");
            text2 = QString::number(saildef.footR);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Luff round position");
            text2 = QString::number(saildef.footRP);
            text3 = "%";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Gaff round position");
            text2 = QString::number(saildef.gaffRP);
            text3 = "%";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Leech round position");
            text2 = QString::number(saildef.leechRP);
            text3 = "%";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Foot round position");
            text2 = QString::number(saildef.footRP);
            text3 = "%";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            // sail setting
            painter.drawText(int(x1/2), y, tr("Sail settings"));
            y = y + h;

            // twist
            text1 = tr("Twist angle");
            text2 = QString::number(saildef.twistDeg);
            text3 = tr("deg");
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            // sheeting
            text1 = tr("Sheeting angle");
            text2 = QString::number(saildef.sheetDeg);
            text3 = tr("deg");
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            painter.drawText(int(x1/2), y, tr("Cloth seams and hems"));
            y = y + h;

            // cloth width, seam and hems width
            text1 = tr("Cloth width");
            text2 = QString::number(saildef.clothW);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y+h;

            text1 = tr("Seams width");
            text2 = QString::number(saildef.seamW);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y+h;

            text1 = tr("Leech hem width");
            text2 = QString::number(saildef.leechHemW);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y+h;

            text1 = tr("Other hem width");
            text2 = QString::number(saildef.hemsW);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y+h;

            painter.drawText(int(x1/2), y, tr("Sail mould"));
            y = y + h;
            x = int(scale*fontsz1);
            text2 = tr("Luff factor");
            text3 = tr("Depth");
            text4 = tr("Leech factor");
            painter.drawText(x2-2*x,y, text2);
            painter.drawText(x3-x,y, text3);
            painter.drawText(x4-2*x,y, text4);
            y = y + h;

            text1 = tr("Top profile");
            text2 = QString::number( saildef.mould.profile[2].getLuff() );
            text3 = QString::number( saildef.mould.profile[2].getDepth()*100 )+ "%";
            text4 = QString::number( saildef.mould.profile[2].getLeech() *50);
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            painter.drawText(x4,y, text4);
            y = y + h;

            text1 = tr("Mid profile at h = ") +QString::number( saildef.mould.vertpos )+"%";
            text2 = QString::number( saildef.mould.profile[1].getLuff() );
            text3 = QString::number( saildef.mould.profile[1].getDepth()*100 )+"%";
            text4 = QString::number( saildef.mould.profile[1].getLeech() *50);
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            painter.drawText(x4,y, text4);
            y = y + h;

            text1 = tr("Bottom profile");
            text2 = QString::number( saildef.mould.profile[0].getLuff() );
            text3 = QString::number( saildef.mould.profile[0].getDepth()*100 )+"%";
            text4 = QString::number( saildef.mould.profile[0].getLeech() *50);
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            painter.drawText(x4,y, text4);

            // now draw rectangle at end of text
            x = int(scale*fontsz1);
            y = y + h;
            w = x4 - x2 + 3*x;
            h = int(scale*2);
            painter.drawRect (x2-x,y, w,h);
            ///
        }
    }
    catch (CException e)
    {
        QMessageBox::information(this, tr("error"), tr("There was a data printing error"));
    }
}


/**
 * Print the current developed sail drawing.
 */
void CFormMain::slotPrintDev()
{
    try
    {  // try printing
        QPrinter myprinter;
        // set landscape printing
        myprinter.setOrientation(QPrinter::Landscape);
        myprinter.setFullPage(FALSE);

        bool ret = myprinter.setup(); // show printing dialog
        if (ret== true)
        {
            CSailPainter painter(&myprinter);

            // set the viewport
            CSailDisp disp;
            disp.setSail(flatsail);
            QRect vRect = painter.viewport();
            disp.setVRect(vRect.width(), vRect.height());
            disp.setZoom(0.9);

            // set coordinate system to match the logical viewport
            painter.setWindow(disp.getLRect());

            QPaintDeviceMetrics pdm(&myprinter);
            //double aspect = (double)pdm.widthMM() / (double)pdm.heightMM();
            unsigned int height = pdm.heightMM();
            real scale = height/50;

            QString text1, text2;
            unsigned int fontsz1 = int(11*scale), x=1, y=1;
            painter.setFont(QFont ("times", fontsz1));

            /*x = 0;
            y = 0;
            text1 = saildef.sailID;
            if (text1.length()<1)   text1="*";
            text1 =  text1;
            */

            // print the panels out one by one
            for (unsigned int i = 0; i < flatsail.panel.size(); i++)
            {
                painter.draw(disp.getSail().panel[i]);
                text2 = "p:"+ QString::number( i );
                painter.drawText(x,y, text2);

                if (i < (disp.getSail().panel.size()-1))
                {
                    myprinter.newPage();
                }
            }
            ///
        }
    }
    catch (CException e)
    {
        QMessageBox::information(this, tr("error"), tr("There was a development printing error"));
    }
}


/**
 * Print the current sail drawing.
 */
void CFormMain::slotPrintDwg()
{  // try printing
    try
    {
        QPrinter myprinter;
        // set landscape printing
        myprinter.setOrientation(QPrinter::Portrait);
        myprinter.setFullPage(FALSE);

        bool ret = myprinter.setup(); // show printing dialog
        if (ret== true)
        {
            CSailPainter painter(&myprinter);

            /*QString text1, text2;
            unsigned int fontsz1=1;
            int x=1, y=1;
            */
            // set the viewport
            CSailDisp disp;
            disp.setSail(sail);
            QRect vRect = painter.viewport();
            disp.setVRect(vRect.width(), vRect.height());
            disp.setZoom(0.90);

            // set coordinate system to match the logical viewport
            painter.setWindow(disp.getLRect());

            /*QPaintDeviceMetrics pdm(&myprinter);
            unsigned int width = pdm.widthMM();
            unsigned int height = pdm.heightMM();
            real scale = height/25;

            fontsz1 = 11 * scale;
            x=0;
            y=int(height/2 * scale);
            painter.setFont(QFont ("times", fontsz1));
            painter.drawText(x, y, "THIS");
            */

            // print the panels out one by one
            for (unsigned int i = 0; i < flatsail.panel.size(); i++)
            {
                painter.draw(disp.getSail().panel[i]);
            }
            ///
        }
    }
    catch (CException e)
    {
        QMessageBox::information(this, tr("error"), tr("There was a drawing printing error"));
    }
}


/**
 * Display the rig dialog
 */
void CFormMain::slotRig()
{
    CFormRig(app, this, true).exec();
}


/**
 * Saves the current sail definition to an XML file.
 */
void CFormMain::slotSave()
{
    if ( filename.isEmpty() )
    {
        slotSaveAs();
        return;
    }

    // try writing to file, catch exception
    try
    {
        CSailDefXmlWriter(saildef,"saildef").write(filename);
        fileAccess(tr("wrote '%1'").arg(filename),filename);
    }
    catch (CException e)
    {
        QMessageBox::information(this, tr("error"), tr("There was an error writing to the selected file"));
    }
}


/**
 * Opens a dialog to select the XML to which the sail definition should be saved.
 */
void CFormMain::slotSaveAs()
{
    QString newname = CSailDefXmlWriter(saildef,"saildef").writeDialog(filename);

    if (!newname.isNull())
    {
        filename = newname;
        fileAccess(tr("wrote '%1'").arg(filename), filename);
    }
}

