/*
 * Copyright (C) 1993-2004 Robert & Jeremy Lain
 * See AUTHORS file for a full list of contributors.
 * 
 * $Id: formrig.cpp,v 1.11 2004/10/26 19:30:37 jeremy_laine Exp $
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "formrig.h"
#include "sailcutqt.h"
#include "sailviewer-panel.h"
#include "rigdef-panel.h"
#include "sailreader-xml.h"
#include "sailwriter-xml.h"

#include <qlayout.h>
#include <qmenubar.h>
#include <qfileinfo.h>

/**
 * The constructor.
 *
 * @param pref The user's preferences
 */
CFormRig::CFormRig(CSailApp *myApp, QWidget *parent, bool modal)
        : QDialog(parent, 0, modal), prefs(&myApp->prefs)
{
    viewer = new CSailViewerPanel(this, WIREFRAME, true, false);
    defpanel = new CRigDefPanel(this);

    QGridLayout *layout = new QGridLayout( this, 1, 1, 11, 6 );
    layout->addItem( new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Fixed ), 0, 0 );
    layout->addWidget(viewer, 1, 0);
    layout->setRowStretch(1, 2);
    layout->addWidget(defpanel, 2, 0);
    layout->setRowStretch(2, 1);

    /* create blank menu bar */
    menubar = new QMenuBar(this);
    menuRecent = new QPopupMenu(this);
    makeMenuMru();

    languageChange();
    resize( QSize(prefs->rigWindowWidth, prefs->rigWindowHeight).expandedTo(minimumSizeHint()) );

    /* connect signals to slots */
    connect(defpanel, SIGNAL(signalUpdate(const CRigDef& )), this, SLOT(slotUpdate(const CRigDef& )));
}


/**
 * This event is received when the user closes the dialog.
 */
void CFormRig::closeEvent(QCloseEvent *e)
{
    prefs->rigWindowHeight = height();
    prefs->rigWindowWidth = width();
    QDialog::closeEvent(e);
}


/**
 * Sets the strings of the subwidgets using the current
 * language.
 */
void CFormRig::languageChange()
{
    setCaption( "Sailcut CAD - " + tr("Rig") );

    /* File menu */
    QPopupMenu * menuFile = new QPopupMenu(this);
    menuFile->insertItem( tr("&New"), this, SLOT( slotNew() ) );

    QPopupMenu *menuAdd = new QPopupMenu(this);
    menuAdd->insertItem( tr("sail &definition"), this, SLOT( slotAddSailDef() ) );
    menuAdd->insertItem( tr("3D &sail"), this, SLOT( slotAddSail() ) );
    menuFile->insertItem( tr("&Add sail"), menuAdd );

    menuFile->insertItem( tr("&Open"), this, SLOT( slotOpen() ) );
    menuFile->insertItem( tr("Open &recent"), menuRecent );
    menuFile->insertSeparator();
    menuFile->insertItem( tr("&Save"), this, SLOT( slotSave() ) );
    menuFile->insertItem( tr("Save &As"), this, SLOT( slotSaveAs() ) );
    menuFile->insertSeparator();
    menuFile->insertItem( tr("&Close"), this, SLOT( accept() ) );

    menubar->clear();
    menubar->insertItem( tr("&File"), menuFile );

    defpanel->languageChange();
    viewer->languageChange();

}


/**
 * Creates the "Open Recent" menu from the Most Recently Used files list.
 */
void CFormRig::makeMenuMru()
{
    menuRecent->clear();
    for ( unsigned int i = 0; i < prefs->mruRigdef.size(); i++)
        menuRecent->insertItem( prefs->mruRigdef[i].data(), this, SLOT ( slotOpenRecent(int) ), 0, i );
}


/**
 * We got a new sail definition, update widgets.
 *
 * @param newdef The new rig definition
 */
void CFormRig::setRigDef(const CRigDef &newdef)
{
    rigdef = newdef;
    viewer->setSail(rigdef.makeViewSail());
    defpanel->setRigDef(rigdef);
}


/**
 * The file menu's "Add->3d sail item" was clicked.
 */
void CFormRig::slotAddSail()
{
    CRigSail rigsail;
    QString newname = CSailXmlReader("3Dsail").readDialog((CSail&)rigsail,"");

    if (!newname.isNull())
    {
        rigsail.type = SAIL3D;
        rigsail.filename = newname.latin1();
        rigsail.name = QFileInfo(newname).fileName().latin1();
        rigdef.rigsail.push_back(rigsail);
        setRigDef(rigdef);
    }

}


/**
 * The file menu's "Add->Sail definition" item was clicked.
 */
void CFormRig::slotAddSailDef()
{
    CSailDef saildef;
    QString newname = CSailDefXmlReader("saildef").readDialog(saildef,"");

    if (!newname.isNull())
    {
        CRigSail rigsail;
        (CSail&)rigsail = CSailWorker(saildef).makeSail();
        rigsail.type = SAILDEF;
        rigsail.filename = newname.latin1();
        rigsail.name = QFileInfo(newname).fileName().latin1();
        rigdef.rigsail.push_back(rigsail);
        setRigDef(rigdef);
    }

}


/**
 * The file menu's New item was clicked
 */
void CFormRig::slotNew()
{
    setRigDef(CRigDef());
}


/**
 * The file menu's Open item was clicked.
 */
void CFormRig::slotOpen()
{
    CRigDef newdef;
    QString newname = CRigDefXmlReader("rigdef").readDialog(newdef,filename);

    if (!newname.isNull())
    {
        filename = newname;
        setRigDef(newdef);
        prefs->mruRigdef.touchEntry(filename);
        makeMenuMru();
    }
}


/**
 * Opens a recently used rig definition.
 *
 * @param index The index of the file that was clicked
 */
void CFormRig::slotOpenRecent(int index)
{
    filename = prefs->mruRigdef[index].data();
    try
    {
        setRigDef(CRigDefXmlReader("rigdef").read(filename));
        prefs->mruRigdef.touchEntry(filename);
    }
    catch (CException e)
    {
        prefs->mruRigdef.removeEntry(filename);
    }
    makeMenuMru();

}


/**
 * The file menu's Save item was clicked.
 */
void CFormRig::slotSave()
{
    if ( filename.isEmpty() )
    {
        slotSaveAs();
        return;
    }

    // try writing to file, catch exception
    try
    {
        CRigDefXmlWriter(rigdef,"rigdef").write(filename);
        prefs->mruRigdef.touchEntry(filename);
        makeMenuMru();
    }
    catch (CException e)
    {
        QMessageBox::information(this, tr("error"), tr("There was an error writing to the selected file"));
    }
}


/**
 * The file menu's Save As item was clicked.
 */
void CFormRig::slotSaveAs()
{
    QString newname = CRigDefXmlWriter(rigdef,"rigdef").writeDialog(filename);

    if (!newname.isNull())
    {
        filename = newname;
        prefs->mruRigdef.touchEntry(filename);
        makeMenuMru();
    }
}


/**
 * The rig definition was modified by the user.
 *
 * @param newdef The new rig definition
 */
void CFormRig::slotUpdate(const CRigDef& newdef)
{
    rigdef = newdef;
    viewer->setSail(rigdef.makeViewSail());
}
